# -*- coding: utf-8 -*-
import os
import random
import signal
import uuid
import time
import threading

from functools import wraps

from apscheduler.schedulers.blocking import BlockingScheduler
from apscheduler.executors.pool import ThreadPoolExecutor

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log, default_log

from intranet.yandex_directory.src.yandex_directory.common.backpressure import is_need_to_close_service
from intranet.yandex_directory.src.yandex_directory.common.db import (
    lock,
    AlreadyLockedError,
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import with_app_context, utcnow
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.utils.ycrid_manager import ycrid_manager
from intranet.yandex_directory.src.yandex_directory.core.models import ScheduledTasksResultModel

executors = {
    'default': ThreadPoolExecutor(20),
}

job_defaults = {
    'max_instances': 1,

    # https://apscheduler.readthedocs.io/en/latest/userguide.html?highlight=coalesce#missed-job-executions-and-coalescing
    # if coalescing is enabled for the job and the scheduler sees one or more queued executions
    # for the job, it will only trigger it once
    'coalesce': True,
}

scheduler = BlockingScheduler(executors=executors, job_defaults=job_defaults)
short_scheduler = BlockingScheduler(executors=executors, job_defaults=job_defaults)
_database_lock_names = set()

await_kill = False
tasks_in_work = 0


def queue_size_logger(logger_name):
    while True:
        try:
            for queue, pool in list(executors.items()):
                queue_size = pool._pool._work_queue.qsize()
                with log.name_and_fields(logger_name, queue=queue, size=queue_size):
                    log.info('Scheduler queue size')
        except:
            log.trace().error('Unhandled exception')

        time.sleep(60)


def run_harakiri_thread(interval_min, interval_max, timeout):
    if interval_min and interval_max:
        # Иногда воркер почему-то начисто зависает, причину я пока не нашёл
        def make_harakiri():
            global await_kill, tasks_in_work
            sleep_time = random.randint(interval_min, interval_max)
            log.warning('Sleep %d seconds before harakiri' % sleep_time)
            time.sleep(sleep_time)

            log.warning('Try make a harakiri...')

            await_kill = True
            start_time = utcnow()
            while tasks_in_work > 0 and (utcnow() - start_time).total_seconds() < timeout:
                log.warning('Sleep, tasks in work %s' % tasks_in_work)
                time.sleep(1)

            log.warning('Make a harakiri...')
            # sys.exit(0) почему-то не работает, приходится эмулировать kill -9
            os.kill(os.getpid(), signal.SIGKILL)

        harakiri_thread = threading.Thread(target=make_harakiri)
        harakiri_thread.start()


def run_scheduler():
    run_harakiri_thread(
        app.config['SCHEDULER_HARAKIRI_INTERVAL_MIN'],
        app.config['SCHEDULER_HARAKIRI_INTERVAL_MAX'],
        app.config['SCHEDULER_HARAKIRI_INTERVAL_TIMEOUT']
    )
    # Стартанём тред в котором будем логгировать размер очереди
    queue_size_logger_thread = threading.Thread(
        target=queue_size_logger,
        name='SchedulerQueueSizeLogger',
        kwargs={'logger_name': 'scheduler'},
    )
    queue_size_logger_thread.daemon = True
    queue_size_logger_thread.start()

    job_names = [job[0].name for job in scheduler._pending_jobs]
    with log.fields(jobs=job_names):
        log.info('Starting scheduled jobs')
        scheduler.start()


def run_short_scheduler():
    run_harakiri_thread(
        app.config['SHORT_SCHEDULER_HARAKIRI_INTERVAL_MIN'],
        app.config['SHORT_SCHEDULER_HARAKIRI_INTERVAL_MAX'],
        app.config['SHORT_SCHEDULER_HARAKIRI_INTERVAL_TIMEOUT']
    )

    # Стартанём тред в котором будем логгировать размер очереди
    queue_size_logger_thread = threading.Thread(
        target=queue_size_logger,
        name='SchedulerQueueSizeLogger',
        kwargs={'logger_name': 'short-scheduler'},
    )
    queue_size_logger_thread.daemon = True
    queue_size_logger_thread.start()

    job_names = [job[0].name for job in short_scheduler._pending_jobs]
    with log.fields(jobs=job_names):
        log.info('Starting short scheduled jobs')
        short_scheduler.start()


def scheduled_job(trigger='interval', **decorator_kwargs):
    return _scheduled_job(
        use_lock=True,
        scheduler_var=scheduler,
        trigger=trigger,
        **decorator_kwargs
    )


def scheduled_job_without_lock(trigger='interval', **decorator_kwargs):
    return _scheduled_job(
        use_lock=False,
        scheduler_var=scheduler,
        trigger=trigger,
        **decorator_kwargs
    )


def short_scheduled_job(trigger='interval', **decorator_kwargs):
    return _scheduled_job(
        use_lock=True,
        scheduler_var=short_scheduler,
        trigger=trigger,
        **decorator_kwargs
    )

def short_scheduled_job_without_lock(trigger='interval', **decorator_kwargs):
    return _scheduled_job(
        use_lock=False,
        scheduler_var=short_scheduler,
        trigger=trigger,
        **decorator_kwargs
    )


def _scheduled_job(use_lock=False, scheduler_var=None, trigger='interval', **decorator_kwargs):
    def decorator(function):
        lock_name = function.__name__ + '_' + app.config['ENVIRONMENT']

        with log.name_and_fields('scheduled-command.' + function.__name__,
                                 use_lock=use_lock,
                                 trigger=trigger,
                                 kwargs=decorator_kwargs):
            # защита от определения нескольких задач,
            # берущих лок с одним и тем же именем
            if lock_name in _database_lock_names:
                log.error('Scheduled job was not registered')
                raise ValueError('Task with same lock_name (%s) is already defined!' % lock_name)
            elif lock_name:
                _database_lock_names.add(lock_name)

            log.info('Scheduled job was registered')

        @with_app_context
        def wrapper(*args, **kwargs):
            global tasks_in_work, await_kill
            ycrid_manager.generate()
            start_at = utcnow()

            fields_to_log = {
                'lock_name': lock_name,
                'trigger': trigger,
                'args': args,
                'kwargs': kwargs,
                'task_id': uuid.uuid4().hex,
            }
            with log.name_and_fields('scheduled-command.' + function.__name__,
                                     **fields_to_log):

                if is_need_to_close_service():
                    log.warning('Service unavailable, skipping scheduled job')
                    return

                if os.path.exists('/disable-background-jobs') or \
                   os.environ.get('DISABLE_BACKGROUND_JOBS', '0') not in ('0', 'no'):
                    log.debug('Background jobs were disabled')
                    return

                if await_kill:
                    default_log.info('Await kill scheduler, new jobs not run')
                    return

                default_log.warning('Increment tasks_in_work')
                tasks_in_work += 1

                log.info('Running scheduled command')

                if use_lock:
                    # пробуем открыть транзакцию и взять лок
                    # если получилось, запускаем функцию
                    with get_meta_connection(for_write=True) as meta_connection:
                        try:
                            log.debug('Getting lock')
                            with lock(meta_connection, lock_name):
                                log.debug('Lock was acquired')
                                try:
                                    result = function(*args, **kwargs)
                                    ScheduledTasksResultModel(meta_connection).create(
                                        name=function.__name__,
                                        start_at=start_at,
                                        finish_at=utcnow(),
                                        success=True,
                                    )
                                    ycrid_manager.reset()
                                    default_log.warning('Decrement tasks_in_work 1')
                                    tasks_in_work -= 1
                                    return result
                                except:
                                    log.trace().error('Unhandled exception in task')
                                    ScheduledTasksResultModel(meta_connection).create(
                                        name=function.__name__,
                                        start_at=start_at,
                                        finish_at=utcnow(),
                                        success=False,
                                    )

                        except AlreadyLockedError:
                            log.debug('Lock is already acquired by somebody else')
                else:
                    try:
                        result = function(*args, **kwargs)
                        ycrid_manager.reset()
                        default_log.warning('Decrement tasks_in_work 2')
                        tasks_in_work -= 1
                        return result
                    except:
                        log.trace().error('Unhandled exception in task')

                log.info('Scheduled command is done')
                ycrid_manager.reset()
                default_log.warning('Decrement tasks_in_work 3')
                tasks_in_work -= 1

        scheduler_var.add_job(wrapper, trigger, name=lock_name, **decorator_kwargs)

        return wrapper

    return decorator
