# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory.common.utils import force_text
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    UserMetaModel,
)
from intranet.yandex_directory.src.yandex_directory.core.templates import get_translations
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    lang_for_notification,
    only_attrs,
)
from intranet.yandex_directory.src.yandex_directory.core.sms.tasks import SendSmsTask
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    UserNotFoundError,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask


def _get_sms_template(lang, tanker_key):
    """
    Шаблон для СМС на выбранном языке
    :param lang: язык
    :param tanker_key: ключ перевода
    :return: str
    """
    translations = get_translations(lang, 'sms')
    tracker_enabled_message = translations.gettext(tanker_key)
    return force_text(tracker_enabled_message)


def send_sms_to_admins(meta_connection, main_connection, org_id, tanker_key, **kwargs):
    """
    Отправка СМС всем администраторам организации
    Переводы тут: https://tanker.yandex-team.ru/?project=directory&branch=master&keyset=Sms
    :param meta_connection: соединение к meta базе
    :param main_connection: соединение к main базе
    :param org_id: ид организации
    :param tanker_key: ключ перевод для СМС
    :param kwargs: параметры для шаблона СМС
    """
    with log.fields(org_id=org_id, tanker_key=tanker_key, kwargs=kwargs):
        log.info('Sending sms to admins')
        uids = list()

        # внутренние админы
        inner_admins = UserModel(main_connection).get_organization_admins(org_id)
        uids += only_attrs(inner_admins, 'id')

        # внешние админы
        outer_admins = UserMetaModel(meta_connection).find(filter_data={
            'org_id': org_id,
            'is_outer': True,
        })
        uids += only_attrs(outer_admins, 'id')
        for uid in uids:
            with log.fields(uid=uid):
                try:
                    lang = lang_for_notification(meta_connection, main_connection, uid, org_id)
                    message_template = _get_sms_template(lang, tanker_key)

                    text = message_template.format(**kwargs)
                    SendSmsTask(main_connection).delay(uid=uid, text=text, org_id=org_id)
                except UserNotFoundError:
                    log.trace().error('Unable to send sms, admin was not found')
                except DuplicatedTask:
                    log.warning('Unable to send sms, task already exists')
