# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow


class ResultException(Exception):
    pass


class UnknownTask(ResultException):
    """
    Нет задачи с таким Id
    """
    pass


class TaskFailed(ResultException):
    """
    Задача или откат завершилась с ошибкой
    """
    pass


class TaskRolledBack(ResultException):
    """
    Задача штатно откатилась
    """
    pass


class TaskInProgress(ResultException):
    """
    Задача в процессе выполнения
    """
    pass


class TaskException(Exception):
    pass


class DuplicatedTask(TaskException):
    """
    Лишь одна задача такого типа и с такими параметрами может выполняться единовременно
    """
    def __init__(self, task_id):
        super(DuplicatedTask, self).__init__()
        self.existing_task_id = task_id


class Defer(TaskException):
    """
    Специальное исключение для сигнала о том, что задача должна быть повторена позже.

    """

    when = None  # дата время или через сколько секунд нужно запустить задачу

    def __init__(self, retry_at=None, countdown=None, suspended=False):
        """
        Откладываем задачу
        :param retry_at: когда повторить задачу
        :type retry_at: datetime
        :param countdown: через сколько секунд повторить задачу
        :param suspended: переводить ли задачу в suspended статус
        :type countdown: int
        """
        assert retry_at or countdown, 'Set retry_at or countdown'
        self.suspended = suspended

        if retry_at:
            assert isinstance(retry_at, datetime), 'Invalid type of retry_at'
            self.when = retry_at
        elif countdown:
            assert isinstance(countdown, (int, float)), 'Invalid type of countdown'
            self.when = utcnow() + timedelta(seconds=countdown)
        super(Defer, self).__init__(self)


class DependencyCreationError(TaskException):
    """
    Ошибка при добавлении зависимости
    """
    pass


class Suspend(TaskException):
    """
    Специальное исключение для сигнала о том, что задача ожидает выполнения подтасков
    """
    pass
