# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
    get_shard,
    get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.departments import (
    is_department_not_exists_in_dir,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import is_user_not_exists_in_dir
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DepartmentModel,
    UserModel,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_department_modify,
    action_user_modify,
)


class MoveUserToDepartmentTask(Task):

    singleton = True

    def do(self, org_id, user_id, parent_dep_id, author_id):
        """
        Перемещаем одного пользователя в другой департамент
        (меняем родительский департамент)
        :param org_id: id организации
        :param user_id: id пользователя, который перемещаем
        :param parent_dep_id: id родительского департамента, куда перемещаем
        """

        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id)

        with get_main_connection(
                shard=shard,
                for_write=True,
        ) as main_connection:

            user_model = UserModel(main_connection)
            with log.fields(shard=shard):
                if is_user_not_exists_in_dir(
                    main_connection,
                    org_id,
                    user_id,
                ):
                    return

                if is_department_not_exists_in_dir(
                    main_connection,
                    org_id,
                    parent_dep_id,
                ):
                    return

                old_user = user_model.get(
                    user_id=user_id,
                    org_id=org_id,
                    fields=['**'],
                )

                user_model.update_one(
                    filter_data={
                        'id': user_id,
                        'org_id': org_id,
                    },
                    update_data={
                        'department_id': parent_dep_id,
                    }
                )

                new_user = user_model.get(
                    user_id=user_id,
                    org_id=org_id,
                    fields=['**'],
                )

                action_user_modify(
                    main_connection,
                    org_id=org_id,
                    author_id=author_id,
                    object_value=new_user,
                    old_object=old_user,
                )


class MoveDepartmentToDepartmentTask(Task):

    singleton = True

    def do(self, org_id, dep_id, parent_dep_id, author_id):
        """
        Перемещаем один департамент в другой департамент
        (меняем родительский департамент)
        :param org_id: id организации
        :param dep_id: id департамента, который перемещаем
        :param parent_dep_id: id родительского департамента, куда перемещаем
        :param author_id: id юзера, от имени которого прооисходит перемещение
        """

        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id)

        with get_main_connection(
            shard=shard, for_write=True
        ) as main_connection:
            department_model = DepartmentModel(main_connection)

            with log.fields(shard=shard):
                if is_department_not_exists_in_dir(
                    main_connection,
                    org_id,
                    dep_id,
                ):
                    return

                if is_department_not_exists_in_dir(
                    main_connection,
                    org_id,
                    parent_dep_id,
                ):
                    return

                old_department = department_model.get(
                    department_id=dep_id,
                    org_id=org_id,
                    fields=[
                        '*',
                        'parent.*',
                        'parents.*',
                        'head.*',
                        'email',
                    ],
                )

                department_model.update_one(
                    id=dep_id,
                    org_id=org_id,
                    data={
                        'parent_id': parent_dep_id,
                    }
                )

                new_department = department_model.get(
                    department_id=dep_id,
                    org_id=org_id,
                    fields=[
                        '*',
                        'parent.*',
                        'parents.*',
                        'head.*',
                        'email',
                    ],
                )

                action_department_modify(
                    main_connection,
                    org_id=org_id,
                    author_id=author_id,
                    object_value=new_department,
                    old_object=old_department,
                )
