# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection
from intranet.yandex_directory.src.yandex_directory.common.models import types as OBJECT_TYPES
from intranet.yandex_directory.src.yandex_directory.connect_services.idm.base_service import WORKING_WITH_IDM_SYSTEMS
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DepartmentModel,
    ResourceModel,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ServiceNotFound
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    enable_service,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.resource.tasks import (
    CreateExistingAccountTask,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_resource_add,
)
from intranet.yandex_directory.src.yandex_directory.connect_services.idm import get_service
from intranet.yandex_directory.src.yandex_directory.core.views.resources import convert_relations
from .responsible import SetResponsibleTask


class CreateResourceTask(Task):

    singleton = True

    def do(self, org_id, service_slug, external_id, relations=None, author_id=None):
        """
        Создаём в организации ресурс и опционально - связываем его с пользователями, отделами или командами.
        """
        self.synchronous_call(self.main_connection, org_id, service_slug, external_id, relations, author_id)

    @staticmethod
    def synchronous_call(main_connection, org_id, service_slug, external_id, relations, author_id):
        log.debug('Creating resource')

        if relations:
            relations = convert_relations(relations)

        resource = ResourceModel(main_connection).create(
            external_id=external_id,
            org_id=org_id,
            service=service_slug,
            relations=relations,
        )

        action_resource_add(
            main_connection,
            org_id=org_id,
            author_id=author_id,
            object_value=resource,
        )


class AuthorShouldBeInRelationsError(Exception):
    """
    Автор должен получить права на ресурс, и это сервис должен явно
    передать его в списке relations
    """


class BindTask(Task):
    """Включает сервис, если нужно, создает отдел outstaff, создаёт ресурс, привязывает к нему пользователей.
       Пользователи создаются в подзадачах, а этот таск их создаёт, а потом дожидается.
    """
    singleton = True

    def do(self, org_id, user_id, service_slug, resource_id, language, relations):
        """
        :param relations: список
        """
        # Сначала нужно проверить, включён ли сервис и если нет, то включить
        with get_meta_connection(for_write=True) as meta_connection:
            self.synchronous_call(meta_connection, self.main_connection, org_id, user_id, service_slug, resource_id, language, relations)

    @staticmethod
    def synchronous_call(meta_connection, main_connection, org_id, user_id, service_slug, resource_id, language, relations):
        if user_id not in [r['object_id'] for r in relations if r['object_type'] == OBJECT_TYPES.TYPE_USER]:
            raise AuthorShouldBeInRelationsError

        service = enable_service(
            meta_connection,
            main_connection,
            org_id,
            service_slug,
        )

        outstaff_department = DepartmentModel(main_connection).get_or_create_outstaff(
            org_id=org_id,
            language=language,
        )
        # Для каждой связи нужно завести сотрудников
        create_user_tasks = {}

        for relation in relations:
            object_type = relation['object_type']
            object_id = relation['object_id']
            if object_type == OBJECT_TYPES.TYPE_USER:
                task = CreateExistingAccountTask(main_connection)
                task.synchronous_call(
                    meta_connection, main_connection,
                    org_id=org_id,
                    uid=object_id,
                    department_id=outstaff_department['id']
                )
                create_user_tasks[object_id] = task

        SetResponsibleTask(
            main_connection,
            depends_on=[]
        ).synchronous_call(
            meta_connection, main_connection,
            service_id=service['id'],
            org_id=org_id,
            responsible_id=user_id,
        )

        try:
            service_client = get_service(service_slug)
            service_client.add_responsible_role(
                main_connection=main_connection,
                relations=relations,
                org_id=org_id,
                service_id=service['id'],
            )

            if service_slug in WORKING_WITH_IDM_SYSTEMS:
                roles = [
                    {
                        'uid': relation['object_id'],
                        'path': relation['name'],
                        'resource_id': resource_id,
                    }
                    for relation in relations
                ]
                roles.append({
                    'uid': org_id,
                    'path': '/organization/associated/',
                    'user_type': 'organization',
                    'resource_id': resource_id,
                })
                service_client.request_roles(org_id, user_id, *roles)
                relations = []

        except ServiceNotFound:
            pass

        # Дальше нужно создать таск про то чтобы ресурс завести
        CreateResourceTask(
            main_connection,
            depends_on=[],
        ).synchronous_call(
            main_connection,
            org_id=org_id,
            service_slug=service_slug,
            external_id=resource_id,
            relations=relations,
            author_id=user_id,
        )
