# -*- coding: utf-8 -*-

import time
import collections
from intranet.yandex_directory.src.yandex_directory import app, gendarme, fouras
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
    get_main_connection,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    find_domid,
    to_punycode,
    to_lowercase,
)

from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_organization_admin_uid,
    is_cloud_uid,
    get_user_info_from_blackbox,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    sync_domain_state,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DomainModel,
    WebmasterDomainLogModel,
    UserModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.user import get_external_org_ids
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.actions import action_user_modify


class TestTask(Task):
    singleton = False

    def do(self, sleep=None):
        log.debug('Running test task')
        if sleep:
            with log.fields(interval=sleep):
                log.debug('Sleeping in test task')
            time.sleep(sleep)
        log.debug('Done')


class SyncSingleDomainTask(Task):
    singleton = True
    tries = 1000
    tries_delay = 30

    """
    Синхронизируем домен с вебмастером.
    Фактически это таск создан для ретрая механизма автоорыва домена.
    """
    def do(self, org_id, domain_name):
        log.debug('Syncing domain')
        admin_id = get_organization_admin_uid(self.main_connection, org_id)
        domain = DomainModel(self.main_connection).get(domain_name=domain_name, org_id=org_id)
        # возможно домена уже нет в организации
        if not domain:
            return
        with get_meta_connection(for_write=True) as meta_connection:
            domain_was_verified = sync_domain_state(
                meta_connection=meta_connection,
                main_connection=self.main_connection,
                org_id=org_id,
                admin_id=admin_id,
                domain=domain
            )

            if not domain_was_verified:
                # проверим последний способ подтверждения, если это pdd_emu или None (нет записи в логах),
                # то продолжим проверку
                last_verification_type = WebmasterDomainLogModel(self.main_connection).get_last_verification_type(
                    org_id,
                    domain_name,
                    admin_id,
                )
                if last_verification_type == 'PDD_EMU':
                    # Если таску больше 7 дней, то просто завершим его.
                    # Возможно пользователь передумал подтверждать домен.
                    if self.get_age() > 7 * 24 * 60 * 60:
                        log.debug('Task expired')
                        return

                    # В течении суток будем перепроверять домен с увеличивающимся интервалом,
                    # а потом раз в час.
                    self.exponential_defer(
                        min_interval=5 * 60,
                        max_interval=60 * 60,
                        const_time=24 * 60 * 60,
                    )
        log.debug('Done')


class UpdateEmailFieldsTask(Task):
    def do(self, master_domain, org_id):
        """
        Обновляем все поля email-ов для контейнеров заданной организации
        """
        log.info('Updating email fields for domain %s', master_domain)

        from intranet.yandex_directory.src.yandex_directory.core.models import DepartmentModel, UserModel, GroupModel
        # собираемся делать запрос вида:
        # UPDATE departments SET email=CONCAT(label,'@demotest05.com') WHERE org_id=6

        for model in [DepartmentModel, UserModel, GroupModel]:
            model(self.main_connection).replace_email_field(
                org_id,
                master_domain,
            )

        log.info('Done updating email fields for domain %s', master_domain)


class SyncExternalIDS(Task):
    """Синхронизирует список external_organization_ids в Паспорте."""
    singleton = True
    default_priority = 1100
    lock_ttl = 600  # даем 10 минут на выполнение таска

    def do(self, org_id, user_id, mix_org_id_in=False):
        # org_id тут нужен, во-первых, для того, чтобы таск
        # был привязан только к той организации, в которую добавился или удалился
        # сотрудник, и этот таск было бы видно в списке тасков организации,
        # а во-вторых, для того, чтобы иметь возможность подмешать его

        if is_cloud_uid(user_id) or user_id in (780889736, 780891246, 780890239):
            log.info('Skipping SyncExternalIDS task for cloud uid {}'.format(user_id))
            return
        with get_meta_connection() as meta_connection:
            org_ids = get_external_org_ids(meta_connection, user_id)
            if mix_org_id_in and org_id not in org_ids:
                org_ids.append(org_id)
            app.passport.set_organization_ids(user_id, org_ids)


class SetOrganizationNameInPassportTask(Task):
    singleton = True
    tries = 1000
    tries_delay = 60

    def do(self, org_id):
        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id=org_id)
            if shard:
                with get_main_connection(shard) as main_connection:
                    # Возьмём мастер-домен
                    domain = DomainModel(main_connection) \
                             .filter(org_id=org_id, master=True) \
                             .fields('name') \
                             .one()
                    # На всякий случай защитимся тут от того, что его удалили
                    # с тех пор, как завелась задача
                    if domain:
                        with log.fields(domain=domain['name']):
                            log.info('Settings correct organization_name for domain in Passport')

                            admin_uid = get_organization_admin_uid(main_connection, org_id)
                            hosted_domains = app.blackbox_instance.hosted_domains(
                                domain_admin=admin_uid,
                            )
                            domid = find_domid(hosted_domains, domain['name'])
                            with log.fields(domain_id=domid):
                                organization_name = 'org_id:{0}'.format(org_id)
                                state = app.passport.set_organization_name(
                                    domid,
                                    organization_name,
                                )
                                if not state:
                                    log.trace().error('Failed to set organization name in passport')
                                    raise RuntimeError('Failed to set organization name in passport')


class CheckDomainInGendarmeAndGenDkimTask(Task):
    def do(self, org_id, domain):
        domain = to_punycode(domain)

        with log.fields(org_id=org_id):
            gendarme.recheck(domain)
            fouras.get_or_gen_domain_key(domain)


class UnblockDomainsTask(Task):
    """Разблокирует все домены организации после модифицирующей операции.
    """
    singleton = True
    tries = 1000
    tries_delay = 30

    def do(self, org_id):
        DomainModel(self.main_connection).unblock(org_id)


class SyncUserData(Task):
    """Обновляем данные пользователя из паспорта"""
    singleton = True
    org_id_is_required = False
    singleton_params = ['user_id']

    def do(self, user_id, **kwargs):

        (login, first, last, gender, birthday, email, cloud_uid) = get_user_info_from_blackbox(user_id)

        changed_data = {
            'nickname': to_lowercase(login),
            'name': {'first': first or login, 'last': last or ('' if is_cloud_uid(user_id) else login)},
            'email': to_lowercase(email),
            'gender': gender,
            'birthday': birthday,
        }
        orgs_by_shard = self._get_user_orgs(user_id=user_id)
        for shard, orgs in orgs_by_shard.items():
            with get_main_connection(for_write=True, shard=shard) as main_connection:
                UserModel(main_connection).update_one(
                    filter_data={'id': user_id},
                    update_data=changed_data,
                )
                for org_id in orgs:
                    user = UserModel(main_connection).get(user_id, org_id)
                    action_user_modify(
                        main_connection,
                        org_id=org_id,
                        author_id=user_id,
                        object_value=user,
                        old_object=user,
                    )

    def _get_user_orgs(self, user_id):
        with get_meta_connection() as meta_connection:
            query = """
            select o.shard, o.id
            from users u
            inner join organizations o on u.org_id = o.id
            where u.is_dismissed = false
            and u.id = %(uid)s
            and u.user_type = 'inner_user'
            """
            orgs_data = meta_connection.execute(query, uid=user_id).fetchall()
        orgs_by_shard = collections.defaultdict(set)
        for shard, org_id in orgs_data:
            orgs_by_shard[shard].add(org_id)
        return orgs_by_shard


class ResetDomainVerification(Task):
    singleton = True
    singleton_params = ['domain_name', 'new_owner_uid', 'old_owner_uid', 'org_id']

    def do(self, domain_name, new_owner_uid, old_owner_uid, org_id):
        from intranet.yandex_directory.src.yandex_directory import webmaster
        webmaster.reset(domain_name, old_owner_uid, new_owner_uid)
