from collections import defaultdict
from datetime import datetime
from datetime import timedelta

from intranet.yandex_directory.src import settings
from yql.api.v1.client import YqlClient

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import get_main_connection, get_meta_connection, \
    get_shard_numbers, get_shard
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import (
    send_first_tracker_disable_warning_emails,
    send_second_tracker_disable_warning_emails,
    send_third_tracker_disable_warning_emails,
)
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationMetaModel
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationServiceModel
from intranet.yandex_directory.src.yandex_directory.core.models import ServiceModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import disable_service, reason
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task
from intranet.yandex_directory.src.yandex_directory.core.utils import only_attrs


def _chunks(lst, chunk_size):
    for i in range(0, len(lst), chunk_size):
        yield lst[i:i + chunk_size]


def _process_candidates_for_sending_first_email(meta_connection):
    month_ago_timestamp = (datetime.today() - timedelta(days=30)).timestamp() * 1000

    yql_client = YqlClient(db='markov', token=app.config['YT_TOKEN'])

    query = yql_client.query(
        '''
        SELECT id
        from `{table}`
        where status = 'active'
        and lastActivityDate <= {month_ago_timestamp}'''.format(
            table=settings.TRACKER_YT_ORGANIZATIONS_TABLE,
            month_ago_timestamp=month_ago_timestamp,
        ),
        syntax_version=1
    )
    query.run()
    result = query.get_results()
    if not result.is_success:
        raise RuntimeError()

    data = []
    for table in result:
        table.fetch_full_data()
        data = table.rows
        break

    service = ServiceModel(meta_connection).get_by_slug('tracker')
    service_id = service['id']

    for data_chunk in _chunks(list(data), 500):
        org_ids = [row[0] for row in data_chunk]
        orgs_meta = OrganizationMetaModel(meta_connection).fields('id', 'shard').filter(id=org_ids).all()
        org_ids_by_shard = defaultdict(list)
        for org_meta in orgs_meta:
            shard = org_meta['shard']
            org_id = org_meta['id']
            org_ids_by_shard[shard].append(org_id)

        for shard in org_ids_by_shard:
            sql = '''select org_id
            from organization_services
            where service_id = %(service_id)s
            and enabled = true
            and last_mail_sent_for_disable is null
            and org_id in %(org_ids)s
            '''
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                org_ids_to_sent_email = only_attrs(
                    main_connection.execute(
                        sql,
                        service_id=service_id,
                        org_ids=tuple(org_ids_by_shard[shard])
                    ).fetchall(),
                    'org_id'
                )
                for org_id in org_ids_to_sent_email:
                    SendFirstTrackerDeactivationWarningEmailTask(main_connection).delay(org_id=org_id)


def _process_candidates_for_sending_second_and_third_email(meta_connection):
    service = ServiceModel(meta_connection).get_by_slug('tracker')
    service_id = service['id']
    yql_client = YqlClient(db='markov', token=app.config['YT_TOKEN'])
    now = datetime.now()
    _3_weeks_delta = timedelta(weeks=3)
    _1_week_delta = timedelta(weeks=1)

    for shard in get_shard_numbers():
        with get_main_connection(shard, for_write=True) as main_connection:
            sql_reset_counter = '''update organization_services
            set last_mail_sent_for_disable=null, mail_count=null
            where service_id = %(service_id)s
            and enabled = false
            and mail_count >= 1
            '''
            main_connection.execute(sql_reset_counter, service_id=service_id)

            sql_get_orgs = '''select org_id, mail_count, last_mail_sent_for_disable
            from organization_services
            where service_id = %(service_id)s
            and enabled = true
            and mail_count >= 1
            '''
            orgs = main_connection.execute(sql_get_orgs, service_id=service_id).fetchall()
            data_by_org = {
                org['org_id']: {
                    'mail_count': org['mail_count'],
                    'last_mail_sent_for_disable': org['last_mail_sent_for_disable'],
                }
                for org in orgs
            }

        for org_ids_chunk in _chunks(list(data_by_org.keys()), 500):
            org_ids_str = ','.join(
                map(str, org_ids_chunk)
            )
            query = yql_client.query(
                '''
                SELECT id, lastActivityDate
                from `{table}`
                where id in ({org_ids})
                '''.format(
                    table=settings.TRACKER_YT_ORGANIZATIONS_TABLE,
                    org_ids=org_ids_str,
                ),
                syntax_version=1
            )
            query.run()
            result = query.get_results()
            if not result.is_success:
                raise RuntimeError()
            data = []
            for table in result:
                table.fetch_full_data()
                data = table.rows
                break

            for row in data:
                org_id = row[0]
                if row[1] is None:
                    continue
                last_activity_date = datetime.utcfromtimestamp(row[1] / 1000)
                mail_count = data_by_org[org_id]['mail_count']
                last_mail_sent_for_disable = datetime.utcfromtimestamp(
                    data_by_org[org_id]['last_mail_sent_for_disable'].timestamp()
                )

                if last_activity_date > last_mail_sent_for_disable:
                    sql_reset_counter_in_org = '''
                        update organization_services
                        set last_mail_sent_for_disable=null, mail_count=null
                        where service_id = %(service_id)s
                        and org_id = %(org_id)s
                    '''
                    with get_main_connection(shard, for_write=True) as main_connection:
                        main_connection.execute(sql_reset_counter_in_org, service_id=service_id, org_id=org_id)
                    continue

                if mail_count == 1 and last_mail_sent_for_disable + _3_weeks_delta <= now:
                    with get_main_connection(shard, for_write=True) as main_connection:
                        SendSecondTrackerDeactivationWarningEmailTask(main_connection).delay(org_id=org_id)
                    continue

                if mail_count == 2 and last_mail_sent_for_disable + _1_week_delta <= now:
                    with get_main_connection(shard, for_write=True) as main_connection:
                        SendThirdEmailAndDisableTrackerTask(main_connection).delay(org_id=org_id)
                    continue


class CheckActivityInTrackerTask(Task):
    singleton = True
    org_id_is_required = False

    def do(self, **kwargs):
        with get_meta_connection() as meta_connection:
            _process_candidates_for_sending_first_email(meta_connection)
            _process_candidates_for_sending_second_and_third_email(meta_connection)


class SendFirstTrackerDeactivationWarningEmailTask(Task):
    singleton = True

    def do(self, org_id):
        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            org_service = OrganizationServiceModel(main_connection).get_by_slug(org_id, 'tracker',
                                                                                fields=['id', 'enabled'])
            if not org_service['enabled']:
                return

            send_first_tracker_disable_warning_emails(main_connection, org_id)

            OrganizationServiceModel(main_connection).update_one(
                id=org_service['id'],
                update_data={
                    'last_mail_sent_for_disable': datetime.now(),
                    'mail_count': 1,
                }
            )


class SendSecondTrackerDeactivationWarningEmailTask(Task):
    singleton = True

    def do(self, org_id):
        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            org_service = OrganizationServiceModel(main_connection).get_by_slug(org_id, 'tracker',
                                                                                fields=['id', 'enabled'])
            if not org_service['enabled']:
                return

            send_second_tracker_disable_warning_emails(main_connection, org_id)

            OrganizationServiceModel(main_connection).update_one(
                id=org_service['id'],
                update_data={
                    'last_mail_sent_for_disable': datetime.now(),
                    'mail_count': 2,
                }
            )


class SendThirdEmailAndDisableTrackerTask(Task):
    singleton = True

    def do(self, org_id):
        with get_meta_connection() as meta_connection:
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                org_service = OrganizationServiceModel(main_connection).get_by_slug(org_id, 'tracker',
                                                                                    fields=['id', 'enabled'])
                if not org_service['enabled']:
                    return

                send_third_tracker_disable_warning_emails(main_connection, org_id)

                OrganizationServiceModel(main_connection).update_one(
                    id=org_service['id'],
                    update_data={
                        'last_mail_sent_for_disable': datetime.now(),
                        'mail_count': 3,
                    }
                )

                disable_service(meta_connection, main_connection, org_id, 'tracker', reason.no_activity)
