# -*- coding: utf-8 -*-
import multiprocessing
import operator
import os
import random
import re
import string
from intranet.yandex_directory.src import blackbox_client
from intranet.yandex_directory.src import settings

from itertools import zip_longest
from collections import (
    defaultdict,
)
from frozendict import frozendict
from functools import (
    partial,
    wraps,
)
from itertools import (
    chain,
)
from threading import RLock

import requests
from cachetools import TTLCache, keys, cached
from concurrent.futures import (
    ThreadPoolExecutor,
    wait,
)
from flask import g

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard,
    get_meta_connection, lock, wait_lock,
)

from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    ConstraintValidationError,
    UserNotFoundError,
    ServiceNotFound,
    UserOrganizationsLimit,
    MasterDomainNotFound,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_USER,
    TYPE_GROUP,
    TYPE_DEPARTMENT,
    ROOT_DEPARTMENT_ID,
)
from intranet.yandex_directory.src.yandex_directory.core.features import enable_features
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    to_punycode,
    NotGiven,
    ignore_first_args,
    try_format_date,
    try_format_datetime,
    json_error,
    get_localhost_ip_address,
    force_text,
    to_lowercase,
    parse_birth_date,
    get_user_data_from_blackbox_by_uid,
    split_by_comma,
    url_join,
    get_user_data_from_blackbox_by_login,
    json_error_not_found,
    format_datetime,
    Ignore,
    check_label_or_nickname_or_alias_is_uniq_and_correct,
    get_user_data_from_blackbox_by_uids,
    get_user_id_from_passport_by_login,
    utcnow, get_domain_info_from_blackbox,
)

PRIVATE_USER_FIELDS = ['first_name', 'middle_name', 'last_name', 'position_plain', 'karma']

# Диапазоны паспортных uid-ов: https://wiki.yandex-team.ru/passport/uids/
# Диапазон обычных пользователей в паспорте: [0; 111*10**13]
RANGE_PASSPORT = (0, 111 * 10 ** 13)

RANGE_PDD = (113 * 10 ** 13, 114 * 10 ** 13)

# Диапазоны облачных uid-ов: https://wiki.yandex-team.ru/Connect/projects/cloud-integration/Jetap-1-Detali/#uidvoblake
RANGE_CLOUD = (
    9000000000000000,
    9001000000000000
)


def get_random_password(password_length=10):
    return ''.join(random.sample(string.ascii_letters + string.digits, password_length))


def only_fields(dictionary, *fields):
    """Возвращает новый словарик только с теми полями, что перечислены.
    """
    return dict((key, value)
                for key, value in list(dictionary.items())
                if key in fields)


def check_if_members(fields):
    """ Проверяем, есть ли поля, относящиесея к members.
    """
    for field in fields:
        if field.startswith('members.') or field == 'members':
            return True
    return False


def prepare_fields(query, **kwargs):
    """Возвращает список полей из строки запроса.
    """
    if kwargs.get('default_fields', False):
        default_fields = kwargs.get('default_fields')
    else:
        default_fields = ['id']
    fields = split_by_comma(query) or default_fields
    # Иногда для обработки members нужно указывать type.
    if kwargs.get('add_members_type', False):
        if 'members.*' not in fields and check_if_members(fields):
            fields.append('members.type')

    return fields


class SkipField(RuntimeError):
    """Это исключение выбрасывается функцией only_hierarchical_fields,
    если ей передан объект, и ни одно из его полей не указанов в fields, как
    поле, которое должно попасть в результирующий словарь.
    """
    pass


def only_hierarchical_fields(obj, fields):
    """Возвращает подмножество ключей со значениями, но в отличии от
    only_fields, нужные поля заданы в словаре, который в свою очередь
    может содержать другие словари. К примеру:

    >>> obj = {
            'foo': 'bar',
            'blah': {
                'id': 42,
                'name': 'Pupkin',
            },
            'minor': [{'id': 100500, 'nickname': 'art'}],
        }
    >>> only_hierarchical_fields(obj, {'foo': True})
    {'foo': 'bar'}
    >>> only_hierarchical_fields(obj, {'blah': {'id': True}})
    {'blah': {'id': 42}}
    >>> only_hierarchical_fields(obj, {'minor': {'nickname': True}})
    {'minor': [{'nickname': 'art'}]
    """
    # В fields может быть передан True, когда идёт рекурсивный вызов
    # и тогда надо просто отдать текущее значение объекта, без
    # фильтрации полей
    if fields is True:
        return obj

    if isinstance(obj, dict):
        # Здесь мы рекурсивно обрабатываем значение, чтобы если для него тоже
        # перечислены поля, то мы выдали только их
        result = {}
        for key, value in list(obj.items()):
            if key in fields and fields[key] is not False:
                try:
                    result[key] = only_hierarchical_fields(value, fields[key])
                except SkipField:
                    # Если поле надо пропустить, то просто не будем
                    # устанавливать его в результирующем словаре.
                    pass

        # Если словарь получился пустой, то значит все его поля объекта
        # отфильтровались и его надо пропустить
        if not result:
            raise SkipField()
        return result

    if isinstance(obj, (list, tuple)):
        return [
            only_hierarchical_fields(item, fields)
            for item in obj
        ]
    else:
        return obj


def fallback_fields(dictionary, *fields):
    """Возвращает новый словарь в котором для тех
    ключей, что перечислены в fields, выбрано первое
    существующее значение.

    При этом, каждое значение в fields должно быть tuple
    с как минимум 2 значениями.

    Например:
        fallback_fields(
            {'object': 'blah', 'object_id': 123, 'name': 'test'},
            ('object', 'object_id'))
    Вернет:
        {'object': 'blah', 'name': 'test'}
    Но:
        fallback_fields(
            {'object_id': 123, 'name': 'test'},
            ('object', 'object_id'))
    Вернет:
        {'object_id': 123, 'name': 'test'}
    """
    result = dictionary.copy()

    for row in fields:
        # сначала ищем первый ключ, для которого
        # в словаре есть значения
        if isinstance(row, str):
            row = (row,)

        while row:
            key = row[0]
            row = row[1:]

            if key in result:
                break

        # потом удаляем из словаря остальные, ключи
        for key in row:
            del result[key]

    return result


def identity(x):
    return x


def serialize(dictionary, *fields, **kwargs):
    """Больше, больше магии!
    """
    return dict((key, kwargs.get(key, identity)(value))
                for key, value in list(dictionary.items())
                if key in fields)


def except_fields(dictionary, *fields):
    """Возвращает новый словарик только со всеми полями,
    за исключением тех, что перечислены в *fields.
    """
    return dict((key, value)
                for key, value in list(dictionary.items())
                if key not in fields)


def except_items(iterable, *items):
    """Возвращает список элементов из iterable за исключением тех
    которые есть есть среди items:

    >>> except_items([1, 2, 3, 4], 1, 3)
    [2, 4]
    """
    return [item for item in iterable
            if item not in items]


def only_ids(items):
    """Возвращает список id, извлекая их из словариков
    в переданной последовательности.
    """
    return [item['id'] for item in items]


def strip_id(text):
    """Убирает из текста суффикс _id.
       Полезно при обработке списка полей, пришедшего в post или patch.
    """
    if text.endswith('_id'):
        return text[:-3]
    return text


def objects_map_by_id(objects, key='id', remove_key=False, transform_key=identity):
    """Возвращает словарь, в котором ключами являются 'id' из словарей,
    переданных в objects, а значениями – сами словари.
    """
    keys_to_remove = [key] if remove_key else []
    return dict(
        (
            transform_key(obj[key]),
            except_fields(obj, *keys_to_remove)
        )
        for obj in objects
    )


def only_attrs(items, key):
    """Возвращает список из значений, извлечённых по указанному ключу из каждого словаря в списке.
        Если в каком-то объекте нет ключа, будет исключение KeyError."""
    return list(map(operator.itemgetter(key), items))


def get_common_parent(path_one, path_two):
    """
    Находит общего предка в дереве
    Args:
        path_one (str): путь до корня, разделитель '.'
    Returns:
        int: id общего предка
    """
    array_one = path_one.split('.')
    array_two = path_two.split('.')
    min_length = min(len(array_one), len(array_two))
    last_common = 0
    last_common_index = 0
    for i in range(1, min_length):
        # первым элементом должен приезжать корневой департамент
        if array_one[i] != array_two[i]:
            return (int(last_common), int(array_one[last_common_index + 1]),
                    int(array_two[last_common_index + 1]))
        last_common_index = i
        last_common = array_one[last_common_index]
    return (1,
            int(array_one[min(len(array_one) - 1, 1)]),
            int(array_two[min(len(array_two) - 1, 1)]),
            )


def is_need_blackbox_info(uid, email):
    return email and 'robot' not in email and is_outer_uid(uid)


def is_outer_uid(uid, is_cloud=False):
    """
    Вернуть True, если пользователь из обычных паспортных учеток. Не доменных.
    """
    if uid:
        if is_cloud:
            return False
        return bool(RANGE_PASSPORT[0] <= int(uid) < RANGE_PASSPORT[1])


def is_cloud_uid(uid):
    """
    Вернуть True, если пользователь из облачных учеток (не паспортных)
    """
    if uid:
        return bool(RANGE_CLOUD[0] <= int(uid) < RANGE_CLOUD[1])


def is_domain_uid(uid):
    """Вернуть True, если пользователь доменный"""
    return not is_outer_uid(uid) and not is_cloud_uid(uid)


def create_organization(meta_connection,
                        main_connection,
                        name,
                        label,
                        domain_part,
                        admin_uid,
                        admin_nickname,
                        admin_first_name,
                        admin_last_name,
                        admin_gender,
                        admin_birthday,
                        language,
                        # robot_login=None,
                        # robot_password=None,
                        # admin_login=None,
                        # admin_password=None,
                        is_sso_enabled=None,
                        is_provisioning_enabled=None,
                        root_dep_label=None,
                        ready=True,
                        user_ip=None,
                        cloud_org_id=None,
                        **additional_fields
                        ):
    """Эта функция заводит организацию в нашей базе,
    заводит домен в нашей базе, соответст. организации
    и добавляет в неё аккаунт админа. При этом:

    1. Админский аккаунт уже должен существовать в паспорте, хотя для тестов это не важно.
    2. Домен организации тоже должен быть заведен в паспорте, но для тестов и это не важно.
    """

    from intranet.yandex_directory.src.yandex_directory.core.models import (
        OrganizationModel,
        OrganizationMetaModel,
        OrganizationSsoSettingsModel,
        UserModel,
        UserMetaModel,
        GroupModel,
    )

    if user_ip is None:
        if hasattr(g, 'user'):
            user_ip = g.user.ip

    label = to_lowercase(label)
    shard = main_connection.engine.db_info['shard']

    # добавим к label случайный постфикс
    if OrganizationMetaModel(meta_connection).filter(label=label).one():
        postfix = get_random_password(5).lower()
        label = '{label}{postfix}'.format(label=label, postfix=postfix)

    organization_meta_instance = OrganizationMetaModel(meta_connection).create(
        label=label,
        shard=shard,
        ready=ready,
        limits={'users_limit': settings.ORG_LIMITS['users_limit']},
        cloud_org_id=cloud_org_id,
    )
    org_id = organization_meta_instance['id']

    # если админский аккаунт не внешний и уже есть в нашей метабазе, то это
    # странная ситуация — значит этот аккаунт пытаются привязать к
    # другой организации
    admin_from_meta = UserMetaModel(meta_connection).find(
        filter_data={
            'id': admin_uid,
            'is_outer': False,
        })

    UserMetaModel(meta_connection).create(
        id=admin_uid,
        org_id=org_id,
    )

    with main_connection.begin_nested():
        # create organization in main database
        organization = OrganizationModel(main_connection).create(
            id=org_id,
            name=name,
            label=label,
            admin_uid=admin_uid,
            language=language,
            **additional_fields
        )
        if is_sso_enabled is not None and is_provisioning_enabled is not None:
            OrganizationSsoSettingsModel(main_connection).insert_or_update(
                org_id,
                is_sso_enabled is True,
                is_provisioning_enabled is True,
            )

        # сохраним домен в базу
        domain = create_domain_for_organization(
            main_connection,
            organization,
            domain_part,
        )

        root_department = create_root_department(
            main_connection,
            org_id,
            language,
            root_dep_label,
        )

        GroupModel(main_connection).get_or_create_robot_group(org_id)

        # Не сохраняем информацию о внешнем админе у нас в базе Users
        admin_user = None
        if not is_outer_uid(admin_uid):
            admin_email = build_email(
                main_connection,
                admin_nickname,
                org_id,
                user_id=admin_uid,
            )
            admin_user = UserModel(main_connection).create(
                id=admin_uid,
                department_id=root_department['id'],
                nickname=admin_nickname,
                name={
                    'first': {
                        'ru': admin_first_name
                    },
                    'last': {
                        'ru': admin_last_name
                    }
                },
                email=admin_email,
                org_id=org_id,
                gender=admin_gender,
                birthday=admin_birthday,
            )
            UserModel(main_connection).make_admin_of_organization(
                org_id=org_id,
                user_id=admin_uid,
            )
            OrganizationModel(main_connection).update_user_count(org_id=org_id)

            enable_features(
                meta_connection,
                main_connection,
                organization,
            )

    return {
        'domain': domain,
        'organization': organization,
        'organization_meta_instance': organization_meta_instance,
        'root_department': root_department,
        'admin_user': admin_user,
        'admin_user_uid': admin_uid,
    }


def create_domain_for_organization(connection, organization, domain_part):
    """
    При заведении организации сохраняем домен к себе в базу.
    Домен прилетает при заведении организации через паспорт
    и является master & display доменом.
    """
    from intranet.yandex_directory.src.yandex_directory.core.models.domain import DomainModel
    domain_name = build_organization_domain(organization['label'], domain_part)
    # сохраняем домен в пуникоде в нашу базу при первом создании организации
    domain_name = to_punycode(domain_name)
    domain_model = DomainModel(connection) # TODO: DIR-9717

    with get_meta_connection() as meta_connection:
        from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
            get_domains_from_db_or_domenator,
            DomainFilter,
        )
        exist_domain = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(name=domain_name, owned=True),
            one=True,
        )

        org_id = organization['id']

        if exist_domain:
            raise ImmediateReturn(
                json_error(
                    422,
                    'domain_already_exists',
                    '%s already exist' % exist_domain[0]['name'],
                )
            )

        domain_model.create(
            name=domain_name,
            org_id=org_id,
            owned=True,
        )
        domain_model.update_one(
            name=domain_name,
            org_id=org_id,
            data={
                'master': True,
                'display': True,
            },
        )
        return domain_model.get(domain_name=domain_name, org_id=org_id)


if os.environ.get('ENVIRONMENT') == 'autotests':
    import sys

    organization_domains_cache_size = 0  # для тестов размер кэша == 0
    # установим бесконечное время жизни,
    # чтобы TTLCache не пытался вызвать expire
    organization_domains_cache_ttl = sys.maxsize
else:
    organization_domains_cache_size = 1024 * 1024  # размер в байтах
    organization_domains_cache_ttl = 300  # 5 минут

organization_domains_cache = TTLCache(
    maxsize=organization_domains_cache_size,
    ttl=organization_domains_cache_ttl,
)
organization_domains_lock = RLock()


@cached(organization_domains_cache,
        key=ignore_first_args(2)(keys.hashkey),
        lock=organization_domains_lock)
def get_domains_by_org_id(meta_connection, main_connection, org_id):
    """
    Возвращает все домены по id организации в виде:
    {
        'all': [<domainname1>, <domainname2>, ...],
        'master': "master_domainname",
        'display': "displaydomainname",
        'owned': [<domainname1>, ...],
    }
    """
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        get_domains_from_db_or_domenator,
        DomainFilter,
        get_master_domain_from_db_or_domenator,
        update_domains_in_db_or_domenator,
        DomainUpdateFilter,
        DomainUpdateData,
    )

    # для обезличенных сервисов и внешних админов
    if main_connection is None:
        shard = get_shard(meta_connection, org_id=org_id)
        with get_main_connection(shard) as main_connection:
            domains = get_domains_from_db_or_domenator(
                meta_connection=meta_connection,
                domain_filter=DomainFilter(org_id=org_id),
                main_connection=main_connection,
            )
    else:
        domains = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(org_id=org_id),
            main_connection=main_connection,
        )

    domains_info = defaultdict(list)
    for domain in domains:
        domains_info['all'].append(domain['name'])
        if domain.get('master'):
            domains_info['master'] = domain['name']
            domains_info['display'] = domain['name']
        if domain.get('owned'):
            domains_info['owned'].append(domain['name'])
    return domains_info


def get_organizations_domains(main_connection, org_ids):
    """
    Словарь org_id -> domains, где domains тоже словарь вида:

    {
        'all': [<domainname1>, <domainname2>, ...],
        'master': "master_domainname",
        'display': "displaydomainname",
    }

    Функции обязательно должен быть передан main_connection,
    и все организации должны быть именно в этом шарде.
    """
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        get_domains_from_db_or_domenator,
        DomainFilter,
    )

    with get_meta_connection() as meta_connection:
        domains = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(org_id=org_ids),
            main_connection=main_connection,
        )

    # В результатах всегда есть словарь с доменами для всех переданных
    # организаций. Даже если у организации почему-то нет ни одного.
    results = {
        org_id: {'all': [], 'master': None, 'display': None}
        for org_id in org_ids
    }

    for domain in domains:
        domains_info = results[domain['org_id']]

        domains_info['all'].append(domain['name'])
        if domain.get('master'):
            domains_info['master'] = domain['name']
            domains_info['display'] = domain['name']

    return results


def get_organization_admin_uid(main_connection, org_id):
    """
    Возвращаем uid админа организации.
    Теперь он хранится в базе main в таблице organizations.
    """
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
    organization = OrganizationModel(main_connection).get(org_id)
    if organization:
        return organization['admin_uid']


def create_user(meta_connection,
                main_connection,
                org_id,
                user_data,
                nickname=None,
                password=None,
                password_mode='plain',
                ignore_login_not_available=False,
                calc_disk_usage=True,
                update_groups_leafs_cache=True,
                user_type='user',
                is_outer=None,
                is_sso=False,
                ):
    from intranet.yandex_directory.src.yandex_directory.core.models.user import (
        UserModel,
        UserMetaModel,
    )
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
        OrganizationModel,
        OrganizationMetaModel,
    )
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import (
        DomainYandexTeam,
        LoginNotavailable,
    )
    from intranet.yandex_directory.src.yandex_directory.core.models.department import (
        DepartmentModel,
    )
    from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
        InvalidDepartmentId,
    )
    from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
        UsersLimitError,
        UserAlreadyExists,
    )
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import (
        DomainInvalidType,
        DomainInvalid,
    )
    from intranet.yandex_directory.src.yandex_directory.common.crutches import sync_domain_status

    with log.fields(org_id=org_id, nickname=nickname):
        log.info('Creating user')

        organization = OrganizationModel(main_connection).get(org_id)
        organization_meta = OrganizationMetaModel(meta_connection).get(org_id)
        if not organization_meta:
            OrganizationMetaModel(meta_connection).raise_organization_not_found_exception(org_id)
        # проверяем не уперлись ли в лимит пользователей
        users_limit = organization_meta['limits'].get('users_limit')
        if user_type == 'user' and users_limit and organization['user_count'] >= users_limit:
            raise UsersLimitError()

        department_id = user_data.get('department_id')
        if department_id is not None:
            department = DepartmentModel(main_connection).get(
                int(department_id), org_id
            )
            if not department:
                raise InvalidDepartmentId()

        # Язык берем из организации, если он не передан
        # страну - пока не передаем
        language = user_data.pop('language', None)
        if not language:
            language = organization['language']

        user_email = build_email(
            main_connection,
            nickname,
            org_id,
            user_id=user_data.get('id'),
        )
        user_data['email'] = user_email
        nickname = nickname.lower()
        user_data['nickname'] = nickname
        # Если нет поля name, то нужно добавить.
        user_data['name'] = user_data.get('name') or {}
        # Если нет имени, то нужно добавить пустую строку
        user_data['name']['first'] = user_data['name'].get('first') or {}
        user_data['name']['first']['ru'] = user_data['name']['first'].get('ru') or ""
        user_data['name']['first']['en'] = user_data['name']['first'].get('en') or ""
        # Если нет фамилии, то нужно добавить пустую строку
        user_data['name']['last'] = user_data['name'].get('last') or {}
        user_data['name']['last']['ru'] = user_data['name']['last'].get('ru') or ""
        user_data['name']['last']['en'] = user_data['name']['last'].get('en') or ""
        if not user_data.get('id'):
            domain = get_master_domain(main_connection, org_id, force_refresh=True)
            if is_yandex_team(domain):
                if user_type != 'yamb_bot':
                    raise DomainYandexTeam

                user_data['id'] = app.team_passport.yambot_add()
            else:
                request_data = {
                    'login': nickname,
                    'firstname': user_data['name']['first']['ru'] or user_data['name']['first']['en'] or user_data['name']['first'].get('tr', ''),
                    'lastname': user_data['name']['last']['ru'] or user_data['name']['last']['en'] or user_data['name']['last'].get('tr', ''),
                    'birthday': user_data.get('birthday'),
                    'gender': user_data.get('gender'),
                    'language': language,
                }
                if password_mode == 'plain':
                    request_data['password'] = password
                elif password_mode == 'hash':
                    request_data['password_hash'] = password

                if user_type == 'yamb_bot':
                    # Указываем, что аккаунт является ботом Ямба
                    # https://wiki.yandex-team.ru/passport/api/bundle/registration/pdd/#parametryzaprosa
                    request_data['with_yambot_alias'] = 1

                try:
                    user_id = _add_account(
                        domain=domain,
                        request_data=request_data,
                        user_email=user_email,
                        ignore_login_not_available=ignore_login_not_available,
                    )
                except (DomainInvalid, DomainInvalidType):
                    if sync_domain_status(
                            main_connection=main_connection,
                            domain=domain,
                            org_id=org_id,
                    ):
                        user_id =_add_account(
                            domain=domain,
                            request_data=request_data,
                            user_email=user_email,
                            ignore_login_not_available=ignore_login_not_available,
                        )
                    else:
                        raise

                user_data['id'] = user_id

        user_data['org_id'] = org_id
        user_data['user_type'] = user_type

        # проверяем, что такого пользователя еще нет в метабазе, т.к.
        # может возникнуть ситуация, когда пользователь успел добавиться в метабазу, но не добавился в main базу
        meta_user = UserMetaModel(meta_connection).filter(
            id=user_data['id'],
            org_id=org_id,
        ).one()
        main_user = UserModel(main_connection).filter(
            id=user_data['id'],
            org_id=org_id,
        ).one()

        if meta_user and main_user:
            raise UserAlreadyExists()

        if not meta_user:
            meta_user = UserMetaModel(meta_connection).create(
                id=user_data['id'],
                org_id=org_id,
                is_outer=is_outer,
            )

        if not main_user:
            main_user = UserModel(main_connection).create(
                calc_disk_usage=calc_disk_usage,
                update_groups_leafs_cache=update_groups_leafs_cache,
                is_sso=is_sso,
                **user_data
            )

        log.info('User created')
        return {
            'user': main_user,
            'user_meta_instance': meta_user,
        }


def _add_account(domain, request_data, user_email, ignore_login_not_available):
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import (
        LoginNotavailable,
    )
    try:
        user_id = app.passport.account_add(
            domain=domain,
            user_data=request_data,
        )
    except LoginNotavailable:
        if ignore_login_not_available:
            user_info = get_user_data_from_blackbox_by_login(user_email)
            if user_info['is_maillist']:
                raise
            user_id = user_info['uid']
        else:
            raise
    return user_id


def create_maillist(main_connection, org_id, label, ignore_login_not_available=False):
    """
    Вернуть UID рассылки, если ее удалось создать.

    :rtype None | int
    """
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import (
        DomainYandexTeam,
        LoginNotavailable,
    )
    from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
        DomainNotFound,
    )
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        DomainFilter,
        get_master_domain_from_db_or_domenator,
        get_domains_from_db_or_domenator,
    )

    try:
        with get_meta_connection() as meta_connection:
            domain = get_domains_from_db_or_domenator(
                meta_connection=meta_connection,
                domain_filter=DomainFilter(org_id=org_id, master=True),
                main_connection=main_connection,
                one=True,
                exc_on_empty_result=MasterDomainNotFound(org_id=org_id),
            )['name']
    except DomainNotFound:
        log.info('Skipped creating maillist, no domain in this org')
        return None

    if domain.startswith('yandex-team'):
        raise DomainYandexTeam

    maillist_email = build_email(main_connection, label, org_id)
    try:
        maillist_uid = app.passport.maillist_add(
            domain=domain,
            login=label,
        )
    except LoginNotavailable:
        if ignore_login_not_available:
            account_info = get_user_data_from_blackbox_by_login(maillist_email)
            if not account_info['is_maillist']:
                raise
            maillist_uid = account_info['uid']
        else:
            raise
    return maillist_uid


def delete_user(meta_connection, main_connection, user_id):
    """
    Удаляем всех юзеров по заданному user_id из main и meta базы
    """
    from intranet.yandex_directory.src.yandex_directory.core.models.user import (
        UserModel,
        UserMetaModel,
    )
    meta_model = UserMetaModel(meta_connection)
    user_model = UserModel(main_connection)

    meta_model.delete(
        filter_data=dict(
            id=user_id,
        )
    )
    users = meta_model.find(filter_data=dict(id=user_id))
    for user in users:
        user_model.delete(
            filter_data=dict(
                org_id=user['org_id'],
                id=user_id
            )
        )


def build_organization_domain(label, domain_part):
    # TODO: избавиться от использования этого метода
    # Избавляться будем в рамках этого таска: https://st.yandex-team.ru/DIR-728
    if label == 'yandex':
        return 'yandex.ru'
    elif label == 'yandex-team':
        return 'yandex-team.ru'

    elif label is None:
        return domain_part
    elif domain_part is None:
        return label
    return label + domain_part


def get_master_domain(main_connection,
                      org_id,
                      force_refresh=False,
                      domain_is_required=True):
    """
    Получаем отображаемый домен (результат кэшируется)
    :param main_connection: соединение к main базе
    :param org_id: id организации
    :param force_refresh: не использовать кэш
    :return: имя домена
    :rtype: str
    """
    from intranet.yandex_directory.src.yandex_directory.common.exceptions import DomainNotFound
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
        DomainFilter,
        get_master_domain_from_db_or_domenator,
    )

    try:
        with get_meta_connection() as meta_connection:
            domain_obj = get_master_domain_from_db_or_domenator(
                org_id=org_id,
                meta_connection=meta_connection,
                main_connection=main_connection,
            )
    except DomainNotFound:
        domain_obj = None
    if domain_obj:
        domain_name = domain_obj['name']
    else:
        domain_name = None
        if domain_is_required:
            raise ImmediateReturn(
                json_error(
                    422,
                    'display_master_should_exists',
                    'Display master domain should exists',
                )
            )
    return domain_name


def prepare_user_with_service(meta_connection, main_connection, user, api_version):
    _expand_services_in_user(meta_connection, main_connection, user, api_version)
    return only_fields(user, 'id', 'org_id', 'nickname', 'services')


def _expand_services_in_user(meta_connection, main_connection, user, api_version, with_disk_paid_space=False):
    if api_version > 6:
        for service in user['services']:
            if with_disk_paid_space and service['slug'] == 'disk':
                from intranet.yandex_directory.src.yandex_directory import disk
                service['has_paid_space'] = disk.is_paid(user['id'])
            del service['id']
    else:
        from intranet.yandex_directory.src.yandex_directory.core.models import (
            ServiceModel, OrganizationServiceModel, UserServiceLicenses
        )

        org_services = OrganizationServiceModel(main_connection).find(
            filter_data={'org_id': user['org_id'], 'enabled': True},
            fields=['service_id', 'ready'],
        )
        user_services = []
        # помним, что в filter_data нельзя передавать пустые списки для фильтрации
        if org_services:
            services = ServiceModel(meta_connection).find(
                filter_data={'id': only_attrs(org_services, 'service_id')},
                fields=['slug', 'name', 'id'],
            )
            user_licenses = []
            licensed_services = list(OrganizationServiceModel(main_connection).get_org_services_with_licenses(
                user['org_id'],
                trial_expired=True,
                only_id=True,
            ).values())
            if licensed_services:
                user_licenses = list(UserServiceLicenses(main_connection).get_users_with_service_licenses(
                    org_id=user['org_id'],
                    service_ids=licensed_services,
                    user_ids=user['id']
                ).keys())
            service_ready = {row['service_id']: row['ready'] for row in org_services}
            for service in services:
                if service['id'] in user_licenses or service['id'] not in licensed_services:
                    service.update({'ready': service_ready[service['id']]})
                    if with_disk_paid_space and service['slug'] == 'disk':
                        from intranet.yandex_directory.src.yandex_directory import disk
                        service['has_paid_space'] = disk.is_paid(user['id'])
                    del service['id']
                    user_services.append(service)
        user['services'] = user_services

    return user


def check_data_relevance(org_id, users):
    from intranet.yandex_directory.src.yandex_directory.core.tasks import SyncUserData
    from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
    from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationModel

    if getattr(g, 'user', None) and g.user.ip:
        user_ip = g.user.ip
    else:
        user_ip = get_localhost_ip_address()

    users_data = get_user_data_from_blackbox_by_uids(
        ip=user_ip,
        uids=[
            obj['id'] for obj in users
            if obj.get('email') and
               'robot' not in obj.get('email') and
               is_outer_uid(obj['id'])
        ],
    )
    changed_users = []
    gender_map = {
        'unspecified': '0',
        'male': '1',
        'female': '2',
    }
    for user in users:
        if not is_outer_uid(user['id']):
            continue
        user_data = users_data.get(str(user['id']))
        if not user_data:
            continue
        is_changed = False
        last_name = user.get('name', {}).get('last') or user.get('last_name')
        first_name = user.get('name', {}).get('first') or user.get('first_name')
        birth_date = user.get('birthday')
        sex = user.get('gender')
        if sex:
            sex = gender_map.get(sex, user_data['sex'])

        if last_name and last_name != user_data['last_name']:
            is_changed = True
        if first_name and first_name != user_data['first_name']:
            is_changed = True
        if sex and sex != user_data['sex']:
            is_changed = True
        if birth_date:
            if not isinstance(birth_date, str):
                birth_date = birth_date.strftime('%Y-%m-%d')
            if birth_date != user_data['birth_date']:
                is_changed = True
        if is_changed:
            changed_users.append(user['id'])

    with get_main_connection(for_write=True, shard=g.shard) as main_connection:
        if changed_users:
            for user_id in changed_users:
                try:
                    SyncUserData(main_connection).delay(org_id=org_id, user_id=user_id)
                except DuplicatedTask:
                    # если таск уже есть - не нужно пробовать создать его снова
                    pass
        OrganizationModel(main_connection).update(
            filter_data={'id': org_id},
            update_data={'last_passport_sync': utcnow()}
        )


def prepare_user(connection, user, expand_contacts=False, api_version=None, org_domain=NotGiven):
    assert isinstance(api_version, int), 'api_version is "{0}", but integer was expected'.format(api_version)

    if api_version > 5:
        return prepare_user_v6(connection, user, expand_contacts=expand_contacts, org_domain=org_domain)

    data = {
        'id': user['id'],
        'login': user['nickname'],
        'nickname': user['nickname'],
        'aliases': user.get('aliases', []),
        'email': build_email(
            connection,
            user['nickname'],
            user['org_id'],
            user_id=user['id'],
            email=user.get('email'),
            org_domain=org_domain,
        ),
        'name': user['name'],
        'gender': user['gender'],
        'groups': list(map(
            partial(prepare_group, connection, api_version=api_version),
            user.get('groups', [])
        )),
        # Позиции в некоторых случаях может не быть
        # https://st.yandex-team.ru/DIR-7020
        'position': user.get('position'),
        'about': user['about'],
        'birthday': try_format_date(user['birthday']) if user['birthday'] else None,
        'contacts': user['contacts'],
        'is_dismissed': user['is_dismissed'],
        'external_id': user.get('external_id'),
        'is_admin': bool(user.get('is_admin', False)),
        'role': user.get('role', ''),
    }
    if expand_contacts:
        data['contacts'] = expand_user_contacts(connection, user, org_domain)
    if user.get('department'):
        data['department'] = prepare_department(
            connection,
            user['department'],
            api_version=api_version,
        )
    else:
        data['department_id'] = user['department_id']

    if 'is_enabled' in user:
        data['is_enabled'] = user['is_enabled']

    if user.get('is_robot', False):
        data['is_robot'] = user['is_robot']

    if user.get('user_type', None) is not None:
        data['user_type'] = user['user_type']

    if user.get('service_slug', None) is not None:
        data['service_slug'] = user['service_slug']

    if user.get('timezone'):
        data['timezone'] = user.get('timezone')

    if user.get('language'):
        data['language'] = user.get('language')

    if user.get('avatar_id'):
        data['avatar_id'] = user['avatar_id']

    return data


def prepare_user_v6(connection, user, expand_contacts=False, api_version=6, org_domain=NotGiven):
    response = {}

    fields_to_copy = [
        'id',
        # 'login', убрали в 5 версии API
        'nickname',
        'name',
        'gender',
        'position',
        'about',
        'contacts',
        'is_dismissed',
        'aliases',
        'external_id',
        'is_robot',
        'service_slug',
        'is_enabled',
        'is_admin',
        'user_type',
        'role',
    ]
    if api_version > 6:
        fields_to_copy.append('services')

    for field in fields_to_copy:
        if field in user:
            response[field] = user[field]

    if expand_contacts and 'nickname' in user and 'org_id' in user and 'contacts' in user:
        response['contacts'] = expand_user_contacts(connection, user, org_domain=org_domain)

    if 'email' in user and 'nickname' in user and 'org_id' in user:
        response['email'] = build_email(
            connection,
            user['nickname'],
            user['org_id'],
            user_id=user['id'],
            org_domain=org_domain,
        )

    if 'groups' in user:
        response['groups'] = list(map(
            partial(prepare_group, connection, api_version=api_version),
            user.get('groups', [])
        ))

    if 'birthday' in user:
        response['birthday'] = try_format_date(user['birthday']) if user['birthday'] else None

    if 'department' in user:
        response['department'] = prepare_department(
            connection,
            user['department'],
            api_version=api_version,
        )
    elif 'department_id' in user:
        response['department_id'] = user['department_id']

    if user.get('timezone'):
        response['timezone'] = user.get('timezone')

    if user.get('language'):
        response['language'] = user.get('language')

    if user.get('avatar_id'):
        response['avatar_id'] = user['avatar_id']

    return response


def prepare_user_with_fields(meta_connection, main_connection, user, fields, api_version, org_domain=NotGiven):
    assert isinstance(api_version, int), 'api_version is "{0}", but integer was expected'.format(api_version)
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        OrganizationModel,
    )

    if 'id' not in fields:
        fields = ['id'] + fields

    # Если были запрошены подполя у отделов, то их надо убрать и заменить
    # просто на поле departments.
    # иначе ничего не преобразуется
    if any(field.startswith('departments.')
           for field in fields):
        fields = [field for field in fields
                  if not field.startswith('departments.')]
        fields.append('departments')

    # То же самое для отдела
    if any(field.startswith('department.')
           for field in fields):
        fields = [field for field in fields
                  if not field.startswith('department.')]
        fields.append('department')

    # То же самое для организации
    if any(field.startswith('organization.')
           for field in fields):
        fields.append('organization')

    for field in fields:
        if field == 'services.disk.has_paid_space':
            continue

        elif field == 'aliases' and field in user:
            user['aliases'] = user.get('aliases', [])
        elif field == 'email' and 'nickname' in user and 'org_id' in user:
            user['email'] = build_email(
                main_connection,
                user['nickname'],
                user['org_id'],
                user_id=user['id'],
                email=user.get('email'),
                org_domain=org_domain,
            )
        elif field == 'groups' and field in user:
            user['groups'] = list(map(
                partial(prepare_group, main_connection, api_version=api_version),
                user.get('groups', [])
            ))
        elif field == 'services':
            if 'services.disk.has_paid_space' in fields:
                with_disk_paid_space = True
            else:
                with_disk_paid_space = False
            _expand_services_in_user(
                meta_connection,
                main_connection,
                user,
                api_version,
                with_disk_paid_space=with_disk_paid_space,
            )
        elif field == 'birthday' and field in user:
            user['birthday'] = try_format_date(user['birthday']) if user['birthday'] else None
        elif field == 'is_admin' and field in user:
            user['is_admin'] = bool(user.get('is_admin', False))
        elif field == 'department' and field in user:
            user['department'] = prepare_department(
                main_connection,
                user['department'],
                api_version=api_version,
            )
        elif field == 'departments' and field in user:
            user['departments'] = [
                prepare_department(
                    main_connection,
                    department,
                    api_version=api_version,
                )
                for department in user['departments']
            ]
        elif field == 'contacts' and field in user:
            user['contacts'] = expand_user_contacts(main_connection, user, org_domain)
        elif field == 'created' and field in user:
            user['created'] = try_format_datetime(user['created'])
        elif field == 'organization.organization_type':
            org_data = OrganizationModel(main_connection).get(
                g.org_id, fields=('organization_type',)
            )
            user['organization'] = org_data

    # отказ от deprecated поля login
    if api_version >= 5 and 'login' in user:
        del user['login']
    public_fields = list(set(fields) - set(PRIVATE_USER_FIELDS))
    return only_fields(user, *public_fields)


def prepare_action(obj):
    result = fallback_fields(
        obj,
        'id',
        'revision',
        'name',
        'object_type',
        'object',
        'old_object',
        'timestamp',
        ('author', 'author_id'),
    )
    return result


def prepare_events(obj):
    return only_fields(
        obj,
        'id',
        'revision',
        'name',
        'object_type',
        'content',
        'object',
        'timestamp',
    )


def prepare_service(connection, service):
    return only_fields(
        service,
        'id',
        'name',
        'slug',
    )


def prepare_group_member(connection, item, api_version, org_domain=NotGiven):
    obj_type = item['type']
    preparers = {
        TYPE_USER: partial(prepare_user, expand_contacts=True, api_version=api_version, org_domain=org_domain),
        TYPE_GROUP: partial(prepare_group, api_version=api_version, org_domain=org_domain),
        TYPE_DEPARTMENT: partial(prepare_department, api_version=api_version),
    }
    prepare = preparers[obj_type]
    return {'type': obj_type, 'object': prepare(connection, item['object'])}


def prepare_group(connection, group, api_version, org_domain=NotGiven):
    # Начиная с 3 версии API мы отдаём только те поля, которые
    # есть во входном словаре

    assert isinstance(api_version, int), 'api_version is "{0}", but integer was expected'.format(api_version)

    response = {
        'id': group['id'],
    }

    if api_version < 3:
        response['email'] = group.get('email')

    fields_to_copy = (
        'name',
        'type',
        'description',
        'label',
        'members_count',
        'external_id',
        'created',
        'maillist_type',
        'uid',
    )

    fields_to_copy_v6 = (
        'aliases',
        'org_id',
        'resource_id',
        'email',
        'removed',
        'all_users',
        'tracker_license',
    )

    for field in fields_to_copy:
        if api_version == 1:
            response[field] = group.get(field)
        else:
            if field in group:
                response[field] = group[field]

    if api_version > 5:
        for field in fields_to_copy_v6:
            if field in group:
                response[field] = group.get(field)

    if 'created' in group:
        response['created'] = try_format_datetime(group['created'])

    if 'author' in group:
        if group.get('author'):
            response['author'] = prepare_user(
                connection,
                group['author'],
                expand_contacts=True,
                api_version=api_version,
                org_domain=org_domain,
            )
        else:
            response['author'] = None
    else:
        if 'author_id' in group or api_version < 3:
            response['author_id'] = group.get('author_id')

    if 'members' in group:
        add_avatar_id_for_user_objects(group['members'])
        response['members'] = list(map(
            partial(prepare_group_member, connection, api_version=api_version, org_domain=org_domain),
            group['members']
        ))

    if 'member_of' in group:
        response['member_of'] = list(map(
            partial(prepare_group, connection, api_version=api_version, org_domain=org_domain),
            group['member_of']
        ))

    if 'admins' in group:
        response['admins'] = list(map(
            partial(
                prepare_user,
                connection,
                expand_contacts=True,
                api_version=api_version,
                org_domain=org_domain,
            ),
            group['admins']
        ))
    return response


def prepare_organization(connection, organization):
    return {
        'id': organization['id'],
    }


def prepare_domain_with_fields(main_connection, domain, fields=None):
    from intranet.yandex_directory.src.yandex_directory import gendarme
    from intranet.yandex_directory.src.yandex_directory.core.utils.domain import domain_is_tech

    from intranet.yandex_directory.src.yandex_directory.core.models import (
        DomainModel,
        OrganizationModel,
    )
    org_id = g.org_id

    if fields is None:
        fields = []

    info = {}

    lower_domain_name = domain['name'].lower()

    if domain['owned']:
        info = {}

        if 'country' in fields:
            country = OrganizationModel(main_connection) \
                .filter(id=org_id) \
                .scalar('country')[0]
            info['country'] = country or 'ru'

        if 'delegated' in fields or 'mx' in fields:
            try:
                data_from_gendarme = gendarme.status(lower_domain_name)
                info['mx'] = data_from_gendarme['mx'].get('match', False)
                info['delegated'] = data_from_gendarme['ns'].get('match', False)
            except gendarme.GendarmeDomainNotFound:
                # Пока Жандарм не проверит домен, будем отдавать дефолтные значения
                info['mx'] = False
                info['delegated'] = False

    response = {'name': lower_domain_name}
    if 'master' in fields:
        response['master'] = domain['master']
    if 'tech' in fields:
        response['tech'] = domain_is_tech(lower_domain_name)
    if 'owned' in fields:
        response['owned'] = domain['owned']
    if 'mx' in fields:
        response['mx'] = info.get('mx', domain['mx'])
    if 'delegated' in fields:
        response['delegated'] = info.get('delegated', domain['delegated'])
    if 'country' in fields:
        response['country'] = info.get('country', 'ru').lower()

    # Эти поля можно будет выпилить, когда полностью перейдём на сервис настроек.
    if 'can_users_change_password' in fields:
        response['can_users_change_password'] = info.get('can_users_change_password', True)
    if 'pop_enabled' in fields:
        response['pop_enabled'] = bool(info.get('pop_enabled', 0))
    if 'imap_enabled' in fields:
        response['imap_enabled'] = bool(info.get('imap_enabled', 0))
    if 'roster_enabled' in fields:
        response['roster_enabled'] = info.get('roster_enabled', False)
    if 'postmaster_uid' in fields:
        response['postmaster_uid'] = info.get('default_uid', 0)
    if 'org_id' in fields:
        response['org_id'] = domain['org_id']
    if 'domain_id' in fields:
        bb_response = get_domain_info_from_blackbox(domain['name'])
        domain_id = None if bb_response is None else bb_response['domain_id']
        response['domain_id'] = domain_id
    return response


def prepare_domain(connection, domain):
    return domain


def prepare_department(connection, department, api_version=None):
    assert isinstance(api_version, int), 'api_version is "{0}", but integer was expected'.format(api_version)

    # Если объекта нет, то его и обрабатывать не надо
    # в таких полях мы возвращаем None.
    # Эта ситуация может возникнуть, если Директория отдаёт
    # в списке пользователей робота, у которого нет отдела

    if not department:
        return None

    response = {
        'id': department['id'],
    }
    # TODO:
    # Рассчёт поля email надо бы перенести в prefetch_related,
    # когда мы полностью переведём всех на ручки, требующие перечисления полей.
    # Хотя, со вложенными объектами департаментов тут могут быть проблемы.

    fields_to_copy = ['name', 'label', 'description', 'external_id',
                      'created', 'email', 'maillist_type', 'members_count',
                      ]

    if app.config['INTERNAL']:
        fields_to_copy.append('uid')

    for field in fields_to_copy:
        if field in department or api_version < 3:
            response[field] = department.get(field)

    if 'parents' in department:
        response['parents'] = list(map(
            partial(prepare_department, connection, api_version=api_version),
            department['parents']
        ))

    if 'removed' in department:
        response['removed'] = bool(department['removed'])

    if 'parent' in department:
        response['parent'] = department['parent']
    else:
        if 'parent_id' in department or api_version < 3:
            response['parent_id'] = department.get('parent_id')

    if 'head' in department:
        if department['head'] is None:
            response['head'] = None
        else:
            response['head'] = prepare_user(
                connection,
                department['head'],
                api_version=api_version,
                expand_contacts=True,
            )

    if 'created' in department:
        response['created'] = try_format_datetime(department['created'])

    fields_to_copy = ['org_id', 'path', 'heads_group_id', 'aliases', 'is_outstaff', 'tracker_license', ]

    for field in fields_to_copy:
        if field in department and api_version > 4:
            response[field] = department.get(field)

    return response


def prepare_resource(resource):
    """
    Подготавливает ресурс для выдачи в API наружу.
    В API в качестве id ресурса используется поле external_id в базе.
    """
    from intranet.yandex_directory.src.yandex_directory.connect_services.idm import get_service

    service = resource['service']
    relations_to_skip = {}
    relations = []
    try:
        service_client = get_service(service)
        relations_to_skip = {service_client.responsible_relation}
    except ServiceNotFound:
        pass
    add_avatar_id_for_users_relations(resource['relations'])
    for relation in resource['relations']:
        prepared_relation = prepare_relation_for_resource(relation, relations_to_skip)
        if prepared_relation:
            relations.append(prepared_relation)

    return {
        'id': resource['external_id'],
        'service': service,
        'relations': relations,
        'metadata': {},
    }


def get_object_type(item):
    if item.get('user_id') or item.get('user'):
        return 'user'
    elif item.get('department_id') or item.get('department'):
        return 'department'
    elif item.get('group_id') or item.get('group'):
        return 'group'
    else:
        raise RuntimeError('Unknown item object type')


def prepare_relation_for_resource(relation, relations_to_skip=None):
    relations_to_skip = relations_to_skip or {}
    object_type = get_object_type(relation)
    relation_name = relation['name']
    if relation_name in relations_to_skip:
        return {}

    return {
        'id': relation['id'],
        'name': relation_name,
        'object_type': object_type,
        'object': prepare_resource_object_func[object_type](
            relation.get(object_type)
        )
    }


def add_avatar_id_for_container(uids_data):
    if uids_data:
        if getattr(g, 'user', None) and g.user.ip:
            user_ip = g.user.ip
        else:
            user_ip = get_localhost_ip_address()

        user_info = get_user_data_from_blackbox_by_uids(user_ip, list(uids_data.keys()))
        for user_data in user_info.values():
            uid = int(user_data['uid'])
            uids_data[uid]['avatar_id'] = user_data['avatar_id']


def add_avatar_id_for_user_objects(response):
    uids_data = {}
    for container in response:
        if container.get('type') == 'user':
            uids_data[container['object']['id']] = container['object']

    add_avatar_id_for_container(uids_data=uids_data)

    return response


def add_avatar_id_for_users_relations(relations):
    uids_data = {}
    for relation in relations:
        if get_object_type(relation) == 'user':
            uids_data[relation['user']['id']] = relation['user']

    add_avatar_id_for_container(uids_data=uids_data)

    return relations


def prepare_resource_for_licenses(resource):
    """
    Подготавливает ресурс для выдачи в API в виде лицензии на сервис.
    """
    add_avatar_id_for_users_relations(resource['relations'])
    response = [prepare_relation_for_resource(i) for i in resource['relations']]
    for item in response:
        del item['id']
        del item['name']
    return response


def add_user_data_from_blackbox(objects):
    """
    Добавляет данные из ЧЯ в результаты запроса, если данных
    пользователя еще нет в нашей базе (пользователя нет в организации)
    """
    uids = {}
    for obj in objects:
        if obj.get('user_id'):
            if obj.get('user') is None:
                uids[obj['user_id']] = obj
            else:
                obj['user']['external'] = False
    if uids:
        if getattr(g, 'user', None) and g.user.ip:
            user_ip = g.user.ip
        else:
            user_ip = get_localhost_ip_address()

        user_info = get_user_data_from_blackbox_by_uids(user_ip, list(uids.keys()))
        for user_data in list(user_info.values()):
            user_data['id'] = int(user_data['uid'])
            user_data['name'] = {
                'first': {
                    'ru': user_data['first_name'],
                },
                'last': {
                    'ru': user_data['last_name'],
                }
            }
            user_data['nickname'] = user_data['login']
            user_data['gender'] = 'male' if user_data['sex'] == '1' else 'female'
            user_data['external'] = True
            uids[user_data['id']]['user'] = user_data

    return objects


def prepare_requests_for_licenses(req):
    """
    Подготавливает запросы на лицензии для выдачи в API
    """
    object_type = get_object_type(req)
    if not req.get(object_type):
        req[object_type] = {'id': req['{}_id'.format(object_type)]}

    response = {
        'object_type': object_type,
        'object': prepare_resource_object_func[object_type](req[object_type]),
        'service_slug': req['service_slug'],
        'resource_id': req.get('external_id'),
        'comment': req.get('comment'),
        'name': 'view',
    }
    if 'author' in req or 'author_id' in req:
        response['author'] = prepare_user_for_resource(req.get('author', {'id': req.get('author_id')}))
    if 'comment' in req:
        response['comment'] = req['comment']
    if 'created_at' in req:
        response['created_at'] = format_datetime(req['created_at'])
    return response


def prepare_group_for_resource(group):
    return only_fields(
        group,
        'id',
        'name',
        'type',
        'members_count',
    )


def prepare_department_for_resource(department):
    return only_fields(
        department,
        'id',
        'name',
        'parent_id',
        'members_count',
    )


def prepare_user_for_resource(user):
    return only_fields(
        user,
        'id',
        'nickname',
        'name',
        'gender',
        'department_id',
        'position',
        'avatar_id',
        'external',
    )


prepare_resource_object_func = {
    'group': prepare_group_for_resource,
    'department': prepare_department_for_resource,
    'user': prepare_user_for_resource,
}


def prepare_relation(relation):
    """Это преобразование для вью, которое отдаёт детали о relation.
    """

    # Если relation содержит None ссылки на пользователя, отдел или группу, их надо убрать из выдачи.
    relation = relation.copy()
    for name in ('user', 'department', 'group'):
        if name in relation and relation[name] is None:
            del relation[name]

    return relation


def prepare_group_member_for_members_view(relation):
    """Этот метод получает на входе relation возвращаемый
    ResourceRelationModel а возвращает dict с полями:

    {
        'type': 'user',
        'object': {'id': 124, 'about': ...}
    }
    """
    response = prepare_relation_for_resource(relation)
    return {
        'type': response['object_type'],
        'object': response['object'],
    }


def simplify_member(member):
    """Преобразует словарик
        {"type": "user", "object": {"id": 100500, ...}}
       в
        {"type": "user", "id": 100500}
    """
    return {
        'type': member['type'],
        'id': member['object']['id'],
    }


def simplify_members(members):
    """Преобразует элементы списка из словариков
        {"type": "user", "object": {"id": 100500, ...}}
       в
        {"type": "user", "id": 100500}
    """
    return list(map(simplify_member, members))


def get_member_instance(connection, member, org_id, extra_data=None):
    """
    :param: extra_data - dict с доп. данными (ex: select_related,
                                                  prefetch_related)
    Получаем объект для member = {
        'type': enum('user', 'group', 'department'),
        'id': INTEGER
    }
    """
    model = get_model_by_type(member['type'])
    member_type_id = member.get('id') or member.get('object', {}).get('id')
    member_key = member.get('type') + '_id'
    get_data = {
        'org_id': org_id,
        member_key: member_type_id,
        'fields': ['*', 'email']
    }
    if extra_data:
        get_data.update(extra_data)
    model_inst = model(connection)
    return model_inst.get(**get_data)


def get_model_by_type(object_type):
    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
    from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
    from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
    TYPE_MODELS_DICT = {
        TYPE_USER: UserModel,
        TYPE_GROUP: GroupModel,
        TYPE_DEPARTMENT: DepartmentModel,
    }
    return TYPE_MODELS_DICT.get(object_type)


def get_all_groups_for_department(connection, org_id, department_id):
    """
    Отдает все группы, в которые так или иначе входит отдел (непосредственно
    или через родительские департаменты)
    """

    from intranet.yandex_directory.src.yandex_directory.core.models.resource import (
        ResourceModel,
        relation_name,
    )
    from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
    if department_id is None or department_id == -1:
        return []
    filter_data = {
        'org_id': org_id,
        'service': 'directory',
        'relation_name': relation_name.include,
        'department_id': department_id,
    }
    resources = ResourceModel(connection).find(
        filter_data=filter_data,
        fields=['id'],
    )
    resource_ids = only_ids(resources)
    if not resource_ids:
        return []
    groups = GroupModel(connection).find({'resource_id': resource_ids})
    return groups


def is_yandex_team_org_id(main_connection, org_id, ignore_cached=False):
    domain = get_master_domain(
        main_connection,
        org_id,
        force_refresh=ignore_cached,
        domain_is_required=False,
    )
    if domain:
        return is_yandex_team(domain)


def is_yandex_team(org_domain):
    """
    Организация yandex-team?
    """
    return bool(org_domain in app.config['YANDEX_TEAM_ORG_DOMAINS'])


def is_org_admin(role):
    """
    Является ли пользователь с id=uid админом организации org_id (внешним или внутренним)
    """
    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserRoles
    return role in UserRoles.admin_roles


def is_outer_org_admin(role):
    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserRoles
    return role == UserRoles.outer_admin


def is_deputy_admin(role):
    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserRoles
    return role == UserRoles.deputy_admin


def is_common_user(role):
    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserRoles
    return role == UserRoles.user


def get_user_role(meta_connection, main_connection, org_id, uid, is_cloud=False):
    """
    Получаем роль пользователя в организации.
    Возможные варианты: outer_admin, deputy_admin, admin, user
    """
    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserMetaModel, UserModel

    user_meta = UserMetaModel(meta_connection).get(
        user_id=uid,
        org_id=org_id,
        is_outer=Ignore,
        fields=['user_type'],
        is_cloud=is_cloud,
    )

    if user_meta:
        if user_meta['user_type'] != 'inner_user':
            return user_meta['user_type']
        user = UserModel(main_connection).get(uid, org_id, fields=['role'], is_cloud=is_cloud)
        if user:
            return user['role']


def build_email(connection, label, org_id, org_domain=NotGiven, user_id=None, email=None):
    """
    Функция генерирует email, который можно использовать в качестве
    ящика для департамента или группы-команды
    """
    if label:
        if user_id and is_outer_uid(user_id):
            # Пользователь может предпочитать использовать yandex.com,
            # но пока для простоты мы это игнорируем, почта дойдёт и так.
            # А в будущем можно брать tld организации.
            org_domain = 'yandex.ru'

        if user_id and is_cloud_uid(user_id):
            if not email:
                from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
                email = UserModel(connection).get(
                    user_id=user_id, is_dismissed=Ignore,
                )['email']

            return email

        else:
            if org_domain is NotGiven:
                org_domain = get_master_domain(connection, org_id)

        if org_domain is None:
            message = 'Domain is required for building email'
            with log.fields(label=label):
                log.error(message)

            raise RuntimeError(message)

        email = '%s@%s' % (
            force_text(label),
            force_text(org_domain),
        )
        return email.lower()


def get_member_localpart(member):
    """
    :param member: сотрудник, группа, департамент
    :return: localpart - email без доменной части
    """
    # пока не работаем с внешними email-ами, можно делать и так:
    # поле nickname есть только у пользователей, а у групп - совсем нет
    # но есть email-поле, поэтому пока можем отсекать
    # доменную часть у email-a группы, чтоб подписывать nickname внутри
    # фуриты (внутри фурита не может напрямую работать с email-ами,
    # особенно если указано allow_external='no').
    # Поэтому нужно помнить, что у группы, департамента и пользователя
    # не должно быть одинаковой localpart-части,
    # а то в этом месте что-нибудь поломается
    # TODO: https://st.yandex-team.ru/DIR-635
    if not member:
        return None

    localpart_or_email = member.get('nickname', member.get('label'))
    if localpart_or_email:
        return localpart_or_email.split('@')[0]
    return None


def expand_user_contacts(main_connection, user, org_domain=NotGiven):
    # TODO: test me
    """
    Разбавляем личные контактные данные пользователя - рабочими email-ами и
    рабочими email-алиасами.
    """
    user_contacts = []
    contacts = user.get('contacts')
    if contacts:
        for c in user['contacts']:
            c['main'] = bool(c.get('main'))
            c['alias'] = False
            c['synthetic'] = False
            user_contacts.append(c)

    main_work_email = [{
        "type": "email",
        "value": build_email(
            main_connection,
            user['nickname'],
            user['org_id'],
            user_id=user['id'],
            org_domain=org_domain,
        ),
        "main": True,
        "alias": False,
        "synthetic": True,
    }]

    work_aliases = []
    aliases = user.get('aliases', [])
    if aliases:
        for alias in aliases:
            work_aliases.append({
                "type": "email",
                "value": build_email(
                    main_connection,
                    alias,
                    user['org_id'],
                    user_id=user['id'],
                    org_domain=org_domain,
                ),
                "main": False,
                "alias": True,
                "synthetic": True,
            })

    # ссылка на staff
    if is_yandex_team_uid(user['id']):
        staff_link = url_join(app.config['STAFF_YANDEX_TEAM_URL'], user['nickname'])
    else:
        from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationModel
        tld = OrganizationModel(main_connection).get_tld_for_email(user['org_id'])
        staff_url = app.config['STAFF_URL']
        staff_link = url_join(staff_url, '{nickname}?org_id={org_id}&uid={uid}').format(
            tld=tld,
            nickname=user['nickname'],
            org_id=user['org_id'],
            uid=user['id'],
        )


    user_contacts.append({
        "type": "staff",
        "value": staff_link,
        "main": False,
        "alias": False,
        "synthetic": True,
    })

    all_contacts = user_contacts + main_work_email + work_aliases
    return all_contacts


def check_technical_group(group_type):
    # Не добавляем 'organization_admin', так как эта функция используется, чтобы
    # запретить изменять технические группы, но админы должны иметь возможность
    # добавлять или удалять других админов.
    tech_group_types = [
        'department_head',
        'robots',
    ]
    return group_type in tech_group_types


def check_objects_exists(conn, org_id, members):
    """
    Проверяет, что все, кого хотим добавить, есть в базе
    Args:
        conn: коннект,
        org_id(integer):
        members(list): [{'type': TYPE_USER, 'id': id}]
        Формат объекта может быть
        {'type|object_type': TYPE_USER, 'id|object_id': id} или
        {'type|object_type': TYPE_USER, 'object': object}
    """

    members_by_type = defaultdict(set)
    allowed_member_types = [TYPE_USER, TYPE_DEPARTMENT, TYPE_GROUP]

    for member in members:
        if 'type' in member and 'object_type' in member:
            raise ConstraintValidationError(
                'invalid_object',
                'Keys type and object_type are mutually exclusive, please, correct member {member}',
                member=member,
            )

        member_type = member.get('type') or member.get('object_type')
        assert member_type is not None, 'Unknown member type'

        if member_type not in allowed_member_types:
            raise ConstraintValidationError(
                'invalid_object_type',
                '"{type}" is invalid object type',
                type=member_type,
            )
        object_id = member.get('id') or member.get('object_id') or member.get('object', {}).get('id')
        members_by_type[member_type].add(object_id)

    for object_type, set_ids in members_by_type.items():
        model = get_model_by_type(object_type)
        filter_data = {
            'org_id': org_id,
            'id': tuple(set_ids),
        }
        count = model(conn).count(filter_data=filter_data)
        if count != len(set_ids):
            raise ConstraintValidationError(
                'objects_not_found',
                'Some objects of type "{type}" were not found in database',
                type=object_type,
            )


def is_inner_uid(uid):
    return not is_outer_uid(uid)


def is_yandex_team_uid(uid):
    # Диапазоны паспортных uid-ов: https://wiki.yandex-team.ru/passport/uids/
    first = 112 * 10 ** 13
    second = 113 * 10 ** 13
    return bool(first <= uid < second)


def int_or_none(value):
    """Если значение не None, то пытаемся привести его к integer.
    """
    if value is not None:
        return int(value)


def get_orgid_and_shard_by_label(meta_connection, label):
    """
    Получаем org_id и shard по label-у домена
    """
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationMetaModel
    organizations = OrganizationMetaModel(meta_connection).find(
        filter_data=dict(label=label)
    )
    if organizations:
        org = organizations[0]
        return org['id'], org['shard']
    return None, None


def remove_comments(text):
    """Убираем все строки, начинающиеся с #
    Символу # может предшествовать любое количество пробелов.
    """
    return re.sub(
        r'^ *#.*\n',
        '',
        text,
        flags=re.MULTILINE,
    )


def ensure_integer(value, field_name):
    """Пытается привести value к числу, и если не получается,
    то кидает ConstraintValidationError.
    """

    try:
        return int(value)
    except ValueError:
        raise ConstraintValidationError(
            code='invalid_value',
            message='Field "{field}" has invalid value',
            field=field_name,
        )


def flatten_paths(paths):
    """
    Принимает список списков с id отделов.
    Каждый список содержит то же самое, что у нас лежит в поле path
    в каждого отдела, но не в виде строки, а в виде списка с числовыми id.

    Функция берёт эти несколько списков, и возвращает один, в котором
    все id уникальны и содержатся в том порядке, как если бы мы отбходили дерево
    отделов, начиная с вершины, и по алгоритму depth-first.

    Например, имея на входе:

    [
      [1, 2, 3, 4]
      [1, 2, 3, 5]
      [1, 6]
    ]

    функция должна вернуть:

    [1, 6, 2, 3, 5, 4]

    Это нужно для того, чтобы правильно обновлять счетчики с количеством
    пользователей в отделах, обходя их с листьев к вершине.
    """
    # для работы алгоритма важно, чтобы входные данные были
    # отсортированы та, чтобы цепочки
    paths.sort()
    # превращаем входные данные в итератор
    paths = list(map(iter, paths))

    results = []
    tuples = zip_longest(*paths)
    # дальше идём по всем сразу

    for ids in tuples:
        current_value = None

        for idx, _id in enumerate(ids):
            if _id is not None:
                if current_value is None:
                    current_value = _id
                else:
                    if _id != current_value:
                        # дошли до ветки, у которой нет
                        # общих частей с другими ветками
                        # сливаем все её данные в results
                        # и на следующей итерации итератор
                        # этой ветки вернёт None
                        results.append(_id)
                        results.extend(paths[idx])

        # добавим в результаты общий id
        results.append(current_value)

    return results


def thread_log(func):
    """
    Декоратор для проброса родительского логера в функции запущенные в отдельных потоках
    """

    def wrapper(*args, **kwargs):
        return func(*args, **kwargs)

    return wrapper


def pmap(func, iterable, pool_size=None):
    """
    Параллельный map
    2*cpu_count параллельных процессов
    :param func: функция
    :param iterable:  список параметров функции
    :rtype: list[concurrent.futures.Future]
    """
    if pool_size is None:
        pool_size = 2 * multiprocessing.cpu_count()

    # В тестах используется один и тот же коннект к базе, поэтому
    # если запускать код в нескольких потоках, то непредсказуемо
    # вылезают странные артефакты.
    if app.config['ENVIRONMENT'] == 'autotests':
        results = [func(**kwargs) for kwargs in iterable]
    else:
        with ThreadPoolExecutor(pool_size) as executor:
            futures = [
                executor.submit(thread_log(func), **kwargs)
                for kwargs in iterable
            ]
            wait(futures)
            results = [f.result() for f in futures]

    return results


def call_passport_if_needed(f):
    """
    Паспортные каллбэки нужно вызывать только для событий, непосредственно касающихся
    отделов, команд или пользователей. Если есть признак directly == False,
    то каллбэк вызывать не надо.
    """

    @wraps(f)
    def wrap(*args, **kwargs):
        content = kwargs.get('content')
        # для событий, у которых directly != True, не нужно вызывать паспортные-методы
        call = True
        # Тут важно, что мы используем при сравнении 'is',
        # потому что в этом случае, при отсутствии признака 'directly',
        # callback будет вызван
        if content and content.get('directly') is False:
            call = False

        if call:
            return f(*args, **kwargs)

    return wrap


class ShardNotFoundError(RuntimeError):
    pass


def paginate_by_shards(shards=None,
                       shard=None,
                       page=1,
                       per_page=None,
                       item_getter=None):
    """Параметр item_getter должен быть функцией, которая
    принимает shard, offset, limit, и возвращает список объектов
    из указанного шарда.
    """

    assert isinstance(shards, list)
    assert isinstance(per_page, int)
    assert isinstance(page, int)
    assert item_getter is not None, 'Argument item_getter should be given'

    shard = shard or shards[0]

    offset = (page - 1) * per_page
    results = []

    if shard not in shards:
        # Если шард не удалось найти, то это ошибка
        # и надо вернуть специальное исключение
        raise ShardNotFoundError()

    for shard in shards[shards.index(shard):]:
        results = item_getter(
            shard=shard,
            offset=offset,
            # Пробуем выбрать на один элемент больше, чтобы понять, будут ли
            # ещё страницы в этом шарде.
            limit=per_page + 1,
        )
        if results:
            break
        else:
            # переходим к следующему шарду,
            # сбрасываем значения на дефолтные
            page = 1
            offset = 0

    next_shard_index = shards.index(shard) + 1
    if next_shard_index < len(shards):
        next_shard = shards[next_shard_index]
    else:
        next_shard = None

    # Если результатов меньше чем ожидалось, значит в этом шарде
    # больше нет интересующих нас данных и надо либо переходить к следующему,
    # либо сказать что данных больше нет.
    if len(results) < per_page + 1:
        if next_shard:
            next_page = 1
        else:
            # Если это последний шард, то линка на следующую
            # страницу быть не должно.
            next_page = None
    else:
        # Если в этом шарде есть ещё данные, то продолжаем итерироваться
        # по ним.
        next_shard = shard
        next_page = page + 1
        # Уберём последний элемент, так как мы
        # его запрашивали только для того, чтобы убедиться
        # есть следующая страница или нет.
        results = results[:per_page]

    return results, next_shard, next_page


def lang_for_notification(meta_connection, main_connection, uid, org_id):
    """
    Язык для отправки уведомлений пользователю
    :param main_connection:
    :param uid: id пользователя (может быть как внешним, так и админом на домене!)
    :return: язык для сообщений
    :rtype: str
    """
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        UserModel,
        UserMetaModel,
        OrganizationModel,
    )

    bb_info = get_user_data_from_blackbox_by_uid(uid)
    if bb_info is None:
        # Такое иногда случается, что учётку из Паспорта удаляют.
        # И тогда blackbox возвращает None.
        # В этих случаях мы наверное не хотим отправлять польователю нотификацию,
        # так как его всё равно нет.
        raise UserNotFoundError()

    if bb_info['language'] in app.config['ORGANIZATION_LANGUAGES']:
        return bb_info['language']

    user = UserModel(main_connection).get(user_id=uid, fields=['org_id'])
    if not user:
        user = UserMetaModel(meta_connection).get(user_id=uid, fields=['org_id'])
    if not user:
        # значит такого пользователя нет ни в main, ни в meta-базе и что-то пошло не так
        # вернем по умолчанию 'en' язык, чтобы не фейлить процессы миграции, синхронизацию доменов и др.
        return 'en'

    return OrganizationModel(main_connection).get(org_id, fields=['language'])['language']


def get_login(email):
    return email.split('@', 1)[0].lower()


def simple_get_from_bb_with_logs(bb_params, uids):
    bb_url = app.config['BLACKBOX']['url']
    bb_params['uid'] = ','.join(map(str, uids))
    try:
        response = requests.get(bb_url, params=bb_params)
        return response.json()['users']
    except Exception:
        with log.name_and_fields('simple_get_from_bb_failed',
                                 uids=uids,
                                 bb_params=bb_params):
            log.trace().error('Method userinfo from BlackBox return some error')
        return []


def grouper(iterable, n, fillvalue=None):
    """
    Collect data into fixed-length chunks or blocks"
    """
    args = [iter(iterable)] * n
    return zip_longest(fillvalue=fillvalue, *args)


# TODO: когда-нибудь выпилить в пользу blackbox_client.batch_userinfo
def get_userinfo_from_bb(uids, attributes, n=20):
    """
    Из метода userinfo ЧЯ отдаем подробные данные с необходимыми атрибутами для каждого uid-a.
    Если uid-ов очень много, то разделяем их на пачки по n(==20) и затем объединим полученные данные в один список.
    :return: iterable
    """

    all_attrs = ','.join(map(str, attributes))

    # здесь напрямую дергаем через requests, потому что один запрос лучше len(uids)
    bb_params = {
        'method': 'userinfo',
        'userip': get_localhost_ip_address(),
        'format': 'json',
        'attributes': all_attrs,
    }

    # fillvalue - значение, которым нужно заполнить список в случае, если кол-во uid-ов не строго делится на n.
    # В fillvalue передаем пустую строку, потому что в дальнейшем делаем преобразование вида map(str, iterator) для
    # запросов в ЧЯ
    results = (
        simple_get_from_bb_with_logs(bb_params, group_of_uids)
        for group_of_uids in grouper(uids, n, fillvalue='')
    )

    return chain(*results)


def change_object_alias(main_connection,
                        obj_id,
                        obj_type,
                        org_id,
                        alias,
                        action_name,
                        action_func,
                        author_id,
                        skip_connect=False):
    # добавляем/удаляем алиас пользователя, группы или отдела
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import AliasNotFound, AliasExists

    model = get_model_by_type(obj_type)
    obj = model(main_connection).get(obj_id, org_id)

    if not obj:
        return json_error_not_found()

    if obj_type == TYPE_USER:
        id_key = 'id'
        name_key = 'nickname'
    else:
        id_key = 'uid'
        name_key = 'label'

    old_aliases = obj['aliases']
    new_aliases = old_aliases

    if action_name == 'add':
        if alias in old_aliases and not skip_connect:
            log.info('Alias "{}" for user {} already exists in db'.format(alias, obj[id_key]))
            raise AliasExists
        try:
            app.passport.alias_add(obj[id_key], alias)
        except AliasExists:
            # Узнали что алиас существует, нужно убедится, что он существует именно
            # у этого пользователя
            master_domain = get_master_domain(main_connection, org_id, domain_is_required=False)
            user_id = get_user_id_from_passport_by_login(f'{alias}@{master_domain}')
            if user_id != obj[id_key]:
                log.info('Alias "{}" for user {} already exists in passport but for another user'.format(alias, obj[id_key]))
                raise
            log.info('Alias "{}" for user {} already exists in passport'.format(alias, obj[id_key]))

        new_aliases.append(alias)
        content = {'alias_added': alias}
    elif action_name == 'delete':
        try:
            app.passport.alias_delete(obj[id_key], alias)
        except AliasNotFound:
            with log.fields(alias=alias, obj_name=obj[name_key]):
                log.warning('Alias not found in Passport')
        if alias in new_aliases:
            new_aliases.remove(alias)
        content = {'alias_deleted': alias}
    updated_obj = obj
    if not skip_connect:
        updated_obj = model(main_connection).update(
            update_data={
                'aliases': new_aliases
            },
            filter_data={
                'id': obj_id,
                'org_id': org_id,
            }
        )
    action_params = {
        'main_connection': main_connection,
        'org_id': org_id,
        'author_id': author_id,
        'object_value': obj,
        'content': content
    }
    action_func(**action_params)
    return updated_obj


def create_root_department(
        main_connection,
        org_id,
        language,
        root_dep_label,
):
    # Создание корневого отдела и админской группы после создания организации
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        DepartmentModel,
        GroupModel,
        OrganizationServiceModel,
    )
    from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG

    english_root_dep_name = 'All employees'

    if language == 'ru':
        root_dep_name = {
            'ru': 'Все сотрудники',
            'en': english_root_dep_name
        }
    else:
        root_dep_name = {
            'ru': english_root_dep_name,
            'en': english_root_dep_name,
        }

    uid = None
    if OrganizationServiceModel(main_connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
        uid = create_maillist(main_connection, org_id, root_dep_label, ignore_login_not_available=True)

    # создать корневой департамент
    root_department = DepartmentModel(main_connection).create(
        id=ROOT_DEPARTMENT_ID,
        org_id=org_id,
        name=root_dep_name,
        label=root_dep_label,
        uid=uid,
    )

    # Создать группу - "Администраторы организации",
    # в которую внешний админ может добавить кого угодно
    GroupModel(main_connection).get_or_create_admin_group(org_id)

    # Создать группу заместители администраторов
    GroupModel(main_connection).get_or_create_deputy_admin_group(org_id)
    return root_department


# TODO: убрать эту проверку, как только перейдём на новый сервис рассылок DIR-1060
def check_label_length(label, is_cloud=False):
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import LoginLong
    if is_cloud:
        if len(label) > app.config['MAX_LABEL_CLOUD_LENGTH']:
            raise LoginLong()
    else:
        if len(label) > app.config['MAX_LABEL_LENGTH']:
            raise LoginLong()


def after_first_domain_became_confirmed(meta_connection, main_connection, org_id, domain_name, force=False):
    # Если домен подтвержден в организации, в которой ещё нет мастер-домена:
    # 1. делаем этот домен мастером
    # 2. обновляем признак подтвержденного домена в метабазе
    # 3. применяем пресеты
    # 4. Создаем рассылку на корневой отдел, проставляем ему uid и label
    # 5. проставляем название организации в паспорте

    from intranet.yandex_directory.src.yandex_directory.core.models import (
        OrganizationMetaModel,
        OrganizationModel,
        DomainModel,
        OrganizationServiceModel
    )
    from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG
    from intranet.yandex_directory.src.yandex_directory.core.models.preset import apply_preset
    from intranet.yandex_directory.src.yandex_directory.core.tasks import SetOrganizationNameInPassportTask
    from intranet.yandex_directory.src.yandex_directory.core.maillist.tasks import CreateMaillistTask

    with log.name_and_fields('domain_verify', org_id=org_id, domain=domain_name):
        log.info('First confirmed domain becomes a master')

        main_updated = DomainModel(main_connection).update(
            filter_data={
                'name': domain_name,
                'org_id': org_id,
            },
            update_data={
                'master': True,
                'display': True,
            },
            force=force
        )

        with log.name_and_fields('domain_verify', main_updated=main_updated):
            log.info('Master and display fields updated in main db')

        org_from_db = OrganizationModel(main_connection).get(
            id=org_id,
            fields=['preset', 'language', 'label'],
        )

        apply_preset(
            meta_connection,
            main_connection,
            org_id,
            org_from_db['preset'] or 'default'
        )
        log.info('Preset applied')

        apply_preset(
            meta_connection,
            main_connection,
            org_id,
            'enable-maillist',
        )
        log.info('Maillist preset applied')

        # проставляем название организации в паспорте
        # чтобы пользователи в этой организации были с атрибутом 1011
        # и списочным атрибутом 1017
        SetOrganizationNameInPassportTask(main_connection).delay(
            # Даём 15 секунд паспорту на то, чтобы информация про смену домена
            # доехала до всех реплик.
            start_in=15,
            org_id=org_id,
        )

    log.info('Updating ws domain')


def get_user_info_from_blackbox(uid, userip=None):
    """Возвращает основную информацию о пользователе:

    - полный логин, с доменной частью.
    - имя
    - фамилию
    - пол в виде строки 'male' или 'female' или None
    - дату рождения в виде datetime или None, если не получается распарсить
    """

    fields = [
        blackbox_client.FIELD_LOGIN,
        blackbox_client.FIELD_FIRSTNAME,
        blackbox_client.FIELD_LASTNAME,
        ('userinfo.sex.uid', 'sex'),
        ('userinfo.birth_date.uid', 'birth_date'),
    ]
    if userip is None:
        userip = get_localhost_ip_address()

    response = app.blackbox_instance.userinfo(
        uid=uid,
        userip=userip,
        dbfields=fields,
        attributes=blackbox_client.CLOUD_UID_ATTRIBUTE,
        emails='getdefault'
    )
    return prepare_data_from_blackbox_response(response)


def prepare_data_from_blackbox_response(bb_response):
    from intranet.yandex_directory.src.yandex_directory.passport.exceptions import BirthdayInvalid

    uid = int(bb_response['uid'])

    fields = bb_response['fields']
    gender_map = {
        '1': 'male',
        '2': 'female',
    }
    try:
        birthday = parse_birth_date(fields.get('birth_date'))
    except BirthdayInvalid:
        birthday = None

    first_name = fields.get('first_name') or fields.get('firstname')
    if not first_name and is_cloud_uid(uid):
        first_name = bb_response.get('attributes', {}).get(blackbox_client.FIRST_NAME_ATTRIBUTE) or \
            bb_response.get('display_name', {}).get('name')

    last_name = fields.get('last_name') or fields.get('lastname')
    if not last_name and is_cloud_uid(uid):
        last_name = bb_response.get('attributes', {}).get(blackbox_client.LAST_NAME_ATTRIBUTE)

    cloud_uid = None
    if is_cloud_uid(uid):
        cloud_uid = bb_response.get('attributes', {}).get(blackbox_client.CLOUD_UID_ATTRIBUTE)

    return (
        fields.get('login'),
        first_name,
        last_name,
        gender_map.get(fields.get('sex')),
        birthday,
        bb_response['default_email'] or '',
        cloud_uid,
    )


def add_existing_user_with_lock(meta_connection,
                                main_connection,
                                org_id,
                                user_id,
                                is_sso=False,
                                set_org_id_sync=False,
                                ):
    with wait_lock(main_connection, f'create_user_{org_id}_{user_id}'):
        add_existing_user(meta_connection, main_connection, org_id, user_id, is_sso=is_sso, set_org_id_sync=set_org_id_sync)


def add_existing_user(meta_connection,
                      main_connection,
                      org_id,
                      user_id,
                      cloud_uid=None,
                      author_id=None,
                      department_id=ROOT_DEPARTMENT_ID,
                      set_org_id_sync=False,
                      cloud_user_data=None,
                      is_sso=False,
                      ):
    """Добавляет в организацию пользователя, который уже есть в паспорте.
    """
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        UserModel,
        UserMetaModel,
    )
    from intranet.yandex_directory.src.yandex_directory.core.actions import action_user_add
    from intranet.yandex_directory.src.yandex_directory.core.tasks import SyncExternalIDS

    # Сперва выясним его имя
    # Имя и фамилию
    if cloud_user_data:
        login = cloud_user_data['login']
        email = cloud_user_data['email']
        first = cloud_user_data['first']
        last = cloud_user_data['last']
        gender = None
        birthday = None
    else:
        (login, first, last, gender, birthday, email, cloud_uid_from_bbemu) = get_user_info_from_blackbox(user_id)
        if cloud_uid is None:
            cloud_uid = cloud_uid_from_bbemu

    nickname = login
    if not is_outer_uid(user_id) and not is_cloud_uid(user_id):
        from intranet.yandex_directory.src.yandex_directory.core.utils.domain import get_domains_from_db_or_domenator, DomainFilter
        nickname, domain_part = login.split('@')
        domain_info = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            main_connection=main_connection,
            domain_filter=DomainFilter(org_id=org_id, name=domain_part, owned=True),
            one=True,
        )
        if not domain_info:
            raise RuntimeError('domain not found in organization')

    if is_domain_uid(user_id):
        check_label_or_nickname_or_alias_is_uniq_and_correct(
            main_connection,
            nickname,
            org_id,
            user_id,
            email=email,
            is_cloud=is_cloud_uid(user_id)
        )

    # Сначала вставим его в метабазу
    # Такие яндексовые учетки (не перенесенные как внешние админы из пдд) считаются внутренними
    UserMetaModel(meta_connection) \
        .create(
            id=user_id,
            org_id=org_id,
            is_outer=False,
            cloud_uid=cloud_uid,
        )

    # А затем и в основную
    UserModel(main_connection) \
        .create(
            id=user_id,
            nickname=nickname,
            name={'first': first or login, 'last': last or ('' if is_cloud_uid(user_id) else login)},
            email=email,
            gender=gender,
            org_id=org_id,
            department_id=department_id,
            birthday=birthday,
            cloud_uid=cloud_uid,
            is_sso=is_sso,
        )
    # Надо получить пользователя с департаментом, чтобы передать его в событие,
    # чтобы обновился счетчик количества пользователей в отделе
    user = UserModel(main_connection).get(
        user_id=user_id,
        org_id=org_id,
        fields=[
            '*',
            'department.*',
        ]
    )
    if not author_id:
        author_id = user_id

    action_user_add(
        main_connection,
        org_id=org_id,
        author_id=author_id,
        object_value=user
    )
    if set_org_id_sync:
        attr = 'do'
    else:
        attr = 'delay'
    getattr(SyncExternalIDS(main_connection), attr)(
        org_id=org_id,
        user_id=user_id,
        mix_org_id_in=True,
    )
    return user


def unfreeze_or_copy(data):
    """Превращает FrozenDict в обычный словарь, или копирует обычный словарь.
       Нужно использовать перед модификацией словаря, поступившего на вход функции.

       Тут выполняется неполное копирование!
    """
    if isinstance(data, frozendict):
        return dict(data)
    if isinstance(data, dict):
        return data.copy()
    return data


def is_member(main_connection, org_id, uid):
    from intranet.yandex_directory.src.yandex_directory.core.models import UserModel
    return UserModel(main_connection) \
        .filter(org_id=org_id, id=uid) \
        .count() > 0


def walk_department(main_connection,
                    org_id,
                    department_id,
                    fields=['id'], # поля которые нужно запрашивать у отделов
                    on_department=lambda arg: None):
    """Идёт по всем вложенным отделам и зовёт на них on_department.
    """
    from intranet.yandex_directory.src.yandex_directory.core.models import DepartmentModel

    path = '*.{}.*{{1,}}'.format(department_id)
    departments = DepartmentModel(main_connection) \
        .filter(
            path__ltree_match=path,
            org_id=org_id,
        ) \
        .fields(*fields)
    for dep in departments:
        on_department(dep)


def walk_group(main_connection,
               org_id,
               group_id,
               # Сейчас выборка членов группы так устроена, что для них отдаются
               # все простые поля. Так что ограничить список требуемых полей можно только
               # для вложенных подотделов
               department_fields=['id'], # поля которые нужно запрашивать у отделов
               on_group=lambda group: None,
               on_department=lambda department: None):
    """Рекурсивно идёт по всем вложенным в группу группам и отделам.
       На подгруппах вызывает функцию on_group, на отделах on_department.
    """
    from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel, GROUP_TYPE_GENERIC

    group_members = GroupModel(main_connection) \
        .filter(
            org_id=org_id,
            id=group_id,
            type=GROUP_TYPE_GENERIC,
        ) \
        .fields('members.*') \
        .one()

    for child in group_members['members']:
        if child['type'] == 'department':
            on_department(child['object'])
            walk_department(
                main_connection,
                org_id,
                child['object']['id'],
                fields=department_fields,
                on_department=on_department,
            )
        if child['type'] == 'group':
            on_group(child['object'])
            walk_group(
                main_connection,
                org_id,
                child['object']['id'],
                department_fields=department_fields,
                on_group=on_group,
                on_department=on_department,
            )


def print_departments(main_connection,
                      org_id,
                      department_id=1):
    """Выводит на экран структуру отделов"""

    from collections import defaultdict
    from intranet.yandex_directory.src.yandex_directory.core.models import DepartmentModel
    children = defaultdict(list)

    def collector(obj):
        children[obj['parent_id']].append(obj)

    fields = ['name', 'label', 'uid', 'parent_id']

    department = DepartmentModel(main_connection) \
        .filter(org_id=org_id, id=department_id) \
        .fields(*fields) \
        .one() \

    collector(department)

    walk_department(
        main_connection,
        org_id,
        department_id,
        fields=fields,
        on_department=collector,
    )

    def printer(parent_id, padding=0):
        current_id = None
        for child in children.get(parent_id, []):
            print('  ' * padding, end=' ')
            print(child['name']['ru'], end=' ')
            if child['label']:
                print('(' + child['label'] + '@', str(child['uid']) + ')')
            else:
                print('')
            printer(child['id'], padding=padding+1)

    printer(department['parent_id'])


def check_organizations_limit(meta_connection, uids, org_to_skip=None, ):
    query = '''
        select id, count(*) from users
        where is_dismissed = False
        and id in %(uids)s
        {org_to_skip}
        group by id
    '''
    if isinstance(uids, (str, int)):
        uids = [uids]
    uids = tuple(map(int, uids))  # в базе uid - число
    if uids:
        if org_to_skip:
            format_org = f'and org_id != {org_to_skip}'
        else:
            format_org = ''

        query = query.format(org_to_skip=format_org)
        result = meta_connection.execute(query, {'uids': uids, }).fetchall()
        for row in result:
            if row['count'] >= settings.USER_ORGANIZATIONS_LIMIT:
                raise UserOrganizationsLimit(user_id=row['id'])
    return True


def is_email_valid(email):
    pattern = re.compile('(^|\s)[-a-z0-9_.\+]+@([-a-z0-9]+\.)+[a-z]{2,6}(\s|$)')
    return pattern.match(email)
