# -*- coding: utf-8 -*-
import os
from functools import wraps

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationServicesAnalyticsInfoModel,
    OrganizationsAnalyticsInfoModel,
    UsersAnalyticsInfoModel,
    DomainsAnalyticsInfoModel,
)
from intranet.yandex_directory.src.yandex_directory.common.yt import utils as yt_utils
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow, format_date
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard_numbers,
    get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.step.client import STEPApiClient
from intranet.yandex_directory.src.yandex_directory.core.task_queue import Task

BATCH_SIZE = 100000


class SendStepEventTask(Task):
    default_priority = 50
    org_id_is_required = False

    def do(self, table):
        STEPApiClient().create_cluster_table_publish_events(table)


def send_step_event(table):
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            result = func(*args, **kwargs)
            # ставим таск на отправку событий в STEP
            shard = get_shard_numbers()[0]
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                SendStepEventTask(main_connection).delay(table=table)
            return result

        return wrapper
    return decorator


class AnalyticsToYtSaver(object):
    def __init__(self, tables_path=None):
        self.tables_path = tables_path or app.config['YT_ANALYTICS_TABLES_PATH']
        self.now_date = utcnow().date()
        self.shards = get_shard_numbers()
        # маппит названия локальных таблиц с аналитикой в функции,
        # которые коипруют эти данные в yt
        self.local_table_map = {
            'domains_analytics_info': self.save_domains_stats_data_to_yt,
            'organizations_services_analytics_info': self.save_org_services_stats_to_yt,
            'organizations_analytics_info': self.save_organizations_stats_data_to_yt,
            'users_analytics_info': self.save_users_stats_data_to_yt,
        }

    def save(self, for_date=None, recreate=False, local_table=None):
        """
        Выгружаем аналитические данные в Yt (DIR-2895):

        По организациям:
            id организации
            дата регистрации организации
            источник регистрации организации (в первую очередь реклама/не реклама).
            статус платности = False  todo: сохранять статус платности
            включенные сервисы
            страна
            lang
            tld
            баланс
            дата первого неоплаченного акта

        По пользователям:
            id организации
            uid
            дата регистрации

        По провалидированным доменам:
            org_id
            name

        По платным сервисам:
            id организации
            слаг
            дата окончания триала
            ready at
            включен/выключен
            причина отключения
            дата включения
            дата выключения

        """

        with log.name_and_fields('analytics_saver', tables_path=self.tables_path):
            log.info('Starting saving analytics data')
            if local_table in self.local_table_map:
                self.local_table_map.get(local_table)(for_date, recreate)
            elif local_table is None:
                self.save_organizations_stats_data_to_yt(for_date, recreate)
                self.save_domains_stats_data_to_yt(for_date, recreate)
                self.save_org_services_stats_to_yt(for_date, recreate)
                self.save_users_stats_data_to_yt(for_date, recreate)
            else:
                return

            log.info('Analytics data has been saved')

    @send_step_event(table='organizations')
    def save_organizations_stats_data_to_yt(self, for_date=None, recreate=False):
        table = 'organizations'
        model = OrganizationsAnalyticsInfoModel
        if not for_date:
            for_date = self.now_date

        schema = [
            {'name': 'id', 'type': 'int64'},
            {'name': 'registration_date', 'type': 'utf8'},
            {'name': 'subscription_plan', 'type': 'utf8'},
            {'name': 'services', 'type': 'any'},
            {'name': 'country', 'type': 'utf8'},
            {'name': 'language', 'type': 'utf8'},
            {'name': 'source', 'type': 'utf8'},
            {'name': 'tld', 'type': 'utf8'},
            {'name': 'balance', 'type': 'utf8'},
            {'name': 'first_debt_act_date', 'type': 'utf8'},
            {'name': 'for_date', 'type': 'utf8'},
            {'name': 'admins', 'type': 'any'},
            {'name': 'deputy_admins', 'type': 'any'},
            {'name': 'organization_type', 'type': 'any'},
            {'name': 'name', 'type': 'utf8'},
            {'name': 'vip', 'type': 'any'},
            {'name': 'feature_ids', 'type': 'any'},
        ]

        self._save_data_to_yt(table, model, schema, for_date, recreate)

    @send_step_event(table='users')
    def save_users_stats_data_to_yt(self, for_date=None, recreate=False):
        table = 'users'
        model = UsersAnalyticsInfoModel
        if not for_date:
            for_date = self.now_date

        schema = [
            {'name': 'org_id', 'type': 'int64'},
            {'name': 'uid', 'type': 'int64'},
            {'name': 'for_date', 'type': 'utf8'},
            {'name': 'created', 'type': 'utf8'},
            {'name': 'email', 'type': 'utf8'},
            {'name': 'licensed_services', 'type': 'any'},
        ]

        self._save_data_to_yt(table, model, schema, for_date, recreate)

    @send_step_event(table='domains')
    def save_domains_stats_data_to_yt(self, for_date=None, recreate=False):
        table = 'domains'
        model = DomainsAnalyticsInfoModel
        if not for_date:
            for_date = self.now_date

        schema = [
            {'name': 'org_id', 'type': 'int64'},
            {'name': 'name', 'type': 'utf8'},
            {'name': 'for_date', 'type': 'utf8'},
            {'name': 'master', 'type': 'boolean'},
        ]
        self._save_data_to_yt(table, model, schema, for_date, recreate)

    @send_step_event(table='org_services')
    def save_org_services_stats_to_yt(self, for_date=None, recreate=False):
        table = 'org_services'
        model = OrganizationServicesAnalyticsInfoModel
        if not for_date:
            for_date = self.now_date

        schema = [
            {'name': 'org_id', 'type': 'int64'},
            {'name': 'slug', 'type': 'utf8'},
            {'name': 'trial_expires', 'type': 'utf8'},
            {'name': 'ready_at', 'type': 'utf8'},
            {'name': 'enabled', 'type': 'boolean'},
            {'name': 'disable_reason', 'type': 'utf8'},
            {'name': 'enabled_at', 'type': 'utf8'},
            {'name': 'disabled_at', 'type': 'utf8'},
            {'name': 'for_date', 'type': 'utf8'},
        ]
        self._save_data_to_yt(table, model, schema, for_date, recreate)

    def _save_data_to_yt(self, table, model, schema, for_date, recreate):
        log.info('Saving %s data...' % table)
        table_path = self._get_analytics_table_path(table, for_date)
        filter_data = {'for_date': for_date}

        if not recreate and yt_utils.is_table_exists(table_path):
            log.info('Table for %s in the YT already exists' % table)
            return

        with yt_utils.yt_client.Transaction():
            if recreate and yt_utils.is_table_exists(table_path):
                yt_utils.remove_table(table_path)
            yt_utils.create_table_if_needed(table_path, schema)
            for shard in self.shards:
                with get_main_connection(shard=shard) as main_connection:
                    total_rows = model(main_connection).count(filter_data=filter_data)
                    for i in range(0, total_rows, BATCH_SIZE):
                        batch_info = model(main_connection).find(
                            filter_data=filter_data,
                            fields=['*'],
                            limit=BATCH_SIZE,
                            skip=i,
                        )
                        yt_utils.append_rows_to_table(
                            table=table_path,
                            rows_data=batch_info,
                        )

    def _get_analytics_table_path(self, table_name, for_date=None):
        if not for_date:
            for_date = self.now_date
        return os.path.join(
            self.tables_path,
            table_name,
            format_date(for_date, only_date=True),
        )

    def get_yt_folders_without_data(self):
        """
        Возвращает список папок YT, в которых нет таблицы с аналитическими данными за сегодня
        """
        clusters_without_data = []

        for folder_name in ('organizations', 'users', 'domains', 'org_services'):
            clusters_without_data += yt_utils.get_yt_clusters_without_table(
                table=self._get_analytics_table_path(folder_name),
                check_empty_tables=True,
                yt_clients=[yt_utils.yt_client],
            )

        return list(set(clusters_without_data))
