# -*- coding: utf-8 -*-

import contextlib
import threading


store = threading.local()


def _ensure_has_attr():
    if not hasattr(store, 'calls'):
        store.calls = []


def _remove_attr():
    if hasattr(store, 'calls'):
        delattr(store, 'calls')

        
def _assert_has_attr():
    if not hasattr(store, 'calls'):
        raise RuntimeError('Please, use add_call only inside calls_at_the_end context manager')

        
def add_call(func, *args, **kwargs):
    _assert_has_attr()
    store.calls.append((func, args, kwargs))


def _perform_calls():
    if hasattr(store, 'calls'):
        for func, args, kwargs in store.calls:
            func(*args, **kwargs)

        store.calls[:] = []


@contextlib.contextmanager
def calls_at_the_end():
    _ensure_has_attr()
    
    try:
        yield
    finally:
        # Второй finally нужен чтобы если по время
        # обработки отложенных вызовов произойдёт исключение,
        # то мы бы корректно подчистили за собой список вызовов
        try:
            _perform_calls()
        finally:
            _remove_attr()
