# -*- coding: utf-8 -*-
from datetime import datetime

from typing import Optional, Union, List

from retrying import retry

import time
from collections import defaultdict
from contextlib import contextmanager

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    log_exception,
    create_domain_in_passport,
    get_domain_info_from_blackbox,
    utcnow,
    force_text,
    delete_domain_in_passport,
    json_error,
)

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
    get_main_connection,
    get_shard,
    retry_on_close_connect_to_server,
)
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_email_to_all_async
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationModel,
    DomainModel,
    UserModel,
    DepartmentModel,
    GroupModel,
)

from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    AdminUidNotFound,
    DomainNotFound,
    ImmediateReturn,
    DuplicateDomain,
    MasterDomainNotFound,
)

from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_organization_admin_uid,
)

from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type
from intranet.yandex_directory.src.yandex_directory.core.resource_history.domain import save_domain_resource_history

from intranet.yandex_directory.src.yandex_directory.passport.exceptions import AccountDisabled, PassportException
from intranet.yandex_directory.src.yandex_directory.core.features import is_feature_enabled, USE_DOMENATOR


def sync_domains_with_webmaster(meta_connection, main_connection, org_id):
    """
    Синхронизируем статусы доменов в нашей базе со статусами в вебмастере.

    """
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        OrganizationMetaModel,
        DomainModel,
    )

    # Если организацию уже успели удалить в результате отката, или даже просто
    # начали откат и проставили ready=False, то её надо просто пропустить.
    org = OrganizationMetaModel(meta_connection).find(
        {'id': org_id},
        fields=['ready'],
        one=True,
    )
    if not org or not org['ready']:
        return

    with log.name_and_fields('sync_domains_state', org_id=org_id):
        domain_model = DomainModel(main_connection)
        # синкаем статусы доменов

        admin_id = get_organization_admin_uid(main_connection, org_id)
        domains = domain_model.find(filter_data={'org_id': org_id})

        for domain in domains:
            try:
                sync_domain_state(
                    meta_connection,
                    main_connection,
                    org_id,
                    admin_id,
                    domain,
                )
            except Exception as e:
                with log.fields(domain=domain):
                    msg = 'Error during sync domain state'
                    log_exception(e, msg)


def sync_domain_state(meta_connection,
                      main_connection,
                      org_id,
                      admin_id,
                      domain):
    """Возвращает True, если домен подтверждён или стал подтверждён в результате выполнения функции."""
    from intranet.yandex_directory.src.yandex_directory import webmaster
    domain_name = domain['name']

    with log.fields(
            org_id=org_id,
            domain=domain_name,
    ):
        if domain_name in app.config['YANDEX_TEAM_ORG_DOMAINS']:
            log.info('Skipping yandex-team domain')
            return

        webmaster.update_domain_state_if_verified(
            meta_connection,
            main_connection,
            org_id,
            admin_id,
            domain,
            send_sms=True,
        )
        # заново получим домен, вдруг у него обновился статус
        domain = DomainModel(main_connection).get(domain_name, org_id)

        if 'validation-error-42' in domain_name:
            # Это нам нужно специально для того, чтобы вручную тестировать случай, когда
            # на этапе валидации домена происходит ошибка:
            # https://st.yandex-team.ru/DIR-6906
            raise RuntimeError('This test domain can\'t be validated')

        return domain['owned']


def assert_can_add_invalidated_alias(meta_connection, main_connection, org_id, alias):
    """
    Проверяем, может ли админ с uid добавить неподтвержденный alias.
    :param: org_id - ид организации
    :param: alias - имя проверяемого домена
    :return: True

    Пытаемся предотвратить случай, когда один админ добавляет к себе в
    несколько организаций один и тот же домен, новый, еще не подтвержденный.

    https://st.yandex-team.ru/DIR-2156

    Если всё хорошо, и домен добавить можно, то функция просто выходит.
    Если нет, то выбрасывает исключение ImmediateReturn с кодом 409 и объяснением
    того, что произошло.
    В параметрах ошибки будет параметр conflicting_org_id - это id той организации
    пользователя, в которой уже есть этот домен.

    """
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        DomainModel,
        UserMetaModel,
        OrganizationMetaModel,
    )

    admin_uid = get_organization_admin_uid(main_connection, org_id)

    # находим все организации у основного админа
    organizations_by_shards = defaultdict(set)
    usermeta_model = UserMetaModel(meta_connection)
    for u in usermeta_model.filter(id=admin_uid, is_dismissed=False).fields('organization.shard', 'org_id').all():
        shard = u['organization']['shard']
        organizations_by_shards[shard].add(u['org_id'])

    for shard, org_ids in organizations_by_shards.items():
        with get_main_connection(shard=shard) as main_connection:
            # организация должна быть со статусом ready
            ready_org_ids = OrganizationMetaModel(meta_connection)\
                .filter(id=org_ids, ready=True) \
                .fields('id') \
                .scalar()
            if not ready_org_ids:
                continue

            # ищем все домены в этих организациях
            found_aliases = DomainModel(main_connection) \
                          .filter(org_id=ready_org_ids, name=alias) \
                          .fields('org_id') \
                          .all()

            # убираем ту же самую организацию
            found_aliases = list([cur_alias for cur_alias in found_aliases if cur_alias['org_id'] != org_id])

            if found_aliases:
                raise DuplicateDomain(
                    conflicting_org_id=found_aliases[0]['org_id'],
                )
    return True


def generate_tech_domain(domain_name, org_id):
    # генерирует технический домен для организации <domain_name>-<org_id>.yaconnect.com
    # в имени точки заменяет на дефис

    return '{domain_name}-{org_id}{domain_part}'.format(
        domain_name=force_text(
            domain_name.replace('.', '-'),
        ),
        org_id=org_id,
        domain_part=force_text(
            app.config['DOMAIN_PART']
        )
    )


@contextmanager
def unblock_domains(org_id=None, connection=None):
    try:
        yield
    finally:
        log.debug('Unblock domains')
        with connection.begin_nested():
            DomainModel(connection).update(
                update_data={'blocked_at': None},
                filter_data={'org_id': org_id},
                force=True
            )


@retry(stop_max_attempt_number=3,
       wait_incrementing_increment=50,
       retry_on_exception=retry_on_close_connect_to_server)
def _disable_domain_in_organization(domain, new_owner_uid):
    org_id = domain['org_id']
    with get_meta_connection() as meta_connection:
        shard = get_shard(meta_connection, org_id)

    with get_main_connection(shard=shard, for_write=True, no_transaction=True) as main_connection:
        domain_name = domain['name']
        new_master = None
        tech = False
        admin_uid = get_organization_admin_uid(main_connection, org_id)

        with main_connection.begin_nested():
            # перед началом передачи нужно заблокировать все домены в организации, т.к. мы будем менять мастер домен
            log.debug('Block domains')
            DomainModel(main_connection).update(
                update_data={
                    'blocked_at': utcnow()
                },
                filter_data={
                    'org_id': org_id,
                    'owned': True
                }
            )

        with unblock_domains(connection=main_connection, org_id=org_id):
            # если данный домен мастер, то
            if domain['master']:
                # если есть другие подтвержденные домены, то делаем один из них мастером
                other_owned_domain = DomainModel(main_connection).find(
                    filter_data={
                        'org_id': org_id,
                        'owned': True,
                        'name__notequal': domain_name,
                    },
                    one=True
                )
                if other_owned_domain:
                    with log.fields(new_master=other_owned_domain['name']):
                        log.debug('Found other owned domain')
                    new_master = other_owned_domain['name']
                else:
                    # если нет других подтвержденных доменов, то заводим технический домен и делаем его мастером
                    tech = True
                    with main_connection.begin_nested():
                        new_master = generate_tech_domain(domain_name, org_id)
                        DomainModel(main_connection).create(
                            new_master,
                            org_id,
                            owned=True,
                        )
                        with log.fields(new_master=new_master):
                            log.debug('Use tech domain')

                        # добавим технический домен в паспорт как алиас
                        create_domain_in_passport(
                            main_connection,
                            org_id=org_id,
                            punycode_name=new_master,
                            admin_uid=admin_uid,
                        )
                        time.sleep(5)  # даем время репликам паспорта
                        new_master_domain = DomainModel(main_connection).get(domain_name=new_master, org_id=org_id)
                        from intranet.yandex_directory.src.yandex_directory.core.actions import action_domain_add
                        action_domain_add(
                            main_connection,
                            org_id=org_id,
                            author_id=admin_uid,
                            object_value=new_master_domain,
                            old_object=new_master_domain,
                        )
                log.debug('Change master domain')
                # меняем мастер домен (в базе и в паспорте)
                with main_connection.begin_nested():
                    DomainModel(main_connection).change_master_domain(org_id=org_id, domain_name=new_master, force=True)
                time.sleep(5)  # даем время репликам паспорта

            # удаляем домен
            log.debug('Remove domain')
            with main_connection.begin_nested():
                DomainModel(main_connection).delete_domain(domain_name, org_id, admin_uid, delete_blocked=True)
            save_domain_resource_history(domain_name, org_id, 'domain_auto_handover', admin_uid)

        # ставим задачи на отправку писем каждому админу организации, о том, что домен был оторван
        log.debug('Create mail send task')
        campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['DISABLE_DOMAIN_EMAIL']
        with main_connection.begin_nested():
            send_email_to_all_async(
                main_connection,
                org_id=org_id,
                domain=domain_name,
                campaign_slug=campaign_slug,
                new_master=new_master,
                tech=tech,
            )

            # ставим таску на перепроверку для старого владельца, чтобы оторвать у него права в вебмастере
            from intranet.yandex_directory.src.yandex_directory.core.tasks import ResetDomainVerification
            from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
            try:
                ResetDomainVerification(main_connection).delay(
                    domain_name=domain_name,
                    new_owner_uid=new_owner_uid,
                    old_owner_uid=admin_uid,
                    org_id=domain['org_id'],
                )
            except DuplicatedTask:
                pass


def disable_domain_in_organization(domain_name, org_id, new_owner_uid):
    # Открепляем домен от организации, где он подтвержден
    # ищем по всем шардам подтвержденный домен с именем domain_name
    # делаем его неподтвержденным
    # и если он был мастером, то делаем другой подтвержденный мастером или, если его нет,
    # то генерируем технический.
    #
    # Тут org_id это org_id той организации, в которой домен только что подтвердился.
    # Он нам нужен только для того, чтобы проверить, включена ли у неё фича автоотрыва домена.

    # ищем по всем шардам подтвержденный домен с именем domain_name
    domain = DomainModel(None).find(
        filter_data={
            'name': domain_name,
            'owned': True,
            # Запрещаем авто-отрыв домена в той же организации
            'org_id__notequal': org_id,
        },
        one=True,
    )

    if not domain:
        return

    # Запрещаем авто-отрыв домена в той же организации
    if domain['org_id'] != org_id:
        with get_meta_connection(for_write=True) as meta_connection:
            shard = get_shard(meta_connection, domain['org_id'])
            with get_main_connection(shard=shard) as main_connection:
                if OrganizationModel(main_connection).get_organization_type(domain['org_id']) == organization_type.portal:
                    # запрещаем отрывать домены у порталов
                    raise RuntimeError('Disable domain in portal is prohibited')

        with log.name_and_fields('disable_domain_in_organization', org_id=domain['org_id'], domain=domain['name']):
            log.info('Disabling domain in organization')
            _disable_domain_in_organization(domain, new_owner_uid)


def delete_domain_with_accounts(main_connection, org_id, check_domain_users=True):
    from intranet.yandex_directory.src.yandex_directory.core.utils import is_outer_uid

    with log.name_and_fields('delete_domain_with_accounts', org_id=org_id), get_meta_connection() as meta_connection:
        try:
            master_domain = get_master_domain_from_db_or_domenator(
                org_id=org_id,
                meta_connection=meta_connection,
                main_connection=main_connection,
            )
        except DomainNotFound:
            log.info('Master domain was not found in DB')
            return

        domain_info = get_domain_info_from_blackbox(master_domain['name'])
        if not domain_info:
            log.info('Master domain was not found in blackbox')
            return
        master_domain_id = domain_info['domain_id']

        admin_uid = get_organization_admin_uid(main_connection, org_id)
        if not admin_uid:
            log.error('Admin uid for organization was not found')
            raise AdminUidNotFound
    registrar_id = OrganizationModel(main_connection).filter(id=org_id).fields('registrar_id').one()['registrar_id']

    # проверяем учетки которые есть в паспорте
    robots = UserModel(main_connection).filter(
        is_robot=True,
        org_id=org_id,
        is_dismissed=False
    ).fields('id').scalar()

    remained_domain_uids = set([u for u in robots if not is_outer_uid(u)])

    inner_admin_uid = None
    if not is_outer_uid(admin_uid):
        inner_admin_uid = admin_uid

    dep_uids = set(DepartmentModel(main_connection).filter(
        org_id=org_id,
        uid__isnull=False,
        removed=False
    ).fields('uid').scalar())

    gr_uids = set(GroupModel(main_connection).filter(
        org_id=org_id,
        uid__isnull=False,
        removed=False
    ).fields('uid').scalar())
    remained_domain_uids = remained_domain_uids.union(dep_uids).union(gr_uids)

    uids_from_passport = set(app.blackbox_instance.account_uids(domain=master_domain['name']))

    with log.name_and_fields('delete_domain_with_accounts',
                             org_id=org_id,
                             master_domain=master_domain['name'].lower(),
                             uids_from_db=remained_domain_uids,
                             uids_from_passport=uids_from_passport,
                             registrar_id=registrar_id,
                             ):
        if check_domain_users:
            # проверяем, что мы собираемся удалить только роботов, рассылки и владельца домена
            if not uids_from_passport.issubset(remained_domain_uids.union({admin_uid})):
                log.error('Unable to delete domain with accounts')
                raise ImmediateReturn(
                    json_error(
                        422,
                        'unable_to_delete_domain_with_accounts',
                        'Unable to delete domain with accounts',
                    )
                )

        # удаляем все алиасы мастер-домена
        if not is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            for alias in DomainModel(main_connection).get_aliases(org_id):
                with log.name_and_fields('delete_domain_with_accounts',
                                         org_id=org_id,
                                         domain_alias=alias['name'].lower()
                                         ):
                    _delete_domain_when_delete_org(
                        main_connection,
                        org_id,
                        admin_uid,
                        alias['name'],
                        registrar_id,
                        master_domain_id=master_domain_id,
                        is_alias=True,
                    )
        if domain_info['blocked']:
            # разблокируем домен иначе не даст удалить учетки
            app.passport.unblock_domain(master_domain_id)
            # и аккаунт того кто удаляет тоже
            if inner_admin_uid:
                app.passport.unblock_user(inner_admin_uid)

        # удаляем из паспорта роботов и рассылку all
        log.info('Trying to delete accounts')
        _delete_users(*remained_domain_uids)
        log.info('Accounts has been deleted')

        if inner_admin_uid:
            app.passport.unset_pdd_admin(inner_admin_uid)
            app.passport.account_delete(inner_admin_uid)

        if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            app.domenator.delete_all_domains(org_id)
        else:
            # удаляем мастер-домен
            for _ in range(5):
                try:
                    _delete_domain_when_delete_org(
                        main_connection,
                        org_id,
                        admin_uid,
                        master_domain['name'],
                        registrar_id,
                    )
                    break
                except PassportException as exc:
                    if exc.error_code == 'domain.users_remaining':
                        # остались еще пользователи
                        if check_domain_users:
                            raise
                        # нужно их поудалять
                        uids_from_passport = set(app.blackbox_instance.account_uids(domain=master_domain['name']))
                        _delete_users(*uids_from_passport)
                        # даем репликам паспорта посинкаться
                        time.sleep(3)
                    else:
                        raise


def _delete_users(*uids):
    for uid in uids:
        for _ in range(3):
            try:
                app.passport.account_delete(uid)
            except AccountDisabled:
                app.passport.unblock_user(uid)
                # дадим репликам паспорта прососаться
                time.sleep(0.5)


def _delete_domain_when_delete_org(
        main_connection,
        org_id,
        admin_uid,
        domain_name,
        registrar_id,
        master_domain_id=None,
        is_alias=False,
):
    # Удаление домена или алиаса из паспорта и базы директории.
    # Специальный метод, используется ТОЛЬКО при удалении организации.
    # Если удялем алиас - надо передать master_domain_name и master_domain_id и флаг is_alias = True
    # Если удаляем мастер - master_domain_name и master_domain_id передавать не надо
    log.info('Trying to delete domain')
    delete_domain_in_passport(domain_name,
                              admin_uid,
                              is_alias,
                              master_domain_id)

    is_owned = bool(DomainModel(main_connection).filter(name=domain_name, org_id=org_id, owned=True).one())

    DomainModel(main_connection).delete(
        {'name': domain_name, 'org_id': org_id},
        force_remove_all=True,
    )
    log.info('Domain has been deleted from db')
    save_domain_resource_history(domain_name, org_id, 'domain_deleted_with_organization', admin_uid)

    # если домен от регистратора, ставим таск на отправку колбека
    if not registrar_id or not is_owned:
        return

    from intranet.yandex_directory.src.yandex_directory.core.registrar.tasks import DomainDeleteCallbackTask
    DomainDeleteCallbackTask(main_connection).delay(registrar_id=registrar_id, domain_name=domain_name)


def delete_domain_from_passport(main_connection, org_id):
    # функция используется для удаления доменов,
    # если у организации не стоит флаг has_owned_domains,
    # но домены при этом могли прорасти в паспорте
    aliases = {}
    master_domains = {}

    with log.name_and_fields('delete_domain_from_passport', org_id=org_id):
        admin_uid = get_organization_admin_uid(main_connection, org_id)
        if not admin_uid:
            log.error('Admin uid for organization requests was not found')
            raise AdminUidNotFound
        registrar_id = OrganizationModel(main_connection).filter(id=org_id).fields('registrar_id').one()['registrar_id']

        # т.к. из базы нельзя понять, кто мастер, придется делать это через бб
        for domain in DomainModel(main_connection).filter(org_id=org_id).all():
            domain_name = domain['name']
            domain_info = get_domain_info_from_blackbox(domain_name)
            if domain_info and domain_info['admin_id'] == admin_uid:
                # проверим, что больше нет организаций в коннекте с такими доменом,
                # если есть, то удалять нельзя
                if len(DomainModel(None).find({
                    'name': domain_name,
                    'org_id__notequal': org_id,
                    'owned': True
                })) > 0:
                    continue

                if domain_info['master_domain']:
                    aliases[domain_name] = domain_info['master_domain']
                else:
                    master_domains[domain_info['domain_id']] = domain_name

        # удаляем все алиасы мастера, который принадлежит той же организации
        for alias_name, master_id in aliases.items():
            if master_id in master_domains:
                with log.fields(domain_alias=alias_name, master_domain=master_domains[master_id]):
                    _delete_domain_when_delete_org(
                        main_connection,
                        org_id,
                        admin_uid,
                        alias_name,
                        registrar_id,
                        master_domain_id=master_id,
                        is_alias=True,
                    )

        for master_id, master_name in master_domains.items():
            # удаляем из паспорта все учетки
            uids = app.blackbox_instance.account_uids(domain=master_name)
            with log.fields(uids=uids):
                log.info('Trying to delete accounts')
                for uid in uids:
                    app.passport.account_delete(uid)
                log.info('Accounts has been deleted')

            # удаляем мастер-домен
            with log.fields(master_domain=master_name):
                _delete_domain_when_delete_org(
                    main_connection,
                    org_id,
                    admin_uid,
                    master_name,
                    registrar_id,
                )


class DomainFilter:
    def __init__(
        self,
        org_id: Optional[Union[int, List[int]]] = None,
        name: Optional[str] = None,
        owned: Optional[bool] = None,
        master: Optional[bool] = None,
        display: Optional[bool] = None,
    ):
        if org_id is None and name is None:
            raise ValueError('Specify org_id or name')
        self.org_id = org_id
        self.name = name
        self.owned = owned
        self.master = master
        self.display = display

    def org_list(self):
        if self.org_id is None:
            return []
        if isinstance(self.org_id, list):
            return self.org_id
        return [self.org_id]

    def dict(self):
        fields = {
            'org_id': self.org_id,
            'name': self.name,
            'owned': self.owned,
            'master': self.master,
            'display': self.display,
        }
        return {
            k: v
            for k, v in fields.items()
            if v is not None
        }


def get_domains_from_db_or_domenator(
    meta_connection,
    domain_filter: DomainFilter,
    main_connection=None,
    one=False,
    exc_on_empty_result=None,
):
    org_ids = domain_filter.org_list()

    # если не передан список организаций ищем в коннекте, и, если ничего не нашли, то потом в доменаторе
    if not org_ids:
        domains = DomainModel(main_connection).find(filter_data=domain_filter.dict(), one=one)
        if not domains:
            domains = app.domenator.private_get_domains(**domain_filter.dict())
        if not isinstance(domains, list):
            domains = [domains]
    else:
        connect_org_ids = []
        domenator_org_ids = []
        for org_id in org_ids:
            if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
                domenator_org_ids.append(org_id)
            else:
                connect_org_ids.append(org_id)

        domains = []
        if connect_org_ids:
            filter_data = domain_filter.dict()
            filter_data['org_id'] = connect_org_ids
            connect_domains = DomainModel(main_connection).find(filter_data=domain_filter.dict(), one=one)
            if isinstance(connect_domains, list):
                domains.extend(connect_domains)
            elif connect_domains:
                domains.append(connect_domains)
        if domenator_org_ids:
            filter_data = domain_filter.dict()
            filter_data['org_id'] = domenator_org_ids
            domenator_domains = app.domenator.private_get_domains(**filter_data)
            domains.extend(domenator_domains)

    if exc_on_empty_result is not None and not domains:
        raise exc_on_empty_result

    if domains:
        if one:
            return domains[0]
        return domains

    if one:
        return None
    return []


def get_master_domain_from_db_or_domenator(org_id, meta_connection, main_connection=None, raise_on_empty_result=True):
    exc_on_empty_result = None
    if raise_on_empty_result:
        exc_on_empty_result = MasterDomainNotFound(org_id=org_id)
    return get_domains_from_db_or_domenator(
        meta_connection=meta_connection,
        domain_filter=DomainFilter(org_id=org_id, master=True),
        main_connection=main_connection,
        one=True,
        exc_on_empty_result=exc_on_empty_result,
    )


class DomainUpdateFilter:
    def __init__(
        self,
        org_id: int,
        name: Optional[str] = None,
        owned: Optional[bool] = None,
        master: Optional[bool] = None,
        display: Optional[bool] = None,
    ):
        self.org_id = org_id
        self.name = name
        self.owned = owned
        self.master = master
        self.display = display

    def dict(self):
        fields = {
            'org_id': self.org_id,
            'name': self.name,
            'owned': self.owned,
            'master': self.master,
            'display': self.display,
        }
        return {
            k: v
            for k, v in fields.items()
            if v is not None
        }


class DomainUpdateData:
    def __init__(
        self,
        master: Optional[bool] = None,
        owned: Optional[bool] = None,
        display: Optional[bool] = None,
        blocked_at: Optional[datetime] = None,
    ):
        self.master = master
        self.display = display
        self.owned = owned
        self.blocked_at = blocked_at

    def dict(self):
        fields = {
            'master': self.master,
            'display': self.display,
            'owned': self.owned,
            'blocked_at': self.blocked_at,
        }
        return {
            k: v
            for k, v in fields.items()
            if v is not None
        }


def update_domains_in_db_or_domenator(
    meta_connection,
    update_filter: DomainUpdateFilter,
    update_data: DomainUpdateData,
    main_connection=None,
    force=False,
):
    org_id = update_filter.org_id
    if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
        app.domenator.private_patch_domains(
            data=update_data.dict(),
            **update_filter.dict(),
        )
    else:
        DomainModel(main_connection).update(
            update_data=update_data.dict(),
            filter_data=update_filter.dict(),
            force=force,
        )


def domain_is_tech(domain_name: str):
    return domain_name.endswith(app.config['DOMAIN_PART'])
