# -*- coding: utf-8 -*-
import calendar
from datetime import datetime

from dateutil.relativedelta import relativedelta

from intranet.yandex_directory.src.yandex_directory.core.utils.services import (
    get_services,
)
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import get_main_connection, get_meta_connection, get_shard


def get_organizations_services(meta_connection, main_connection, org_ids):
    """
    Возвращает словарь org_id -> [service, service,...].

    Этой функции обязательно нужен main_connection и в списке org_ids
    должны быть только организации из того самого шарда, к которому установлен
    коннект.

    Если в организации нет подключенных сервисов, то для неё в результирующем
    словаре будет пустой список.

    :param meta_connection:
    :param main_connection:
    :param org_ids: список id организаций.
    """
    from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationServiceModel

    # проверим есть ли подключенные сервисы
    find_services_kwargs = {
        'filter_data': {'org_id': org_ids},
        'fields': ['org_id', 'service_id', 'ready'],
    }

    result = dict((org_id, []) for org_id in org_ids)
    # Словарь со всеми известными сервисами.
    all_services = get_services(meta_connection)
    get_service = all_services.get

    organizations_services = OrganizationServiceModel(main_connection).find(**find_services_kwargs)

    for row in organizations_services:
        service_info = get_service(row['service_id']).copy()
        # готов ли сервис к работе для данной организации
        service_info['ready'] = row['ready']
        result[row['org_id']].append(service_info)

    return result


def get_organization_language(main_connection, org_id):
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel

    organization = OrganizationModel(main_connection).get(org_id, fields=['language'])
    if organization:
        return organization['language']


def get_organization_maillist_type(main_connection, org_id):
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel

    organization = OrganizationModel(main_connection).get(org_id, fields=['maillist_type'])
    if organization:
        return organization['maillist_type']


def _get_days_in_month(date):
    return calendar.monthrange(date.year, date.month)[1]


def get_balance_and_current_month_price(org_id):
    """
    Возвращает сумму за услуги оказанные организации, по которым еще не выставлены
    счета
    """
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        OrganizationBillingInfoModel,
        OrganizationLicenseConsumedInfoModel,
        OrganizationBillingConsumedInfoModel,
    )
    from intranet.yandex_directory.src.yandex_directory.core.models.service import (
        OrganizationServiceModel,
        AuthorizationError,
        ServiceNotFound,
    )
    from intranet.yandex_directory.src.yandex_directory.common.utils import ensure_date
    from intranet.yandex_directory.src.yandex_directory.core.billing.utils import (
        calculate_licensed_users,
        calculate_base_licenses,
        get_promocode_id,
        get_price_for_users,
        TrackerBillingStatusModel,
    )

    with get_meta_connection() as meta_connection:
        shard = get_shard(meta_connection, org_id)

        with get_main_connection(shard=shard, for_write=True) as main_connection:
            balance = 0
            current_month_price = 0

            billing_info = OrganizationBillingInfoModel(main_connection).get(org_id=org_id)
            if not billing_info:
                return balance, current_month_price

            client_id = billing_info['client_id']
            balance_info = app.billing_client.get_balance_info(client_id)
            balance = balance_info['balance']

            end_date = datetime.today().replace(hour=0, minute=0, second=0, microsecond=0)
            end_date = ensure_date(end_date)

            OrganizationLicenseConsumedInfoModel(main_connection).save_user_service_licenses(
                org_id=org_id,
                for_date=end_date,
            )
            OrganizationBillingConsumedInfoModel(main_connection).calculate(
                org_id=org_id,
                for_date=end_date,
                rewrite=True,
            )
            try:
                tracker_service = OrganizationServiceModel(main_connection).get_by_slug(
                    org_id, 'tracker',
                    fields=['*']
                )
            except (AuthorizationError, ServiceNotFound):
                # если трекер в организации не включали - долг считать не будем
                return balance, current_month_price

            if tracker_service['trial_expires'] and tracker_service['trial_expires'] >= end_date:
                # если триал не истек - долг считать не будем
                return balance, current_month_price

            org_billing_status = TrackerBillingStatusModel(main_connection).filter(
                payment_date__gt=end_date,
                payment_status=False,
                org_id=org_id,
            ).one()

            previous_payment = TrackerBillingStatusModel(main_connection).filter(
                payment_date__lte=end_date,
                payment_status=True,
                org_id=org_id,
            ).order_by('-payment_date').one()

            if org_billing_status:
                from_date = None
                if previous_payment:
                    from_date = previous_payment['payment_date']

                if from_date == end_date:
                    # организация сегодня уже платила, не нужно считать долг
                    current_month_price = 0
                else:
                    promocode_id = get_promocode_id(
                        main_connection=main_connection,
                        org_id=org_id,
                        to_date=end_date,
                        from_date=from_date,
                    )

                    base_licenses = calculate_base_licenses(
                        main_connection=main_connection,
                        org_id=org_id,
                        for_date=from_date or end_date - relativedelta(months=1),
                    )
                    max_period_licenses_count = calculate_licensed_users(
                        meta_connection=meta_connection,
                        org_id=org_id,
                        to_date=end_date + relativedelta(days=1),
                        from_date=from_date,
                        base_licenses=base_licenses,
                    )

                    total, current_month_price = get_price_for_users(
                        users_count=max_period_licenses_count,
                        promocode_id=promocode_id,
                    )

    return float(balance), current_month_price


def is_sso_turned_on(main_connection, org_id):
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
    organization = OrganizationModel(main_connection).get(org_id, fields=['is_sso_enabled'])
    if organization:
        return organization['is_sso_enabled']


def is_provisioning_turned_on(main_connection, org_id):
    from intranet.yandex_directory.src.yandex_directory.core.models.organization import OrganizationModel
    organization = OrganizationModel(main_connection).get(org_id, fields=['is_provisioning_enabled'])
    if organization:
        return organization['is_provisioning_enabled']
