# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.models import (
    ServiceModel,
    GroupModel,
    UserModel,
    RobotServiceModel,
    OrganizationModel,
    UserGroupMembership,
)
from intranet.yandex_directory.src.yandex_directory.core.views.users import (
    UserListView,
    UserDetailView,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    ServiceNotFound,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import TYPE_USER
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    utcnow,
    get_user_login_from_passport_by_uid,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_ids,
    get_random_password,
    get_domains_by_org_id,
    is_yandex_team,
    is_outer_uid,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.actions import action_service_robot_add


def create_robot_for_service_and_org_id(meta_connection, main_connection, service_slug, org_id):
    """
    Создаем робота для организации с заданным org_id, в которой подключен сервис slug.
    1. Проверяем поле robot_required
    2. Проверяем, что организация подключена к этому сервису
    3. Для организаций с org_id создаем робота с ником: robot_<service_slug> и рандомным паролем (пароль не храним)
    4. Департамент у таких роботов = None
    5. Генерируется только событие <user_added>, не генерируются
                события про изменение департамента
                событие об изменении количества сотрудников в отделе
                события об изменении всех связанных ресурсов
    6. Если еще не существует, то создаем группу Роботы у организаций, в которых подключен сервис slug
    (Группа без label, тип: robots)
    7. Обновляем у группы Роботов состав, добавляем нового робота
    8. Добавляем связь между роботом и сервисом
    :return: uid созданного робота
    :rtype: int
    """

    with log.name_and_fields('create_robot_for_service', slug=service_slug):
        service = ServiceModel(meta_connection).filter(slug=service_slug).one()

        if not service:
            log.error('Service with %s is not found' % service_slug)
            raise ServiceNotFound('Service with %s is not found' % service_slug)

        if not service.get('robot_required', False):
            log.info('Robot is not required for org_id = %s and slug = %s' % (org_id, service_slug))
            return

        service_id = service['id']

        if service['robot_uid']:
            # Если сервис использует в качестве робота
            # одну и ту же учётку, то нам надо достать её nickname
            # из blackbox
            robot_uid = service['robot_uid']
            robot_nickname = get_user_login_from_passport_by_uid(robot_uid)
            if not robot_nickname:
                with log.fields(robot_uid=robot_uid):
                    message = 'Robot with uid not found in blackbox'
                    log.error(message)
                    raise RuntimeError(message)
        else:
            # В противном случае - сгенерим никнейм на основе слага сервиса
            # robot-{slug}
            robot_uid = None
            robot_nickname = get_robot_nickname(service_slug)

        robot_user = UserModel(main_connection) \
                     .filter(org_id=org_id, nickname=robot_nickname) \
                     .fields('id') \
                     .one()
        organization_model = OrganizationModel(main_connection)
        # Если робот не существует, то создадим его
        if not robot_user:
            password = get_random_password()
            organization = organization_model.get(id=org_id)

            # DIR-3186 Запрещаем заводить роботов тимной организации
            org_domains = get_domains_by_org_id(meta_connection, main_connection, org_id)
            if is_yandex_team(org_domains['master']):
                log.info('Unable to create a robot for yandex-team. Skipping.')
                return

            if service['robot_name']:
                # Кастомное имя
                if organization['language'] == 'ru':
                    name = service['robot_name']['ru']
                else:
                    name = service['robot_name']['en']
            else:
                # Стандартное имя
                if organization['language'] == 'ru':
                    name = 'Робот сервиса %s' % service['slug'].capitalize()
                else:
                    name = 'Robot for %s' % service['slug'].capitalize()

            name = {
                'en': '%s Robot' % service['slug'].capitalize(),
                'ru': name
            }

            data = {
                'gender': 'male',
                'password': password,
                'nickname': robot_nickname,
                # не хотим, чтоб робот был виден в общей структуре организации
                'department_id': None,
                'birthday': utcnow().date(),
                'name': {
                    'first': name,
                    'last': {
                        'en': '',
                        'ru': '',
                    }
                }
            }
            # Если роботный пользователь уже существует в Паспорте, то мы укажем его uid
            # и _post_user не будет пытаться создать его внутри домена.
            if robot_uid:
                data['id'] = robot_uid
                data['is_outer'] = False

            try:
                robot_user = UserListView._post_user(
                    meta_connection,
                    main_connection,
                    organization['id'],
                    data,
                    organization['admin_uid'],
                    user_type='robot',
                    service_slug=service_slug,
                    api_version=1,
                )
            except ImmediateReturn:
                log.trace().info('Robot %s already exists' % robot_nickname)

            log.info('Robot created, trying to find him in database and to bind to the service')

        robot_uid = robot_user['id']
        with log.fields(robot_uid=robot_uid, robot_nickname=robot_nickname):
            group_model = GroupModel(main_connection)
            robot_group = group_model.get_or_create_robot_group(org_id)
            if not UserGroupMembership(main_connection).filter(
                org_id=org_id,
                group_id=robot_group['id'],
                user_id=robot_uid
            ).count():
                # Добавляем робота в роботную группу
                # Получим всех роботов, вместе с сервисными
                robots = organization_model.get_robots(org_id)
                current_robots_uids = only_ids(robots)
                member_uids = [robot_uid] + current_robots_uids
                members = [{'type': TYPE_USER, 'id': uid} for uid in member_uids]
                data = {
                    'members': members,
                }

                log.info(
                    'Updating "Robots" group with these members: {0}'.format(
                        member_uids
                    )
                )
                group_model.update_one(org_id, robot_group['id'], data, allow_empty_admins=True)

            if not RobotServiceModel(main_connection) \
               .filter(org_id=org_id, uid=robot_uid, service_id=service_id) \
               .count():
                # Добавляем связь между роботом и сервисом
                log.info('Connecting robot to service through robot_services table')
                RobotServiceModel(main_connection).create(org_id, robot_uid, service_id, service_slug)
                action_service_robot_add(
                    main_connection,
                    org_id=org_id,
                    author_id=None,
                    object_value=service,
                )
                log.info('Robot created, returning uid')
        return robot_uid


def get_robot_nickname(service_slug):
    return '{}{}'.format(app.config['ROBOT_ACCOUNT_NICKNAME_PREFIX'], service_slug)


def change_is_enabled_status_for_robot(meta_connection,
                                       main_connection,
                                       service_slug,
                                       org_id,
                                       is_enabled):
    # активируем/блокируем робота сервиса service_id
    with log.name_and_fields('change_is_enabled_status_for_robot',
                             org_id=org_id,
                             service_slug=service_slug,
                             is_enabled=is_enabled):
        service = ServiceModel(meta_connection).get_by_slug(service_slug)
        if not service:
            log.error('Service not found')
            raise ServiceNotFound('Service is not found')

        robot_service = RobotServiceModel(main_connection).find(
            filter_data={
                'org_id': org_id,
                'service_id': service['id'],
            },
            one=True
        )
        if not robot_service:
            log.warning('No robot for service')
            return

        robot_uid = robot_service['uid']
        with log.fields(robot_uid=robot_uid):
            if is_outer_uid(robot_uid):
                log.info('Unable to block robot because he is "portal" user')
            else:
                UserDetailView._change_is_enabled_status_for_user(
                    main_connection,
                    robot_uid,
                    is_enabled,
                    org_id,
                    author_id=None,
                )


def singleton_robot_uids(meta_connection):
    """
    Вернуть uid роботов-синглтонов.
    """
    return ServiceModel(meta_connection).fields('robot_uid').filter(robot_uid__gt=0).scalar()
