# -*- coding: utf-8 -*-
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    get_domain_info_from_blackbox,
    to_punycode,
    ensure_date,
    utcnow,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    is_outer_uid,
    get_organization_admin_uid,
)
from intranet.yandex_directory.src.yandex_directory.core.events.utils import get_callbacks_for_events
from intranet.yandex_directory.src.yandex_directory.core.utils.analytics import AnalyticsToYtSaver
from intranet.yandex_directory.src.yandex_directory.core.task_queue import (
    Task,
    TASK_STATES,
)
from intranet.yandex_directory.src.yandex_directory.core.models.license import TrackerLicenseLogModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import TrackerBillingStatusModel
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationServiceModel,
    UserServiceLicenses,
    UserModel,
    UserMetaModel,
    TaskModel,
    OrganizationModel,
    DomainsAnalyticsInfoModel,
    OrganizationServicesAnalyticsInfoModel,
    OrganizationsAnalyticsInfoModel,
    UsersAnalyticsInfoModel,
    DomainModel,
    AdminContactsAnalyticsInfoModel,
    EventModel,
    OrganizationRevisionCounterModel,
)

from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    delete_domain_with_accounts,
    delete_domain_from_passport,
)

from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_meta_connection,
    get_shard_numbers,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.robots import (
    create_robot_for_service_and_org_id,
    change_is_enabled_status_for_robot,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import (
    TaskFailed,
    DuplicatedTask,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_organization_owner_change,
    action_organization_delete,
    action_resource_delete,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.base import TERMINATE_STATES
from intranet.yandex_directory.src.yandex_directory.core.models import ResourceModel
from intranet.yandex_directory.src.yandex_directory.core.tasks import SyncExternalIDS
from intranet.yandex_directory.src.yandex_directory.core.events import event
from intranet.yandex_directory.src.yandex_directory.connect_services.idm import get_service
from intranet.yandex_directory.src.yandex_directory.connect_services.idm.base_service import OperationNotInIdmError
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ServiceNotFound, AuthorizationError
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationBillingInfoModel
from intranet.yandex_directory.src.yandex_directory.common.models.types import ROOT_DEPARTMENT_ID
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import (
    create_portal_user,
    restore_user,
)
from intranet.yandex_directory.src.yandex_directory.core.features import is_feature_enabled, USE_DOMENATOR


class SaveAnalyticsToYTTask(Task):
    singleton = True
    org_id_is_required = False
    tries = 5
    tries_delay = 180
    lock_ttl = 4 * 60 * 60  # 4 часа
    default_priority = 500

    def do(self, for_date=None, recreate=False, table=None):
        metadata = self.get_metadata()
        if metadata:
            # в мета данных хранится словарь с id и state зависимых тасков по шардам
            not_success_states = []
            for shard, task_info in metadata.items():
                # пройдем по всем шардам и обновим состояния тасков
                if task_info['state'] != TASK_STATES.success:
                    # проверяем статус тасков, которые еще не завершились успехом
                    with get_main_connection(shard=shard) as main:
                        task = TaskModel(main).get(task_info['id'])
                        if not task:
                            # таск может быть удален, потому что завершился успешно
                            task = {'state': TASK_STATES.success}
                        metadata[shard]['state'] = task['state']
                        if task['state'] != TASK_STATES.success:
                            not_success_states.append(task['state'])

            self.set_metadata(metadata)
            if not not_success_states:
                # если массив пустой, значит все таски завершились успешно
                AnalyticsToYtSaver().save(for_date, recreate, table)
            elif all(state in (TASK_STATES.in_progress, TASK_STATES.free) for state in not_success_states):
                # если массив состоит только из тасков in_progress или free, то отложим выполнение этого таска
                self.defer(countdown=600)      # откладываем выполнение на 10 минут
            else:
                # во всех других случаях фейлим этот таск,
                # чтобы он не работал вечно
                raise TaskFailed
        else:
            # если таск запущен командой save-analytics-to-yt, то мета данных не будет
            # и таск просто запустит копирование имеющихся данных в YT
            AnalyticsToYtSaver().save(for_date, recreate, table)


TABLE_MODEL_MAP = {
    'users_analytics_info': UsersAnalyticsInfoModel,
    'domains_analytics_info': DomainsAnalyticsInfoModel,
    'organizations_services_analytics_info': OrganizationServicesAnalyticsInfoModel,
    'organizations_analytics_info': OrganizationsAnalyticsInfoModel,
    'admin_contacts_analytics_info': AdminContactsAnalyticsInfoModel,
}


class SaveAnalyticsLocal(Task):
    singleton = True
    org_id_is_required = False
    tries = 5
    tries_delay = 180
    lock_ttl = 1 * 60 * 60  # 1 час
    default_priority = 500

    def do(self, table):
        model = TABLE_MODEL_MAP.get(table)
        if model:
            shard = self.main_connection.engine.db_info['shard']
            with log.name_and_fields('save_analytics', shard=shard, table=table):
                model(self.main_connection).save_analytics()
        else:
            raise RuntimeError('Unknown table %s' % table)


def save_analytics(table=False, recreate=False, show_progress=False, copy_to_yt=False):
    shards = get_shard_numbers()
    if table:
        tables = [table]
    else:
        tables = list(TABLE_MODEL_MAP.keys())

    for table in tables:
        with get_main_connection(shard=shards[0], for_write=True) as main_connection:
            # проверим, что для данной таблицы нет таска на копирование в YT,
            # чтобы избежать ошибок DuplicatedTask
            copy_to_yt_task = TaskModel(main_connection).filter(
                task_name=SaveAnalyticsToYTTask.get_task_name(),
                params__contains={'table': table},
            ).order_by('-created_at').one()
            if copy_to_yt_task and copy_to_yt_task['state'] not in TERMINATE_STATES:
                log.info('Copy to YT task already exists')
                if show_progress:
                    print('Для таблицы {table} уже запущен таск на копирование в YT с id {task_id} {state}'.format(
                        table=table,
                        task_id=copy_to_yt_task['id'],
                        state=copy_to_yt_task['state']
                    ))
                    metadata = Task(main_connection, task_id=copy_to_yt_task['id']).get_metadata()
                    if metadata:
                        print('Метаданные таска:')
                        for shard, task_info in metadata.items():
                            print('Шард {shard}: подтаск {subtask_id}, state {subtask_state}'.format(
                                shard=shard,
                                subtask_id=task_info['id'],
                                subtask_state=task_info['state'],
                            ))
                continue
        # сложим в словарик все id тасков по шардам,
        # которых нужно дождаться перед началом копирования в YT
        dependent_tasks = {}
        for shard in shards:
            with get_main_connection(shard, for_write=True) as connection:
                try:
                    task = SaveAnalyticsLocal(connection).delay(table=table)
                    task_id = task.task_id
                except DuplicatedTask as exc:
                    with log.fields(existing_task_id=exc.existing_task_id, table=table, shard=shard):
                        log.info('Task for table and shard already in the queue')
                        task_id = exc.existing_task_id

                task_from_db = TaskModel(connection).filter(
                    id=task_id,
                ).one()
                dependent_tasks[shard] = {
                    'id': task_from_db['id'],
                    'state': task_from_db['state'],
                }
        if app.config['COPY_ANALYTICS_TO_YT'] or copy_to_yt:
            with get_main_connection(shard=shards[0], for_write=True) as main_connection:
                """
                Cоздадим таск на копирование в YT для каждой модели, который будет ждать,
                пока все dependent_tasks выполнятся. сделано через мета данные,
                потому что таски находятся в разных шардах и dependency_task_id не подходят.

                Запускаем на одном шарде, потому что сам таск внутри ходит по всем шардам
                для транзакционного создания таблиц в YT, складываем данные сразу для всех шардов
                """
                task = SaveAnalyticsToYTTask(main_connection).delay(
                    metadata=dependent_tasks,
                    table=table,
                    recreate=recreate
                )
                if show_progress:
                    print('Создан таск {task_id} на копирование в YT для таблицы {table}'.format(
                        task_id=task.task_id,
                        table=table,
                    ))


class SaveAnalyticsMetaTask(Task):
    singleton = True
    tries = 5
    tries_delay = 180
    lock_ttl = 900  # 15 минут
    org_id_is_required = False

    def do(self):
        # таск нужен для крона, на случай,
        # если таски на сохранение данных локально не создались с первого раза
        # такое может быть, например, из-за проблем с логированием
        save_analytics()


class UpdateLicenseCache(Task):
    singleton = True

    def do(self, org_id, **kwargs):
        if kwargs.get('service_id'):
            org_services = [{'service_id': kwargs['service_id']}]
        else:
            org_services = OrganizationServiceModel(self.main_connection).find(
                filter_data={
                    'org_id': org_id,
                    'enabled': True,
                    'ready': True,
                    'resource_id__isnull': False,
                },
                fields=['service_id']
            )
        for org_service in org_services:
            UserServiceLicenses(self.main_connection).update_licenses_cache(
                org_id=org_id,
                service_id=org_service['service_id'],
            )


class CreateRobotTask(Task):
    singleton = True

    def do(self, org_id, **kwargs):
        if not OrganizationModel(self.main_connection).exists(org_id):
            log.info('Organization was removed.')
            return

        if 'service_slug' not in kwargs:
            raise RuntimeError('service_slug is required for CreateRobotTask')
        service_slug = kwargs['service_slug']

        with get_meta_connection(for_write=True) as meta_connection:
            create_robot_for_service_and_org_id(
                meta_connection=meta_connection,
                main_connection=self.main_connection,
                service_slug=service_slug,
                org_id=org_id,
            )

    def delay(self, service_slug, org_id):
        with log.fields(service_slug=service_slug, org_id=org_id):
            log.info('Delaying task to create service robot')
            try:
                return super(CreateRobotTask, self).delay(service_slug=service_slug, org_id=org_id)
            except DuplicatedTask:
                log.info('Task for this robot is already in the queue')


class ChangeIsEnabledRobotStatusTask(Task):
    singleton = True

    def do(self, service_slug, org_id, is_enabled):
        with get_meta_connection() as meta_connection:
            change_is_enabled_status_for_robot(
                meta_connection=meta_connection,
                main_connection=self.main_connection,
                service_slug=service_slug,
                org_id=org_id,
                is_enabled=is_enabled,
            )

    def delay(self, service_slug, org_id, is_enabled):
        with log.fields(service_slug=service_slug, org_id=org_id, is_enabled=is_enabled):
            log.info('Delaying task to change is_enabled robot status')
            try:
                return super(ChangeIsEnabledRobotStatusTask, self).delay(
                    service_slug=service_slug,
                    org_id=org_id,
                    is_enabled=is_enabled,
                )
            except DuplicatedTask:
                log.info('Task for this robot is already in the queue')


class DeleteOrganizationTask(Task):
    default_priority = 2000
    singleton = True
    tries = 1

    def _unbind_resources(self, main_connection, org_id, user_id):
        remaining_resources = ResourceModel(main_connection).filter(
            org_id=org_id,
            service__in=['metrika', 'direct'],
        )
        for resource in remaining_resources:
            service_slug = resource['service']
            try:
                service = get_service(service_slug)
                service.unbind_resource(
                    org_id=org_id,
                    author_id=user_id,
                    resource_id=resource['external_id'],
                )
            except (OperationNotInIdmError, ServiceNotFound):
                pass

            ResourceModel(main_connection).delete(
                filter_data={'id': resource['id']}
            )

            action_resource_delete(
                main_connection,
                org_id=org_id,
                author_id=user_id,
                object_value=resource,
                old_object=resource,
            )

    def do(self, org_id, user_id):
        from intranet.yandex_directory.src.yandex_directory.core.actions.organization import _notify_organization_deleted

        log.info('Deleting organization')
        with get_meta_connection(for_write=True) as meta_connection:
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                self._unbind_resources(
                    main_connection=main_connection,
                    org_id=org_id,
                    user_id=user_id,
                )

                inner_admins = UserModel(main_connection).get_organization_admins(org_id)
                today = ensure_date(utcnow())
                org_billing_status = TrackerBillingStatusModel(main_connection).filter(
                    payment_date__gte=today,
                    payment_status=False,
                ).one()
                if org_billing_status:
                    payment_needed = True
                    try:
                        tracker_service = OrganizationServiceModel(main_connection).get_by_slug(
                            org_id, 'tracker',
                            fields=['*']
                        )
                    except (AuthorizationError, ServiceNotFound):
                        pass
                    else:
                        if tracker_service['trial_expires'] >= today:
                            # сервис еще в триале, удалим данные о следующих платежах
                            TrackerBillingStatusModel(main_connection).delete(filter_data={'org_id': org_id})
                            payment_needed = False

                    if payment_needed:
                        billing_info = OrganizationBillingInfoModel(main_connection).get(org_id=org_id)
                        if billing_info:
                            TrackerBillingStatusModel(main_connection).update(
                                filter_data={'org_id': org_id},
                                update_data={'client_id': billing_info['client_id']},
                            )

                OrganizationBillingInfoModel(main_connection).delete({'org_id': org_id})

                if OrganizationModel(main_connection).has_owned_domains(org_id):
                    # Если домены есть - удаляем
                    delete_domain_with_accounts(main_connection, org_id, check_domain_users=False)
                else:
                    # удаляем домены из паспорта
                    delete_domain_from_passport(main_connection, org_id)

                organization = OrganizationModel(main_connection).filter(id=org_id).one()

                for admin in inner_admins:
                    if not UserModel(main_connection).is_user_admin(user_id=admin['id'], exclude={org_id}):
                        app.passport.reset_admin_option(admin['id'])

                admin_uid = get_organization_admin_uid(
                    main_connection,
                    org_id
                )
                if not UserModel(main_connection).is_user_owner(user_id=admin_uid, exclude={org_id}):
                    app.passport.unset_pdd_admin(admin_uid)

                action_organization_delete(
                    main_connection=main_connection,
                    org_id=org_id,
                    author_id=user_id,
                    object_value=organization,
                )

                # получим всех подписчиков события удаления организации до очистки базы
                organization_deleted_event = EventModel(main_connection).fields('*', 'environment').filter(
                    org_id=org_id,
                    name=event.organization_deleted
                ).one()
                new_events_subscribers = get_callbacks_for_events(
                    meta_connection,
                    main_connection,
                    [organization_deleted_event],
                    organization['organization_type'],
                )

                # удаляем остатки организации в базе директории
                OrganizationModel(main_connection).remove_all_data_for_organization(org_id)

                # Создаём задачу, чтобы обновить в паспорте список org_id для учётки админа.
                # Это нужно, чтобы все сервисы понимали, что он больше не состоит в этой организации.
                try:
                    SyncExternalIDS(main_connection).delay(
                        org_id=org_id,
                        user_id=user_id,
                    )
                except DuplicatedTask:
                    pass

                try:
                    _notify_organization_deleted(
                        meta_connection,
                        main_connection,
                        new_events_subscribers,
                        organization_deleted_event,
                    )
                    log.info('Notify subscribers')
                except Exception:
                    log.trace().error('Unable to notify about deleted organization')
                OrganizationRevisionCounterModel(main_connection).increment_revisions_for_user(
                    meta_connection,
                    user_id,
                )
        log.info('Organization has been deleted')


class ChangeOrganizationOwnerTask(Task):
    singleton = True
    tries = 10
    tries_delay = 90
    singleton_params = ['org_id']

    def do(self, org_id, old_owner_uid, new_owner_uid):
        if not self._is_new_owner_uid_valid(org_id, old_owner_uid, new_owner_uid):
            log.info('New owner uid is not valid')
            return

        if get_organization_admin_uid(self.main_connection, org_id) != old_owner_uid:
            log.info('Old owner uid is not equal to admin_uid in database')
            return

        self._update_passport(org_id, old_owner_uid, new_owner_uid)
        self._update_connect(org_id, old_owner_uid, new_owner_uid)

    def _is_new_owner_uid_valid(self, org_id, old_owner_uid, new_owner_uid):
        # новый владелец внешний админ
        if is_outer_uid(new_owner_uid):
            return True
        # новый владелец другой сотрудник организации
        if UserModel(self.main_connection).filter(id=new_owner_uid, org_id=org_id, is_robot=False).one():
            return True
        return False

    def _update_connect(self, org_id, old_owner_uid, new_owner_uid):
        log.info('Start changing admin in Connect')
        with get_meta_connection(for_write=True) as meta_connection:

            user_meta = UserMetaModel(meta_connection)

            # если новый админ из текущей организации просто дадим админские права
            if UserModel(self.main_connection).get(user_id=new_owner_uid, org_id=org_id):
                UserModel(self.main_connection).make_admin_of_organization(org_id, new_owner_uid)
            else:
                user = user_meta \
                    .filter(org_id=org_id, id=new_owner_uid) \
                    .fields('user_type', 'is_dismissed') \
                    .one()
                if user:
                    if user['user_type'] != 'inner_user':
                        UserMetaModel(meta_connection).delete(
                            filter_data={'org_id': org_id, 'id': new_owner_uid}
                        )
                        create_portal_user(
                            meta_connection=meta_connection,
                            main_connection=self.main_connection,
                            uid=new_owner_uid,
                            org_id=org_id,
                            department_id=ROOT_DEPARTMENT_ID,
                            author_id=old_owner_uid,
                        )

                    if user['is_dismissed']:
                        restore_user(
                            meta_connection=meta_connection,
                            main_connection=self.main_connection,
                            uid=new_owner_uid,
                            org_id=org_id,
                            department_id=ROOT_DEPARTMENT_ID,
                            author_id=old_owner_uid,
                        )
                else:
                    create_portal_user(
                        meta_connection=meta_connection,
                        main_connection=self.main_connection,
                        uid=new_owner_uid,
                        org_id=org_id,
                        department_id=ROOT_DEPARTMENT_ID,
                        author_id=old_owner_uid,
                    )
                    UserModel(self.main_connection).make_admin_of_organization(org_id, new_owner_uid)

            old_organization = OrganizationModel(self.main_connection).get(org_id)
            # поменяем основного админа организации
            OrganizationModel(self.main_connection).update_one(org_id, {'admin_uid': new_owner_uid})
            updated_organization = old_organization.copy()
            updated_organization['admin_uid'] = new_owner_uid

            if not UserModel(self.main_connection).is_user_owner(user_id=old_owner_uid, exclude={org_id}):
                app.passport.unset_pdd_admin(old_owner_uid)

            action_organization_owner_change(
                self.main_connection,
                org_id=org_id,
                author_id=old_owner_uid,
                object_value=updated_organization,
                old_object=old_organization,
                content={'new_admin': new_owner_uid},
            )


    def _update_passport(self, org_id, old_owner_uid, new_owner_uid):
        # обновим админа в паспорте
        with get_meta_connection() as meta_connection:
            if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
                app.domenator.update_admin(org_id, new_owner_uid)
                return

        for domain in DomainModel(self.main_connection).filter(owned=True, org_id=org_id).order_by('-master').all():
            punycode_name = to_punycode(domain['name'])
            domain_info = get_domain_info_from_blackbox(punycode_name)
            if not domain_info:
                continue
            domain_id = domain_info['domain_id']
            if domain_info['admin_id'] == new_owner_uid:
                continue
            if domain_info['admin_id'] == old_owner_uid:
                app.passport.domain_edit(domain_id, {'admin_uid': new_owner_uid})


class UpdateOrganizationMembersCountTask(Task):
    singleton = True

    def do(self, org_id):
        from intranet.yandex_directory.src.yandex_directory.core.models import (
            UserModel,
            OrganizationModel,
        )

        new_user_count = UserModel(self.main_connection).count({
            'is_robot': False,
            'is_dismissed': False,
            'org_id': org_id,
        })
        OrganizationModel(self.main_connection) \
            .filter(id=org_id) \
            .update(user_count=new_user_count)
