# coding: utf-8

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from datetime import timedelta
from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
    get_main_connection,
    get_shard_numbers,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.core.models import UserModel
from intranet.yandex_directory.src.yandex_directory.core.tasks import SyncExternalIDS
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.utils import is_cloud_uid


def _get_changed_uids():
    """Cобирает данные про пользователей по шардам.

    Возвращает мап в котором ключами являются uid,
    а значениями - set с org_id.
    Если пользователь уволен в организации, то он также
    будет присутствовать в ответе, но set будет пустой."""
    shards  = get_shard_numbers()

    delay = app.config['ORG_IDS_INCONSISTENCIES_DELAY']
    window = app.config['ORG_IDS_INCONSISTENCIES_WINDOW']
    from_timestamp = utcnow() - timedelta(seconds=window + delay)
    to_timestamp = utcnow() - timedelta(seconds=delay)

    result = dict()
    for shard in shards:
        with get_main_connection(shard=shard) as connection:
            users = UserModel(connection) \
                    .filter(updated_at__between=(from_timestamp, to_timestamp)) \
                    .fields('org_id', 'is_dismissed') \
                    .all()
            for user in users:
                result.setdefault(user['id'], set())
                if not user['is_dismissed']:
                    result[user['id']].add(user['org_id'])
    return result


def _get_uids_from_bb(uids):
    """Получить данные о пользователях и их организациях из Паспорта.

    Возвращает данные в таком же формате, как и get_changed_uids.
    Но список org_id берёт из 1017 атрибута.
    """
    data = app.blackbox_instance.batch_userinfo(
        uids,
        userip='127.0.0.1',
        attributes='1017',
    )
    result = dict()
    for user in data:
        uid = user['uid']
        if uid:
            org_ids  = user['attributes'].get('1017', '')
            org_ids = org_ids.split(',')
            org_ids = set(map(int, [_f for _f in org_ids if _f]))
            result[int(user['uid'])] = org_ids

    return result


def _compare(from_dir, from_bb):
    """
    Сравнивает данные из паспорта и из Директории.
    Возвращает количество пользователей, для которых есть расхождения в списке организаций.
    """
    empty = set()
    difference = 0

    for uid, org_ids_from_dir in list(from_dir.items()):
        if is_cloud_uid(uid) or uid in (780889736, 780891246, 780890239):
            continue
        org_ids_from_bb = from_bb.get(uid, empty)
        if org_ids_from_dir != org_ids_from_bb:
            with log.fields(user_id=uid,
                            org_ids_from_bb=list(org_ids_from_bb),
                            org_ids_from_dir=list(org_ids_from_dir)):
                log.warning('Organization ids are out of sync, launching SyncExternalIDs')
                with get_meta_connection() as meta_connection:
                    for org_id in org_ids_from_dir:
                        shard = get_shard(meta_connection, org_id=org_id)
                        with get_main_connection(shard=shard, for_write=True) as main_connection:
                            try:
                                SyncExternalIDS(main_connection).delay(
                                    org_id=org_id,
                                    user_id=uid,
                                )
                            except DuplicatedTask:
                                pass
            difference += 1

    return difference


def update_cache():
    """Обновляет значения метрики в кэше.
    """
    return 0
    # uids_from_directory = _get_changed_uids()
    # uids = list(uids_from_directory.keys())
    # uids_from_blackbox = _get_uids_from_bb(uids)
    # number_of_problems = _compare(uids_from_directory, uids_from_blackbox)

    # app.cache.set(
    #     app.config['ORG_IDS_INCONSISTENCIES_CACHE_KEY'],
    #     number_of_problems,
    #     app.config['ORG_IDS_INCONSISTENCIES_CACHE_TTL'],
    # )
    # return number_of_problems


def get_inconsistencies_count():
    """Возвращает количество учёток, для которых в Паспорте
       хранится неправильный набор org_id.

       Вызывается в ручке которая отдаёт статистику для Yasm.
    """
    # Кэш подновляется по крону, поэтому в нормальном режиме
    # данные всегда должны браться из кэша.
    #
    # Однако на всякий случай я тут оставил рассчёт и обновление
    # данных в кэше.
    value = app.cache.get(app.config['ORG_IDS_INCONSISTENCIES_CACHE_KEY'])
    return value or update_cache()
