# -*- coding: utf-8 -*-

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.exceptions import DomainUserCanNotBeAddedByInvite
from intranet.yandex_directory.src.yandex_directory.common.models.types import ROOT_DEPARTMENT_ID
from intranet.yandex_directory.src.yandex_directory.core.features import (
    is_feature_enabled,
    MULTIORG,
    is_multiorg_feature_enabled_for_user,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DepartmentModel,
    OrganizationRevisionCounterModel,
    ResourceModel,
    OrganizationServiceModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import update_service_licenses
from intranet.yandex_directory.src.yandex_directory.core.models.user import (
    UserModel, Ignore,
    UserMetaModel,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    add_existing_user,
    get_organization_admin_uid,
    is_outer_uid,
    only_attrs,
    check_organizations_limit,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.restore import restore_user
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.utils.invite import use_invite
from intranet.yandex_directory.src.yandex_directory.common.utils import get_localhost_ip_address


class UserNotFoundInBlackbox(Exception):
    pass


class UserAlreadyMemberOfOrganization(Exception):
    pass


class UserAlreadyInThisOrganization(Exception):
    pass


def is_user_not_exists_in_dir(connection, org_id, user_id):
    user = UserModel(connection).get(user_id=user_id, org_id=org_id)
    if user is None:
        return True
    return False


def remaining_users_number(main_connection, org_id, admin_id):
    # Возвращаем количество всех оставшихся пользователей, кроме самого админа
    return UserModel(main_connection).filter(
        org_id=org_id,
        is_robot=False,
        id__notequal=admin_id,
    ).count()


def is_organization_owner(main_connection, org_id, uid, **kwargs):
    return get_organization_admin_uid(main_connection, org_id) == uid


def is_outer_admin(meta_connection, user_id, is_cloud=False):
    """Проверяет, является ли пользователь внешним админом хотя бы в одной организации."""
    if is_cloud:
        return False
    return UserMetaModel(meta_connection) \
        .filter(
            id=user_id,
            # Получим внешних админов и замов
            user_type__notequal='inner_user',
        ) \
        .count() > 0


def add_user_by_invite(meta_connection, main_connection, uid, user_ip, invite, set_org_id_sync=False,):
    if not is_outer_uid(uid):
        raise DomainUserCanNotBeAddedByInvite()

    org_id = invite['org_id']
    department_id = invite['department_id']

    # создадим портального пользователя в директории
    user = create_portal_user(
        meta_connection=meta_connection,
        main_connection=main_connection,
        uid=uid,
        org_id=org_id,
        department_id=department_id,
        user_ip=user_ip,
        set_org_id_sync=set_org_id_sync,
    )
    add_license = invite['add_license']
    if add_license:
        resource_id = OrganizationServiceModel(main_connection).get_licensed_service_resource_id(
            org_id,
            'tracker'
        )
        relations = [
            {
                'user_id': uid,
                'name': 'member'
            }
        ]
        ResourceModel(main_connection).add_relations(
            resource_id=resource_id,
            org_id=org_id,
            relations=relations
        )
        update_service_licenses(main_connection, org_id, 'tracker', uid)

    # Обновим инвайт
    use_invite(meta_connection, invite, uid)
    return user


def create_portal_user(meta_connection,
                       main_connection,
                       uid,
                       org_id,
                       cloud_uid=None,
                       department_id=None,
                       user_ip=None,
                       author_id=None,
                       set_org_id_sync=False,
                       cloud_user_data=None,
                       ):
    other_orgs_filter_data = {
        'org_id__notequal': org_id,
        'is_outer': False,
        'is_dismissed': False,
        'id': uid,
    }
    other_orgs = UserMetaModel(meta_connection).filter(**other_orgs_filter_data).scalar('org_id')

    if len(other_orgs) > 0:
        if (
            not is_feature_enabled(meta_connection, org_id, MULTIORG)
            and not is_multiorg_feature_enabled_for_user(meta_connection, uid)
        ):
            raise UserAlreadyMemberOfOrganization
        else:
            check_organizations_limit(meta_connection, uid)

        # Обновим ревизию во всех организациях пользователя, чтобы после добавления
        # в организацию он получил свежие данные
        OrganizationRevisionCounterModel(main_connection).increment_revisions_for_user(
            meta_connection,
            uid,
        )

    if user_ip is None:
        user_ip = get_localhost_ip_address()

    if not author_id:
        author_id = uid

    if cloud_uid is None:
        user = app.blackbox_instance.userinfo(uid=uid, userip=user_ip)
        if not user['uid']:
            raise UserNotFoundInBlackbox

    # Если отдел существует - создадим пользователя сразу в этом отделе.
    # Если не существует или не передан - в корневом.
    if not department_id or not DepartmentModel(main_connection). \
            filter(id=department_id, org_id=org_id). \
            count():
        log.info('Department from invite does not exist, using default')
        department_id = ROOT_DEPARTMENT_ID

    filter_data = {'id': uid, 'org_id': org_id, 'is_dismissed': Ignore}
    if cloud_uid is not None:
        filter_data['cloud_uid'] = cloud_uid
    exists_in_this_org = (
        UserModel(main_connection).filter(**filter_data).fields('is_dismissed').one()
    )
    if not exists_in_this_org:
        user = add_existing_user(
            meta_connection=meta_connection,
            main_connection=main_connection,
            org_id=org_id,
            user_id=uid,
            cloud_uid=cloud_uid,
            author_id=author_id,
            department_id=department_id,
            set_org_id_sync=set_org_id_sync,
            cloud_user_data=cloud_user_data,
        )
    else:
        if exists_in_this_org['is_dismissed']:
            log.info('Restored user who was dismissed')
            user = restore_user(meta_connection, main_connection,
                                uid, org_id,
                                department_id=department_id,
                                author_id=author_id
                                )
        else:
            raise UserAlreadyInThisOrganization()
    return user


def get_sorted_user_ready_org_ids(meta_connection, uid, org_ids=None, is_cloud=False):
    sql = '''select o.id from users u
    inner join organizations o on u.org_id = o.id
    where u.is_dismissed = false
    and {user_field} = %(uid)s
    and o.ready = true
    {org_ids_where}
    order by u.created, u.org_id'''

    org_ids_where = ''
    sql_params = {'uid': uid}
    if org_ids is not None:
        if not org_ids:
            return []
        org_ids_where = 'and u.org_id in %(org_ids)s'
        sql_params['org_ids'] = tuple(org_ids)
    if is_cloud:
        user_field = 'u.cloud_uid'
    else:
        user_field = 'u.id'

    sql = sql.format(org_ids_where=org_ids_where, user_field=user_field)
    result = meta_connection.execute(sql, sql_params).fetchall()

    return only_attrs(result, 'id')
