# coding: utf-8

from requests.exceptions import HTTPError

from intranet.yandex_directory.src.yandex_directory.core import exceptions
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    is_outer_uid,
    is_yandex_team_uid,
    prepare_department,
)
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.models.types import ROOT_DEPARTMENT_ID
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    AuthorizationError,
    ServiceNotFound,
)

from .responsible import is_responsible_in_org


def dismiss_user(meta_connection,
                 main_connection,
                 org_id,
                 user_id,
                 author_id,
                 old_user=None,
                 # эти аргументы нужны для того, чтобы можно было
                 # пропустить походы во внешние сервисы при генерации документации
                 # про события
                 skip_disk=False,
                 skip_passport=False):
    """
    Увольнение сотрудника.

    Нельзя уволить ответственного.

    Args:
        org_id (int): id организации пользователя
        user_id (int): id пользователя
        author_id (int): id автора увольнения
        old_user (dict): Объект пользователя до увольнения.
                         Ожидается, что он будет содержать groups с группами пользователя,
                         которые необходимы для action_user_dismiss
    """
    from intranet.yandex_directory.src.yandex_directory import disk
    from intranet.yandex_directory.src.yandex_directory.core.models.group import (
        GroupModel,
        UserGroupMembership,
    )
    from intranet.yandex_directory.src.yandex_directory.core.models.user import UserMetaModel, UserModel, UserDismissedModel
    from intranet.yandex_directory.src.yandex_directory.core.tasks import SyncExternalIDS
    from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
    from intranet.yandex_directory.src.yandex_directory.core.models.disk_usage import DiskUsageModel
    from intranet.yandex_directory.src.yandex_directory.core.models import (
        ResourceRelationModel,
        OrganizationServiceModel,
        ServiceModel,
        OrganizationRevisionCounterModel,
    )
    from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask

    user_meta_model = UserMetaModel(meta_connection)
    user_model = UserModel(main_connection)

    if is_responsible_in_org(main_connection, org_id, user_id):
        try:
            forms_service = OrganizationServiceModel(main_connection).get_by_slug(
                org_id=org_id, service_slug='forms',
                fields=['id', 'responsible_id'],
            )
        except (AuthorizationError, ServiceNotFound):
            raise exceptions.UnableToDismissServiceResponsible()
        if forms_service['responsible_id'] == user_id:
            OrganizationServiceModel(main_connection).update(
                filter_data={'id': forms_service['id']},
                update_data={'responsible_id': None}
            )
        else:
            raise exceptions.UnableToDismissServiceResponsible()


    if not old_user:
        old_user = user_model.find(
            filter_data={
                'id': user_id,
                'org_id': org_id,
            },
            fields=[
                # Сохраним слепок пользователя в базу со всеми полями
                '**'
            ],
            limit=1,
            one=True,
        )

    user_meta_model.update(
        dict(is_dismissed=True),
        filter_data=dict(id=user_id, org_id=org_id)
    )

    user_model.update(
        {
            'is_dismissed': True,
            'external_id': None,
        },
        {
            'org_id': org_id,
            'id': user_id
        }
    )
    user = user_model.find(
        filter_data={
            'id': user_id,
            'org_id': org_id,
            'is_dismissed': True,
        },
        # Сохраним в слепок пользователя в базу со всеми полями
        fields=['**'],
        limit=1,
        one=True,
    )
    if not user:
        log.warning('User was not found in database')
        return

    admin_groups = GroupModel(main_connection).find(filter_data={
        'org_id': org_id,
        'admin_uid': user_id,
    })

    department = None
    if user['department_id']:
        department = DepartmentModel(main_connection).get(
            department_id=user['department_id'],
            org_id=org_id,
            # Сохраним в слепок отдела в базу со всеми полями
            fields=['**'],
        )
        department = prepare_department(
            main_connection,
            department,
            api_version=1,
        )

    UserDismissedModel(main_connection).create(
        org_id=org_id,
        uid=user_id,
        groups=user['groups'],
        groups_admin=admin_groups or [],
        department=department,
    )

    # выгоняем из всех групп и переносим в корневой отдел
    # Удалим все записи из кэша групп.
    #
    # Это костыль, необходимый до тех пор, пока мы не переделаем
    # код так, чтобы руководители отдела связывались с группой
    # руководителей посредством связей.
    # Когда будет исправлен TODO в DepartmentModel.create,
    # этот delete можно будет выпилить.
    UserGroupMembership(main_connection).delete(
        {
            'org_id': org_id,
            'user_id': user_id,
        }
    )

    # А вот этот код - правильный, его надо будет оставить.
    user_model.update_one(
        update_data={
            'department_id': ROOT_DEPARTMENT_ID,
            'groups': []
        },
        filter_data={
            'org_id': org_id,
            'id': user_id,
            'is_dismissed': True,
        },
        # Передаём сюда None, чтобы
        # удалить человека из групп всех типов.
        group_types=None,
    )

    # Так же, удалим человека из всех команд, где он был админом
    GroupModel(main_connection).remove_admin_from_all_groups(
        org_id,
        user_id,
    )

    # отбираем права администратора
    user_model.revoke_admin_permissions(org_id=org_id, user_id=user_id)

    # DIR-6249 тут нужно не удалять те доменные учетки, которые состоят в нескольких организациях
    if is_outer_uid(user_id):
        with log.fields(nickname=user['nickname'], uid=user_id):
            log.info('Forcefully skipping passport, because users is not from connect')
        skip_passport = True
        try:
            SyncExternalIDS(main_connection) \
                .delay(org_id=org_id, user_id=user_id)
        except DuplicatedTask:
            log.info('DuplicatedTask: SyncExternalIDS already in queue')

    if not skip_passport:
        # Удаляем в Паспорте
        # возможно, это надо делать асинхронно
        with log.fields(nickname=user['nickname'], uid=user_id):
            if not user['is_enabled']:
                # если пользователь заблокирован, то разблокируем его прежде, чем удалить
                log.info('Unblocking account')
                try:
                    user_model.change_is_enabled_status(org_id, author_id, user_id, True)
                except HTTPError as exc:
                    # если пытаемся разблокировать пользователя, которого нет, паспорт возвращает 404
                    # 404 игнорируем только при удалении пользователя, во всех других случаях HTTPError
                    # при смене статуса пользователя - это ошибка
                    if exc.response.status_code == 404:
                        log.warning('Unknown UID (unblock users on_dismiss)')
                    else:
                        log.trace().error('HTTPError when unblock users on_dismiss')
                        raise

            log.info('Removing Passport account')
            app.passport.account_delete(user['id'])

    # если у пользователя была лицензия на диск, сходим в диск и отберем место
    # здесь используется просто поиск по таблице, а не get_licensed_service_resource_id,
    # чтобы не ловить ненужные исключения
    disk = ServiceModel(meta_connection).get_by_slug('disk')
    if disk:
        org_disk = OrganizationServiceModel(main_connection).filter(
            org_id=org_id,
            service_id=disk['id']
        ).one()
        if org_disk and org_disk['resource_id'] and ResourceRelationModel(main_connection).filter(
                user_id=user_id,
                org_id=org_id,
                resource_id=org_disk['resource_id'],
        ).count():
            from intranet.yandex_directory.src.yandex_directory.connect_services.partner_disk.tasks import DeleteSpacePartnerDiskTask

            DeleteSpacePartnerDiskTask(main_connection).delay(
                org_id=org_id,
                uid=user_id,
                resource_id=org_disk['resource_id'],
                author_id=author_id,
            )

    # Пересчитаем лимит дискового пространства
    DiskUsageModel(main_connection).update_organization_limits(org_id)

    from intranet.yandex_directory.src.yandex_directory.core.actions import action_user_dismiss
    action_user_dismiss(
        main_connection,
        org_id=org_id,
        author_id=author_id,
        object_value=user,
        old_object=old_user,
    )

    ResourceRelationModel(main_connection).delete(
        filter_data={
            'org_id': org_id,
            'user_id': user_id,
        }
    )
    OrganizationRevisionCounterModel(main_connection).increment_revisions_for_user(
        meta_connection,
        user_id,
    )
