# coding: utf-8

from collections import defaultdict
from concurrent.futures import (
    ThreadPoolExecutor,
    as_completed,
)

from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shards_by_org_ids,
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    OrganizationServiceModel,
)


def _get_services_user_responsible_for_on_this_shard(main_connection, *uids):
    """
    Вернуть пары сервис-организация, где пользователь ответственный в данном шарде.

    :return: [ ( service_id, org_id ) ]
    """
    result = {uid: [] for uid in uids}
    for org_serv in OrganizationServiceModel(main_connection).filter(
        responsible_id__in=uids,
    ).all():
        result[org_serv['responsible_id']].append(
            (org_serv['service_id'], org_serv['org_id'])
        )
    return result


def __execute_me_in_new_thread(shard, *uids):
    with get_main_connection(shard) as connection:
        return _get_services_user_responsible_for_on_this_shard(connection, *uids)


def get_services_user_responsible_for(*uids):
    from intranet.yandex_directory.src.yandex_directory.core.models.user import (
        UserMetaModel,
    )

    if not uids:
        return {}

    with get_meta_connection() as meta_connection:
        org_ids = set(UserMetaModel(meta_connection).filter(id__in=uids).fields('org_id').scalar())
    if not org_ids:
        return {uid: [] for uid in uids}

    shards_numbers = {
        shard
        for shard in list(get_shards_by_org_ids(*org_ids).values())
        if shard is not None
    }

    with ThreadPoolExecutor(len(shards_numbers)) as executor:
        shard_results = [
            executor.submit(__execute_me_in_new_thread, shard, *uids)
            for shard in shards_numbers
        ]

    result = defaultdict(list)
    for future in as_completed(shard_results, timeout=10):
        for uid, responsible_data in list(future.result().items()):
            result[uid] += responsible_data

    return result


def is_responsible_in_org(main_connection, org_id, uid):
    """
    Вернуть True если пользователь ответственный за любой сервис в указанной организации.

    :rtype: bool
    """
    return bool(OrganizationServiceModel(main_connection).filter(
        org_id=org_id,
        responsible_id=uid,
    ).count())


def is_responsible(uid):
    """
    Вернуть True если пользователь ответственный за любой сервис на всех шардах.

    :rtype: bool
    """
    return bool(
        get_services_user_responsible_for(uid)[uid]
    )
