# coding: utf-8

import time
from flask import request, Response

from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DomainModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import to_punycode
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    no_scopes,
    no_auth,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    get_domains_from_db_or_domenator,
    DomainFilter,
)

RESPONSE_BASE = '<?xml version="1.0" encoding="utf-8"?><pddinfo>{response}</pddinfo>'


class SODomainInfoView(View):
    """
    Ручка для спамообороны с информацией про домен.
    Без авторизации и скоупов.
    """

    @internal
    @no_auth
    @no_scopes
    def get(self, meta_connection, _):
        domain_name = request.args.to_dict().get('domain')
        if not domain_name:
            return xml_error('no_domain')

        with log.fields(domain=domain_name):
            domain = get_domains_from_db_or_domenator(
                meta_connection=meta_connection,
                domain_filter=DomainFilter(name=domain_name, owned=True),
                one=True,
            )
            if not domain:
                return xml_error('not_found')

            org_id = domain['org_id']
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard) as main_connection:
                domains = (
                    DomainModel(main_connection)
                    .filter(org_id=org_id, owned=True)
                    .scalar('name')
                )
                org_info = (
                    OrganizationModel(main_connection)
                    .filter(id=org_id)
                    .fields('created', 'user_count', 'admin_uid', 'karma', 'ip')
                    .one()
                )

            karma = org_info['karma']
            if karma is None:
                log.warning('No karma for domain, using default 50')
                karma = 50

            ip = org_info['ip']
            if ip is None:
                log.warning('No ip for domain, using default 0.0.0.0')
                ip = '0.0.0.0'

            unixtime_created = time.mktime(org_info['created'].timetuple())
            domain_info = {
                'ip': ip,
                'karma': karma,
                'firsttime': unixtime_created,
                'mailboxcnt': org_info['user_count'],
                'admlogin': org_info['admin_uid'],
                'org_id': org_id,
            }
            return xml_pddinfo_response(domain_info, domains)


def xml_response(data, status_code=200):
    return Response(
        data,
        status=status_code,
        mimetype='text/xml; charset=utf-8',
    )


def xml_error(error_text):
    response = '<error>{error_text}</error>'.format(error_text=error_text)
    return xml_response(RESPONSE_BASE.format(response=response))


def xml_pddinfo_response(info_data, domains):
    text_response = ''
    for k, v in info_data.items():
        text_response += '<{key}>{value}</{key}>'.format(key=k, value=v)

    text_domains = ''
    for d in domains:
        text_domains += '<domain>{domain_name}</domain>'.format(domain_name=to_punycode(d))
    response = text_response + '<domains>{text_domains}</domains>'.format(text_domains=text_domains)
    return xml_response(RESPONSE_BASE.format(response=response))
