# -*- coding: utf-8 -*-
import inspect
import operator

from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log


def build_cached_view_methods_map():
    """
    Сохраняет в каждый View класс атрибут _methods_map для удобного получения метода, который
    обрабатывает запрос в нужной версии API:
    {
        (get, 1): get,
        (get, 5): get,
        (get, 7): get_7,
        (get, 8): get_7,
        (get, 9): get_9,
    }
    """
    from intranet.yandex_directory.src.yandex_directory import app

    log.info('Building API methods with versions map...')
    for view in list(app.view_functions.values()):
        if hasattr(view, 'view_class'):
            view.view_class._methods_map = get_methods_map_for_view(view.view_class)
    log.info('API methods map has been prepared')


def get_methods_map_for_view(view):
    """
    Для класса View строит соответствия HTTP методов с версиями названию метода класса:
    {
        (get, 1): get,
        (get, 5): get,
        (get, 7): get_7,
    }
    """
    from intranet.yandex_directory.src.yandex_directory import app

    # https://st.yandex-team.ru/DIR-3571
    # Делаем исключение по версионированию для /monitoring/ & /ping/
    if view.__name__ in app.config['VIEW_CLASSES_WITHOUT_VERSION']:
        return {
            ('get', 1): 'get',
        }

    methods_map = {}
    for http_method in [x.lower() for x in view.methods]:
        # получим все методы API обрабатывающие запросы типа http_method
        def is_http_method(method):

            http_methods = {
                'get', 'head', 'post', 'put',
                'delete', 'connect', 'options',
                'trace', 'patch',
            }
            method_name = getattr(method, '__name__', None)
            if not method_name:
                return False

            if not any(
                    method_name.startswith(hm) for
                    hm in http_methods
            ):
                return False

            splitted_method_name = method_name.split('_')

            # если просто совпало начало названия метода c названием http метода
            if len(splitted_method_name) > 1 and not splitted_method_name[1].isdigit():
                return False

            if len(splitted_method_name) > 2 or splitted_method_name[0] != http_method:
                return False

            return True

        view_methods = [
            (m[0], _get_api_version_from_method_name(m[0])) for m in inspect.getmembers(view, predicate=is_http_method)
        ]
        # отсортируем список методов в порядке возрастания версии API
        view_methods.sort(key=operator.itemgetter(1))

        for supported_version in app.config['SUPPORTED_API_VERSIONS']:
            for method, version in view_methods:
                if version > supported_version:
                    break
                methods_map[(http_method, supported_version)] = method

    return methods_map


def _get_api_version_from_method_name(method):
    """
    Получает версию API из названия метода. Если версия не указана, она считается равной 1
    Например:
        get_1 -> 1
        get_17 -> 17
        patch -> 1
    """
    splitted_method = method.split('_')

    version = 1

    if len(splitted_method) == 2:
        version = int(splitted_method[1])
    elif len(splitted_method) == 1:
        version = 1
    else:
        raise RuntimeError('Can not get API version for method %s' % method)

    return version
