# coding: utf-8
import re

from flask.views import MethodView
from flask import (
    request,
    g,
)
from retrying import retry

from intranet.yandex_directory.src.yandex_directory.common.sqlalchemy import EntityNotFoundError
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.caching.utils import make_key
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    ConstraintValidationError,
    BadRequest,
    APIError,
    ReadonlyModeError,
)
from intranet.yandex_directory.src.yandex_directory.passport.exceptions import PassportException
from intranet.yandex_directory.src.yandex_directory.passport.client import build_passport_exception_response
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    not_modified_response,
    json_response,
    json_error,
    json_error_exception,
    json_error_forbidden,
    json_error_bad_request,
    make_internationalized_strings,
    build_billing_exception_response,
    log_exception,
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection,
    get_main_connection,
    retry_on_close_connect_to_server,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import UnknownFieldError
from intranet.yandex_directory.src.yandex_directory.core.exceptions import BillingException


def retry_dispatch_if_connect_error(exc):
    # повторим при ошибке "неожиданное закрытие соединения к БД"
    if retry_on_close_connect_to_server(exc):
        return True
    return False


def no_cache(function):
    """Этот декоратор нужно использовать для того, чтобы
    отключить кэширование всего ответа для избранных GET ручек.

    Это может быть нужно в тех случаях, когда ручка отдаёт данные, которые меняются без
    изменения ревизии организации, как например данные о регистраторе из ПДД.
    """
    function.no_cache = True
    return function


CACHE_KEY_HEADER_RE = re.compile(r'^X-\S+-SERVICE-HOST$', re.IGNORECASE)


class View(MethodView):
    """Этот базовый view ловит исключения, чтобы показывать
    их в json формате, а так же ловит объекты Response,
    которые может кидать view, если ему необходимо завершиться
    преждевременно.
    """
    # список методов, которые требуют коннекта к базе на запись
    writing_methods = ('post', 'patch', 'put', 'delete')
    allowed_ordering_fields = ['id']
    _methods_map = {}

    def is_not_modified(self):
        """Проверяет, что в текущем запросе использован etag и ревизия организации с тех пор не изменилась.
        """
        revision = getattr(g, 'revision', None)
        # Проверять etag надо только если известна ревизия организации.
        # То есть, etags у нас сейчас будут работать только для ручек,
        # где известна конкретная организация
        if revision:
            # Все етаги являются строками, поэтому и ревизию надо
            # привести к строке.
            revision = str(revision)
            # Тут мы применяем list, чтобы из объекта requests.Etags
            # сделать просто список строк
            etags = list(request.if_none_match)
            # И если среди них есть ревизия, значит
            # данные в организации не менялись
            if revision in etags:
                return True

        return False

    def dispatch_request(self, *args, **kwargs):
        api_version = request.api_version
        route = g.route
        ctx = dict(
            api_version=api_version,
            http_host=request.host,
            route=route,
        )

        with log.fields(**ctx):

            if self.is_not_modified():
                log.debug('Not modified')
                return not_modified_response()

            try:
                response = self.dispatch(*args, **kwargs)
            except UnknownFieldError as e:
                message_params = dict(
                    field=e.field,
                    supported_fields=', '.join(sorted(e.supported_fields)),
                )

                if app.config['ENVIRONMENT'] == 'autotests':
                    message_params['model'] = str(e.model_class)
                log.trace().warning('UnknownFieldError')
                response = json_error(
                    422,
                    'unknown_field',
                    'Unknown field {field}. Supported fields are: {supported_fields}.',
                    **message_params
                )
            except ImmediateReturn as e:
                if app.config['ENVIRONMENT'] == 'autotests':
                    # Сохраним в лог трейс, чтобы проще было разбираться откуда
                    # был совершен выход.
                    log.trace().error('ImmediateReturn from view')

                response = e.response
            except APIError as e:
                msg = 'APIError: {}'.format(e.message)
                log_exception(e, msg)

                response = json_error(
                    e.status_code,
                    e.code,
                    e.message,
                    **e.params
                )
            except PassportException as exc:
                if exc.http_code >= 400 and exc.http_code < 500:
                    log.warning('PassportException')
                else:
                    log.error('PassportException')
                response = build_passport_exception_response(exc)
            except BadRequest:
                response = json_error_bad_request()
            except ConstraintValidationError as e:
                log.trace().warning('ConstraintValidationError')
                response = json_error(
                    422,
                    e.error_code,
                    e.error_message,
                    **e.params
                )
            except BillingException as exc:
                log.trace().error('BillingException')
                response = build_billing_exception_response(exc)
            except EntityNotFoundError:
                log.trace().warning('Entity not found')
                # обрабатывается через DefaultErrorHandlingExtension
                raise
            except Exception as e:
                log.trace().error('Unhandled exception')
                return json_error_exception(e)
        return response

    @retry(stop_max_attempt_number=3,
           wait_incrementing_increment=50,
           retry_on_exception=retry_dispatch_if_connect_error)
    def dispatch(self, *args, **kwargs):
        # для POST, PATCH, DELETE, а так же для случая,
        # когда указан заголовок X-Database: master,
        # используем мастер-базу
        for_write = request.method.lower() in self.writing_methods or request.headers.get('x-database') == 'master'
        # всегда начинаем транзакцию для избежания неконсистентного чтения
        try:
            with get_meta_connection(for_write=for_write) as meta_connection:
                # определим шард пользователя
                if getattr(g, 'shard', None) is None:
                    # в некоторых случаях, шард может быть неизвестен, например когда
                    # пришел внешний админ и у него не указан org-id
                    return self._dispatch_request(
                        meta_connection,
                        None,  # main_connection
                        *args,
                        **kwargs
                    )
                with get_main_connection(g.shard, for_write=for_write) as main_connection:
                    return self._dispatch_request(
                        meta_connection,
                        main_connection,
                        *args,
                        **kwargs
                    )
        except ReadonlyModeError:
            log.trace().warning('The Directory is in read-only mode')
            return json_error(503, 'read_only_mode', 'The Directory is in read-only mode')

    def _get_cache_key(self):
        key_data = [
            g.revision,
            getattr(request, 'api_version', 1),
            getattr(g, 'org_id', None),
            getattr(getattr(g, 'user', None), 'uid', None),
            getattr(getattr(g, 'service', None), 'id', None),
            getattr(g, 'scopes', None),
            request.full_path,
        ]
        headers = []
        for header, value in request.headers:
            header_upper = header.upper()
            if CACHE_KEY_HEADER_RE.match(header_upper):
                headers.append(
                    (header_upper, value.lower())
                )

        headers = sorted(headers, key=lambda t: t[0])
        key_data = key_data + headers
        return tuple(key_data)

    def _dispatch_request(self, meta_connection, main_connection, *args, **kwargs):
        """
        Вызывает нужный для request.method метод view с учетом переданной версии API в заголовке X-API-Version
        """
        method = self._get_method_for_current_request()

        # устанавливает глобальные переменные, что бы использовать их в логировании
        g.view_class = self
        g.method = method

        # если метод не найден, значит его нет для всех поддерживаемых версий API ниже запрошенной
        if method is None:
            with log.fields(method=request.method):
                log.warning('Requested unsupported API version for')
                raise ImmediateReturn(
                    json_error(405, 'method_not_allowed', 'Method Not Allowed')
                )

        no_cache = getattr(method, 'no_cache', None)
        permission_required = getattr(method, 'permissions', None)
        # Для каждой view должны быть в явном виде указаны права
        if permission_required is None:
            raise ImmediateReturn(json_error_forbidden())

        if (
                request.method == 'GET'
                and getattr(g, 'revision', None) is not None \
                and not no_cache):
            # Согл. документации request.full_path включает в себя GET-параметры
            # Явно включаем версию API в ключ (отсутсвует в full_path)
            key = make_key(
                prefix='view',
                key_data=self._get_cache_key(),
            )

            result = app.cache.get(key)
            with log.fields(from_cache=(result is not None)):
                if result is None:
                    result = method(meta_connection, main_connection, *args, **kwargs)
                    app.cache.set(key, result)
                    app.stats_aggregator.inc('cache_get_request_miss_summ')
                else:
                    app.stats_aggregator.inc('cache_get_request_hit_summ')

            return result

        return method(meta_connection, main_connection, *args, **kwargs)

    def _get_method_for_current_request(self):
        api_version = getattr(request, 'api_version', 1)
        method_name = self._methods_map.get((request.method.lower(), api_version))

        if method_name and request.method == 'HEAD' and not hasattr(self, method_name):
            # если не нашли метода-обработчика HEAD запроса, попробуем найти его для GET
            method_name = self._methods_map.get(('get', api_version))

        if method_name:
            return getattr(self, method_name, None)

    def _get_ordering_fields(self):
        fields = []
        ordering_args = request.args.to_dict().get('ordering', [])
        if isinstance(ordering_args, str):
            ordering_args = ordering_args.split(',')
        for field in ordering_args:
            if field.startswith('-'):
                allowed = field[1:] in self.allowed_ordering_fields
            else:
                allowed = field in self.allowed_ordering_fields
            if not allowed:
                raise ImmediateReturn(
                    json_response(
                        {
                            'message': 'Bad ordering parameter',
                            'key': 'bad_ordering_param',
                        }
                    )
                )
            fields.append(field)
        # при None в модели будет использован порядок сортировки по умолчанию
        return fields or None

    def normalization(self, data, schema=None):
        """
        Этот метод вызывается ещё до того, как входный параметры будут проходить
        проверку по JSON схеме.

        Его можно использовать для предварительной обработки параметров или их
        дополнительной проверки.
        """
        if schema and request.api_version >= 5:
            data = make_internationalized_strings(
                data,
                schema,
            )
        return data

    @staticmethod
    def _raise_if_no_permission(permissions, required_permission):
        if required_permission not in permissions:
            with log.fields(required_permissions=[required_permission]):
                log.debug('User has no permissions')
            raise ImmediateReturn(json_error_forbidden())
