# coding: utf-8

from intranet.yandex_directory.src import settings
import collections
import urllib.parse
from flask import request, g
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard_numbers,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    APIError,
    OrganizationAlreadyHasDirectResource,
    UserNotFoundInOrganizationError,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import TYPE_USER
from intranet.yandex_directory.src.yandex_directory.core.models import (
    ResourceModel,
    UserMetaModel,
    UserModel,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import tvm
from intranet.yandex_directory.src.yandex_directory.connect_services.yandexsprav.client import YandexSpravInteractionException
from intranet.yandex_directory.src.yandex_directory.connect_services.metrika.client.exceptions import MetrikaInteractionException
from intranet.yandex_directory.src.yandex_directory.common import http_client

from ..resources import RESOURCE_SCHEMA
from ...features import is_feature_enabled, MULTIORG, is_multiorg_feature_enabled_for_user

BIND_SERVICE = {
    'title': 'B2Service',
    'type': 'object',
    'properties': {
        'slug': schemas.STRING,
        'resource': RESOURCE_SCHEMA,
    },
    'required': ['slug', 'resource'],
    'additionalProperties': False
}

BIND_ORG_SCHEMA = {
    'title': 'BindOrganization',
    'type': 'object',
    'properties': {
        'service': BIND_SERVICE,
        'organization': dict(type=['object', 'null']),
        'org_id': schemas.INTEGER_OR_NULL,
    },
    'required': ['service'],
    'additionalProperties': False
}

BIND_ORGS_SCHEMA = {
    'title': 'BindOrganizations',
    'type': 'object',
    'properties': {
        'service': BIND_SERVICE,
    },
    'required': ['service'],
    'additionalProperties': False
}

BIND_ACCESS_SKIP_CHECK = settings.BIND_ACCESS_SKIP_CHECK


class BindValidationError(APIError):
    """
    Ресурс нельзя привязать потому что он нарушает текущую политику ограничений.

    Бросаем НАСЛЕДНИКОВ, когда сервис не готов принять ресурс.
    """
    code = 'unknown_binding_error'
    status_code = 409
    message = 'Unknown API error.'


class UserIsAlreadyInAnotherOrg(BindValidationError):
    """
    Нельзя состоять в какой-то другой организации.
    """
    log_level = 'WARNING'
    with_trace = False
    code = 'person_is_already_in_some_org'
    message = 'Some person is already in some org'
    description = 'Нельзя передать объект в организацию, потому что один из пользователей уже состоит в другой организации'


class ResourceIsAlreadyExists(BindValidationError):
    """
    Не позволяем создать рессурс, если рессурс с таким external_id
    уже есть в этой организации для этого сервиса
    """
    code = 'resource_is_already_exists'
    description = 'Объект уже привязан к организации'

    def __init__(self, org_id, service_slug, resource_id):
        super(ResourceIsAlreadyExists, self).__init__(
            message='Resource "{resource_id}" is already exists for service "{service_slug}" in org "{org_id}"',
            org_id=org_id,
            service_slug=service_slug,
            resource_id=resource_id,
        )


class AuthorizationError(APIError):
    """
    Ресурс нельзя привязать из-за отсутствия у пользователя прав на ресурс
    """
    code = 'no_access_for_resource'
    message = "Person doesn't have access to resource"
    status_code = 403
    log_level = 'WARNING'
    description = 'Нет прав для передачи объекта в организацию'


class AuthenticationError(APIError):
    """
    Недостаточно пользовательских данных для проверки прав на ресурс
    """
    code = 'auth_data_missing'
    message = 'Not enough auth data to check permission'
    log_level = 'WARNING'
    status_code = 422
    description = 'Для передачи объекта в организацию нужно аутентифицироваться'
    headers = {}


class ExternalServiceError(BindValidationError):
    """
    Не смогли проверить права из-за ошибки внешнего сервиса
    """
    code = 'binding.external_service_error'
    message = 'External service error acquired'
    description = 'Ошибка внешнего сервиса'


class AuthorShouldBeInRelationsApiError(BindValidationError):
    """
    Автор не находится в списке relations
    """
    code = 'author_should_be_in_relations_error'
    message = 'Author should be in relations'
    description = 'Автор должен находится в списке связей'


def check_in_some_org(meta_connection,
                      responsible_id,
                      others_ids,
                      existing_org_id=None):
    from intranet.yandex_directory.src.yandex_directory.core.utils import is_outer_uid, check_organizations_limit

    all_people = list(others_ids) + [responsible_id]
    if existing_org_id and is_feature_enabled(meta_connection, existing_org_id, MULTIORG) \
            and all(is_outer_uid(uid) for uid in all_people):
        check_organizations_limit(meta_connection, all_people, org_to_skip=existing_org_id)
        return

    filter_data = dict(id__in=all_people, is_dismissed=False, is_outer=False)
    if existing_org_id:
        filter_data['org_id__notequal'] = existing_org_id
    multi_org_data = UserMetaModel(meta_connection)\
        .filter(**filter_data)\
        .fields('id', 'org_id')\
        .all()
    users_in_orgs = collections.defaultdict(set)
    for data in multi_org_data:
        if is_multiorg_feature_enabled_for_user(meta_connection, data['id']):
            check_organizations_limit(meta_connection, [data['id']], org_to_skip=existing_org_id)
        else:
            with log.fields(error=dict(uid=data['id'], org_id=data['org_id'])):
                log.warning('User is already in at least one org')
                users_in_orgs[data['org_id']].add(data['id'])

    if users_in_orgs:
        error_details = []
        for org_id, user_ids in users_in_orgs.items():
            shard = get_shard(meta_connection, org_id)
            with get_main_connection(shard=shard) as main_connection:
                users_data = UserModel(main_connection).filter(
                   org_id=org_id, id__in=user_ids
                ).fields('id', 'nickname', 'org_id').all()
                error_details.extend(users_data)

        raise UserIsAlreadyInAnotherOrg(error_details=error_details)


def check_is_resource_exists(service_slug, resource_id):
    if not resource_id:
        return
    for shard in get_shard_numbers():
        with get_main_connection(shard=shard) as main_connection:
            resource = (ResourceModel(main_connection)
                        .filter(external_id=resource_id, service=service_slug).fields('org_id')
                        .one())
            if resource:
                raise ResourceIsAlreadyExists(resource['org_id'], service_slug, resource_id)


def check_access_for_bind_direct(responsible_id, resource_id, relations, **kwargs):
    if not resource_id:
        return

    for relation in relations:
        if (
                relation['object_id'] == responsible_id and
                relation['name'] == '/user/chief/'
        ):
            return

    with log.fields(resource_id=resource_id, user_id=responsible_id, service_slug='direct'):
        log.warning('User have no access to this resource')
        raise AuthorizationError


def check_access_for_bind(**kwargs):
    access_checker = {
        'forms': check_access_for_bind_forms,
        'metrika': check_access_for_bind_metrika,
        'yandexsprav': check_access_for_bind_yandexsprav,
        'direct': check_access_for_bind_direct,
    }
    service_slug = kwargs['service_slug']
    if service_slug not in BIND_ACCESS_SKIP_CHECK:
        access_checker[service_slug](**kwargs)
    else:
        log.warning('Explicitly allow any user to bind any resource')


def check_access_for_bind_metrika(responsible_id, resource_id, **kwargs):
    if not resource_id:
        return

    try:
        can_transfer = app.metrika.check_transfer_possible(
            resource_id,
            responsible_id,
        )
        if not can_transfer:
            raise AuthorizationError()
    except MetrikaInteractionException:
        raise ExternalServiceError()


def check_access_for_bind_forms(existing_org_id, resource_id, **kwargs):
    if not existing_org_id or not resource_id:
        return
    service_ticket = tvm.tickets['forms']
    user_ticket = g.user_ticket
    if not user_ticket:
        raise AuthenticationError
    path = 'admin/api/v2/surveys/{}/?detailed=0'.format(resource_id)
    url = urllib.parse.urljoin(settings.FORMS_HOST, path)
    headers = {
        'X-Ya-Service-Ticket': service_ticket,
        'X-Ya-User-Ticket': user_ticket,
        'X-ORGS': str(existing_org_id),
    }
    response = http_client.request(
        'get',
        url=url,
        headers=headers,
    )

    if response.status_code != 200:
        with log.fields(resource_id=resource_id, org_id=existing_org_id,
                        response_status_code=response.status_code, response_text=response.text,
                        ):
            if response.status_code == 404:
                # формы отдают 404 если у пользователя нет доступа до формы
                log.warning('User have no access to this resource')
                raise AuthorizationError

            log.error('Error from forms api acquired')
            raise ExternalServiceError


def check_access_for_bind_yandexsprav(resource_id, responsible_id, **kwargs):
    if not resource_id:
        return
    try:
        user_has_permission = app.yandexsprav.check_permission(resource_id, responsible_id)
    except YandexSpravInteractionException:
        log.trace().error('Error from yandexsprav api acquired')
        raise ExternalServiceError

    if not user_has_permission:
        with log.fields(resource_id=resource_id, user_id=responsible_id, service_slug='yandexsprav'):
            log.warning('User have no access to this resource')
            raise AuthorizationError


def check_single_direct_resource(meta_connection, service_slug, existing_org_id):
    if not existing_org_id or service_slug != 'direct':
        return

    shard = get_shard(meta_connection, existing_org_id)

    with get_main_connection(shard=shard) as main_connection:
        resource = ResourceModel(main_connection).filter(
            service=service_slug, org_id=existing_org_id,
        ).one()
        if resource:
            raise OrganizationAlreadyHasDirectResource()


def check_in_current_org(meta_connection, existing_org_id, responsible_id):
    if not existing_org_id:
        return

    shard = get_shard(meta_connection, existing_org_id)
    with get_main_connection(shard=shard) as main_connection:
        user_in_org = UserModel(main_connection).filter(
            org_id=existing_org_id,
            id=responsible_id,
            is_dismissed=False,
        ).count()
        if not user_in_org:
            raise UserNotFoundInOrganizationError


def restriction_apply_to_request(meta_connection,
                                 responsible_id,
                                 others_ids,
                                 existing_org_id,
                                 service_slug,
                                 resource_id,
                                 is_dry_run,
                                 relations,
                                 ):
    """
    Проверить можно ли привязать в данный момент ресурс.

    Все новые ограничения добавляйте сюда как вызовы функций,
    которые реально выполняют проверку и бросают наследника BindValidationError.
    """
    check_is_resource_exists(service_slug, resource_id)
    check_single_direct_resource(
        meta_connection=meta_connection,
        service_slug=service_slug,
        existing_org_id=existing_org_id,
    )
    check_in_some_org(meta_connection, responsible_id, others_ids, existing_org_id)
    check_in_current_org(meta_connection, existing_org_id, responsible_id)
    check_access_for_bind(
        meta_connection=meta_connection, responsible_id=responsible_id,
        others_ids=others_ids, existing_org_id=existing_org_id,
        service_slug=service_slug, resource_id=resource_id,
        relations=relations,
    )


def gather_uids_from_request(relations):
    """
    Из request выбрать список UIDs которые должны иметь доступ к ресурсу

    :rtype int[]
    """
    uids = set()
    for relation in relations:
        if relation['object_type'] == TYPE_USER:
            uids.add(int(relation['object_id']))
    return uids
