# coding: utf-8

from flask import request, g
from typing import List

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    requires,
    scopes_required,
    no_permission_required,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard,
    get_main_connection,
    get_main_connection_for_new_organization,
    get_meta_connection,
)

from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    DirectServiceInteractionError,
    UserAlreadyMemberOfOrganization,
    UserOrganizationsLimit,
)
from intranet.yandex_directory.src.yandex_directory.common.models import types as OBJECT_TYPES
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
)
from intranet.yandex_directory.src.yandex_directory.connect_services.direct.client import DirectClient
from intranet.yandex_directory.src.yandex_directory.connect_services.direct.client.client import DirectRoleDto
from intranet.yandex_directory.src.yandex_directory.connect_services.direct.client.exceptions import DirectInteractionException
from intranet.yandex_directory.src.yandex_directory.core.features import CAN_WORK_WITHOUT_OWNED_DOMAIN
from intranet.yandex_directory.src.yandex_directory.core.features import (
    is_feature_enabled,
    set_feature_value_for_organization,
)
from intranet.yandex_directory.src.yandex_directory.core.models import UserMetaModel
from intranet.yandex_directory.src.yandex_directory.core.tasks.resource import (
    BindTask,
    AuthorShouldBeInRelationsError,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.views.organization.logic import create_organization_without_domain
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
)
from .logic import (
    AuthenticationError,
    AuthorizationError,
    BIND_ORG_SCHEMA,
    BIND_ORGS_SCHEMA,
    gather_uids_from_request,
    BindValidationError,
    restriction_apply_to_request,
    AuthorShouldBeInRelationsApiError,
    OrganizationAlreadyHasDirectResource,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import UserAlreadyMemberOfOrganization as  BaseUserAlreadyMemberOfOrganization


class BindOrganizationView(View):

    def get_request_data(self, data):
        service = data['service']
        service_slug = service['slug']
        uid = g.user.passport_uid
        resource = service.get('resource', {})
        resource_id = resource.get('id')
        relations = resource.get('relations', [])

        if service_slug == 'direct':
            relations = self._get_relations_for_direct(resource_id)
        return service, service_slug, uid, resource, resource_id, relations,

    @internal
    @uses_schema(BIND_ORG_SCHEMA)
    @requires(org_id=False, user=True)
    @scopes_required([scope.write_organization, scope.write_resources, scope.write_services])
    @no_permission_required
    def post(self, meta_connection, non_main_connection, data):
        """
        Связать организацию с ресурсом сервиса.

        Позволяет связать существующую организацию (или создать новую) с ресурсом
        сервиса, и определить список людей, которые имеют доступ к этому ресурсу.

        В существующей организации включает фичу `CAN_WORK_WITHOUT_OWNED_DOMAIN`.

        Пользователь, инициировавший запрос, становится ответственным за сервис, если сервис
        до этого не был включён в этой организации.

        По-умолчанию ручка ничего не пишет в базу, лишь проверяет
        возможность создания такой организации. На это поведение нужно опираться, когда
        другой сервис решает показывать ли виджет связывания.

        Чтобы записать изменения надо вызывать ручку с парметром `?dry-run=0`.

        Источник создания организации можно задать, если передать поле `source` внутри
        объекта организации. Например, входные данные могут быть такими:

        ```
        {
             "organization": {
                  "name": "Рога и копыта",
                  "source": "metrika_permission_management_page",
             },
             "service": {
                 "slug": "metrika",
                 "resource": "43245464565"
             }
        }
        ```

        При успешном выполнении в режиме `dry-run`, возвращает:

        ```
        {"binding_available": true}
        ```

        При успешном выполнении в пишущем режиме:

        ```
        {"org_id": 100500}
        ```

        В случае 409 в ответе будет код, по которому можно понять почему привязать не удалось:

        ```
        {"status": "person_is_already_in_some_org"}
        ```

        ---
        tags:
          - Сотрудники
          - Организации
          - Связи ресурсов
        parameters:
          - in: query
            name: "dry-run"
            type: string
            description: "1 если нужно проверить саму возможность операции, без выполнения"
          - in: body
            name: body
        responses:
          200:
            description: "Возвращает {task_id: int, org_id: int}."
          404:
            description: "Переданная организация не существует."
          409:
            description: "Этот ресурс так привязать нелья."
        """
        is_dry_run = request.args.get('dry-run', '1') == '1'
        organization = data.get('organization', {})
        org_id = data.get('org_id')
        user_ip = g.user.ip
        language = organization.get('language', 'ru')

        service, service_slug, uid, resource, resource_id, relations = self.get_request_data(data)

        with log.fields(uid=uid, service_slug=service_slug, org_id=org_id,
                        resource={
                            'id': resource_id,
                            'service': {
                                'slug': service_slug,
                            },
                        },
                        ):
            try:
                restriction_apply_to_request(
                    meta_connection,
                    responsible_id=uid,
                    others_ids=gather_uids_from_request(relations),
                    existing_org_id=org_id,
                    service_slug=service_slug,
                    resource_id=resource_id,
                    is_dry_run=is_dry_run,
                    relations=relations,
                )
            except BindValidationError:
                log.warning('Impossible to bind resource to organization')
                raise

            if is_dry_run:
                return json_response({'binding_available': True})
            else:
                if not org_id:
                    with get_main_connection_for_new_organization(for_write=True) as main_connection:
                        org_id = create_organization_without_domain(
                            meta_connection,
                            main_connection,
                            organization,
                            uid,
                            user_ip,
                        )
                        shard = main_connection.shard
                else:
                    shard = get_shard(meta_connection, org_id)
                    if not is_feature_enabled(meta_connection, org_id, CAN_WORK_WITHOUT_OWNED_DOMAIN):
                        set_feature_value_for_organization(
                            meta_connection,
                            org_id,
                            CAN_WORK_WITHOUT_OWNED_DOMAIN,
                            True,
                        )

                with get_main_connection(for_write=True, shard=shard) as main_connection:
                    try:
                        BindTask(main_connection).synchronous_call(
                            meta_connection=meta_connection,
                            main_connection=main_connection,
                            org_id=org_id,
                            user_id=uid,
                            service_slug=service_slug,
                            resource_id=resource_id,
                            language=language,
                            relations=relations,
                        )
                    except AuthorShouldBeInRelationsError:
                        raise AuthorShouldBeInRelationsApiError()
                    except BaseUserAlreadyMemberOfOrganization:
                        raise UserAlreadyMemberOfOrganization()

            return json_response(
                {
                    'org_id': org_id,
                }
            )

    @staticmethod
    def _get_relations_for_direct(resource_id):
        direct_client = app.direct  # type: DirectClient
        try:
            roles = direct_client.get_roles(resource_id)  # type: List[DirectRoleDto]
        except DirectInteractionException:
            raise DirectServiceInteractionError()

        relations = []
        direct_chief_id = None
        for role in roles:
            role_name = role.path.strip('/').split('/')[-1]

            if role_name == 'chief':
                if direct_chief_id is not None:
                    raise DirectServiceInteractionError(message='More than one chief found')

                direct_chief_id = role.uid

            relations.append({
                'object_type': OBJECT_TYPES.TYPE_USER,
                'object_id': role.uid,
                'name': '/user/{}/'.format(role_name),
            })

        if direct_chief_id is None:
            raise DirectServiceInteractionError(message='Role chief not found')

        return relations


class CheckOrganizationsBindingAvailable(BindOrganizationView):
    @internal
    @uses_schema(BIND_ORGS_SCHEMA)
    @requires(org_id=False, user=True)
    @scopes_required([scope.write_organization, scope.write_resources, scope.write_services])
    @no_permission_required
    def post(self, meta_connection, non_main_connection, data):
        """
        Возвращает список организаций и статусы возможно или нет привязать ресурс к организациям,
        с указанием причины отказа.

        Входные данные должны содержать следующие поля:
        ```
            {
                "service":
                    {
                        "slug": "metrika",
                        "resource":
                            {
                                "id": "43245464565"
                            }
                     }
            }
        ```
        В объект "resourse" можно указывать "relations", объекты которого должны быть списком.

        Выходные данные:
        ```
            [
                {"org_id": 123, "binding_available": True},
                {"org_id": 1234, "binding_available": False, "reason": "resource_is_already_exists"},
            ]
        ```

        ---
        tags:
          - Организации
        parameters:
          - in: body
            name: body
        responses:
          200:
            description: "Возвращает {org_id: int, binding_available: True} или
            {org_id: int, binding_available: False, reason: str}"
        """

        service, service_slug, uid, resource, resource_id, relations = self.get_request_data(data)

        org_ids = UserMetaModel(meta_connection).filter(
            id=uid,
            is_dismissed=False,
            user_type='inner_user',
        ).scalar('org_id')

        result_orgs = []
        for org_id in org_ids:
            temp_data = {'org_id': org_id, 'binding_available': True}
            with log.fields(uid=uid, service_slug=service_slug, org_id=org_id,
                            resource={
                                'id': resource_id,
                                'service': {
                                    'slug': service_slug,
                                },
                            },
                            ):
                try:
                    restriction_apply_to_request(
                        meta_connection,
                        responsible_id=uid,
                        others_ids=gather_uids_from_request(relations),
                        existing_org_id=org_id,
                        service_slug=service_slug,
                        resource_id=resource_id,
                        is_dry_run=True,
                        relations=relations,
                    )
                except (
                        AuthenticationError,
                        AuthorizationError,
                        BindValidationError,
                        OrganizationAlreadyHasDirectResource,
                        UserOrganizationsLimit,
                ) as e:
                    temp_data['binding_available'] = False
                    temp_data['reason'] = e.code
                result_orgs.append(temp_data)
        return json_response(result_orgs)
