# coding: utf-8
import random
import string

from flask import request

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_permission_required,
    requires,
    scopes_required,
    internal,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.yarm import check_server
from intranet.yandex_directory.src.yandex_directory.core.yarm.exceptions import (
    YarmLoginError,
    YarmConnectionError,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema_for_get,
)

CHECK_MAIL_SERVER_SCHEMA = {
    'title': 'Check mail server',
    'type': 'object',
    'properties': {
        'host': schemas.STRING,
        'port': schemas.INTEGER,
        'protocol': schemas.MAIL_PROTOCOL,
        'ssl': schemas.INTEGER,
    },
    'required': ['host', 'port', 'protocol'],
    'additionalProperties': True
}


FAKE_CREDENTIALS = 'fake_credentials'
FAKE_LOGIN_LENGTH = 8


class CHECK_STATUSES:
    ok = 'ok'
    timeout = 'timeout'
    error = 'check_server_error'


class CheckMailServerView(View):
    @internal
    @scopes_required([scope.migrate_emails])
    @no_permission_required
    @uses_schema_for_get(CHECK_MAIL_SERVER_SCHEMA)
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection):
        """
        Проверка настроек почтового сервера.
        Если настройки сервера корректны, возвращаем
            {'status': 'ok'}
        Если не дождались ответа от почтового сервера, возвращаем
            {'status': 'timeout'}
        Если произошла какая-то другая ошибка, возвращаем
            {'status': 'check_server_error'}
        ---
        tags:
          - Настройки почтового сервера

        parameters:
          - in: query
            name: host
            required: true
            type: string
            description: host почтового сервера
          - in: query
            name: port
            required: true
            type: integer
            description: port почтового сервера
          - in: query
            name: ssl
            required: false
            type: integer
            description: используется ли ssl соединение (по умолчанию 1 - используется)
          - in: query
            name: protocol
            required: true
            type: string
            description: протокол, imap или pop3

        responses:
          200:
            description: Возвращаем статус проверки почтового сервера.
        """
        args = request.args.to_dict()

        host = args.get('host')
        port = int(args.get('port'))
        ssl = args.get('ssl')
        ssl = bool(int(ssl)) if ssl is not None else True
        protocol = args.get('protocol').lower()

        imap = (protocol == 'imap')

        with log.fields(host=host, port=port, ssl=ssl, protocol=protocol):
            log.info('Checking mail server')
            try:
                # передаем в check_server фэйковые логин и пароль,
                # но остальные настройки корректные.
                # Логин рандомный, чтобы у сборщиков не срабатывал rate limit на перебор пароля
                check_server(
                    ''.join(random.choice(string.ascii_lowercase) for _ in range(FAKE_LOGIN_LENGTH)),
                    FAKE_CREDENTIALS,
                    host,
                    port,
                    uid=None,
                    imap=imap,
                    ssl=ssl,
                )
                status = CHECK_STATUSES.ok
            except YarmLoginError:
                # если check_server вернул 'login error', значит, достучался до почтового сервера
                # но логин и пароль неверные (мы же передали фейковые)
                # возвращаем OK
                status = CHECK_STATUSES.ok
            except YarmConnectionError:
                log.trace().error('Mail server timeout')
                status = CHECK_STATUSES.timeout
            except Exception:
                log.trace().error('Checking mail server error')
                status = CHECK_STATUSES.error

            return json_response({'status': status})
