from flask import request
import string

from intranet.yandex_directory.src.yandex_directory.auth.decorators import no_auth, no_scopes
from intranet.yandex_directory.src.yandex_directory.common.exceptions import AuthenticationError
from intranet.yandex_directory.src.yandex_directory.connect_services.cloud.grpc.access import CloudAccessServiceClient
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.common.utils import json_response
from intranet.yandex_directory.src.yandex_directory.common.db import get_shard_numbers, get_main_connection
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationMetaModel
from intranet.yandex_directory.src import settings


SELECT_USER_ORGS_SQL = '''
SELECT o.id, o.name from organizations o
JOIN users u on u.org_id = o.id
WHERE o.organization_type IN ('cloud', 'cloud_partner') AND u.is_dismissed=False AND u.cloud_uid=%(cloud_uid)s
'''


def modinv(value, modulo):
    is_odd_iteration = True

    value1 = 1
    value3 = value
    modulo1 = 0
    modulo3 = modulo

    while modulo3 > 0:
        q = value3 // modulo3
        t3 = value3 % modulo3
        t1 = value1 + q * modulo1

        # swap
        value1 = modulo1
        modulo1 = t1
        value3 = modulo3
        modulo3 = t3
        is_odd_iteration = not is_odd_iteration

    if value3 != 1:
        raise Exception('modular inverse does not exist for ' + str(value))
    else:
        return value1 if is_odd_iteration else (modulo - value1)


_PRIME = 604462909807314587353111

BASE_32_DIGIT_MASK = (1 << 5) - 1

_BASE_32_DIGITS = string.digits + string.ascii_lowercase
_DIGIT_TO_INT = {_BASE_32_DIGITS[i]: i for i in range(0, 32)}
_SUFFIX_LEN = 16


def _to_base_32(number):
    result = ""
    while number != 0:
        digit = _BASE_32_DIGITS[number & BASE_32_DIGIT_MASK]
        number //= 32
        result = digit + result
    return result


def generate_tracker_id(cluster_id, connect_id):
    inverse = modinv(connect_id, _PRIME)
    suffix = _to_base_32(inverse)
    return cluster_id + "g" + suffix[:_SUFFIX_LEN].rjust(_SUFFIX_LEN, "0")


def tracker_id_to_connect_id(tracker_id):
    binary_str = ""
    for digit in tracker_id[4:]:
        binary_str = binary_str + "{0:b}".format(_DIGIT_TO_INT[digit]).rjust(5, "0")
    inversed_connect_id = int(binary_str, 2)
    return modinv(inversed_connect_id, _PRIME)


def connect_id_to_tracker_id(connect_id):
    prefix = settings.CLOUD_TRACKER_ID_PREFIX
    return generate_tracker_id(prefix, connect_id)


class UserCloudOrganizationsView(View):
    @no_auth
    @no_scopes
    def get(self, meta_connection, _):
        iam_token = request.headers.get('x-yacloud-subjecttoken')
        if not iam_token:
            raise AuthenticationError('no x-yacloud-subjecttoken header', headers={'x-yacloud-subjecttoken'})

        # спека тут
        # https://a.yandex-team.ru/arc/trunk/arcadia/cloud/bitbucket/private-api/yandex/cloud/priv/servicecontrol/v1/access_service.proto
        response = CloudAccessServiceClient().get_cloud_uid_by_iam_token(iam_token)
        cloud_uid = response.id

        shards = get_shard_numbers()
        orgs = []
        for shard in shards:
            with get_main_connection(shard=shard) as main_connection:
                shard_orgs = main_connection.execute(SELECT_USER_ORGS_SQL, cloud_uid=cloud_uid).fetchall()
            orgs.extend(shard_orgs)

        org_ids = [org['id'] for org in orgs]
        metas = []
        if org_ids:
            metas = OrganizationMetaModel(meta_connection).find({'id': org_ids}, fields=['id', 'cloud_org_id'])
        id_to_cloud_org_id = {}
        for meta in metas:
            id_to_cloud_org_id[meta['id']] = meta['cloud_org_id']

        result = [
            {
                'entityType': 'yc.tracker',
                'entityId': connect_id_to_tracker_id(row['id']),
                'cloudOrgId': id_to_cloud_org_id[row['id']],
                'displayName': row['name']
            }
            for row in orgs
        ]

        id_to_filter = request.args.get('tracker_id')
        if id_to_filter:
            result = [item for item in result if item['entityId'] == id_to_filter]

        response = {
            'entities': result,
        }

        return json_response(response)
