# -*- coding: utf-8 -*-
from copy import deepcopy
from flask import request, g
from functools import partial

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
    uses_out_schema,
)

from intranet.yandex_directory.src.yandex_directory.common.utils import (
    build_list_response,
    json_response,
    json_error,
    json_error_required_field,
    json_error_not_found,
    check_label_or_nickname_or_alias_is_uniq_and_correct,
    split_by_comma,
    to_lowercase,
    update_maillist_label,
)
from intranet.yandex_directory.src.yandex_directory.common.schemas import (
    I18N_STRING,
    INTEGER,
    INTEGER_OR_NULL,
    STRING_OR_NULL,
    DATETIME,
    STRING,
    BOOLEAN,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    CantMoveDepartmentToItself,
    UnableToCreateMailList,
    CantMoveDepartmentToDescendant,
    DepartmentNotEmpty,
    RootDepartmentCantBeRemoved,
)
from intranet.yandex_directory.src.yandex_directory.core.models.department import (
    DepartmentNotEmptyError,
    DepartmentNotFoundError,
    RootDepartmentRemovingError,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    DepartmentModel,
    OrganizationServiceModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    prepare_department,
    build_email,
    prepare_fields,
    create_maillist,
    change_object_alias,
)

from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_department_add,
    action_department_modify,
    action_department_alias_add,
    action_department_alias_delete,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    department_permissions,
    global_permissions,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import TYPE_DEPARTMENT
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    permission_required,
    no_permission_required,
    requires,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.departments import (
    is_department_equal_itself,
    is_child_department,
)
from intranet.yandex_directory.src.yandex_directory.core.tasks.move import (
    MoveUserToDepartmentTask,
    MoveDepartmentToDepartmentTask,
)

DEPARTMENT_BASE_SCHEMA = {
    'title': 'Department',
    'type': 'object',
    'properties': {
        'name': I18N_STRING,
        'description': I18N_STRING,
        'label': STRING_OR_NULL,
        'parent_id': INTEGER,
        'parent': {
            'type': ['object', 'null']
        },
        'head_id': INTEGER_OR_NULL,
        'external_id': STRING_OR_NULL,
        'maillist_type': {'enum': ['inbox', 'shared', 'both']},
    },
    'anyOf': [
        {'required': ['name', 'parent_id']},
        {'required': ['name', 'parent']}
    ],
    'additionalProperties': False,
}

DEPARTMENT_OUT_SCHEMA = deepcopy(DEPARTMENT_BASE_SCHEMA)
DEPARTMENT_OUT_SCHEMA['properties']['parent_id'] = INTEGER_OR_NULL
DEPARTMENT_OUT_SCHEMA['properties']['head'] = {'type': ['object', 'null']}
DEPARTMENT_OUT_SCHEMA['properties']['parent'] = {'type': ['object', 'null']}
DEPARTMENT_OUT_SCHEMA['properties']['parents'] = {'type': ['array', 'null']}
DEPARTMENT_OUT_SCHEMA['properties']['email'] = {'type': ['string', 'null']}
DEPARTMENT_OUT_SCHEMA['properties']['id'] = INTEGER
DEPARTMENT_OUT_SCHEMA['properties']['members_count'] = INTEGER
DEPARTMENT_OUT_SCHEMA['properties']['created'] = DATETIME
DEPARTMENT_OUT_SCHEMA['properties']['removed'] = {'type': ['boolean', 'null']}
DEPARTMENT_OUT_SCHEMA['properties']['tracker_license'] = {'type': ['boolean', 'null']}

DEPARTMENT_OUT_SCHEMA['properties']['description'] = {
    'type': ['object', 'string', 'null']
}
DEPARTMENT_OUT_SCHEMA['properties']['name'] = {
    'type': ['object', 'string', 'null']
}
DEPARTMENT_OUT_SCHEMA['properties']['label'] = STRING_OR_NULL
DEPARTMENT_OUT_SCHEMA['properties']['org_id'] = INTEGER_OR_NULL
DEPARTMENT_OUT_SCHEMA['properties']['aliases'] = {'type': ['array', 'null']}
DEPARTMENT_OUT_SCHEMA['properties']['heads_group_id'] = INTEGER_OR_NULL
DEPARTMENT_OUT_SCHEMA['properties']['path'] = STRING_OR_NULL
DEPARTMENT_OUT_SCHEMA['properties']['is_outstaff'] = BOOLEAN
DEPARTMENT_OUT_SCHEMA['properties']['uid'] = INTEGER_OR_NULL
del DEPARTMENT_OUT_SCHEMA['anyOf']

DEPARTMENT_MOVE_SCHEMA = deepcopy(DEPARTMENT_BASE_SCHEMA)
DEPARTMENT_MOVE_SCHEMA['properties'] = {
    'user_ids': {
        'type': 'array',
        'items': INTEGER
    },
    'dep_ids': {
        'type': 'array',
        'items': INTEGER
    },
    'to_dep_id': INTEGER
}
DEPARTMENT_MOVE_SCHEMA['anyOf'] = [
    {
        'required': ['user_ids', 'to_dep_id'],
    },
    {
        'required': ['dep_ids', 'to_dep_id']
    }
]

DEPARTMENT_ARRAY_OUT_SCHEMA = {
    'title': 'Department object list',
    'type': 'array',
    'items': DEPARTMENT_OUT_SCHEMA
}

DEPARTMENT_CREATE_SCHEMA = deepcopy(DEPARTMENT_BASE_SCHEMA)
DEPARTMENT_CREATE_SCHEMA['title'] = 'CreateDepartment'

DEPARTMENT_UPDATE_SCHEMA = deepcopy(DEPARTMENT_BASE_SCHEMA)
DEPARTMENT_UPDATE_SCHEMA['title'] = 'Update department'
del DEPARTMENT_UPDATE_SCHEMA['anyOf']

DEPARTMENT_CREATE_ALIAS_SCHEMA = {
    'type': 'object',
    'title': 'Add department alias',
    'properties': {
        'name': STRING,
    },
    'required': [
        'name',
    ],
    'additionalProperties': False
}


def check_that_user_exists(connection,
                           org_id,
                           user_id,
                           error_code,
                           error_message):
    if user_id is not None:
        user = UserModel(connection).get(user_id=user_id, org_id=org_id)
        if user is None:
            raise ImmediateReturn(
                json_error(
                    422,
                    error_code,
                    error_message,
                    uid=user_id,
                )
            )


def check_that_department_exists(connection,
                                 org_id,
                                 department_id,
                                 error_code,
                                 error_message):
    if not DepartmentModel(connection).count(
        filter_data={
            'id': department_id,
            'org_id': org_id
        }
    ):
        raise ImmediateReturn(
            json_error(
                422,
                error_code,
                error_message,
                id=department_id,
            )
        )


class DepartmentView(View):
    def normalization(self, data, schema=None):
        if 'parent_id' not in data and 'parent' in data:
            data['parent_id'] = data.get('parent', {}).get('id')
        if 'head_id' not in data and 'head' in data:
            data['head_id'] = data.get('head', {}).get('id')

        return super(DepartmentView, self).normalization(data, schema=schema)


class DepartmentListView(DepartmentView):
    allowed_ordering_fields = ['name', 'id', 'tracker_license']

    @uses_out_schema(DEPARTMENT_ARRAY_OUT_SCHEMA)
    @scopes_required([scope.read_departments, scope.write_departments])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Список отделов

        Максимально, на странице может выводиться до 1000 сотрудников. Количество
        данных на страницу, задается параметром per_page.

        Если передан get-параметр parent_id, то включаем фильтрацию по родителю.
        Корневой отдел организации имеет идентификатор равный 1. При создании организации корневой отдел
        создается автоматически.


        Пример выдачи:

            {
                ...
                "result": [
                    {
                        "id": 181,
                        "email": "group_entertainment",
                        "name": {
                            "en": "Group of entertainment services development",
                            "ru": "Группа разработки развлекательных сервисов"
                        },
                        "parent": {
                            id: 935,
                            "name": {
                                "en": "Service of auxiliary projects",
                                "ru": "Служба разработки вспомогательных сервисов"
                            },
                            parent_id: 100
                        }
                    }
                    ...
                ]
            }

        Поля сущности департамента:

        * **id** - идентификатор
        * **label** - уникальный человеко-читаемый id департамента ([label поле](#label-polya))
        * **email** - email рассылки департамента
        * **name** - название ([локализованное поле](#lokalizovannyie-polya))
        * **parent** - родительский департамент
        * **parents** - список родительских департаментов, начиная от корневого департамента, заканчивая непосредственным предком

        ---
        tags:
          - Отделы
        parameters:
          - in: query
            name: id
            required: false
            type: string
            description: список id отделов, перечисленных через запятую
          - in: query
            name: parent_id
            required: false
            type: string
            description: список родителей отделов, перечисленных через запятую
          - in: query
            name: page
            type: integer
            description: текущая страница
          - in: query
            name: per_page
            type: integer
            description: какое кол-во объектов выведено на странице
          - in: query
            name: ordering
            type: string
            description: сортировка выдачи по указанному полю. Поддерживаемые поля - name, id
        responses:
          200:
            description: Список департаментов
        """
        request_args = request.args.to_dict()

        response = build_list_response(
            model=DepartmentModel(main_connection),
            model_filters=self._get_filters(),
            model_fields=[
                '*',
                'parents.*',
                'parent.*',
                'head.*',
                'email',
            ],
            path=request.path,
            query_params=request_args,
            prepare_result_item_func=partial(
                prepare_department,
                main_connection,
                api_version=request.api_version,
            ),
            # здесь задан специальный лимит, потому что без этого
            # плохо работает портал:
            # https://st.yandex-team.ru/DIR-974
            max_per_page=1000,
            order_by=self._get_ordering_fields(),
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    @uses_out_schema(DEPARTMENT_ARRAY_OUT_SCHEMA)
    @scopes_required([scope.read_departments, scope.write_departments])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get_6(self, meta_connection, main_connection):
        """
        Информация обо всех департаментах организации.

        Если не передана информация о необходимых полях, будет
        возвращено только поле id для каждой организации.

        Необходимые поля возвращаются, если они указаны в поле fields.
        Например:

        ```
        /v6/departments/?fields=head,org_id,label
        ```

        По-умолчанию, количество элементов, отдаваемых на странице – 1000.

        Если в ключе "links" есть словарь с ключом "next", то необходимо получить
        следующую страницу по ссылке из этого ключа.

        Пример ответа:

            {
                u'links': {},
                u'page': 1,
                u'pages': 1,
                u'per_page': 20,
                u'result': [
                    {
                        u'aliases': [],
                        u'created': u'2017-07-14T08:42:32.136247Z',
                        u'description': None,
                        u'email': None,
                        u'external_id': None,
                        u'head': None,
                        u'heads_group_id': 1,
                        u'id': 1,
                        u'label': None,
                        u'members_count': 1,
                        u'name': u'\\u0412\\u0441\\u0435 \\u0441\\u043e\\u0442\\u0440\\u0443\\u0434\\u043d\\u0438\\u043a\\u0438',
                        u'org_id': 20802,
                        u'parent': None,
                        u'parents': [],
                        u'path': u'1',
                        u'removed': False,
                        u'maillist_type': u'inbox'
                    }
                ],
                u'total': 1
             }

        ---
        tags:
          - Отделы
        parameters:
          - in: query
            name: fields
            type: string
            description: список требуемых полей, перечисленных через запятую
        responses:
          200:
            description: Список словарей с информацией об отделах
        """
        fields = prepare_fields(request.args.get('fields'))

        query_params = request.args.to_dict()

        response = build_list_response(
            model=DepartmentModel(main_connection),
            model_filters=self._get_filters(),
            model_fields=fields,
            path=request.path,
            query_params=query_params,
            prepare_result_item_func=partial(
                prepare_department,
                main_connection,
                api_version=request.api_version,
            ),
            # здесь задан специальный лимит, потому что без этого
            # плохо работает портал:
            # https://st.yandex-team.ru/DIR-974
            max_per_page=1000,
            order_by=self._get_ordering_fields(),
        )

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    @uses_schema(DEPARTMENT_CREATE_SCHEMA)
    @scopes_required([scope.write_departments])
    @permission_required([global_permissions.add_departments])
    def post(self, meta_connection, main_connection, data):
        """
        Создать новый отдел

        Поле `name` является обязательным и не должно быть пустым.
        Если валидация не прошла, то возвращается 422 код ошибки
        и её описание в словаре `{"errors": [...]}`

        Поле `label` является не обязательным.

        ---
        tags:
          - Отделы
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Департамент создан
          403:
            description: {permissions}
          422:
            description: Какая-то ошибка валидации

        """
        org_id = g.org_id
        name = data.get('name')
        description = data.get('description')
        parent_id = data.get('parent_id')
        head_id = data.get('head_id')
        label = data.get('label')
        external_id = data.get('external_id')
        maillist_type = data.get('maillist_type')

        label = to_lowercase(label) or None

        # для организаций без домена (в метабазе у них has_owned_domains=False)
        # или с неподтвержденным доменом можно создать только отдел без label
        if not OrganizationModel(main_connection).has_owned_domains(org_id) and label:
            raise UnableToCreateMailList()

        check_label_or_nickname_or_alias_is_uniq_and_correct(
            main_connection,
            label,
            org_id,
        )

        department_model = DepartmentModel(main_connection)

        check_that_department_exists(
            main_connection,
            org_id,
            parent_id,
            error_code='parent_department_not_found',
            error_message='Unable to find parent department with id={id}'
        )

        check_that_user_exists(
            main_connection,
            org_id,
            head_id,
            error_code='head_of_department_not_found',
            error_message='Unable to find person with head_id={uid}',
        )
        uid = None
        if label and OrganizationServiceModel(main_connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
            uid = create_maillist(main_connection, org_id, label, ignore_login_not_available=True)
        department = department_model.create(
            uid=uid,
            name=name,
            label=label,
            description=description,
            parent_id=parent_id,
            org_id=org_id,
            head_id=head_id,
            external_id=external_id,
            maillist_type=maillist_type,
        )

        department_from_db = department_model.get(
            department_id=department['id'],
            org_id=org_id,
            fields=[
                '*',
                'parent.*',
                'parents.*',
                'head.*',
                'email',
            ],
        )
        action_department_add(
            main_connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value=department,
        )

        return json_response(
            prepare_department(
                main_connection,
                department_from_db,
                api_version=request.api_version,
            ),
            status_code=201,
        )

    def _get_ordering_fields(self):
        # временно переопределим метод, чтобы сортировать по нужным полям

        ordering_fields = super(DepartmentListView, self)._get_ordering_fields()
        if ordering_fields and ('name' in ordering_fields or '-name' in ordering_fields):
            name = 'name' if 'name' in ordering_fields else '-name'
            ordering_fields[ordering_fields.index(name)] = name + '_plain'
        return ordering_fields

    def _get_filters(self):
        filters = {'org_id': g.org_id}

        if request.args.get('id'):
            ids = split_by_comma(request.args.get('id'))
            if len(ids) == 1:
                ids = ids[0]
            filters['id'] = ids

        if request.args.get('parent_id'):
            parent_ids = split_by_comma(request.args.get('parent_id'))
            if len(parent_ids) == 1:
                parent_ids = parent_ids[0]
            filters['parent_id'] = parent_ids

        if request.args.get('suggest'):
            filters['suggest'] = request.args.get('suggest')

        if request.args.get('uid') and app.config['INTERNAL']:
            uid = split_by_comma(request.args.get('uid'))
            if len(uid) == 1:
                uid = uid[0]
            filters['uid'] = uid

        return filters


class DepartmentDetailView(DepartmentView):
    @uses_out_schema(DEPARTMENT_OUT_SCHEMA)
    @scopes_required([scope.read_departments, scope.write_departments])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, department_id):
        """
        Информация об отделе

        ---
        tags:
          - Отделы
        parameters:
          - in: path
            name: department_id
            required: true
            type: string
        responses:
          200:
            description: Словарь с информацией об отделе
        """
        department = DepartmentModel(main_connection).get(
            department_id=department_id,
            org_id=g.org_id,
            fields=[
                '*',
                'parent.*',
                'parents.*',
                'head.*',
                'email',
            ]
        )
        if department:
            return json_response(
                prepare_department(
                    main_connection,
                    department,
                    api_version=request.api_version,
                )
            )
        else:
            return json_error_not_found()

    @uses_out_schema(DEPARTMENT_OUT_SCHEMA)
    @scopes_required([scope.read_departments, scope.write_departments])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get_6(self, meta_connection, main_connection, department_id):
        """
        Информация об отделе

        Необходимые поля возвращаются, если они указаны в поле fields.
        Например:

        ```
        /v6/departments/12345/?fields=head,org_id,label
        ```

        Если не передана информация о необходимых полях, будет
        возвращено только поле id для каждой организации.

        Пример ответа:

        ```
        {
            u'aliases': [],
            u'created': u'2017-07-14T08:52:56.712207Z',
            u'description': None,
            u'email': None,
            u'external_id': None,
            u'head': None,
            u'heads_group_id': 1,
            u'id': 1,
            u'label': None,
            u'members_count': 1,
            u'name': u'\\u0412\\u0441\\u0435 \\u0441\\u043e\\u0442\\u0440\\u0443\\u0434\\u043d\\u0438\\u043a\\u0438',
            u'org_id': 20803,
            u'parent': None,
            u'parents': [],
            u'path': u'1',
            u'removed': False
        }
        ```

        ---
        tags:
          - Отделы
        parameters:
          - in: path
            name: department_id
            required: true
            type: string
          - in: query
            name: fields
            type: string
            description: список требуемых полей, перечисленных через запятую
        responses:
          200:
            description: Словарь с информацией об отделе
        """
        fields = prepare_fields(request.args.get('fields'))
        real_fields = fields[:]

        if 'head.contacts' in fields:
            if 'head.nickname' not in fields:
                real_fields.append('head.nickname')
            if 'head.org_id' not in fields:
                real_fields.append('head.org_id')

        department = DepartmentModel(main_connection).get(
            department_id=department_id,
            org_id=g.org_id,
            fields=real_fields,
        )
        if department:
            prepared_department = prepare_department(
                main_connection,
                department,
                api_version=request.api_version,
            )

            if 'head.contacts' in fields and prepared_department['head']:
                if 'head.nickname' not in fields:
                    prepared_department['head'].pop('nickname', None)
                if 'head.org_id' not in fields:
                    prepared_department['head'].pop('org_id', None)

            return json_response(prepared_department)
        else:
            return json_error_not_found()

    @uses_schema(DEPARTMENT_UPDATE_SCHEMA)
    @scopes_required([scope.write_departments])
    @permission_required([department_permissions.edit], TYPE_DEPARTMENT)
    @requires(org_id=True, user=True)
    def patch(self, meta_connection, main_connection, data, department_id):
        """
        Изменить информацию об отделе

        Чтобы сменить руководителя, нужно передать его UID в поле
        head_id. Если нужно убрать руководителя, но никем не заменять,
        то в этом поле должен быть передан null.

        ---
        tags:
          - Отделы
        parameters:
          - in: path
            name: department_id
            required: true
            type: string
          - in: body
            name: body
        responses:
          200:
            description: Департамент обновлен
          422:
            description: Параметры не соответствуют схеме
          403:
            description: {permissions}

        """
        org_id = g.org_id
        department_model = DepartmentModel(main_connection)

        old_department = department_model.get(
            department_id=department_id,
            org_id=org_id,
            fields=[
                '*',
                'parent.*',
                'parents.*',
                'head.*',
            ],
        )
        if not old_department:
            return json_error_not_found()

        update_dict = {}

        if 'name' in data:
            update_dict['name'] = data['name']
        if 'external_id' in data:
            update_dict['external_id'] = data['external_id']
        if 'description' in data:
            update_dict['description'] = data['description']
        if 'label' in data:
            label = to_lowercase(data['label']) or None
            update_dict['label'] = label
            if label != old_department['label']:
                update_dict = update_maillist_label(
                    main_connection,
                    label,
                    org_id,
                    update_dict,
                    old_department['uid']
                )
        if 'head_id' in data:
            update_dict['head_id'] = data['head_id']
            check_that_user_exists(
                main_connection,
                org_id,
                update_dict['head_id'],
                error_code='head_of_department_not_found',
                error_message='Unable to find person with head_id={uid}',
            )
        if 'parent_id' in data:
            update_dict['parent_id'] = data['parent_id']
            check_that_department_exists(
                main_connection,
                org_id,
                data['parent_id'],
                error_code='parent_department_not_found',
                error_message='Unable to find parent department with id={id}'
            )
        if 'maillist_type' in data:
            update_dict['maillist_type'] = data['maillist_type']

        department_model.update_one(
            id=department_id,
            org_id=org_id,
            data=update_dict
        )
        department_from_db = department_model.get(
            department_id=department_id,
            org_id=org_id,
            fields=[
                '*',
                'parent.*',
                'parents.*',
                'head.*',
                'email',
            ],
        )
        action_department_modify(
            main_connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value=department_from_db,
            old_object=old_department,
        )
        return json_response(
            prepare_department(
                main_connection,
                department_from_db,
                api_version=request.api_version,
            )
        )

    def _get_filters(self):
        filters = {'org_id': g.org_id}

        if request.args.get('id'):
            ids = request.args.get('id').split(',')
            if len(ids) == 1:
                ids = ids[0]
            filters['id'] = ids

        if request.args.get('parent_id'):
            parent_ids = request.args.get('parent_id').split(',')
            if len(parent_ids) == 1:
                parent_ids = parent_ids[0]
            filters['parent_id'] = parent_ids

        return filters

    @scopes_required([scope.write_departments])
    @permission_required([global_permissions.remove_departments], TYPE_DEPARTMENT)
    @requires(org_id=True, user=False)
    def delete(self, meta_connection, main_connection, department_id):
        """
        Удалить отдел
        ---
        tags:
          - Отделы
        parameters:
          - in: path
            name: department_id
            required: true
            type: string
        responses:
          200:
            description: Отдел удален.
          404:
            description: Отдел не найден.
          422:
            description: Невозможно удаление этого отдела (Отдел должен быть пустым для удаления).
          403:
            description: {permissions}
        """
        try:
            DepartmentModel(main_connection).remove(
                department_id=department_id,
                org_id=g.org_id,
                author_uid=g.user.passport_uid,
                force=False,
                real_remove=False,
            )
            # TODO: надо в проверках на уровене модели
            #       бросать ConstraintValidationError
        except DepartmentNotFoundError:
            return json_error_not_found()

        except DepartmentNotEmptyError:
            raise DepartmentNotEmpty()

        except RootDepartmentRemovingError:
            raise RootDepartmentCantBeRemoved()

        else:
            return json_response({}, status_code=204)


class DepartmentBulkMove(DepartmentView):
    # Эту ручку мы пока не хотим выставлять наружу
    @internal
    @uses_schema(DEPARTMENT_MOVE_SCHEMA)
    @scopes_required([scope.write_departments])
    @permission_required([department_permissions.edit], TYPE_DEPARTMENT)
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Перемещение департаментов и сотрудников в
        другой департамент.

        На входе принимает словарь с такими ключами:

        user_ids: список id-шников пользователей, которых хотим переместить.
        dep_ids: id-шники департаментов, которые хотим переместим.
        to_dep_id: id-департамента, в который хотим всех перенести.

        Возвращает словарь: {"tasks": [1, 2, 3,...]} с id асинхронных задач
        созданных для перемещения отделов и пользователей.

        ---
        tags:
          - Отделы
        parameters:
          - in: body
            name: body
        responses:
          202:
            description: Таски по перемещению успешно созданы, но выполнение еще не закончено.
          422:
            description: Параметры не соответствуют схеме
          409:
            description: Произошла ошибка в момент перемещения одного из поддепартаментов
          403:
            description: {permissions}

        """
        org_id = g.org_id

        user_ids = data.get('user_ids', [])
        dep_ids = data.get('dep_ids', [])
        to_dep_id = data.get('to_dep_id', None)

        if not to_dep_id:
            return json_error_required_field()

        check_that_department_exists(
            main_connection,
            org_id,
            to_dep_id,
            error_code='department_not_found',
            error_message='Unable to find department with id={id}'
        )

        for dep_id in dep_ids:
            if is_department_equal_itself(dep_id, to_dep_id):
                raise CantMoveDepartmentToItself()

            if is_child_department(main_connection, org_id, dep_id, to_dep_id):
                raise CantMoveDepartmentToDescendant(child_id=to_dep_id, parent_id=dep_id)

        #  После успешных проверок, запускаем перемещение департаментов и сотрудников
        all_tasks = []
        for dep_id in dep_ids:
            task = MoveDepartmentToDepartmentTask(main_connection).delay(
                org_id=org_id,
                dep_id=dep_id,
                parent_dep_id=to_dep_id,
                author_id=g.user.passport_uid,
            )
            all_tasks.append(task.task_id)

        for user_id in user_ids:
            task = MoveUserToDepartmentTask(main_connection).delay(
                org_id=org_id,
                user_id=user_id,
                parent_dep_id=to_dep_id,
                author_id=g.user.passport_uid,
            )
            all_tasks.append(task.task_id)

        #  TODO: переделать в соотв. с таском: https://st.yandex-team.ru/DIR-3635
        return json_response({'tasks': all_tasks}, 202)


class DepartmentAliasesListView(DepartmentView):
    @uses_schema(DEPARTMENT_CREATE_ALIAS_SCHEMA)
    @scopes_required([scope.write_departments])
    @permission_required([department_permissions.edit], TYPE_DEPARTMENT)
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data, department_id):
        """
        Добавить алиас отделу
        ---
        tags:
          - Отделы
        parameters:
          - in: path
            name: department_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          201:
            description: Алиас добавлен
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """
        updated_department = change_object_alias(
            main_connection=main_connection,
            obj_id=department_id,
            obj_type=TYPE_DEPARTMENT,
            org_id=g.org_id,
            alias=data['name'],
            action_name='add',
            action_func=action_department_alias_add,
            author_id=g.user.passport_uid,
        )

        return json_response(
            data=prepare_department(
                main_connection,
                updated_department,
                api_version=request.api_version,
            ),
            status_code=201
        )


class DepartmentAliasesDetailView(DepartmentView):
    @scopes_required([scope.write_departments])
    @permission_required([department_permissions.edit], TYPE_DEPARTMENT)
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, department_id, alias):
        """
        Удалить алиас отдела

        ---
        tags:
          - Отделы
        parameters:
          - in: path
            name: department_id
            required: true
            type: integer
          - in: path
            name: alias
            required: true
            type: string
        responses:
          204:
            description: Алиас удален
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """

        change_object_alias(
            main_connection=main_connection,
            obj_id=department_id,
            obj_type=TYPE_DEPARTMENT,
            org_id=g.org_id,
            alias=alias,
            action_name='delete',
            action_func=action_department_alias_delete,
            author_id=g.user.passport_uid,
        )

        return json_response({}, status_code=204)
