# coding: utf-8

from flask import g
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from datetime import datetime

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_scopes,
    no_permission_required,
    requires,
    internal,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    DiskUsageModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    utcnow,
    json_response,
    get_object_or_404,
)
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema
from intranet.yandex_directory.src.yandex_directory.common import schemas


DISK_USAGE_SCHEMA = {
    'title': 'DiskUsage',
    'type': 'object',
    'properties': {
        'usage': schemas.INTEGER,
    },
    'required': ['usage'],
}


class DiskUsageView(View):
    methods = ['get', 'post']

    @internal
    @no_permission_required
    @no_scopes
    @requires(org_id=True, user=False)
    def get_4(self, meta_connection, main_connection):
        """
        Получить данные про использование пространства в облаке.

        Пример ответа:

            {
                "free": 4200,
                "limit": 100500,
            }

        ---
        tags:
            - Дисковое пространство
        responses:
            200:
                description: Корректный ответ.
        """

        return self._get_disk_usage(
            main_connection,
            g.org_id,
        )


    @internal
    @no_permission_required
    @requires(org_id=True, user=False)
    @no_scopes
    @uses_schema(DISK_USAGE_SCHEMA)
    def post_4(self, meta_connection, main_connection, data):
        """
        Зарепортить использование диска в конкретном сервисе.

        Ручка должна дёргаться сервисом, когда изменяется объем используемого
        организацией дискового пространства. Например после загрузки аттача.

        Можно дергать её не синхронно, а раз в сутки.

        В теле POST запроса должен быть JSON:

            {{
                "usage": 100500,
            }}

        Где цифра - количество байт которые занимают данные организации.

        Ответ ручки, такой же как и на GET.
        Пример ответа:

            {{
                "free": 4200,
                "limit": 100500,
            }}

        ---
        tags:
            - Дисковое пространство
        responses:
            200:
                description: Корректный ответ.
        """

        usage = data['usage']

        model = DiskUsageModel(main_connection)
        filter_data = {'org_id': g.org_id, 'service_id': g.service.id}
        obj = model.find(filter_data, one=True)
        if obj:
            model.update(
                update_data={'usage': usage, 'updated': utcnow()},
                filter_data=filter_data,
            )
        else:
            model.create(
                org_id=g.org_id,
                service_id=g.service.id,
                usage=usage,
            )

        model.update_organization_limits(g.org_id)

        return self._get_disk_usage(
            main_connection,
            g.org_id,
        )


    def _get_disk_usage(self, main_connection, org_id):
        """Вспомогательная функция, для того,
        чтобы формировать одинаковый ответ и в GET и в POST ручке.
        """
        data = get_object_or_404(
            model_instance=OrganizationModel(main_connection),
            id=org_id,
            fields=['disk_usage', 'disk_limit'],
        )
        usage = data['disk_usage']
        limit = data['disk_limit']

        response = {
            'free': limit - usage,
            'limit': limit,
        }
        return json_response(response)
