# coding: utf-8

from flask import g

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    permission_required,
    scopes_required,
    internal,
    requires,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    DomainOccupied, DomainAddToSsoOrganization,
)
from intranet.yandex_directory.src.yandex_directory.common.schemas import (
    STRING,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    check_domain_is_correct,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_domain_add,
)
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationModel
from intranet.yandex_directory.src.yandex_directory.core.models.domain import (
    DomainModel,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    global_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import get_organization_admin_uid
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
)
from intranet.yandex_directory.src.yandex_directory.webmaster import (
    set_domain_as_owned_in_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.features import is_feature_enabled, USE_DOMENATOR
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.utils.organization import is_sso_turned_on

DOMAIN_ADD_SCHEMA = {
    'title': 'Domain',
    'type': 'object',
    'properties': {
        'name': STRING,
    },
    'required': ['name'],
    'additionalProperties': False
}


class AddOwnedDomainsView(View):

    @internal
    @uses_schema(DOMAIN_ADD_SCHEMA)
    @scopes_required([scope.pdd_proxy])
    @permission_required([global_permissions.add_domains])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Добавить подтвержденный домен.

        Возможные коды ошибок:
        * **already_alias** - добавляемый домен уже является чьим-то алиасом
        * **not_found_in_dns** - домен не нашелся в DNS
        * **occupied** - такой домен уже занят другим админом
        * **domain_prohibited** - домен невалидный
        * **bad_domain** - не соответствующее RFC доменное имя
        * **alias_is_self** - добавляемый алиас совпадает с доменом
        * **alias_not_empty** - невозможно сделать алиасом, т.к. на домене есть ящики
        * **alias_blocked** - нельзя добавить в качестве алиаса, т.к. домен заблокирован
        * **alias_in_connect** - нельзя сделать алиасом, т.к. такой домен уже подключен к Коннекту
        * **duplicate_domain** - для этой организации домен уже добавлен

        ---
        tags:
          - Домены
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Домен добавлен
          422:
            description: Какая-то ошибка валидации
          409:
            description: Логическая ошибка
          403:
            description: Нет прав на добавление домена
        """
        domain_name = data['name'].lower()
        org_id = g.org_id

        existent_domain = DomainModel(None) \
            .filter(name=domain_name) \
            .one()

        if existent_domain and existent_domain['owned']:
            raise DomainOccupied(domain=domain_name)

        if is_sso_turned_on(main_connection, org_id):
            raise DomainAddToSsoOrganization()

        admin_uid = get_organization_admin_uid(main_connection, org_id)

        if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            app.domenator.add_owned_domain(org_id, admin_uid, domain_name)
            return json_response(
                {},
                status_code=201,
            )

        # проверяем, корректный ли домен
        check_domain_is_correct(
            meta_connection,
            main_connection,
            domain_name,
            admin_uid,
            org_id,
            ignore_duplicate_domain_in_same_organization=True
        )

        if not existent_domain or existent_domain['org_id'] != org_id:
            domain_model = DomainModel(main_connection)
            domain = domain_model.create(
                domain_name,
                org_id,
                owned=False,
                via_webmaster=True,
            )

            action_domain_add(
                main_connection,
                org_id=org_id,
                author_id=g.user.passport_uid,
                object_value=domain,
                old_object=None,
            )

        organization = OrganizationModel(main_connection).get(org_id)
        set_domain_as_owned_in_connection(
            meta_connection,
            main_connection,
            org_id,
            organization['admin_uid'],
            domain_name,
        )

        return json_response(
            {},
            status_code=201,
        )
