# coding: utf-8
from binascii import Error as BinasciiError

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import internal, requires, scopes_required, no_permission_required
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.components import component_registry
from intranet.yandex_directory.src.yandex_directory.common.exceptions import NotFoundError
from intranet.yandex_directory.src.yandex_directory.common.utils import json_response
from intranet.yandex_directory.src.yandex_directory.core.registrar import CryptToken, PlainToken
from intranet.yandex_directory.src.yandex_directory.core.views.base import View, no_cache
from intranet.yandex_directory.src.yandex_directory.meta.repositories import DomainTokenRepository
from intranet.yandex_directory.src.yandex_directory.meta.models import DomainToken


class GetDomainAndAdminUidView(View):

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_domain_token])
    @no_permission_required
    @no_cache
    def get(self, meta_connection, _, token, pdd_version):
        """
        Получить домен и uid того кто сгенерил токен по токену и версии пдд.

        Пример выдачи:

            {
                "uid": 12345,
                "domain": "my_domain.com',
            }

        ---
        tags:
          - Доменные токены
        parameters:
          - in: path
            name: token
            required: true
            type: string
          - in: path
            name: pdd_version
            required: true
            type: string
            enum: ['old', 'new']
        responses:
           200:
              description:  Информация о домене.
        """
        if app.config['DOMENATOR_DOMAIN_TOKEN_PROXY']:
            admin_id, domain = app.domenator.get_token_info(token, pdd_version)
            return json_response({
                'uid': admin_id,
                'domain': domain,
            })

        domain_token_repository = component_registry().domain_token_repository  # type: DomainTokenRepository

        try:
            if pdd_version == 'new':
                serialized_token = CryptToken(domain_token_repository).serialize_token(token)
            else:
                serialized_token = PlainToken(domain_token_repository).serialize_token(token)
        except (TypeError, BinasciiError):
            # Эта ошибка кидается в тех случаях, когда с токеном что-то не так и он не может быть
            # декодирован из base64
            raise NotFoundError()

        domain_token = domain_token_repository.find_one_by_token_and_pdd_version(serialized_token, pdd_version)
        if not domain_token:
            raise NotFoundError()

        return json_response({
            'uid': domain_token.admin_id,
            'domain': domain_token.domain,
        })


class DomainTokenView(View):
    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_domain_token])
    @no_permission_required
    def post(self, meta_connection, _, uid, domain_name, pdd_version):
        """
        Получить токен.

        Пример выдачи:

            {
                "token": "some-token",
            }

        ---
        tags:
          - Доменные токены
        parameters:
          - in: path
            name: uid
            required: true
            type: integer
          - in: path
            name: domain_name
            required: true
            type: string
          - in: path
            name: pdd_version
            required: true
            type: string
            enum: ['old', 'new']
        responses:
           200:
              description:  Токен.
        """

        admin_id = uid
        if app.config['DOMENATOR_DOMAIN_TOKEN_PROXY']:
            token = app.domenator.create_token(admin_id, domain_name, pdd_version)
        else:
            domain_token_repository = component_registry().domain_token_repository  # type: DomainTokenRepository

            if pdd_version == 'new':
                token_manager = CryptToken(domain_token_repository)
            else:
                token_manager = PlainToken(domain_token_repository)

            domain_token = domain_token_repository.find_one_by_pdd_info(admin_id, domain_name, pdd_version)

            if domain_token:
                token = token_manager.unserialize_token(domain_token.token)
            else:
                token = token_manager.generate_token(domain_name, admin_id)

                domain_token = DomainToken(
                    pdd_version=pdd_version,
                    admin_id=admin_id,
                    domain=domain_name,
                    token=token_manager.serialize_token(token),
                )

                domain_token_repository.save(domain_token)
        if isinstance(token, bytes):
            token = token.decode('utf-8')
        return json_response({
            'token': token,
        })

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_domain_token])
    @no_permission_required
    def delete(self, meta_connection, _, uid, domain_name, pdd_version):
        """
        Удалить токен.

        ---
        tags:
          - Доменные токены
        parameters:
          - in: path
            name: uid
            required: true
            type: integer
          - in: path
            name: domain_name
            required: true
            type: string
          - in: path
            name: pdd_version
            required: true
            type: string
            enum: ['old', 'new']
        responses:
           200:
              description:  Токен удален.
        """

        admin_id = uid

        if app.config['DOMENATOR_DOMAIN_TOKEN_PROXY']:
            app.domenator.delete_token(admin_id, domain_name, pdd_version)

        else:
            domain_token_repository = component_registry().domain_token_repository  # type: DomainTokenRepository

            domain_token = domain_token_repository.find_one_by_pdd_info(admin_id, domain_name, pdd_version)

            if not domain_token:
                raise NotFoundError

            domain_token_repository.delete(domain_token)

        return json_response({}, status_code=204)
