# -*- coding: utf-8 -*-

from flask import g, request

from intranet.yandex_directory.src.yandex_directory import webmaster, app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    permission_required,
    no_permission_required,
    scopes_required,
    requires,
    internal,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope,
    check_scopes, ScopeCheckMethod
)
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_shard, get_shard_numbers,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    NoScopesError,
    DomainNotFound,
    DomainAddToSsoOrganization,
    ParameterNotCorrect,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import (
    UnknownFieldError,
)
from intranet.yandex_directory.src.yandex_directory.common.schemas import (
    STRING,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    to_punycode,
    json_response,
    json_error,
    json_error_not_found,
    create_domain_in_passport,
    check_domain_is_correct, get_domain_info_from_blackbox,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_domain_add,
)
from intranet.yandex_directory.src.yandex_directory.core.models.domain import (
    DomainModel,
    WebmasterDomainLogModel,
    domain_action,

)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    global_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.tasks.tasks import (
    SyncSingleDomainTask,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_organization_admin_uid,
    prepare_fields,
    prepare_domain_with_fields,
    after_first_domain_became_confirmed,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.organization import is_sso_turned_on
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema, uses_schema_for_get,
)

from intranet.yandex_directory.src.yandex_directory.webmaster import (
    WebmasterError,
)

from .logic import create_domain_verified_via_webmaster
from ...features import is_feature_enabled, USE_DOMENATOR
from ...models import OrganizationModel
from ...utils.domain import domain_is_tech

DOMAIN_ADD_SCHEMA = {
    'title': 'Domain',
    'type': 'object',
    'properties': {
        'name': STRING,
    },
    'required': ['name'],
    'additionalProperties': False
}

SUPPORTED_CHECKS = ['webmaster.' + check.lower()
                    for check in webmaster.ALLOWED_VERIFICATION_TYPES]

DOMAIN_CHECK_OWNERSHIP_SCHEMA = {
    'title': 'Domain',
    'type': 'object',
    'properties': {
        'verification_type': {
            'enum': SUPPORTED_CHECKS
        },
    },
    'required': ['verification_type'],
    'additionalProperties': False
}

DOMAIN_SEARCH_SCHEMA = {
    'title': 'Domain',
    'type': 'object',
    'properties': {
        'domains': {
            'type': ['array'],
            'items': {'type': ['string']},
        },
        'fields': {
            'type': ['array'],
            'items': {'type': ['string']},
        },
    },
    'required': ['domains'],
    'additionalProperties': False,
}


class DomainsListView(View):

    @uses_schema(DOMAIN_ADD_SCHEMA)
    @scopes_required([scope.write_domains])
    @permission_required([global_permissions.add_domains])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Добавить новый домен.

        Возможные коды ошибок:
        * **already_alias** - добавляемый домен уже является чьим-то алиасом
        * **not_found_in_dns** - домен не нашелся в DNS
        * **occupied** - такой домен уже занят другим админом
        * **domain_prohibited** - домен невалидный
        * **bad_domain** - не соответствующее RFC доменное имя
        * **alias_is_self** - добавляемый алиас совпадает с доменом
        * **alias_not_empty** - невозможно сделать алиасом, т.к. на домене есть ящики
        * **alias_blocked** - нельзя добавить в качестве алиаса, т.к. домен заблокирован
        * **alias_in_connect** - нельзя сделать алиасом, т.к. такой домен уже подключен к Коннекту
        * **duplicate_domain** - для этой организации домен уже добавлен

        ---
        tags:
          - Домены
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Домен добавлен
          422:
            description: Какая-то ошибка валидации
          409:
            description: Логическая ошибка
          403:
            description: Нет прав на добавление домена
        """

        name = data['name']
        org_id = g.org_id

        if is_sso_turned_on(main_connection, org_id):
            raise DomainAddToSsoOrganization()

        if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            admin_uid = get_organization_admin_uid(main_connection, org_id)
            app.domenator.add_domain(org_id, name, admin_uid)
            OrganizationModel(main_connection).increment_revision(org_id=org_id)
        else:
            create_domain_verified_via_webmaster(meta_connection, main_connection, name, org_id)

        return json_response(
            {'name': name.lower(), 'owned': False},
            status_code=201,
        )

    @no_permission_required
    @scopes_required([scope.read_domains, scope.write_domains])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Список доменов.

        Пример выдачи:

            [
                {
                    "can_users_change_password": true,
                    "country": "ru",
                    "default_theme": null,
                    "display": false,
                    "imap_enabled": false,
                    "logo_url": null,
                    "master": false,
                    "name": "wololo.test",
                    "pop_enabled": false,
                    "postmaster_uid": 0,
                    "roster_enabled": false,
                    "owned": true,
                    "delegated": false,
                    "mx": true,
                },
                ...
            ]

        Поля в выдаче:
        * **name** - fqdn
        * **display** - отображаемый домен. Значит, что во всех смежных
        сервисах этот домен будет отображаться для доменной части email-а.
        (Зачем такое и всякие подробности тут:
        https://st.yandex-team.ru/PASSP-12673)
        * **master** - головной или технический домен, как он есть в Паспорте.
        * **owned** - Признак в базе Директории: подтвержден домен или нет.
        * **mx** - MX-ы домена = mx.yandex.net.
        * **delegated** - NS записи смотрят на dns1.yandex.net и dns2.yandex.net
        (если delegated==True, то домен подтверждает автоматически и MX-запись
        начинает смотреть на mx.yandex.net)

        ---
        tags:
          - Домены
        responses:
          200:
            description: Список доменов
        """
        org_id = g.org_id

        results = []
        domain_model = DomainModel(main_connection)
        if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            domains = app.domenator.get_domains(
                org_ids=[org_id],
                fields=['name', 'display', 'master', 'validated', 'via_webmaster', 'created_at', 'validated_at',
                        'owned', 'mx', 'delegated'],
            )
        else:
            domains = domain_model.find(filter_data={'org_id': org_id})

        for domain in domains:
            lower_domain_name = domain['name'].lower()

            results.append({
                'name': lower_domain_name,
                'display': domain['display'],
                'master': domain['master'],
                'tech': domain_is_tech(lower_domain_name),

                # TODO: почти все данные тут фейковые
                # ручку надо депрекейтить
                # mx, delegated, pop_enabled, imap_enabled, roster_enabled,postmaster_uid мы больше не знаем
                'validated': domain['validated'],
                'via_webmaster': domain['via_webmaster'],
                'created_at': domain['created_at'],
                'validated_at': domain['validated_at'],
                'owned': domain['owned'],
                'mx': domain['mx'],
                'delegated': domain['delegated'],
                'country': 'ru',
                'can_users_change_password': 'yes',
                'pop_enabled': 0,
                'imap_enabled': 0,
                'roster_enabled': 'no',
                'postmaster_uid': 0,
            })

        return json_response(results, allowed_sensitive_params=['can_users_change_password'])

    @no_permission_required
    @scopes_required([scope.read_domains, scope.write_domains])
    @requires(org_id=False, user=False)
    def get_6(self, meta_connection, main_connection):
        """
        Список доменов

        Пример выдачи:

            [
                {
                    "can_users_change_password": true,
                    "country": "ru",
                    "default_theme": null,
                    "display": false,
                    "imap_enabled": false,
                    "logo_url": null,
                    "master": false,
                    "name": "wololo.test",
                    "pop_enabled": false,
                    "postmaster_uid": 0,
                    "roster_enabled": false,
                    "owned": true,
                    "delegated": false,
                    "mx": true,
                },
                ...
            ]

        Поля в выдаче:
        * **name** - имя домена
        * **master** - головной или технический домен, как он есть в Паспорте.
        * **owned** - Признак в базе Директории: подтвержден домен или нет.
        * **mx** - MX-ы домена = mx.yandex.net.
        * **delegated** - NS записи смотрят на dns1.yandex.net и dns2.yandex.net
        (если delegated==True, то домен подтверждает автоматически и MX-запись
        начинает смотреть на mx.yandex.net)

        Из ПДД:
        * **country** - страна домена определяет язык интерфейса
        по-умолчанию для всех ящиков в домене
        * **can_users_change_password** - запрет/разрешение на смену пароля
        для пользователей
        * **pop_enabled** - включён ли доступ по протоколу POP на домене
        * **imap_enabled** - включён ли доступ по протоколу IMAP на домене
        * **roster_enabled** - включён ли общий Jabber-ростер на домене
        (т.к. jabber хотят закрыть, то возможно не нужен этот статус)
        * **postmaster_uid** - **default_uid** в терминах ПДД. uid
        умолчательного ящика в домене (ящика, на который приходят письма на
        несуществующие адреса), либо 0, если такой ящик не настроен.
        Поле переименовано и не как в ПДД, т.к. ящик по-умолчанию и
        "аккаунт-помойка" для недошедших писем - разные вещи.

        В этой версии можно указать нужные поля.
        ---
        tags:
          - Домены
        parameters:
          - in: query
            name: fields
            required: False
            type: string
          - in: query
            name: name
            required: False
            type: string
          - in: query
            name: registrar_id
            required: False
            type: string
        responses:
          200:
            description: Список доменов
        """
        org_id = None
        single_org_mode=False
        if request.headers.get('x-org-id') and g.org_id:
            org_id = g.org_id
            single_org_mode=True
        registrar_id = request.args.get('registrar_id')
        if registrar_id and not check_scopes(g.scopes, [scope.manage_registrar]):
            raise NoScopesError(scope.manage_registrar)
        domain_name = request.args.get('name')
        admin_uid = g.user and g.user.passport_uid
        org_ids = g.user and g.user.org_ids or []

        # нет организаций для поиска
        if not (org_id or org_ids or registrar_id):
            return json_response([])

        filter_data = {}

        # DIR-5369 теперь эта ручка может вызываться в нескольких вариантах
        # org_id или имя domain + внешний админ
        # или с параметром registrar_id
        if registrar_id is not None:
            org_ids = []
            shards = get_shard_numbers()
            for shard in shards:
                with get_main_connection(shard) as connection:
                    organizations = OrganizationModel(connection).find(fields=['id'], filter_data={
                        'registrar_id': registrar_id
                    })
                    for organization in organizations:
                        org_ids.append(organization['id'])
        elif org_id is not None:
            org_ids = [org_id]
        elif admin_uid is not None:
            org_ids = g.user.org_ids

        # Фильтр по имени можно применить в любом случае
        if domain_name is not None:
            filter_data['name'] = domain_name

        fields = prepare_fields(request.args.get('fields'), default_fields=['name'])
        if 'org_id' not in fields:
            fields.append('org_id')

        shard_by_org_id = {}
        if 'country' in fields:
            for org_id in org_ids:
                shard_by_org_id[org_id] = get_shard(meta_connection, org_id)

        supported_fields = {'tech', 'country', 'can_users_change_password', 'pop_enabled', 'imap_enabled',
                            'postmaster_uid', 'domain_id'}
        supported_fields.update(DomainModel.all_fields)
        # Поля не поддерживаемые в 6 версии
        unsupported_fields = ['display', 'validated', 'via_webmaster', 'validated_at', 'created_at', 'blocked_at']
        for field in unsupported_fields:
            supported_fields.remove(field)
        for field in fields:
            if field not in supported_fields:
                raise UnknownFieldError(DomainModel.__class__, field, supported_fields)

        model_fields = [field for field in fields if field in DomainModel.all_fields]
        if 'org_id' not in model_fields:
            model_fields.append('org_id')
        if 'owned' not in model_fields:
            model_fields.append('owned')

        connect_org_ids = []
        domenator_org_ids = []
        for org_id in org_ids:
            if app.config['USE_DOMENATOR_ENABLED_FOR_DOMAINS'] and is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
                domenator_org_ids.append(org_id)
            else:
                connect_org_ids.append(org_id)

        def prepare_connect_domain(domain):
            if main_connection is not None:
                return prepare_domain_with_fields(
                    main_connection,
                    domain,
                    fields=fields,
                )
            # поиске по всем шардам
            org_id = domain['org_id']
            g.org_id = org_id  # используется в prepare_domain_with_fields

            if 'country' in fields:
                shard = shard_by_org_id[org_id]
                with get_main_connection(shard) as connection:
                    return prepare_domain_with_fields(
                        connection,
                        domain,
                        fields=fields,
                    )
            else:
                return prepare_domain_with_fields(
                    None,
                    domain,
                    fields=fields,
                )


        def prepare_domenator_domain(domain):
            prepared_domain = domain
            if 'can_users_change_password' in fields:
                prepared_domain['can_users_change_password'] = True
            if 'pop_enabled' in fields:
                prepared_domain['pop_enabled'] = False
            if 'imap_enabled' in fields:
                prepared_domain['imap_enabled'] = False
            if 'roster_enabled' in fields:
                prepared_domain['roster_enabled'] = False
            if 'postmaster_uid' in fields:
                prepared_domain['postmaster_uid'] = 0
            if 'country' in fields:
                with get_main_connection(shard=shard_by_org_id[domain['org_id']]) as main_connection:
                    country = OrganizationModel(main_connection) \
                        .filter(id=domain['org_id']) \
                        .scalar('country')[0]
                prepared_domain['country'] = country or 'ru'
            return prepared_domain

        domains = []
        if connect_org_ids:
            if not single_org_mode:
                main_connection = None
            filter_data = {'org_id': connect_org_ids}
            if domain_name:
                filter_data['name'] = domain_name

            connect_domains = DomainModel(main_connection).find_all(filter_data=filter_data, fields=model_fields)
            prepared_connect_domains = list(map(prepare_connect_domain, connect_domains))
            domains.extend(prepared_connect_domains)

        if domenator_org_ids:
            domenator_domains = app.domenator.get_domains(
                org_ids=domenator_org_ids,
                domain=filter_data.get('name'),
                fields=fields
            )
            prepared_domenator_domains = list(map(prepare_domenator_domain, domenator_domains))
            domains.extend(prepared_domenator_domains)

        return json_response(domains, allowed_sensitive_params={'can_users_change_password'})


class DomainsSearchView(View):
    @internal
    @no_permission_required
    @uses_schema(DOMAIN_SEARCH_SCHEMA)
    @scopes_required([scope.read_domains, scope.work_with_any_organization], method=ScopeCheckMethod.AND)
    @requires(org_id=False, user=False)
    def post(self, meta_connection, main_connection, data):
        fields = data.get('fields', ['org_id'])
        domain_names = data['domains']

        domains = DomainModel(None).find_all(
            filter_data={'name__in': domain_names},
            fields=['name'] + fields
        )

        result = dict()
        for domain_info in domains:
            name = domain_info['name']

            if name not in result:
                result[name] = []

            result[name].append(
                {field: domain_info[field] for field in fields}
            )

        return json_response(result)


class DomainDetailView(View):

    @scopes_required([scope.write_domains])
    @permission_required([global_permissions.remove_domain])
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, domain_name):
        """
        Удалить домен.
        Пример выдачи:
            {
                "domain": "domain.com"
            }
        Возможные коды ошибок:
        * **not_found** - домен не найден в базе Директории
        * **master** - нельзя удалять отображаемый мастер домен
        * **alias_blocked** - нельзя удалить по причине заблокированности домена
        ---
        tags:
          - Домены
        parameters:
          - in: path
            name: domain_name
            required: true
            type: string
        responses:
          200:
            description: Алиас домена удален.
          404:
            description: Алиас домена не найден.
          422:
            description: Невозможно удаление этого алиаса.
          403:
            description: Нет права на удаление.
        """
        # TODO: https://st.yandex-team.ru/DIR-2028

        if is_feature_enabled(meta_connection, g.org_id, USE_DOMENATOR):
            admin_uid = get_organization_admin_uid(main_connection, g.org_id)
            app.domenator.delete_domain(domain_name, g.org_id, admin_uid, g.user.passport_uid)
        else:
            try:
                DomainModel(main_connection).delete_domain(domain_name, org_id=g.org_id, author_id=g.user.passport_uid)
            except DomainNotFound:
                return json_error_not_found()

        return json_response({'domain': domain_name})


class DomainCheckOwnershipView(View):
    @internal
    @uses_schema(DOMAIN_CHECK_OWNERSHIP_SCHEMA)
    @scopes_required([scope.write_domains])
    @permission_required([global_permissions.add_domains])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data, domain_name):
        """Настоящий докстринг будет ниже."""
        org_id = g.org_id
        lower_domain_name = domain_name.lower()
        admin_id = get_organization_admin_uid(main_connection, org_id)

        if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            app.domenator.check_ownership(org_id, lower_domain_name, data['verification_type'])
        else:
            # Разобъем способ подтверждения на webmaster и код самого способа подтверждения
            via, verification_type = data['verification_type'].split('.', 1)
            domain_model = DomainModel(main_connection)
            domain = domain_model.get(lower_domain_name, g.org_id)
            if not domain:
                return json_error_not_found()

            # Запустим в вебмастере проверку
            webmaster.verify(
                lower_domain_name,
                admin_id,
                verification_type,
            )

            WebmasterDomainLogModel(main_connection).create(
                org_id=g.org_id,
                uid=g.user.passport_uid,
                name=lower_domain_name,
                action=domain_action.verify,
                verification_type=verification_type
            )

            try:
                SyncSingleDomainTask(main_connection).delay(
                    org_id=g.org_id,
                    domain_name=lower_domain_name,
                )
            except DuplicatedTask:
                pass

        # получим домен заново, чтобы отдать его статус
        from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
            get_domains_from_db_or_domenator,
            DomainFilter,
        )
        domain = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(org_id=org_id, name=lower_domain_name),
            one=True,
        )
        return json_response(
            {
                'domain': domain_name,
                'owned': domain['owned'],
            }
        )

    post.__doc__ = """
        Запустить подтверждения домена.

        Для запуска, нужно передать параметром verification_type
        способ подтверждения, выбранный пользователем.

        Способ подтверждения должен быть строкой со значением из списка:

{supported_checks}

        Возвращает словарик вида:

        ```
        {{
           "domain": "example.com",
           "owned": false,
        }}
        ```

        Если owned = true, значит проверка выполнилась и домен уже сейчас подтвердился.
        А если false, то значит проверка побежала в фоне, и статус домена может измениться позже.

        ---
        tags:
          - Домены
        parameters:
          - in: path
            name: domain_name
            required: true
            type: string
          - in: body
            name: body

        responses:
          200:
            description: Процесс валидации запущен.
          400:
            description: Переданный способ подтверждения не поддерживается Вебмастером.
        """.format(
        supported_checks='\n'.join(
            '        * ' + item for item in SUPPORTED_CHECKS
        )
    )


class DomainOwnershipInfoView(View):

    @internal
    @scopes_required([scope.write_domains])
    @permission_required([global_permissions.add_domains])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, domain_name):
        """
        Способы подтверждения.

        Отдаёт информацию о том, подтверждён ли домен, и какие способы подтверждения можно использовать.

        Если домен подтверждён, ручка отдаёт:

        ```
        {
            "domain": "some.example.com",
            "status": "owned",
            "preferred_host": "some.example.com",
        }
        ```

        Если требуется подтверждение, то:

        ```
        {
            "domain": "some.example.com",
            "status": "need-validation",
            "preferred_host": "some.example.com",
            "methods": [
                {
                    "method": "webmaster.file",
                    "filename": "123456.html",
                    "content": "123456",
                    "weight": 0,
                },
                ...
            ],
            "last_check": {
                "date": "2018-03-02T13:39:49.839Z",
                "method": : "webmaster.html_file",
                "fail_type": "page_unavailable",
                "fail_reason": "connection_error",
            }
        }
        ```

        В имени метода могут быть префиксы `webmaster.`.

        ---
        tags:
          - Домены
        parameters:
          - in: path
            name: domain_name
            required: true
            type: string

        responses:
          200:
            description: В ответе будет словарь со статусом подтверждения домена и возможными способами подтверждения.
        """

        lower_domain_name = domain_name.lower()
        org_id = g.org_id

        domain = None
        if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
            result = app.domenator.ownership_info(org_id=org_id, domain_name=lower_domain_name)
            return json_response(result)
        else:
            domain = DomainModel(main_connection).get(lower_domain_name, org_id)

        admin_uid = get_organization_admin_uid(main_connection, org_id)

        if not domain:
            return json_error_not_found()

        result = {
            'domain': domain_name,
            'status': 'need-validation',
            'preferred_host': domain_name,
        }
        # Сюда положим информацию о последней проверке, если она была
        last_check = None

        if domain['owned']:
            result['status'] = 'owned'
        else:
            info = webmaster.info(
                lower_domain_name,
                admin_uid,
                ignore_errors='USER__HOST_NOT_ADDED'
            )
            # Если в webmaster-е нет такого домена, значит либо он удалился по таймауту,
            # либо пользователь сам удалил его через интерфейс webmaster-а
            # Надо добавить домен и снова запросить инфо
            if not info.get('data'):
                webmaster.add(
                    lower_domain_name,
                    admin_uid,
                )
                try:
                    info = webmaster.info(
                        lower_domain_name,
                        admin_uid,
                    )
                except WebmasterError:
                    log.trace().error('Getting domain info from webmaster failed')
                    raise

            data = info['data']
            verification_status = data.get('verificationStatus')

            preferred_host = info.get('request', {}).get('domainVerificationState', {}).get('preferredHost')
            if preferred_host:
                # выдача там вида http:www.aite-dev1.yaconnect.com:80
                try:
                    result['preferred_host'] = preferred_host.split(':')[1]
                except IndexError:
                    log.warning(f'Key error while parsing preferredHost: {preferred_host}')

            # если домен в процессе подтверждения  отдаем статус 'in-progress'
            if verification_status == 'IN_PROGRESS':
                result['status'] = 'in-progress'

            # если домен подтвержден в webmaster отметим его подтвержденным и в директории
            if verification_status == 'VERIFIED':
                result['status'] = 'in-progress'
                shard = main_connection.engine.db_info['shard']
                with get_main_connection(shard=shard, for_write=True) as write_main_connection:
                    try:
                        SyncSingleDomainTask(write_main_connection).delay(org_id=org_id,
                                                                          domain_name=lower_domain_name)
                    except DuplicatedTask:
                        pass

            verification_uin = data['verificationUin']
            methods = webmaster.list_applicable(
                lower_domain_name,
                admin_uid,
            )

            # Если в ответе есть информация про последнюю попытку проверки,
            # то прокинем её в наш собственный ответ
            if 'lastVerificationAttemptTime' in data:
                last_check = {
                    'date': data['lastVerificationAttemptTime'],
                    'method': 'webmaster.' + data['verificationType'].lower(),
                }

                if 'lastVerificationFailInfo' in data:
                    last_check['fail_type'] = data['lastVerificationFailInfo'].get('type', 'unknown').lower()
                    if 'reason' in data['lastVerificationFailInfo']:
                        last_check['fail_reason'] = data['lastVerificationFailInfo'].get('reason',
                                                                                         'unknown').lower()

            result['methods'] = [
                {
                    'method': 'webmaster.' + method.lower(),
                    'weight': 0,
                    'code': verification_uin,
                }
                for method in methods
            ]

            # Если информация про последнюю проверку известна, то
            # добавим её в ответ
            if last_check:
                result['last_check'] = last_check
        return json_response(result)


class DomainConnectView(View):

    @internal
    @uses_schema(DOMAIN_ADD_SCHEMA)
    @scopes_required([scope.write_domains])
    @permission_required([global_permissions.add_domains])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Добавить новый технический домен.
        ---
        tags:
          - Домены
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Домен добавлен
          422:
            description: Какая-то ошибка валидации
          409:
            description: Логическая ошибка
          403:
            description: Нет прав на добавление домена
        """
        org_id = g.org_id

        if is_sso_turned_on(main_connection, org_id):
            raise DomainAddToSsoOrganization()

        admin_uid = get_organization_admin_uid(main_connection, org_id)
        domain_name = to_punycode(data['name'].lower())
        domain_model = DomainModel(main_connection)

        # проверяем, корректный ли домен
        punycode_name = check_domain_is_correct(
            meta_connection,
            main_connection,
            domain_name,
            admin_uid,
            org_id,
            connect=True,
        )

        # добавляем домен в Паспорт
        add_master = create_domain_in_passport(
            main_connection,
            org_id,
            punycode_name,
            admin_uid,
        )
        domain = domain_model.create(
            domain_name,
            org_id,
            owned=True,
        )

        action_domain_add(
            main_connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value=domain,
            old_object=None,
        )

        if add_master:
            after_first_domain_became_confirmed(
                meta_connection,
                main_connection,
                org_id,
                domain_name,
                force=True,
            )

        return json_response(
            {'name': domain_name, 'owned': True},
            status_code=201,
        )
