from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import scopes_required, requires, \
    no_permission_required, internal
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.models.types import ROOT_DEPARTMENT_ID
from intranet.yandex_directory.src.yandex_directory.common.schemas import STRING, INTEGER, BOOLEAN, INTEGER_OR_NULL
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_email_to_all_async
from intranet.yandex_directory.src.yandex_directory.core.maillist.tasks import CreateMaillistTask
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationModel
from intranet.yandex_directory.src.yandex_directory.core.models.preset import apply_preset
from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG, \
    OrganizationServiceModel
from intranet.yandex_directory.src.yandex_directory.core.registrar.tasks import DomainVerifiedCallbackTask
from intranet.yandex_directory.src.yandex_directory.core.sms.tasks import sms_domain_confirmed
from intranet.yandex_directory.src.yandex_directory.core.tasks import UpdateEmailFieldsTask
from intranet.yandex_directory.src.yandex_directory.core.utils import get_organization_admin_uid
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema
from intranet.yandex_directory.src.yandex_directory.common.db import get_shard, get_main_connection
from intranet.yandex_directory.src.yandex_directory.common.utils import json_response

DOMAIN_OCCUPIED_SCHEMA = {
    'title': 'Domain',
    'type': 'object',
    'properties': {
        'domain': STRING,
        'new_owner': {
            'type': 'object',
            'properties': {
                'org_id': INTEGER,
                'new_domain_is_master': BOOLEAN,
            },
            'required': ['org_id', 'new_domain_is_master'],
        },
        'old_owner': {
            'type': 'object',
            'properties': {
                'org_id': INTEGER,
                'new_master': STRING,
                'tech': BOOLEAN,
            },
            'required': ['org_id', 'new_master', 'tech'],
        },
        'registrar_id': INTEGER_OR_NULL,
    },
    'required': ['domain', 'new_owner'],
    'additionalProperties': False
}


class DomainOccupiedView(View):
    @internal
    @uses_schema(DOMAIN_OCCUPIED_SCHEMA)
    @scopes_required([scope.domenator])
    @no_permission_required
    @requires(org_id=False, user=False)
    def post(self, meta_connection, _, data):
        domain_name = data['domain']

        if data.get('old_owner'):
            old_owner_org_id = data['old_owner']['org_id']
            new_master = data['old_owner']['new_master']
            tech = data['old_owner']['tech']
            campaign_slug = app.config['SENDER_CAMPAIGN_SLUG']['DISABLE_DOMAIN_EMAIL']

            shard = get_shard(meta_connection, old_owner_org_id)
            with get_main_connection(shard=shard, for_write=True) as main_connection:
                send_email_to_all_async(
                    main_connection,
                    org_id=old_owner_org_id,
                    domain=domain_name,
                    campaign_slug=campaign_slug,
                    new_master=new_master,
                    tech=tech,
                )
                UpdateEmailFieldsTask(main_connection).delay(
                    master_domain=new_master,
                    org_id=old_owner_org_id,
                )

        if data.get('registrar_id'):
            registrar_id = data['registrar_id']
            DomainVerifiedCallbackTask(main_connection).delay(registrar_id=registrar_id, domain_name=domain_name)

        new_owner_org_id = data['new_owner']['org_id']
        shard = get_shard(meta_connection, new_owner_org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            if data['new_owner']['new_domain_is_master']:
                org_from_db = OrganizationModel(main_connection).get(
                    id=new_owner_org_id,
                    fields=['preset', 'language', 'label'],
                )
                apply_preset(
                    meta_connection,
                    main_connection,
                    new_owner_org_id,
                    org_from_db['preset'] or 'default'
                )

                label = 'all'
                if OrganizationServiceModel(main_connection).is_service_enabled(new_owner_org_id, MAILLIST_SERVICE_SLUG):
                    CreateMaillistTask(main_connection).delay(
                        # Даём 15 секунд паспорту на то, чтобы информация про смену домена
                        # доехала до всех реплик.
                        start_in=15,
                        org_id=new_owner_org_id,
                        department_id=ROOT_DEPARTMENT_ID,
                        label=label,
                        ignore_login_not_available=True,
                    )

            admin_id = get_organization_admin_uid(main_connection, new_owner_org_id)
            sms_domain_confirmed(
                meta_connection,
                main_connection,
                new_owner_org_id,
                admin_id,
                domain_name,
            )

        return json_response({}, status_code=200)
