# coding: utf-8

from flask import request, g

from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.utils import prepare_events
from intranet.yandex_directory.src.yandex_directory.core.models.event import EventModel


from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_not_found,
    build_list_response,
)
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    no_permission_required,
    requires,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.core.models.action import ActionModel


class EventListView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.read_events])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        События в организации

        Эта ручка выдает список событий  в организации.
        Поддерживаются фильтры по name, revision, revision__gt.
        Если фильтры не указаны, то будут выдаваться все события.

        Порядок сортировки: от более новых к более старым.
    ---
    tags:
    - Действия
    parameters:
      - in: query
        name: revision
        type: integer
        description: все события в ревизии
      - in: query
        name: revision__gt
        type: integer
        description: все события в ревизии версии больше чем переданная
      - in: query
        name: revision__lt
        type: integer
        description: все события в ревизии версии меньше чем переданная
      - in: query
        name: name
        type: string
        description: тип события. Например user_add, user_modify и т.п.
      - in: query
        name: page
        type: integer
        description: текущая страница
      - in: query
        name: per_page
        type: integer
        description: какое кол-во объектов выведено на странице
    responses:
      200:
        description: Список событий
        """

        filters = self._get_filters()

        if not self._check_revision_filters(main_connection, filters):
            return json_error_not_found()

        response = build_list_response(
            model=EventModel(main_connection),
            model_filters=self._get_filters(),
            path=request.path,
            query_params=request.args.to_dict(),
            prepare_result_item_func=prepare_events
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def _get_filters(self):
        filters = {
            'org_id': g.org_id
        }

        def process_parameter(name):
            value = request.args.get(name)
            if value:
                filters[name] = value

        process_parameter('name')
        process_parameter('revision')
        process_parameter('revision__gt')
        process_parameter('revision__lt')
        return filters

    def _check_revision_filters(self, main_connection, filters):
        """
        Проверям что у нас есть действия по указанным в фильтре ревизиям.
        Если нет фильтров по ревизиям то вернем `True`.
        :param filters: фильтры для модели `EventModel`
        :type filters: dict
        :rtype: bool
        """
        am = ActionModel(main_connection)

        if 'revision' in filters:
            # есть ли действия с такой ревизией
            return am.count({'org_id': filters['org_id'], 'revision': int(filters['revision'])}) > 0

        if 'revision__gt' in filters:
            # есть ли действия с ревизией больше чем указанная
            return int(filters['revision__gt']) < am.get_max_revision(filters['org_id'])
        return True
