# -*- coding: utf-8 -*-
from copy import deepcopy

from flask import request, g
from functools import partial

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
    uses_out_schema,
)
from intranet.yandex_directory.src.yandex_directory.common.schemas import (
    I18N_STRING,
    I18N_STRING_REQUIRED,
    INTEGER,
    INTEGER_OR_NULL,
    DATETIME,
    STRING_OR_NULL,
    STRING,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    UnableToCreateMailList,
    CantChangeMembersForGroupType,
    CantDeleteGroupType,
    CantChangeMembersForNurKz,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    build_list_response,
    json_response,
    json_error,
    json_error_forbidden,
    get_object_or_404,
    check_label_or_nickname_or_alias_is_uniq_and_correct,
    to_lowercase,
    update_maillist_label,
    NotGiven,
)

from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    group_permissions,
    global_permissions,
)

from intranet.yandex_directory.src.yandex_directory.core.utils import (
    is_member,
    prepare_group,
    prepare_group_member_for_members_view,
    get_member_instance,
    get_organization_admin_uid,
    only_ids,
    ensure_integer,
    check_technical_group,
    prepare_fields,
    create_maillist,
    change_object_alias,
    add_avatar_id_for_users_relations,
    unfreeze_or_copy,
    get_master_domain,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    DepartmentModel,
    GroupModel,
    OrganizationServiceModel,
    OrganizationModel,
)

from intranet.yandex_directory.src.yandex_directory.core.models.resource import (
    ResourceRelationModel,
    relation_name,
)
from intranet.yandex_directory.src.yandex_directory.core.models.group import (
    GROUP_TYPE_GENERIC,
    TECHNICAL_TYPES,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import TYPE_GROUP
from intranet.yandex_directory.src.yandex_directory.common.models.base import FilterForEmptyResult

from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_group_modify,
    action_group_admins_change,
    action_group_alias_add,
    action_group_alias_delete,
)
from intranet.yandex_directory.src.yandex_directory.auth import raise_error_if_unable_to_use_group_type
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    permission_required,
    no_permission_required,
    requires,
    scopes_required,
)


GROUP_MEMBER = {
    'title': 'Group member',
    'type': 'object',
    'properties': {
        'type': {
            'enum': ['user', 'group', 'department'],
        },
        'id': INTEGER,
    },
    'anyOf': [{'required': ['type', 'id']}, {'required': ['type', 'object']}],
}

LIST_OF_GROUP_MEMBERS = {
    'title': 'List of group members',
    'type': 'array',
    'items': GROUP_MEMBER,
}

GROUP_ADMIN = {
    'title': 'Group admin',
    'type': 'object',
    'properties': {
        'id': INTEGER,
    },
    'required': ['id'],
}

LIST_OF_GROUP_ADMINS = {
    'title': 'List of group admins',
    'type': 'array',
    'items': GROUP_ADMIN,
}

GROUP_SCHEMA = {
    'title': 'Group',
    'type': 'object',
    'properties': {
        'type': STRING,
        'name': I18N_STRING_REQUIRED,
        'label': STRING_OR_NULL,
        'description': I18N_STRING,
        'members': LIST_OF_GROUP_MEMBERS,
        'admins': LIST_OF_GROUP_ADMINS,
        'external_id': STRING_OR_NULL,
        'maillist_type': {'enum': ['inbox', 'shared', 'both']},
    },
    'required': ['name'],
    'additionalProperties': False
}
GROUP_CREATE_SCHEMA = deepcopy(GROUP_SCHEMA)
GROUP_CREATE_SCHEMA['title'] = 'Group create schema'

GROUP_OUT_SCHEMA = deepcopy(GROUP_SCHEMA)
GROUP_OUT_SCHEMA['properties']['id'] = INTEGER
GROUP_OUT_SCHEMA['properties']['author'] = {'type': ['object', 'null']}
GROUP_OUT_SCHEMA['properties']['member_of'] = {'type': ['array', 'null']}
GROUP_OUT_SCHEMA['properties']['created'] = DATETIME
GROUP_OUT_SCHEMA['properties']['email'] = {'type': ['string', 'null']}
GROUP_OUT_SCHEMA['properties']['members_count'] = INTEGER
GROUP_OUT_SCHEMA['properties']['uid'] = INTEGER_OR_NULL

GROUP_OUT_SCHEMA_V6 = deepcopy(GROUP_OUT_SCHEMA)
# В 6 версии ручек мы убираем интернациализацию строк.
GROUP_OUT_SCHEMA_V6['properties']['name'] = STRING
GROUP_OUT_SCHEMA_V6['properties']['description'] = STRING_OR_NULL
GROUP_OUT_SCHEMA_V6['properties']['org_id'] = INTEGER
GROUP_OUT_SCHEMA_V6['properties']['resource_id'] = STRING
GROUP_OUT_SCHEMA_V6['properties']['author_id'] = INTEGER_OR_NULL
GROUP_OUT_SCHEMA_V6['properties']['tracker_license'] = {'type': ['boolean', 'null']}
GROUP_OUT_SCHEMA_V6['properties']['removed'] = {'type': ['boolean']}
GROUP_OUT_SCHEMA_V6['properties']['aliases'] = {'type': ['array', 'null']}
GROUP_OUT_SCHEMA_V6['properties']['all_users'] = {'type': ['array', 'null']}
GROUP_OUT_SCHEMA_V6['required'] = []

GROUP_ARRAY_OUT_SCHEMA = {
    'title': 'Group object list',
    'type': 'array',
    'items': GROUP_OUT_SCHEMA
}

GROUP_ARRAY_OUT_SCHEMA_V6 = {
    'title': 'Group object list',
    'type': 'array',
    'items': GROUP_OUT_SCHEMA_V6
}

GROUP_UPDATE_SCHEMA = deepcopy(GROUP_SCHEMA)
GROUP_UPDATE_SCHEMA['title'] = 'Update group'
del GROUP_UPDATE_SCHEMA['properties']['type']
del GROUP_UPDATE_SCHEMA['required']


CREATE_MEMBER_SCHEMA = deepcopy(GROUP_MEMBER)
CREATE_MEMBER_SCHEMA['title'] = 'Create member'

GROUP_MEMBERS_BULK_UPDATE_SCHEMA = {
    'title': 'Group members bulk update',
    'type': 'array',
    'items': {
        'title': 'Operation',
        'type': 'object',
        'properties': {
            'operation_type': {
                'enum': ['add', 'remove']
            },
            'value': GROUP_MEMBER,
        },
        'required': ['operation_type', 'value'],
        'additionalProperties': False
    }
}

GROUP_ADMINS_BULK_UPDATE_SCHEMA = {
    'title': 'Group admins bulk update',
    'type': 'array',
    'items': {
        'title': 'Operation',
        'type': 'object',
        'properties': {
            'operation_type': {
                'enum': ['add', 'remove']
            },
            'value': GROUP_ADMIN,
        },
        'required': ['operation_type', 'value'],
        'additionalProperties': False,
    }
}

GROUP_CREATE_ALIAS_SCHEMA = {
    'type': 'object',
    'title': 'Add group alias',
    'properties': {
        'name': STRING,
    },
    'required': [
        'name',
    ],
    'additionalProperties': False
}


def try_to_add_member(connection, group, member):
    # TODO: эту валидацию надо перенести в модель

    org_id = group['org_id']
    group_id = group['id']

    error_message = 'Unable to find "{type}" with id "{id}"'
    error_message_params = dict(
        type=member['type'],
        id=member['id']
    )
    model_instance = None
    get_kwargs = None
    if member['type'] == 'user':
        model_instance = UserModel(connection)
        get_kwargs = {
            'org_id': org_id,
            'user_id': member['id']
        }
    elif member['type'] == 'department':
        model_instance = DepartmentModel(connection)
        get_kwargs = {
            'org_id': org_id,
            'department_id': member['id']
        }
    elif member['type'] == 'group':
        model_instance = GroupModel(connection)
        get_kwargs = {
            'org_id': org_id,
            'group_id': member['id']
        }

    obj = model_instance.get(**get_kwargs)
    if not obj:
        raise ImmediateReturn(
            json_error(
                422,
                'member_not_found',
                error_message,
                **error_message_params
            )
        )

    GroupModel(connection).add_member(
        org_id=org_id,
        group_id=group_id,
        member=member
    )


class GroupListView(View):
    allowed_ordering_fields = ['name', 'tracker_license']

    @no_permission_required
    @uses_out_schema(GROUP_ARRAY_OUT_SCHEMA)
    @scopes_required([scope.read_groups, scope.write_groups])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Список групп

        Пример выдачи:

            {
                ...
                "result": [
                    {
                        id: 30141,
                        "name": {
                            "ru": "Мужчины",
                            "en": "Males"
                        }
                    }
                    ...
                ]
            }

        Поля сущности группы:

        * **id** - идентификатор
        * **name** - название ([локализованное поле](#lokalizovannyie-polya))
        * **type** - тип группы
        * **description** - описание ([локализованное поле](#lokalizovannyie-polya))
        * **label** - уникальный человеко-читаемый id группы ([label поле](#label-polya))
        * **email** - email рассылки группы
        * **author** - создатель группы
        * **created** - дата/время создания группы
        * **member_of** - список групп, в которые входит текущая группа. Доступно только в detail-view
        * **members** - список объектов, входящих в состав группы,
          каждый элемент должен быть в таком формате:
          `{'type': 'user', 'id': 123}` или
          `{'type': 'user', 'object': {'id': 123, 'name': ...}}`

        ---
        tags:
          - Группы
        parameters:
          - in: query
            name: id
            required: false
            type: string
            description: список id групп, перечисленных через запятую
          - in: query
            name: type
            required: false
            type: string
            description: список типов групп, перечисленных через запятую
          - in: query
            name: admin_uid
            required: false
            type: string
            description: это поле позволяет запросить только те группы, которые может администрировать указанный пользователь
          - in: query
            name: department_id
            required: false
            type: integer
            description: это поле позволяет запросить только те группы, в которых есть указанный департамент (непосредственно)
          - in: query
            name: page
            type: integer
            description: текущая страница
          - in: query
            name: per_page
            type: integer
            description: какое кол-во объектов выведено на странице
        responses:
          200:
            description: Список групп
        """
        response = build_list_response(
            model=GroupModel(main_connection),
            model_filters=self._get_filters(main_connection),
            path=request.path,
            query_params=request.args.to_dict(),
            prepare_result_item_func=partial(
                prepare_group,
                main_connection,
                api_version=request.api_version,
            ),
            model_fields=[
                '*',
                'members.*',
                'author.*',
                'email',
            ],
            max_per_page=1000,
            order_by=self._get_ordering_fields()
        )

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    @no_permission_required
    @uses_out_schema(GROUP_ARRAY_OUT_SCHEMA_V6)
    @scopes_required([scope.read_groups, scope.write_groups])
    @requires(org_id=True, user=False)
    def get_6(self, meta_connection, main_connection):
        """
        Информация обо всех группах организации.

        Если не передана информация о необходимых полях, будет
        возвращено только поле id для каждой группы.

        Необходимые поля возвращаются, если они указаны в поле fields.
        Например:

        ```
        /v6/groups/?fields=email,org_id,label,author.nickname
        ```

        Максимальное количество элементов, отдаваемых на странице – 1000.

        Если в ключе "links" есть словарь с ключом "next", то необходимо получить
        следующую страницу по ссылке из этого ключа.

        Пример ответа:

        ```
        {
            u'links': {},
            u'page': 1,
            u'pages': 1,
            u'per_page': 20,
            u'result': [
                {
                    u'all_users': [{u'id': 1135075862814215}],
                    u'author': None,
                    u'id': 2,
                    u'label': None,
                    u'member_of': [],
                    u'members_count': 1,
                    u'org_id': 40236,
                    u'type': u'organization_admin'
                }
            ],
            u'total': 1
        }
        ```

        ---
        tags:
          - Группы
        parameters:
          - in: query
            name: fields
            type: string
            description: список требуемых полей, перечисленных через запятую
        responses:
          200:
            description: Список словарей с информацией об группах
        """
        fields = prepare_fields(request.args.get('fields'), add_members_type=True)
        query_params = request.args.to_dict()
        org_domain = NotGiven
        if 'members' in fields:
            org_domain = get_master_domain(main_connection, g.org_id, domain_is_required=False)

        response = build_list_response(
            model=GroupModel(main_connection),
            model_filters=self._get_filters(main_connection),
            path=request.path,
            query_params=query_params,
            prepare_result_item_func=partial(
                prepare_group,
                main_connection,
                api_version=request.api_version,
                org_domain=org_domain,
            ),
            model_fields=fields,
            max_per_page=1000,
            order_by=self._get_ordering_fields()
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    @uses_schema(GROUP_CREATE_SCHEMA)
    @scopes_required([scope.write_groups])
    @permission_required([global_permissions.add_groups])
    def post(self, meta_connection, main_connection, data):
        """
        Создать новую группу

        Пользователь, создающий группу, автоматически включается в её
        состав и становится её админом.

        Поле `name` является обязательным и не должно быть пустым.

        Если валидация не прошла, то возвращается 422 код ошибки
        и её описание в словаре `{"errors": [...]}`

        Поле `label` является обязательным для групп в обычных (не yandex-team)
        организациях и не обязательно для групп в yandex-team-организации.
        Для организаций с неподвержденным доменом и организациям без домена поле lable передавать не надо,
        иначе возвращается 422 ошибка.

        Опциональное можно указать поле `type` отличное от "generic",
        но для этого нужен скоуп вида `create_groups_of_type_<type>`.

        ---
        tags:
          - Группы
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Группа создана
          409:
            description: Группа с заданым label уже существует
          422:
            description: Какая-то ошибка валидации
        """
        group_type = data.get('type', GROUP_TYPE_GENERIC).lower()
        raise_error_if_unable_to_use_group_type(group_type)

        label = data.get('label')
        org_id = g.org_id

        label = to_lowercase(label) or None

        # для организаций без домена (в метабазе у них has_owned_domains=False)
        # или с неподтвержденным доменом можно создать только группу без label
        if not OrganizationModel(main_connection).has_owned_domains(org_id) and label:
            raise UnableToCreateMailList()

        check_label_or_nickname_or_alias_is_uniq_and_correct(main_connection, label, org_id)

        new_group = self.inner_post_group(
            main_connection,
            g.org_id,
            data,
            label,
            g.user,
            can_be_created_without_admin=True,
        )
        return json_response(
            prepare_group(main_connection, new_group, api_version=request.api_version),
            status_code=201,
        )

    @staticmethod
    def inner_post_group(main_connection,
                         org_id,
                         data,
                         label,
                         global_user,
                         can_be_created_without_admin=False):

        # https://st.yandex-team.ru/DIR-858
        # Если в заголовке пришел внешний администратор, то:
        # - не включаем его в группу
        # - не делаем его админом (группа админ должен быть передан в POST-запросе)
        # - не делаем его автором (группа без автора)
        # - но можем написать от его имени в лог

        members = data.get('members', [])
        admins = [admin['id'] for admin in data.get('admins', [])]
        author_id = None
        uid = global_user.passport_uid

        is_outer = not is_member(
            main_connection,
            org_id,
            uid,
        )

        if is_outer and not admins and not can_be_created_without_admin:
            # не даём создавать группу без админов без установленного параметра
            raise ImmediateReturn(
                json_error(
                    422,
                    'group_without_admin',
                    'Group should contains one admin at least',
                )
            )
        elif not is_outer:
            members.append(
                dict(
                    type='user',
                    id=uid,
                )
            )
            # если админы не переданы явно, создающий группу пользователь
            # становится им автоматически
            if not admins:
                admins.append(uid)
            author_id = uid

        uid = None
        if label and OrganizationServiceModel(main_connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
            uid = create_maillist(main_connection, org_id, label, ignore_login_not_available=True)
        group_model = GroupModel(main_connection)

        new_group = group_model.create(
            name=data.get('name'),
            org_id=org_id,
            description=data.get('description'),
            members=members,
            label=label,
            author_id=author_id,
            admins=admins,
            aliases=data.get('aliases'),
            type=data.get('type', 'generic'),
            external_id=data.get('external_id'),
            action_author_id=global_user.passport_uid,
            maillist_type=data.get('maillist_type'),
            uid=uid,
        )

        return new_group

    def _get_ordering_fields(self):
        # временно переопределим метод, чтобы сортировать по нужным полям

        ordering_fields = super(GroupListView, self)._get_ordering_fields()
        if ordering_fields and ('name' in ordering_fields or '-name' in ordering_fields):
            name = 'name' if 'name' in ordering_fields else '-name'
            ordering_fields[ordering_fields.index(name)] = name + '_plain'
        return ordering_fields

    def _get_filters(self, main_connection):
        filters = {
            'org_id': g.org_id
        }

        if request.args.get('id'):
            ids = request.args.get('id').split(',')
            if len(ids) == 1:
                ids = ids[0]
            filters['id'] = ids

        if request.args.get('type'):
            types = request.args.get('type').split(',')
            filters['type'] = types

        if request.args.get('admin_uid'):
            filters['admin_uid'] = request.args.get('admin_uid')

        if request.args.get('suggest'):
            filters['suggest'] = request.args.get('suggest')

        if request.args.get('uid') and app.config['INTERNAL']:
            filters['uid'] = ensure_integer(request.args.get('uid'), 'uid')

        if request.args.get('department_id'):
            relations = ResourceRelationModel(main_connection).find(
                    filter_data={
                        'org_id': g.org_id,
                        'department_id': request.args.get('department_id'),
                        'name': relation_name.include,
                    }
                )
            resource_ids = [rel['resource_id'] for rel in relations]
            if not resource_ids:
                return FilterForEmptyResult
            filters['resource_id'] = resource_ids

        return filters


class GroupDetailView(View):
    @no_permission_required
    @uses_out_schema(GROUP_OUT_SCHEMA)
    @scopes_required([scope.read_groups, scope.write_groups])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, group_id):
        """
        Информацию о группе
        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
        responses:
          200:
            description: Словарь с информацией о группе.
        """
        model = GroupModel(main_connection)
        org_id = g.org_id
        group = get_object_or_404(
            model_instance=model,
            org_id=org_id,
            group_id=group_id,
            fields=[
                '*',
                'members.*',
                'admins.*',
                'author.*',
                'email',
            ],
        )
        group['member_of'] = model.get_parents(org_id=org_id, group_id=group['id'])
        return json_response(
            prepare_group(
                main_connection,
                group,
                api_version=request.api_version,
            )
        )

    @no_permission_required
    @uses_out_schema(GROUP_OUT_SCHEMA_V6)
    @scopes_required([scope.read_groups, scope.write_groups])
    @requires(org_id=True, user=False)
    def get_6(self, meta_connection, main_connection, group_id):
        """
        Информация о группе

        Необходимые поля возвращаются, если они указаны в поле fields.
        Например:

        ```
        /v6/groups/12345/?fields=org_id,label
        ```

        Если не передана информация о необходимых полях, будет
        возвращено только поле id для каждой группы.
        Пример ответа:

        ```
        {
            u'admins': [{u'id': 1134067112506421}],
            u'created': u'2017-08-03T15:24:25.950822Z',
            u'description': u'',
            u'id': 3,
            u'label': None,
            u'resource_id': u'5983402abfbe8319bb361e80',
            u'type': u'generic'}
        }
        ```

        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: string
          - in: query
            name: fields
            type: string
            description: список требуемых полей, перечисленных через запятую
        responses:
          200:
            description: Словарь с информацией об отделе
        """

        fields = prepare_fields(request.args.get('fields'), add_members_type=True)

        model = GroupModel(main_connection)
        org_id = g.org_id
        group = get_object_or_404(
            model_instance=model,
            org_id=org_id,
            group_id=group_id,
            fields=fields,
        )
        return json_response(
            prepare_group(
                main_connection,
                group,
                api_version=request.api_version,
            )
        )

    @uses_schema(GROUP_UPDATE_SCHEMA)
    @scopes_required([scope.write_groups])
    @permission_required([group_permissions.edit], TYPE_GROUP)
    def patch(self, meta_connection, main_connection, data, group_id):
        """
        Изменить группу

        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          200:
            description: Информация обновлена.
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """
        org_id = g.org_id
        model = GroupModel(main_connection)
        group_id = ensure_integer(group_id, 'group_id')

        group_before_update = get_object_or_404(
            model_instance=model,
            group_id=group_id,
            org_id=org_id,
            # Тут нам нужны все поля, так как мы эту группу в Action сохраним
            fields=(
                '*',
                'admins.*',
                'members.*',
                'author.*',
                'email',
            ),
        )


        # Запрещаем редактирование технических групп
        if check_technical_group(group_before_update['type']):
            return json_error_forbidden()

        # TODO: надо это в prefetch_related утащить
        group_before_update['member_of'] = model.get_parents(
            org_id=org_id,
            group_id=group_id,
        )

        # если вдруг хотим изменить группу админов организации, то должны
        # проверить, не собираются ли удалить uid владельца
        if group_before_update['type'] == 'organization_admin':
            admin_uid = get_organization_admin_uid(main_connection, org_id)
            members_before_update = [member['object']['id'] for member in group_before_update['members']]
            new_members = only_ids(data.get('members', []))
            if admin_uid in members_before_update and admin_uid not in new_members:
                # раньше мы пытались вернуть в сообщении об ошибке uid админа
                # но кажется, что нет смысла показывать пользователям
                # наши "кишки"
                return json_error_forbidden()

        if 'label' in data:
            data = unfreeze_or_copy(data)
            label = to_lowercase(data['label']) or None
            data['label'] = label
            if label != group_before_update['label']:
                data = update_maillist_label(
                    main_connection,
                    label,
                    org_id,
                    data,
                    group_before_update['uid']
                )

        # если у группы и раньше не было админов, то теперь разрешаем её редактировать в любом случае DIR-5294
        model.update_one(org_id, group_id, data, allow_empty_admins=True)

        # заново получаем из базы, чтобы убедиться что всё хорошо
        # обновилось
        group = model.get(
            org_id=org_id,
            group_id=group_id,
            fields=(
                '*',
                'admins.*',
                'members.*',
                'author.*',
                'email',
            ),
        )
        group['member_of'] = model.get_parents(org_id=org_id, group_id=group['id'])

        action_group_modify(
            main_connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value=group,
            old_object=group_before_update,
        )

        if group['admins'] != group_before_update['admins']:
            action_group_admins_change(
                main_connection,
                org_id=org_id,
                author_id=g.user.passport_uid,
                object_value=group,
                old_object=group_before_update,
            )

        return json_response(
            prepare_group(
                main_connection,
                group,
                api_version=request.api_version,
            )
        )

    @scopes_required([scope.write_groups])
    @permission_required([group_permissions.edit], TYPE_GROUP)
    def delete(self, meta_connection, main_connection, group_id):
        """
        Удалить группу
        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
        responses:
          200:
            description: Группа удалена.
          404:
            description: Группа не найдена.
          422:
            description: Запрещено удаление для этой группы.
          403:
            description: {permissions}
        """
        org_id = g.org_id
        group_model = GroupModel(main_connection)
        # получаем группу
        group = get_object_or_404(
            model_instance=group_model,
            group_id=group_id,
            org_id=org_id,
            fields=[
                '*',
                'members.*',
                'email',
            ],
        )
        raise_error_if_unable_to_use_group_type(group['type'])

        # если группа не пользовательская (а, например, техническая
        # или представляет админов организации), то её нельзя удалить
        if group.get('type') in TECHNICAL_TYPES:
            raise CantDeleteGroupType(type=group.get('type'))
        return self.inner_delete_group(
            main_connection,
            org_id,
            group,
            g.user,
        )

    @staticmethod
    def inner_delete_group(connection,
                           org_id,
                           group,
                           global_user):

        group_model = GroupModel(connection)
        # если пользовательная группа, то делаем некоторые действия до удаления
        # передаем пустой список member-ов
        group_model.delete(filter_data={
            'id': group['id'],
            'org_id': org_id,
        }, action_author_id=global_user.passport_uid)

        # status_code == 200, по просьбе diokuz@,
        # "Яндексовская библиотека, которую мы используем, не поддерживает
        # коды без контента" (c) diokuz@
        # ждем выполнения issue: https://github.com/pasaran/descript/issues/122
        return json_response(data={'message': 'No Content'})


class GroupMembersView(View):
    @no_permission_required
    @scopes_required([scope.read_groups, scope.write_groups])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, group_id):
        """
        Получить состав группы

        Возвращает список групп, отделов и пользователей, входящих непосредственно в данную группу.

        Объекты в списке, это словари типа:

            {
                'type': 'user',
                'object': {{'id': 123, 'name': ...}}
            }

        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
        responses:
          200:
            description: Список сущностей, входящих в группу.
        """
        org_id = g.org_id
        group = get_object_or_404(
            model_instance=GroupModel(main_connection),
            group_id=group_id,
            org_id=org_id
        )

        members = ResourceRelationModel(main_connection).find(
            filter_data=dict(
                resource_id=group['resource_id'],
                org_id=org_id,
                name=relation_name.include,
            ),
            fields=[
                '*',
                'user.*',
                'group.*',
                'department.*',
            ]
        )
        add_avatar_id_for_users_relations(members)
        members = list(map(prepare_group_member_for_members_view, members))
        return json_response(members)

    @uses_schema(CREATE_MEMBER_SCHEMA)
    @scopes_required([scope.write_groups])
    @permission_required([group_permissions.edit], TYPE_GROUP)
    def post(self, meta_connection, main_connection, data, group_id):
        """
        Добавить члена группы

        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          200:
            description: Созданный член группы.
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """
        org_id = g.org_id

        group = get_object_or_404(
            model_instance=GroupModel(main_connection),
            group_id=group_id,
            org_id=org_id
        )

        # Запрещаем редактирование технических групп
        if check_technical_group(group['type']):
            return json_error_forbidden()

        try_to_add_member(main_connection, group=group, member=data)

        filter_data = {
            'resource_id': group['resource_id'],
            'org_id': org_id,
            '%s_id' % data['type']: data['id']
        }

        return json_response(
            prepare_group_member_for_members_view(
                ResourceRelationModel(main_connection).find(
                    filter_data=filter_data,
                    fields=[
                        '*',
                        data['type'] + '.*',
                    ],
                )[0]
            ),
            status_code=201
        )

class GroupMembersBulkAdminUpdateView(View):
    @uses_schema(GROUP_ADMINS_BULK_UPDATE_SCHEMA)
    @scopes_required([scope.write_groups])
    @permission_required([group_permissions.edit], TYPE_GROUP)
    def post(self, meta_connection, main_connection, data, group_id):
        """
        Балковое обновление членов групп.

        Позволяет разом обновить состав команды.

        Пример входных данных:

        ```
        [
          {
            "operation_type": "add",
            "value": {
              "type": "user",
              "id": уид-пользователя,
            }
          },
          {
            "operation_type": "remove",
            "value": {
              "type": "user",
              "id": уид-пользователя-админа,
            }
          },
        ]

        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
            description: id группы
          - in: body
            name: body
        responses:
          200:
            description: Операция выполнена успешно
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """
        org_id = g.org_id
        model = GroupModel(main_connection)
        group_before_update = get_object_or_404(
            model_instance=model,
            group_id=group_id,
            org_id=org_id,
            # Тут нам нужны все поля, так как мы эту группу в Action сохраним
            fields=(
                '*',
                'admins.*',
            ),
        )
        if group_before_update.get('type') != 'generic':
            raise CantChangeMembersForGroupType(type=group_before_update.get('type'))

        # Запрещаем редактирование технических групп
        if check_technical_group(group_before_update['type']):
            return json_error_forbidden()
        operations = data

        add_admins = [i['value'] for i in operations if i['operation_type'] == 'add']
        remove_admins = [i['value'] for i in operations if i['operation_type'] == 'remove']

        old_admins = get_object_or_404(
            model_instance=model,
            group_id=group_id,
            org_id=org_id,
            fields=(
                'admins.id',
                'admins.user_type'
            ),
        )

        # По старым данным и действиям создаем новый список админов
        new_admins = set()
        for admin in old_admins['admins']:
            new_admins.add((admin['id'], admin['user_type']))
        for admin in add_admins:
            new_admins.add((admin['id'], admin['type']))
        for admin in remove_admins:
            if ((admin['id'], admin['type']) in new_admins):
                new_admins.remove((admin['id'], admin['type']))
            else:
                error_message = 'user "{id}" is not admin'
                error_message_params = dict(
                    id=admin['id']
                )
                raise ImmediateReturn(
                    json_error(
                        422,
                        'can_not_remove_admin',
                        error_message,
                        **error_message_params
                    )
                )
        data = {'admins': []}
        for admin in new_admins:
            data['admins'].append({'type': admin[1], 'id': admin[0]})
        # Теперь разрешаем группы без админов
        model.update_one(org_id, group_id, data, allow_empty_admins=True)

        # заново получаем из базы, чтобы убедиться что всё хорошо
        # обновилось
        group = model.get(
            org_id=org_id,
            group_id=group_id,
            fields=(
                '*',
                'admins.*',
            ),
        )

        if group['admins'] != group_before_update['admins']:
            action_group_admins_change(
                main_connection,
                org_id=org_id,
                author_id=g.user.passport_uid,
                object_value=group,
                old_object=group_before_update,
            )
        return json_response(
            prepare_group(
                main_connection,
                group,
                api_version=request.api_version,
            )
        )

class GroupMembersBulkUpdateView(View):
    @uses_schema(GROUP_MEMBERS_BULK_UPDATE_SCHEMA)
    @scopes_required([scope.write_groups])
    @permission_required([group_permissions.edit], TYPE_GROUP)
    def post(self, meta_connection, main_connection, data, group_id):
        """
        Балковое обновление членов групп.

        Позволяет разом обновить состав команды.

        Пример входных данных:

        ```
        [
          {
            "operation_type": "add",
            "value": {
              "type": "user",
              "id": уид-пользователя,
            }
          },
          {
            "operation_type": "add",
            "value": {
              "type": "group",
              "id": id-группы,
            }
          },
          {
            "operation_type": "remove",
            "value": {
              "type": "department",
              "id": id-отдела,
            }
          }
        ]

        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
            description: id группы
          - in: body
            name: body
        responses:
          200:
            description: Операция выполнена успешно
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """
        org_id = g.org_id
        if org_id == 3105515:
            # https://st.yandex-team.ru/DIR-10214
            raise CantChangeMembersForNurKz()
        model = GroupModel(main_connection)

        group_before_update = get_object_or_404(
            model_instance=model,
            group_id=group_id,
            org_id=org_id
        )

        if group_before_update.get('type') != 'generic':
            raise CantChangeMembersForGroupType(type=group_before_update.get('type'))


        # Ex: data = {
        #   'operation_type': 'add|remove',
        #   'value': list({
        #       'type': enum('user', 'group', 'department'),
        #       'id': INTEGER,
        #    }, ...)
        # }
        operations = data

        add_members = [i['value'] for i in operations if i['operation_type'] == 'add']
        remove_members = [i['value'] for i in operations if i['operation_type'] == 'remove']

        group_kwargs = dict(
            group_id=group_id,
            org_id=org_id,
            fields=(
                '*',
                'admins.*',
                'members.*',
                'author.*',
                'email',
            ),
        )

        with main_connection.begin_nested():
            for member in add_members:
                member_instance = get_member_instance(
                    main_connection,
                    member,
                    org_id,
                )

                group_before_update = model.get(**group_kwargs)
                try_to_add_member(main_connection, group=group_before_update, member=member)
                group = model.get(**group_kwargs)

                action_group_modify(
                    main_connection,
                    org_id=org_id,
                    author_id=g.user.passport_uid,
                    object_value=group,
                    old_object=group_before_update,
                )

            for member in remove_members:
                member_instance = get_member_instance(
                    main_connection,
                    member,
                    org_id,
                )

                group_before_update = model.get(**group_kwargs)
                model.remove_member(
                    org_id=group_before_update['org_id'],
                    group_id=group_before_update['id'],
                    member=member
                )
                group = model.get(**group_kwargs)

                action_group_modify(
                    main_connection,
                    org_id=org_id,
                    author_id=g.user.passport_uid,
                    object_value=group,
                    old_object=group_before_update,
                )

        members = ResourceRelationModel(main_connection).find(
            filter_data=dict(
                resource_id=group['resource_id'],
                org_id=org_id,
            ),
            fields=[
                '*',
                'user.*',
                'group.*',
                'department.*',
            ]
        )
        members = list(map(prepare_group_member_for_members_view, members))

        return json_response(members)


class GroupAliasesListView(View):
    @uses_schema(GROUP_CREATE_ALIAS_SCHEMA)
    @scopes_required([scope.write_groups])
    @permission_required([group_permissions.edit], TYPE_GROUP)
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data, group_id):
        """
        Добавить алиас группе
        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          201:
            description: Алиас добавлен
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """

        updated_group = change_object_alias(
            main_connection=main_connection,
            obj_id=group_id,
            obj_type=TYPE_GROUP,
            org_id=g.org_id,
            alias=data['name'],
            action_name='add',
            action_func=action_group_alias_add,
            author_id=g.user.passport_uid,
        )

        return json_response(
            data=prepare_group(
                main_connection,
                updated_group,
                api_version=request.api_version,
            ),
            status_code=201
        )


class GroupAliasesDetailView(View):
    @scopes_required([scope.write_groups])
    @permission_required([group_permissions.edit], TYPE_GROUP)
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, group_id, alias):
        """
        Удалить алиас группы

        ---
        tags:
          - Группы
        parameters:
          - in: path
            name: group_id
            required: true
            type: integer
          - in: path
            name: alias
            required: true
            type: string
        responses:
          204:
            description: Алиас удален
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """

        change_object_alias(
            main_connection=main_connection,
            obj_id=group_id,
            obj_type=TYPE_GROUP,
            org_id=g.org_id,
            alias=alias,
            action_name='delete',
            action_func=action_group_alias_delete,
            author_id=g.user.passport_uid,
        )

        return json_response({}, status_code=204)
