# coding: utf-8

import hashlib
import math
import urllib.request
import urllib.parse
import urllib.error

from datetime import datetime

from flask import (
    g,
    request,
)

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_permission_required,
    requires,
    internal,
    scopes_required,
    permission_required,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    UserDoesNotExist,
    TooManyInvites,
    ServiceNotLicensed,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import UserAlreadyMemberOfOrganization as ApiUserAlreadyMemberOfOrganization
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_invite_add,
    action_invite_use,
    action_invite_deactivate,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard,
    get_main_connection,
    get_meta_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    ROOT_DEPARTMENT_ID,
)
from intranet.yandex_directory.src.yandex_directory.common.schemas import (
    INTEGER,
    STRING,
    LIST_OF_EMAILS,
    BOOLEAN,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error,
    json_error_not_found,
    get_object_or_404,
    build_list_response,
    generate_secret_key_for_invitation,
    utcnow, json_error_invalid_value,
)
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_invite_email
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationModel,
    DepartmentModel,
    InviteModel,
    OrganizationBillingInfoModel,
    OrganizationServiceModel,
    UserServiceLicenses,
)
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    OrganizationIsWithoutContract,
    OrganizationHasDebt,
)
from intranet.yandex_directory.src.yandex_directory.core.models.invite import invite_type
from intranet.yandex_directory.src.yandex_directory.core.models.service import get_service_url
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import global_permissions
from intranet.yandex_directory.src.yandex_directory.core.utils.invite import (
    validate_invite,
    CodeDisabled,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import (
    add_user_by_invite,
    UserNotFoundInBlackbox,
    UserAlreadyMemberOfOrganization,
    UserAlreadyInThisOrganization,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
    uses_schema_for_get,
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import vip_reason, check_has_debt
from intranet.yandex_directory.src.settings import SUPPORT_TLDS

INVITE_CODE_ADD_SCHEMA = {
    'title': 'Invite',
    'type': 'object',
    'properties': {
        'department_id': INTEGER,
        'wait': INTEGER,
        'mail_campaign_slug': STRING,
        'counter': INTEGER,
        'ttl': INTEGER,
        'service_slug': STRING,
        'add_license': BOOLEAN,
    },
    'required': ['department_id'],
    'additionalProperties': False
}

USE_INVITE_SCHEMA = {
    'title': 'Invite',
    'type': 'object',
    'properties': {
        'tld': STRING,
    },
    'required': [],
    'additionalProperties': True,
}

INVITE_EMAILS_SCHEMA = {
    'title': 'InviteEmails',
    'type': 'object',
    'properties': {
        'tld': STRING,
        'department_id': INTEGER,
        'emails': LIST_OF_EMAILS,
        'wait': INTEGER,
        'mail_campaign_slug': STRING,
        'add_license': BOOLEAN,
        'campaign_slug': STRING,
    },
    'required': ['emails'],
    'additionalProperties': False
}

INVITE_LIST_SCHEMA = {
    'title': 'InvitesList',
    'type': 'object',
    'properties': {
        'enabled': INTEGER,
        'ordering': STRING,
    },
    'additionalProperties': False
}


def json_error_code_disabled(code):
    return json_error(
        409,
        'code_disabled',
        'Code "{code}" is disabled',
        code=code,
    )


def json_error_code_expired(code):
    return json_error(
        409,
        'code_expired',
        'Code "{code}" is expired',
        code=code,
    )


class InviteView(View):
    allowed_ordering_fields = ['created_at', 'last_use']

    @internal
    @uses_schema(INVITE_CODE_ADD_SCHEMA)
    @scopes_required([scope.use_invites])
    @permission_required([global_permissions.invite_users])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Создаем код приглашения.
        Если в организации не найден заданный отдел - создаем код для корневого отдела.

        Параметры которые можно передать в теле:
         - counter - сколько раз должен действовать инвайт
         - ttl - время жизни кода в секундах
         - service_slug - слаг сервиса от которого создается инвайт

        Пример ответа:

            {
                "code": "c3020d3e15884c179eec8cf56a5111bc"
            }

        ---
        parameters:
            - in: body
              name: body
        tags:
            - Код приглашения
        responses:
            201:
                description: Создали новый код приглашения.
        """
        code = create_person_invite(
            meta_connection,
            main_connection,
            g.org_id,
            g.user.passport_uid,
            data.get('service_slug') if data.get('service_slug', None) else g.service.identity,
            data,
        )

        return json_response(
            data={
                'code': code,
            },
            status_code=201
        )

    @internal
    @uses_schema_for_get(INVITE_LIST_SCHEMA)
    @scopes_required([scope.use_invites])
    @permission_required([global_permissions.invite_users])
    @requires(org_id=True, user=True)
    def get(self, meta_connection, _):
        """
        Возвращается список инвайтов.

        По умолчанию возвращаются все инвайты в данной организации.
        Чтобы получать только активные или неактивные, надо передать параметр enabled=1 или enabled=0
        Пример:

        /invites/?enabled=1 или /invites/?enabled=0

        Можно указать поле для сортировки ordering: created_at или last_use,
        или со знаком '-' для сортировки по убыванию
        Пример:

        /invites/?ordering=created_at или /invites/?ordering=-last_use

        Пример ответа:
            [
                {
                    'code': '038bbc21-579d-4509-9477-392fac485eb1',
                    'department_id': 1,
                    'enabled': 'True',
                    'counter': 5,
                    'author_id': 123123123123,
                    'created_at': '2017-08-25T12:17:26.638279+00:00',
                    'last_use': '2017-08-25T12:17:26.638279+00:00',
                    'invite_type': 'personal',
                },
                ...
            ]

        ---
        tags:
          - Invites
        responses:
          200:
            description: Список инвайт-кодов
          403:
            description: Недоступно для этого пользователя
          422:
            description: Ошибка в переданных данных
        """
        response = build_list_response(
            model=InviteModel(meta_connection),
            model_filters=self._get_filters(g.org_id),
            model_fields=[
                'code',
                'department_id',
                'created_at',
                'last_use',
                'author_id',
                'counter',
                'enabled',
                'invite_type',
                'valid_to',
            ],
            path=request.path,
            query_params=request.args.to_dict(),
            order_by=self._get_ordering_fields(),
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def _get_filters(self, org_id):
        filters = dict()
        filters['org_id'] = org_id
        enabled = request.args.get('enabled')
        if enabled:
            if int(enabled) == 1:
                filters['enabled'] = True
            elif int(enabled) == 0:
                filters['enabled'] = False
        return filters


class InviteUseView(View):
    @internal
    @uses_schema(USE_INVITE_SCHEMA)
    @scopes_required([scope.use_invites])
    @no_permission_required
    @requires(org_id=False, user=True)
    def post(self, meta_connection, main_connection, data, code):
        """
        Добавляем аккаунт в организацию. Помечаем код приглашения как использованный.
        Возвращаем id организации, в которую добавлен пользователь.


        Пример ответа:

            {
                "org_id": 123,
                "wait": 60,
                "redirect_to": "http://tracker.yandex.ru/"
            }

        ---
        tags:
            - Код приглашения
        parameters:
            - in: path
              name: code
              required: true
              type: string
            - in: body
              name: body
        responses:
            201:
                description: Аккаунт добавлен в организацию
            404:
                description: Код приглашения не найден или организация не найдена
            409:
                description: Срок действия кода истек или код уже использован, либо пользователь уже состоит в другой организации
            422:
                description: Пользователь не найден в ЧЯ



        """
        uid = g.user.passport_uid
        user_ip = g.user.ip

        with log.name_and_fields('use_invite_code', code=code, uid=uid):
            with get_meta_connection(for_write=True) as write_meta_connection:
                log.info('Validating invite code')
                invite_model = InviteModel(write_meta_connection)
                # тут делаем select for update
                # чтобы защититься от параллельного декремента счётчика использования инвайта
                invite = get_object_or_404(invite_model, code=code, for_update=True)
                try:
                    validate_invite(invite)
                except CodeDisabled:
                    return json_error_code_disabled(code)

                org_id = invite['org_id']
                shard = get_shard(write_meta_connection, org_id=org_id)
                if not shard:
                    log.warning('Organization does not exist')
                    return json_error_not_found()

                if invite['valid_to'] and invite['valid_to'] < utcnow():
                    log.warning('Code is expired')
                    return json_error_code_expired(code)

                log.info('Invite code validated')

                log.info('Trying to add user by invite')
                with get_main_connection(for_write=True, shard=shard) as write_main_connection:
                    try:
                        add_user_by_invite(
                            write_meta_connection,
                            write_main_connection,
                            uid,
                            user_ip,
                            invite,
                            set_org_id_sync=True,
                        )
                        action_invite_use(
                            write_main_connection,
                            org_id=org_id,
                            author_id=uid,
                            object_value=invite,
                        )
                    except UserNotFoundInBlackbox:
                        raise UserDoesNotExist()
                    except (UserAlreadyMemberOfOrganization, UserAlreadyInThisOrganization):
                        raise ApiUserAlreadyMemberOfOrganization
                log.info('User by invite was added')

        tld = data.get('tld', 'ru')

        wait = invite['wait']
        service_slug = invite['service_slug']

        if service_slug == 'portal':
            params = (
                ('org_id', org_id),
                ('mode', 'portal'),
                ('retpath', '/portal/home'),
            )
        else:
            params = (
                ('org_id', org_id),
                ('retpath', get_service_url(meta_connection, service_slug, tld)),
            )

        sk = generate_secret_key_for_invitation(uid)

        params += (('sk', sk), )
        url = '/portal/context?' + urllib.parse.urlencode(params)

        return json_response(
            data={
                'org_id': org_id,
                'wait': wait,
                'redirect_to': url,
            },
            status_code=201,
        )


class InviteDetailView(View):
    @internal
    @scopes_required([scope.use_invites])
    @no_permission_required
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection, code):
        """
        Информация о коде приглашения.

        Пример ответа:

            {
                "code": "c3020d3e15884c179eec8cf56a5111bc",
                "last_use": "12-12-2017",
                "org_name": "Яндекс",
            }

            {
                "code": "code_disabled",
                "params":
                {
                    "code": "c3020d3e15884c179eec8cf56a5111bc",
                    "last_use": "12-12-2017",
                    "org_name": "Яндекс"
                }
            }

        ---
        tags:
            - Код приглашения
        responses:
            200:
                description: Приглашение.
            409:
                description: Произошла ошибка при обработке запроса.
        """
        invite_model = InviteModel(meta_connection)
        invite = get_object_or_404(invite_model, code=code)

        try:
            validate_invite(invite)
        except CodeDisabled:
            return json_error_code_disabled(code)

        org_id = invite['org_id']
        with log.fields(org_id=org_id):
            shard = get_shard(meta_connection, org_id=org_id)
            if shard is None:
                log.error('Shard for organization does not exist')
                return json_error_not_found()
            with get_main_connection(shard=shard) as main_connection:
                organization = OrganizationModel(main_connection).get(org_id, fields=['name'])
        if not organization:
            log.error('Organization does not exist')
            return json_error_not_found()

        data = {
            'code': code,
            'org_name': organization['name'],
            'last_use': invite['last_use'],
            'valid_to': invite['valid_to'],
        }

        return json_response(
            data=data,
        )


class InviteEmailsView(View):
    @internal
    @uses_schema(INVITE_EMAILS_SCHEMA)
    @scopes_required([scope.use_invites])
    @permission_required([global_permissions.invite_users])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Создаем код приглашения и задачи на отправку писем с приглашениями.
        Counter в коде приглашения равен количеству переданных адресов.

        ---
        tags:
            - Приглашение пользователей
        responses:
            200:
                description: Задачи на отправку писем поставлены.
            422:
                description: Ошибка в переданных данных.
        """
        org_id = g.org_id
        organization = OrganizationModel(main_connection).get(org_id, fields=['language', 'name', 'tld'])
        if not organization:
            return json_error_not_found()

        tld = data.get('tld')
        if tld:
            if tld not in SUPPORT_TLDS:
                return json_error_invalid_value('tld')
        else:
            tld = organization['tld']

        emails = data.get('emails')
        code = create_person_invite(meta_connection, main_connection, org_id, g.user.passport_uid, g.service.identity, data, is_cloud=g.user.is_cloud)
        invite_link = app.config['INVITE_LINK_TEMPLATE'].format(tld=tld, code=code)

        for email in emails:
            send_invite_email(
                main_connection,
                email,
                invite_link,
                org_id,
                organization['name'],
                organization['language'],
                tld,
                campaign_slug=data.get('campaign_slug')
            )
        return json_response(
            data={},
        )


class DepartmentInviteView(View):
    @internal
    @scopes_required([scope.use_invites])
    @no_permission_required
    @requires(org_id=True, user=True)
    def get(self, meta_connection, _, department_id):
        """
        Актуальный инвайт-код для отдела (invite_type=department).

        Пример ответа:

            {
                "code": "c3020d3e15884c179eec8cf56a5111bc",
            }

        ---
        tags:
            - Код приглашения
        responses:
            200:
                description: Приглашение.
            404:
                description: Актуальный код не найден.
        """
        invite = InviteModel(meta_connection).filter(
            org_id=g.org_id,
            department_id=department_id,
            enabled=True,
            invite_type=invite_type.department,
        ).one()
        if invite:
            return json_response({'code': invite['code']})
        else:
            return json_error_not_found()

    @internal
    @scopes_required([scope.use_invites])
    @no_permission_required
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, department_id):
        """
        Создание инвайт-кода для отдела (invite_type=department).
        Если активный код уже есть - он становится неактивным, и создается новый.

        Пример ответа:

            {
                "code": "c3020d3e15884c179eec8cf56a5111bc",
            }

        ---
        tags:
            - Код приглашения
        responses:
            201:
                description: Приглашение создано.
            404:
                description: Отдел не найден.
        """
        with log.name_and_fields('create_department_invite', org_id=g.org_id, department_id=department_id):
            log.info('Creating department invite code')
            if not DepartmentModel(main_connection).get(department_id, g.org_id):
                log.warning('Department was not found')
                return json_error_not_found()

            with get_meta_connection(for_write=True) as write_meta_connection:
                shard = get_shard(write_meta_connection, org_id=g.org_id)
                with get_main_connection(for_write=True, shard=shard) as write_main_connection:
                    invite = InviteModel(write_meta_connection).find(
                        filter_data={
                            'org_id': g.org_id,
                            'department_id': department_id,
                            'enabled': True,
                            'invite_type': invite_type.department,
                        },
                        for_update=True,
                        one=True,
                    )
                    if invite:
                        with log.fields(old_invite_code=invite['code']):
                            log.info('Deactivate old invite code')
                            InviteModel(write_meta_connection).\
                                filter(code=invite['code']).\
                                update(enabled=False)

                            action_invite_deactivate(
                                write_main_connection,
                                org_id=g.org_id,
                                author_id=g.user.passport_uid,
                                object_value=invite,
                            )

                    invite_code = InviteModel(write_meta_connection).create(
                        g.org_id,
                        department_id,
                        g.user.passport_uid,
                        invite_type=invite_type.department,
                    )
                    action_invite_add(
                        write_main_connection,
                        org_id=g.org_id,
                        author_id=g.user.passport_uid,
                        object_value={'code': invite_code},
                    )

                    with log.fields(invite_code=invite_code):
                        log.info('Department invite code was created')

            return json_response(
                data={
                    'code': invite_code,
                },
                status_code=201,
            )

    @internal
    @scopes_required([scope.use_invites])
    @no_permission_required
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, department_id):
        """
        Деактивация инвайт-кода для отдела (invite_type=department).

        ---
        tags:
            - Код приглашения
        responses:
            204:
                description: Код удален.
        """
        invite = InviteModel(meta_connection).filter(
            org_id=g.org_id,
            department_id=department_id,
            enabled=True,
            invite_type=invite_type.department,
        ).update(enabled=False)
        if invite:
            action_invite_deactivate(
                main_connection,
                org_id=g.org_id,
                author_id=g.user.passport_uid,
                object_value=invite,
            )
        return json_response({}, status_code=204)


def create_person_invite(meta_connection,
                         main_connection,
                         org_id,
                         author_id,
                         service_slug,
                         data,
                         is_cloud=False,
                         ):
    enabled_invites = InviteModel(meta_connection).filter(
            org_id=org_id, enabled=True, valid_to__gt=utcnow(), valid_to_opt=True
    ).count()

    # Пока у нас нет нормального rate limit, ограничим и максимальное число
    # приглашений, так как возможно спамеры найдут способ их отключать
    # и генерить заново
    total_invites = InviteModel(meta_connection).filter(org_id=org_id).count()

    vip_features = OrganizationModel(main_connection).filter(id=org_id).fields('vip').one()['vip']

    if vip_reason.whitelist not in vip_features and (
        enabled_invites >= app.config['MAX_INVITES_COUNT']
        or total_invites >= app.config['MAX_TOTAL_INVITES_COUNT']
    ):
        raise TooManyInvites()

    department_id = data.get('department_id')
    if not department_id or not DepartmentModel(main_connection).get(department_id, org_id):
        department_id = ROOT_DEPARTMENT_ID

    wait = data.get('wait', app.config['DEFAULT_WAIT_BEFORE_REDIRECT'])
    mail_campaign_slug = data.get('mail_campaign_slug', None)
    emails = data.get('emails')
    counter = len(emails) if emails else data.get('counter')
    ttl = data.get('ttl')

    add_license = data.get('add_license')
    if add_license:
        if service_slug != 'tracker':
            raise ServiceNotLicensed()
        billing_info = OrganizationBillingInfoModel(main_connection).get(org_id)
        if not billing_info:
            service = OrganizationServiceModel(main_connection) \
                .get_by_slug(org_id, 'tracker', fields=['service_id'])
            user_count = UserServiceLicenses(main_connection) \
                .filter(org_id=org_id, service_id=service['service_id']) \
                .count()
            if user_count + (counter or 0) > settings.TRACKER_FREE_LICENSES:
                raise OrganizationIsWithoutContract()
        if billing_info and check_has_debt(
            first_debt_act_date=billing_info['first_debt_act_date'],
            balance=billing_info['balance']
        )[0]:
            raise OrganizationHasDebt()


    if is_cloud:
        admins = OrganizationModel(main_connection).get_admins(org_id)
        if admins:
            author_id = admins[0]['id']

    code = InviteModel(meta_connection).create(
        org_id=org_id,
        department_id=department_id,
        author_id=author_id,
        wait=wait,
        counter=counter,
        mail_campaign_slug=mail_campaign_slug,
        service_slug=service_slug,
        ttl=ttl,
        add_license=add_license,
    )
    action_invite_add(
        main_connection,
        org_id=org_id,
        author_id=author_id,
        object_value={'code': code},
    )
    return code
