# coding: utf-8
import collections

from flask import request, g

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import no_auth, no_scopes, no_permission_required, requires, internal
from intranet.yandex_directory.src.yandex_directory.auth.middlewares import get_client_ip_from
from intranet.yandex_directory.src.yandex_directory.common.db import get_main_connection
from intranet.yandex_directory.src.yandex_directory.common.utils import json_response
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.exceptions import ValidationQueryParametersError


class OrganizationsAuthInfoMixin(object):
    def _get_organizations_info_by_uids(self, meta_connection, uids):
        if not uids:
            return []
        uids = list(map(str, uids))
        uids.sort()

        # получаем данные из кэша
        cache_key = 'orgs_by_uids:' + ','.join(uids)
        cache_data = {}
        if app.cache.exists(cache_key):
            cache_data = app.cache.get(cache_key)

        # идем в метабазу за списками организаций по пользователям
        orgs_meta = self._get_uid_orgs_meta(meta_connection, uids)

        # вычисляем хэш по списку организаций
        hash_value = hash(','.join([
            str(org_meta['uid']) + ':' + str(org_meta['org_id'])
            for org_meta in orgs_meta
        ]))

        # если вычисленное значение хэша равно значению хэша в кэше,
        # значит списки организаций пользователей не поменялись и можно вернуть закэшированные данные
        if cache_data.get('hash') == hash_value:
            return cache_data['data']

        # группируем организации по шардам, чтобы потом сходить за данными
        # группируем uid'ы по организациям, чтобы потом собрать результат
        org_ids_by_shard = collections.defaultdict(list)
        uids_by_org_id = collections.defaultdict(list)
        user_join_date_by_uid_and_org_id = collections.defaultdict(dict)
        for org_meta in orgs_meta:
            uid = str(org_meta['uid'])
            org_id = org_meta['org_id']
            shard = org_meta['shard']
            user_join_date = org_meta['created']

            org_ids_by_shard[shard].append(org_id)
            uids_by_org_id[org_id].append(uid)
            user_join_date_by_uid_and_org_id[uid][org_id] = user_join_date

        # по шардам идем за данными организаций
        orgs_data = self._get_orgs_data(org_ids_by_shard)

        # собираем результат
        result = self._build_result(orgs_data, uids_by_org_id, user_join_date_by_uid_and_org_id)

        # сохраняем результат в кэш на 15 минут
        app.cache.set(cache_key, {
            'hash': hash_value,
            'data': result
        }, 15 * 60)

        return result

    def _get_uid_orgs_meta(self, meta_connection, uids):
        query = """
                select uids.uid, uo.org_id, uo.shard, uo.created
                from (
                         values {placeholders}
                     ) uids(uid)
                inner join lateral (
                    select u.org_id, o.shard, u.created
                    from users u
                             inner join organizations o on u.org_id = o.id
                    where
                        u.is_dismissed = false
                    and
                        u.id = uids.uid
                    order by u.created, u.org_id
                    limit 16) uo on true
                order by uids.uid, uo.org_id
            """

        placeholders = ', '.join(['(%s)'] * len(uids))
        query = query.format(placeholders=placeholders)

        uids = list(map(int, uids))  # в базе uid - число
        return meta_connection.execute(query, *uids).fetchall()

    def _get_orgs_data(self, org_ids_by_shard):
        from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationModel
        orgs_data = []
        for shard, org_ids in list(org_ids_by_shard.items()):
            with get_main_connection(shard=shard) as main_connection:
                orgs_data += OrganizationModel(main_connection).filter(id=org_ids) \
                    .fields('id', 'name', 'logo').all()

        return orgs_data

    def _build_result(self, orgs_data, uids_by_org_id, user_join_date_by_uid_and_org_id):
        result = collections.defaultdict(list)

        for org_detail in orgs_data:
            org_id = org_detail['id']
            org_name = org_detail['name']

            org_logo = None
            if org_detail.get('logo') is not None:
                org_logo = org_detail['logo'].get('orig', {}).get('url', None)

            org_uids = uids_by_org_id[org_id]
            for uid in org_uids:
                result[uid].append({
                    'id': org_id,
                    'name': org_name,
                    'logo': org_logo,
                })

        def sort_key(org_data):
            org_id = org_data['id']
            return str(user_join_date_by_uid_and_org_id[uid][org_id]) + str(org_id)

        for uid in result:
            result[uid].sort(key=sort_key)

        return result


class OrganizationsByYandexSessionView(View, OrganizationsAuthInfoMixin):
    methods = ['get']

    @no_scopes
    @no_permission_required
    @no_auth
    def get(self, meta_connection, main_connection):
        """
        Возвращает список организаций пользователей по куке Session_id

        ---
        tags:
          - Мультиавторизация
        responses:
          200:
            description: Список организаций для каждого пользователя
        """
        session_id = request.cookies.get('Session_id')
        if session_id is None:
            return json_response([])

        host = request.host or 'yandex.ru'

        if getattr(g, 'user', None) and g.user.ip:
            user_ip = g.user.ip
        else:
            user_ip = get_client_ip_from(request.headers)

        # получаем список пользователей в сессии
        uids = app.blackbox_instance.get_uids_by_session(
            request=request,
            sessionid=session_id,
            userip=user_ip,
            host=host,
        )
        uids = list(filter(None, uids))
        result = self._get_organizations_info_by_uids(meta_connection, uids)

        return json_response(result)


class OrganizationsByUidsView(View, OrganizationsAuthInfoMixin):
    methods = ['get']

    @internal
    @no_scopes
    @no_permission_required
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection):
        """
        Возвращает список организаций пользователей по переданному списку пользователей

        ---
        tags:
          - Мультиавторизация
        parameters:
          - in: query
            name: uids
            required: False
            type: string
            description: список uid'ов через запятую
        responses:
          200:
            description: Список организаций для каждого пользователя
        """
        raw_uids = request.args.get('uids', None)
        uids = []

        if raw_uids:
            uids = raw_uids.split(',')
        for uid in uids:
            try:
                int(uid)
            except ValueError:
                raise ValidationQueryParametersError(query_params='uid')

        result = self._get_organizations_info_by_uids(meta_connection, uids)

        return json_response(result)
