# -*- coding: utf-8 -*-
import csv
import uuid
from io import StringIO
from collections import Counter

import chardet

from flask import g, request, Response

from intranet.yandex_directory.src.env_settings.base import YANDEX_MAIL_SERVERS
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    permission_required,
    requires,
    internal,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common import json
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ImmediateReturn, InvalidValue, DuplicateLogin as ApiDuplicateLogin, \
    InvalidMigration
from intranet.yandex_directory.src.yandex_directory.common.schemas import STRING, INTEGER, BOOLEAN
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_error_required_field,
    json_error_not_found,
    force_text,
    json_response,
    validate_data_by_schema,
    check_label_or_nickname_or_alias_is_uniq_and_correct)
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    MigrationFileTooLarge,
    RequiredFieldsMissed,
    MigrationFileParsingError,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    MailMigrationFileModel,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import global_permissions
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    ensure_integer,
    build_email,
    get_master_domain,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.passport import PassportApiClient
from intranet.yandex_directory.src.yandex_directory.passport.exceptions import (
    PassportException,
    LoginNotavailable,
    LoginNotAvailable,
    PasswordLikelogin,
    YandexMailToOrgWithDomain,
)
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema_for_get,
)

# максимальный размер миграционного файла
MAX_FILE_SIZE = 1024 * 1024 + 1
# максимальная длинная поля
MAX_FIELD_LENGTH = 300

# поля в csv
EMAIL = 'email'
PASSWORD = 'password'
FIRST_NAME = 'first_name'
LAST_NAME = 'last_name'
NEW_LOGIN = 'new_login'
NEW_PASSWORD = 'new_password'

# обязательные
REQUIRED_FIELDS = [
    EMAIL,
    PASSWORD,
]

# проблемы, которые могут возникнуть при валидации файла для миграции
TO_LONG_FIELD = {
    FIRST_NAME: 'first_name_is_too_long',
    LAST_NAME: 'last_name_is_too_long',
}
INVALID_EMAIL_FORMAT = 'invalid_email_format'
INVALID_RECORD_LENGTH = 'invalid_record_length'  # record should have the same number of fields as header
LINE_PARSING_ERROR = 'line_parsing_error'
# Если пользователь пытается с одного ящика собирать почту в несколько аккаунтов
EMAIL_ALREADY_USED = 'email_already_used'

MAIL_MIGRATION_JSON_DATA_SCHEMA = {
    'title': 'Migrate mail',
    'type': 'object',
    'properties': {
        'email': STRING,
        'password': STRING,
        'first_name': STRING,
        'last_name': STRING,
        'new_login': STRING,
        'new_password': STRING,
    },
    'required': [
        'email',
        'password'
    ],
}
LIST_OF_MAIL_MIGRATION_JSON = {
    'title': 'List of mail migration JSONs',
    'type': 'array',
    'items': MAIL_MIGRATION_JSON_DATA_SCHEMA,
}

MAIL_MIGRATION_JSON = {
    'title': 'List of mail migration JSONs',
    'type': 'object',
    'data': {
        'accounts_list': LIST_OF_MAIL_MIGRATION_JSON,
        'host': STRING,
        'port': INTEGER,
        'imap': BOOLEAN,
        'ssl': BOOLEAN,
        'no_delete_msgs': BOOLEAN,
        'sync_abook': BOOLEAN,
    },
    'required': [
        'accounts_list',
        'host',
        'port',
    ]
}

MAIL_MIGRATION_STATUS_SCHEMA = {
    'title': 'Check migration status for mail migration',
    'type': 'object',
    'properties': {
        'mail_migration_id': STRING
    },
    'additionalProperties': True
}


class MailMigrationView(View):

    @internal
    @permission_required([global_permissions.migrate_emails])
    @scopes_required([scope.migrate_emails])
    @requires(org_id=True, user=False)
    def post(self, meta_connection, main_connection):
        """
        Создание мигратора ящиков

        Эта ручка возвращает id задаци на перенос ящиков
        ---
        tags:
          - Миграция ящиков
        consumes:
          - multipart/form-data
        parameters:
          - in: formData
            name: migration_file
            type: file
            required: true
            description: файл с ящиками для переноса
          - in: formData
            name: host
            required: true
            type: string
            description:  сервер, с которого нужно собирать почту
          - in: formData
            name: port
            required: true
            type: integer
            description: порт на сервере, к которому нужно подключаться
          - in: formData
            name: imap
            default: true
            type: boolean
            description: использовать imap иначе pop3
          - in: formData
            name: ssl
            type: boolean
            default: true
            description: нужно используя SSL  (по умолчанию Да)
          - in: formData
            name: no_delete_msgs
            type: boolean
            default: true
            description: письма с удаленного сервера удалять не надо  (по умолчанию Да)
          - in: formData
            name: sync_abook
            type: boolean
            default: true
            description: сборщика нужно выполнить процедуру сбора контактов  (по умолчанию Да)
          - in: formData
            name: mark_archive_read
            default: true
            type: boolean
            description: письма пришедшие в исходный ящик до того, как был создан сборщик, нужно при сборе пометить прочитанными (по умолчанию Да)
        responses:
          200:
            description: Ид задачи сбора ящиков
        """
        # Если есть файл, то работаем с ним
        if request.headers['Content-Type'].startswith('multipart/form-data'):
            files = request.files or {}
            migration_file = files.get('migration_file')
            if not migration_file:
                return json_error_required_field('migration_file')
            else:
                return self._migrate(migration_file.read(MAX_FILE_SIZE), request.form, main_connection, "file")
        # Если нет файла, пробуем получить json и сделать с ним то же самое
        data = request.get_json()
        if data and data.get('accounts_list', False):
            response = self._validate_json_by_schema(data, MAIL_MIGRATION_JSON)
            if response:
                return response
            migration_file_content = _get_migration_file_from_data(data.get('accounts_list', False))
            return self._migrate(migration_file_content, data, main_connection, "json")
        else:
            return json_error_required_field('accounts_list')

    @internal
    @permission_required([global_permissions.migrate_emails])
    @scopes_required([scope.migrate_emails])
    @uses_schema_for_get(MAIL_MIGRATION_STATUS_SCHEMA)
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Возвращаем состояние миграции почты

        Пример ответа:

            [
                {'stage': 'accounts-creating', 'state': 'success'},
                {'stage': 'collectors-creating', 'state': 'success'},
            ]

        ---
        responses:
            200:
                description: Получаем данные о миграции почты
            403:
                description: Доступ запрещен
            404:
                description: миграция не найдена
        """

        # этот импорт тут, чтобы избежать циклических зависимостей
        from intranet.yandex_directory.src.yandex_directory.core.mail_migration.utils import MailMigration

        args = request.args.to_dict()
        mail_migration_id = args.get('mail_migration_id')
        if mail_migration_id:
            try:
                uuid.UUID(mail_migration_id)
            except ValueError:
                return json_error_not_found()
        migration_data = MailMigration(main_connection, mail_migration_id, g.org_id).get_migration_progress()

        if not migration_data:
            return json_error_not_found()
        return json_response(
            data=migration_data
        )

    def _migrate(self, migration_file_content, data, main_connection, migration_source):
        host = data.get('host')
        if not host:
            return json_error_required_field('host')
        port = data.get('port')
        if not port:
            return json_error_required_field('port')
        port = ensure_integer(port, 'port')
        imap, ssl, no_delete_msgs, sync_abook, mark_archive_read = _get_and_validate_additional_params(
            data)

        org_id = g.org_id
        errors, errors_to_mail, duplicates = _validate_migration_file(
            main_connection,
            org_id,
            migration_file_content,
            host,
        )
        if duplicates:
            raise ApiDuplicateLogin(duplicate_logins=duplicates)
        if errors:
            etc = []
            for item in sorted(errors.items()):
                if migration_source == "file":
                    error = {'line': item[0], 'errors': item[1]}
                else:
                    error = {'line': item[0], 'message': item[1]}
                if item[0] in errors_to_mail:
                    error['email'] = errors_to_mail[item[0]]
                etc.append(error)
            errors = etc
            raise InvalidMigration(
                error_code='migration_' + migration_source + '_is_invalid',
                message='Migration ' + migration_source + ' is invalid and has following errors: {errors}',
                errors=errors,
            )

        encoding_info = chardet.detect(migration_file_content)
        decoded_content = migration_file_content.decode(
            encoding=encoding_info['encoding']
        )

        file_id = MailMigrationFileModel(main_connection).create(
            org_id=org_id,
            file=decoded_content,
        )['id']
        # этот импорт тут, чтобы избежать циклический зависимостей
        from intranet.yandex_directory.src.yandex_directory.core.mail_migration.utils import start_migration
        mail_migration_task_id = start_migration(
            main_connection,
            str(file_id),
            org_id,
            host,
            port,
            imap,
            ssl,
            no_delete_msgs,
            sync_abook,
            mark_archive_read,
        )

        return json_response(
            data={'mail_migration_id': mail_migration_task_id},
            status_code=200,
        )

    def _validate_json_by_schema(self, json_data, schema):
        data = self.normalization(json_data, schema=schema)
        errors = validate_data_by_schema(data, schema=schema)

        if errors:
            response = {
                'message': 'Schema validation error',
                'code': 'schema_validation_error',
                'params': {
                    'schema': schema,
                    'errors': errors,
                }
            }
            return Response(
                json.dumps(response),
                status=422,
                mimetype='application/json; charset=utf-8',
            )
        return None


def _validate_migration_file(main_connection, org_id, migration_file_content, host):
    """
    Валидируем .csv файл
    """
    seen_logins = Counter()

    with log.fields(org_id=org_id):
        if len(migration_file_content) == MAX_FILE_SIZE:
            raise MigrationFileTooLarge()
        log.debug('Start parsing migration file')

        encoding_info = chardet.detect(migration_file_content)
        migration_file = StringIO(
            migration_file_content.strip().decode(
                encoding_info['encoding']
            )
        )

        errors = {}
        errors_to_mail = {}
        try:
            reader = csv.DictReader(migration_file)
        except Exception:
            log.trace().error('Migration file has invalid format')
            raise MigrationFileParsingError()
        try:
            fields = reader.fieldnames
            header_length = len(fields)
            if not set(REQUIRED_FIELDS).issubset(set(fields)):
                raise RequiredFieldsMissed(REQUIRED_FIELDS)
            org_domain = get_master_domain(main_connection, org_id)

            used_emails = set()
            for row in reader:
                try:
                    problems = _validate_record(
                        main_connection,
                        org_id,
                        org_domain,
                        row,
                        header_length,
                        host,
                        seen_logins,
                    )
                except DuplicateLogin:
                    # Если есть повторяющиеся логины - не делаем лишних проверок и запросов в паспорт
                    continue

                email = row.get(EMAIL)
                if email:
                    if email in used_emails:
                        problems.append(EMAIL_ALREADY_USED)
                    used_emails.add(email)

                if problems:
                    errors[reader.line_num] = problems
                    if row.get(EMAIL):
                        errors_to_mail[reader.line_num] = row.get(EMAIL)
            log.debug('Migration file validated')
        except csv.Error:
            log.trace().error('Line parsing error')
            errors[reader.line_num] = LINE_PARSING_ERROR
        duplicates = []
        for login in seen_logins:
            if seen_logins[login] > 1:
                duplicates.append(login)
        return errors, errors_to_mail, duplicates


def _validate_record(main_connection, org_id, org_domain, record, header_length, host, seen_logins):
    """
    Валидируем запись .csv файла
    """
    problems = []
    try:
        # если число полей в записи не совпадает с числом полей в хедере - вернуть ошибку
        if len(record) != header_length:
            problems.append(INVALID_RECORD_LENGTH)
            return problems

        # проверяем что email валидный
        email_problem = _is_valid_email(record.get(EMAIL))
        if email_problem:
            problems.append(email_problem)

        # если задан новый логин, валидируем через API паспорта его
        # если нет - валидируем email
        login = record.get(NEW_LOGIN) or record.get(EMAIL) or ''
        login_problem = _validate_login(org_id, org_domain, login, seen_logins)
        if login_problem:
            problems.append(login_problem)

        # если задан новый пароль, валидируем через API паспорта его
        # если нет - валидируем старый
        password = record.get(NEW_PASSWORD) or record.get(PASSWORD)
        password_problem = _validate_password(password)
        if password_problem:
            problems.append(password_problem)

        # Проверяем, что длинна first_name и last_name не больше MAX_FIELD_LENGTH
        for field in [FIRST_NAME, LAST_NAME]:
            value = record.get(field)
            if value and len(value) > MAX_FIELD_LENGTH:
                problems.append(TO_LONG_FIELD[field])

        # Проверяем, что логин не совпадает с паролем
        if password == login:
            problems.append(PasswordLikelogin.code)

        # Нельзя мигрировать почту с яндексовского домена в организацию с тем же доменом
        if host in YANDEX_MAIL_SERVERS and org_domain == record.get(EMAIL).split('@')[1]:
            login = record.get(NEW_LOGIN, None)
            if not login:
                problems.append(YandexMailToOrgWithDomain.code)
            else:
                try:
                    check_label_or_nickname_or_alias_is_uniq_and_correct(main_connection, login, org_id)
                except (ImmediateReturn, LoginNotavailable, LoginNotAvailable):
                    problems.append(YandexMailToOrgWithDomain.code)

    except Exception as e:
        log.trace().error('Unexpected record parsing error')
        problems.append(str(e))
    return problems

def _is_valid_email(email):
    # Проверяем что строка - валидный email
    if not email or '@' not in email:
        return INVALID_EMAIL_FORMAT

def _validate_password(password):
    # валидируем пароль через API паспорта
    try:
        PassportApiClient().validate_password(password)
    except PassportException as e:
        return e.code
    except Exception as e:
        return str(e)

def _validate_login(org_id, org_domain, login, seen_logins):
    # валидируем логин через API паспорта
    if '@' in login:
        login = login[:login.index('@')]

    seen_logins[login] += 1
    if seen_logins[login] > 1:
        raise DuplicateLogin

    domain_login = build_email(
        None,
        login,
        org_id=org_id,
        org_domain=org_domain,
    )
    try:
        PassportApiClient().validate_login(domain_login)
    except (LoginNotavailable, LoginNotAvailable):
        # если такой аккаунт уже есть в паспорте, мы не ругаемся, просто дальше создавать его не будем,
        # и почту перенесем в существующий аккаунт
        pass
    except PassportException as e:
        return e.code
    except Exception as e:
        return str(e)

def _get_and_validate_additional_params(data):
    params = ['ssl', 'no_delete_msgs', 'sync_abook', 'mark_archive_read']
    protocol = data.get('protocol', 'imap').lower()
    values = [(protocol == 'imap')]
    for param in params:
        val = str(data.get(param, 'true')).lower()
        if val not in ['false', 'true']:
            raise InvalidValue(
                message='Field "{field}" has invalid value. Valid value "true" or "false"',
                field=param,
            )
        values.append(val == 'true')
    return tuple(values)


def _get_migration_file_from_data(data):
    info = [EMAIL, PASSWORD, FIRST_NAME, LAST_NAME, NEW_LOGIN, NEW_PASSWORD]
    migration_file = ""
    migration_file += EMAIL + ',' + PASSWORD + ',' + FIRST_NAME +\
                     ',' + LAST_NAME + ',' + NEW_LOGIN + ',' + NEW_PASSWORD + '\n'
    for mail_info in data:
        for field in info:
            if field in mail_info:
                migration_file += force_text(mail_info[field])
            migration_file += ','
        migration_file = migration_file[:len(migration_file) - 1] + '\n'
    migration_file = migration_file.encode('utf-8')
    return migration_file


class DuplicateLogin(Exception):
    """
    Такой логин уже есть в файле.
    """
    pass


