import time

from intranet.yandex_directory.src import settings
from sqlalchemy.exc import IntegrityError

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import get_meta_connection, get_main_connection
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (UserDoesNotExist, DomainUserError,
                                                                              ImmediateReturn)
from intranet.yandex_directory.src.yandex_directory.common.utils import json_error, json_error_required_field, \
    json_error_invalid_value
from intranet.yandex_directory.src.yandex_directory.core.actions import action_organization_add
from intranet.yandex_directory.src.yandex_directory.core.actions.department import on_department_add
from intranet.yandex_directory.src.yandex_directory.core.features import (
    enable_features,
    is_multiorg_feature_enabled_for_user,
    is_feature_enabled,
    USE_DOMENATOR,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserMetaModel, OrganizationModel, GroupModel, UserModel,
    OrganizationMetaModel, OrganizationRevisionCounterModel
)
from intranet.yandex_directory.src.yandex_directory.core.models.preset import apply_preset
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserRoles
from intranet.yandex_directory.src.yandex_directory.core.registrar.tasks import DomainRegistrationProcedureTask
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    add_existing_user, check_organizations_limit, create_root_department,
    is_outer_uid
)
from intranet.yandex_directory.src.yandex_directory.core.views.domains import create_domain_verified_via_webmaster
from intranet.yandex_directory.src.yandex_directory.core.views.organization.utils import assert_not_spammer
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type


def create_organization_without_domain(
        meta_connection,
        main_connection,
        data,
        uid,
        user_ip,
):
    """Создаёт организацию без домена и делает учётку с uid - первым сотрудником
       и админом этой организации.
       Аргумент data должен быть словарём, в котором ожидаются ключи:
       - language - язык пользователя, по-умолчанию: ru
       - tld - top level domain с которого происходит регистрация организации, по-умолчанию: ru
       - source - источник из которого к нам пришел пользователь, по-умолчанию: unknown
       - maillist_type - тип рассылок, по-умолчанию: inbox (это сделано на будущее, пока не используется)
       - country - страна организации, по-умолчанию: страна пользователя из Паспорта
       - name - название организации, по-умолчанию будет иметь вид #100500
       - label - короткое уникальное имя, будет использоваться для технического домена.
                 По-умолчанию: org-100500.
       - preset - идентификатор набора сервисов, которые надо включить, по-умолчанию: without-domain
       - organization_type - тип создаваемой организации, по-умолчанию: common
       - clouds - список облаков облачной организации, используется для синка пользователей
       Созданной организации включается фича CAN_WORK_WITHOUT_OWNED_DOMAIN,
       чтобы в неё можно было приглашать внешние учётки.
    """
    language = data.get(
        'language',
        'ru',  # язык по-умолчанию - русский
    )
    tld = data.get('tld', 'ru')
    source = data.get('source', 'unknown')
    maillist_type = data.get('maillist_type', 'inbox')
    clouds = data.get('clouds', [])
    cloud_org_id = data.get('cloud_org_id')

    log.info('Creating organization without domain')
    user = app.blackbox_instance.userinfo(
        uid=uid,
        userip=user_ip,
        dbfields=[('userinfo.country.uid', 'country')],
    )
    if not user['uid']:
        raise UserDoesNotExist()
    country = data.get('country', user['fields'].get('country'))

    already_exists = UserMetaModel(meta_connection).filter(
        id=uid, is_outer=False, is_dismissed=False
    ).count()
    if already_exists:
        if is_multiorg_feature_enabled_for_user(meta_connection, uid):
            check_organizations_limit(meta_connection, uid)
        else:
            log.warning('User already a member of some organization')
            raise ImmediateReturn(
                json_error(
                    409,
                    'has_organization',
                    'User already a member of some organization',
                )
            )

    meta_model = OrganizationMetaModel(meta_connection)
    org_id = meta_model.get_id_for_new_organization()
    org_type = data.get('organization_type', 'common')

    org_name = data.get('name', '#{}'.format(org_id))
    if org_type not in organization_type.cloud_types:
        assert_not_spammer(org_name)

    partner_id = data.get('partner_id')
    if partner_id is not None:
        partner_meta_org = OrganizationMetaModel(meta_connection).get(partner_id)
        if not partner_meta_org:
            raise ImmediateReturn(json_error_invalid_value('partner_id'))
        with get_main_connection(shard=partner_meta_org['shard']) as partner_main_connection:
            if not OrganizationModel(partner_main_connection).is_partner_organization(partner_id):
                raise ImmediateReturn(json_error_invalid_value('partner_id'))

    label = data.get('label', 'org-{}'.format(org_id))
    if already_exists:
        OrganizationRevisionCounterModel(main_connection).increment_revisions_for_user(
            meta_connection,
            uid,
        )
    # создаём организацию c признаком неподтвержденного домена в метабазе
    try:
        with meta_connection.begin_nested():
            organization_meta_instance = meta_model.create(
                id=org_id,
                label=label,
                shard=main_connection.shard,
                cloud_org_id=cloud_org_id,
                limits={'users_limit': settings.ORG_LIMITS['users_limit']},
            )
    except IntegrityError:
        existing_models = meta_model.filter(cloud_org_id=cloud_org_id).scalar('id')
        ids_for_message = ','.join(map(str, existing_models))
        raise ImmediateReturn(
            json_error(
                status_code=400,
                error_code='Conflict',
                error_message=f'Organizations with the same cloud_org_id already exist: {ids_for_message}',
                org_id=ids_for_message,
            )
        )

    org_id = organization_meta_instance['id']

    # применяем пресет without-domain или тот, что был указан фронтом
    preset = data.get('preset', 'without-domain')

    organization = OrganizationModel(main_connection).create(
        id=org_id,
        name=org_name,
        label=label,
        admin_uid=uid,
        language=language,
        source=source,
        tld=tld,
        country=country,
        maillist_type=maillist_type,
        preset=preset,
        organization_type=org_type,
        ip=user_ip,
        clouds=clouds,
        partner_id=data.get('partner_id'),
    )
    # событие создания организации
    initial_revision = action_organization_add(
        main_connection,
        org_id=org_id,
        author_id=uid,
        object_value=organization,
    )
    root_department = create_root_department(
        main_connection,
        org_id,
        language,
        # У корневого отдела не должно быть рассылки,
        # так как в этот момент у организации нет ни одного домена
        root_dep_label=None,
    )

    # Эта функция обычно срабатывает при создании action
    # и внутри генерит пару событий.
    # Тут мы не создаём сам action, потому что
    # отдел создан в рамках той же ревизии, которая появилась
    # в результате действия organization_add.
    on_department_add(
        main_connection,
        org_id=org_id,
        revision=initial_revision,
        department=root_department,
    )

    group_model = GroupModel(main_connection)
    group_model.get_or_create_robot_group(org_id)

    # И добавим в организацию первого сотрудника
    user = add_existing_user(
        meta_connection,
        main_connection,
        org_id,
        uid,
    )
    # TODO: Пользователю надо проставить специальный атрибут в Паспорте
    # но мы этого пока не делаем, потому что нужна доработка на стороне Паспорта.
    UserModel(main_connection).change_user_role(
        org_id,
        uid,
        UserRoles.admin,
        # Тут пользователь сам добавляет себя в админы только что созданной организации.
        admin_uid=uid,
        old_user=user,
    )
    app.passport.set_admin_option(uid)

    apply_preset(
        meta_connection,
        main_connection,
        org_id,
        preset,
    )

    # Включаем фичи, если надо.
    enable_features(
        meta_connection,
        main_connection,
        organization,
    )

    log.info('Organization without domain was created')
    return org_id


def create_organization_with_domain(
        meta_connection,
        data,
        uid,
        user_ip,
        registrar_id=None,
):
    """Создаёт организацию c доменом и делает учётку с uid - владельцем.
       Если передан registrar_id, то привязываем домен к регистратору и устанавливаем source 'registrar'
    """
    if not is_outer_uid(uid):
        raise DomainUserError()

    language = data.get(
        'language',
        'ru',  # язык по-умолчанию - русский
    )
    domain_name = data.get('domain_name')
    tld = data.get('tld')
    org_name = data.get('name', domain_name)
    assert_not_spammer(org_name)

    source = data.get('source', 'unknown')
    if registrar_id is not None:
        source = 'registrar'
    maillist_type = data.get('maillist_type', 'inbox')

    label = (domain_name + '-' + str(time.time())).replace('.', '-')

    with log.name_and_fields('domain_verify', domain=domain_name, registrar_id=registrar_id):
        log.info('Creating organization with domain')
        user = app.blackbox_instance.userinfo(
            uid=uid,
            userip=user_ip,
            dbfields=[('userinfo.country.uid', 'country')],
        )
        if not user['uid']:
            raise UserDoesNotExist()
        country = data.get('country', user['fields'].get('country'))
        check_organizations_limit(meta_connection, uid)

        with get_meta_connection(for_write=True) as write_meta_connection:
            shard = OrganizationMetaModel.get_shard_for_new_organization()
            with get_main_connection(shard, for_write=True) as write_main_connection:
                # создаём организацию c признаком неподтвержденного домена в метабазе
                organization_meta_instance = OrganizationMetaModel(write_meta_connection).create(
                    label=label,
                    shard=shard,
                    limits={'users_limit': settings.ORG_LIMITS['users_limit']},
                )

                OrganizationRevisionCounterModel(write_main_connection).increment_revisions_for_user(
                    write_meta_connection,
                    uid,
                )
                org_id = organization_meta_instance['id']
                # создаём организацию в базе

                preset = data.get('preset', 'default')
                organization = OrganizationModel(write_main_connection).create(
                    id=org_id,
                    name=org_name,
                    label=label,
                    admin_uid=uid,
                    language=language,
                    source=source,
                    tld=tld,
                    country=country,
                    maillist_type=maillist_type,
                    preset=preset,
                    registrar_id=registrar_id,
                    ip=user_ip,
                )
                # событие создания организации
                initial_revision = action_organization_add(
                    write_main_connection,
                    org_id=org_id,
                    author_id=uid,
                    object_value=organization,
                )
                root_department = create_root_department(
                    write_main_connection,
                    org_id,
                    language,
                    # У корневого отдела не должно быть рассылки,
                    # так как в этот момент у организации нет ни одного домена
                    root_dep_label=None,
                )
                on_department_add(
                    write_main_connection,
                    org_id=org_id,
                    revision=initial_revision,
                    department=root_department,
                )

                # И добавим в организацию первого сотрудника
                user = add_existing_user(
                    write_meta_connection,
                    write_main_connection,
                    org_id,
                    uid,
                )
                # TODO: Пользователю надо проставить специальный атрибут в Паспорте
                # но мы этого пока не делаем, потому что нужна доработка на стороне Паспорта.
                UserModel(write_main_connection).change_user_role(
                    org_id,
                    uid,
                    UserRoles.admin,
                    # Тут пользователь сам добавляет себя в админы только что созданной организации.
                    admin_uid=uid,
                    old_user=user,
                )
                app.passport.set_admin_option(uid)

                # применяем пресет no-owned-domain, чтобы для этой организации был доступен дашборд
                apply_preset(
                    write_meta_connection,
                    write_main_connection,
                    org_id,
                    'no-owned-domain',
                )
                # Включаем фичи, если надо.
                enable_features(
                    write_meta_connection,
                    write_main_connection,
                    organization,
                )

                # создаем домен
                if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
                    app.domenator.add_domain(org_id, domain_name, uid)
                else:
                    create_domain_verified_via_webmaster(
                        write_meta_connection,
                        write_main_connection,
                        domain_name,
                        org_id,
                    )

                GroupModel(write_main_connection).get_or_create_robot_group(org_id)

                if registrar_id is not None:
                    DomainRegistrationProcedureTask(write_main_connection).delay(
                        registrar_id=registrar_id,
                        domain_name=domain_name,
                        org_id=org_id,
                    )

            log.info('Organization with domain was created')

            return org_id
