import collections
import threading
from copy import copy, deepcopy
from itertools import groupby

from flask import request

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_meta_connection, use_connections,
    get_main_connection, get_shard_numbers
)
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_admin_after_create_organization_email
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel, OrganizationModel, OrganizationMetaModel,
    OrganizationServiceModel, UserMetaModel
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    prepare_user_with_fields, only_attrs,
    paginate_by_shards, ShardNotFoundError, is_outer_uid
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import get_sorted_user_ready_org_ids
from intranet.yandex_directory.src.yandex_directory.core.views.organization import constants
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ImmediateReturn
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    is_spammy, utcnow, json_error_invalid_value, url_join, format_date, json_error
)


def _process_fields(meta_connection,
                    main_connection,
                    organizations,
                    fields=None,
                    version=1):
    """
    Раскрывает некоторые поля в объекте организация.

    Если fields == None, то раскрываются все поля.
    Если это список, то раскрываются только перечисленные.
    """
    if fields is None:
        fields = ('id', 'head', 'services', 'domains')

    _process_organizations_heads(
        meta_connection,
        main_connection,
        organizations,
        api_version=request.api_version,
    )

    _process_organizations_services(
        meta_connection,
        main_connection,
        organizations,
        version=version,
    )


def _process_organizations_heads(meta_connection,
                                 main_connection,
                                 organizations,
                                 api_version):
    """Заменяет в организации ключ head_id на head,
    подтягивая из базы главу компании.
    """

    # Пропустим head через prepare_user
    for org in organizations:
        if 'head' in org:
            org.pop('head_id', None)
            head = org['head']

            if head:
                org['head'] = prepare_user_with_fields(
                    meta_connection,
                    main_connection,
                    head,
                    fields=UserModel.simple_fields,
                    api_version=api_version,
                )

    return organizations


def _clean_organization(organization, api_version=1):
    """
    Удаляем из данных об организации приватные поля
    перед выдачей данных из ручки
    :param organization:
    :return:
    """
    for field in constants.ORGANIZATION_PRIVATE_FIELDS:
        if not app.config['INTERNAL'] or field not in constants.ORGANIZATION_PRIVATE_FIELDS_INTERNAL_EXCLUSIONS:
            organization.pop(field, None)
    if api_version > 3:
        for field in constants.ORGANIZATION_PRIVATE_FIELDS_V4:
            organization.pop(field, None)
        if api_version > 5 and 'domains' in organization:
            organization['domains'].pop('domain', None)

    return organization


def _delay_welcome_email(org_id, admin_nickname, uid):
    """
    Отложенно отправляем welcome письмо c задержкой env:MAIL_SEND_DELAY
    :param org_id:  организация
    :type org_id: int
    :param uid: пользователь
    :type uid: int
    """

    @use_connections(for_write=True)
    def _send_welcome_email(meta_connection, main_connection, org_id, uid):
        # отправляем письмо
        with app.app_context():
            send_admin_after_create_organization_email(
                meta_connection=meta_connection,
                main_connection=main_connection,
                org_id=org_id,
                uid=uid,
                admin_nickname=admin_nickname,
            )

    thread = threading.Timer(
        app.config['MAIL_SEND_DELAY'],
        _send_welcome_email,
        kwargs={'org_id': org_id, 'uid': uid}
    )
    thread.daemon = True
    thread.start()


def _get_organizations_info(meta_connection, org_ids, fields=None):
    """
    Отдаем полную информацию про организации по списку id-шников
    :rtype: list[dict]
    """
    if not org_ids:
        return []

    OrganizationModel.raise_on_prohibited_fields(fields, ('ip', 'karma'))

    meta_organizations = OrganizationMetaModel(meta_connection).find(
        {'id': org_ids},
        fields=['id', 'shard', 'ready', 'cloud_org_id'],
    )
    # Сгруппируем организации по шарду, чтобы не открывать коннекты
    # по многу раз.
    by_shard = lambda org: org['shard']
    meta_organizations.sort(key=by_shard)
    grouped_by_shard = groupby(meta_organizations, by_shard)

    organizations = []
    for shard, metas in grouped_by_shard:
        with get_main_connection(shard) as main_connection:
            # Одним запросом возьмем все организации из этого шарда
            shard_org_ids = []
            id_to_cloud_id = {}

            for meta in metas:
                if meta['ready']:
                    shard_org_ids.append(meta['id'])
                    id_to_cloud_id[meta['id']] = meta['cloud_org_id']

            # DIR-3191: если shard_org_ids - пусто, значит еще нет организаций или они не готовы
            if not shard_org_ids:
                continue

            has_cloud_org_id = 'cloud_org_id' in fields
            fields = [field for field in fields if field != 'cloud_org_id']

            shard_orgs = OrganizationModel(main_connection).find(
                {
                    'id': shard_org_ids,
                },
                fields=fields,
            )
            if has_cloud_org_id:
                for shard_org in shard_orgs:
                    shard_org['cloud_org_id'] = id_to_cloud_id[shard_org['id']]

            # Развернём вложенные поля для всех организаций
            # этого шарда.
            _process_fields(
                meta_connection,
                main_connection,
                shard_orgs,
                fields=fields,
                version=request.api_version,
            )
            organizations.extend(shard_orgs)

    return organizations


def _get_all_admin_organizations(request_user):
    """
    Возвращает список id организаций, где указанный пользователь - админ
    """
    orgs_ids = []
    for shard in get_shard_numbers():
        with get_main_connection(shard=shard) as main_connection:
            shard_orgs = UserModel(main_connection).filter(
                id=request_user.passport_uid,
                role='admin',
            ).scalar('org_id')
            orgs_ids.extend(shard_orgs)
    return orgs_ids


def _get_service_organizations_from_shard(meta_connection,
                                          shard,
                                          service_id,
                                          ready,
                                          fields,
                                          skip=None,
                                          limit=None):
    """Возвращает организации включенные для сервиса из указанного шарда.

    Если указаны параметры skip и limit, то выдаёт не все организации
    а только указанный slice.

    В любом случае, даже если ничего не найдено, возвращает список.
    """

    with get_main_connection(shard) as main_connection:
        filter_data = {'service_id': service_id}
        # фильтр по готовности сервиса в организации
        if ready is not None:
            filter_data['ready'] = ready

        service_organizations = OrganizationServiceModel(main_connection).find(
            filter_data=filter_data,
            # кроме org_id нам от этого запроса ничего не нужно
            fields=['org_id'],
            order_by=['org_id'],
            skip=skip,
            limit=limit,
        )
        org_ids = only_attrs(service_organizations, 'org_id')
        # Может случиться такое, что в шарде пока нет
        # ни одной организации, для которой подключен
        # сервис, и тогда список организаций получать не надо.
        # Тесты этот момент упустили, так как у нас автотесты
        # гоняются на одном шарде
        if org_ids:
            organizations = OrganizationModel(main_connection).find(
                {'id': org_ids},
                fields=fields,
            )
            _process_fields(
                meta_connection,
                main_connection,
                organizations,
                fields=fields,
                version=request.api_version,
            )
        else:
            organizations = []

        return organizations


def _get_service_organizations(meta_connection,
                               service_id,
                               fields=None,
                               ready=None,
                               query_params=None):
    """
    Получить список организациий по заданному service_id
    :rtype: list[dict]
    """
    results = []
    links = {}

    query_params = query_params or {}
    per_page = query_params.get('per_page')

    # Пока мы поддерживаем версии API < 4,
    # нужна эта ветка, отдающая все организации из всех шардов сразу
    if per_page is None:
        for shard in get_shard_numbers():
            organizations = _get_service_organizations_from_shard(
                meta_connection,
                shard,
                service_id,
                ready,
                fields
            )
            results += organizations
    else:
        # Определим, организации из какого шарда надо отдавать
        shards = get_shard_numbers()
        try:
            shard = int(query_params.get('shard', shards[0]))
        except ValueError:
            raise ImmediateReturn(
                json_error_invalid_value('shard')
            )

        try:
            page = max(1, int(query_params.get('page', '1')))
        except ValueError:
            raise ImmediateReturn(
                json_error_invalid_value('page')
            )

        try:
            per_page = int(per_page)
        except ValueError:
            raise ImmediateReturn(
                json_error_invalid_value('per_page')
            )

        def get_data(shard, offset, limit):
            return _get_service_organizations_from_shard(
                meta_connection,
                shard,
                service_id,
                ready,
                fields,
                skip=offset,
                limit=limit,
            )

        try:
            results, next_shard, next_page = paginate_by_shards(
                shards=shards,
                shard=shard,
                page=page,
                per_page=per_page,
                item_getter=get_data,
            )
        except ShardNotFoundError:
            # Если шард не удалось найти, то это ошибка
            # и надо вернуть BadRequest
            raise ImmediateReturn(
                json_error_invalid_value('shard')
            )

        def build_url(**kwargs):
            params = deepcopy(query_params)
            params.update(kwargs)

            return url_join(
                app.config['SITE_BASE_URI'],
                request.path,
                force_trailing_slash=True,
                query_params=params,
            )

        # Если есть следующая страница, то формируем на неё ссылку
        if next_shard and next_page:
            next_link = build_url(
                shard=next_shard,
                per_page=per_page,
                page=next_page,
            )
        else:
            # Если это последний шард, то линка на следующую
            # страницу быть не должно.
            next_link = None

        if next_link:
            links['next'] = next_link

    return results, links


def _get_all_organizations_from_shard(meta_connection,
                                      shard,
                                      fields,
                                      id_list=None,
                                      skip=None,
                                      limit=None):
    """Возвращает все организации из указанного шарда.
    Если указаны ids, то все организации из шарда в пределах ids.

    Если указаны параметры skip и limit, то выдаёт не все организации
    а только указанный slice.

    В любом случае, даже если ничего не найдено, возвращает список.
    """
    filter_data = {
        'ready': True,
        'shard': shard,
    }
    if id_list is not None:
        if not id_list:
            # если пришел пустой массив, то искать не будем
            return []
        filter_data['id'] = id_list

    with get_main_connection(shard) as main_connection:
        # Дополнительный запрос в метабазу нужен для того, чтобы выбрать только
        # "ready=True" организации
        meta_organizations = OrganizationMetaModel(meta_connection).find(
            filter_data=filter_data,
            # кроме org_id нам от этого запроса ничего не нужно
            fields=['id'],
            order_by=['id'],
            skip=skip,
            limit=limit,
        )
        org_ids = only_attrs(meta_organizations, 'id')
        # Может случиться такое, что в шарде пока нет
        # ни одной организации, для которой подключен
        # сервис, и тогда список организаций получать не надо.
        if org_ids:
            organizations = OrganizationModel(main_connection).find(
                {'id': org_ids},
                fields=fields,
            )
            _process_fields(
                meta_connection,
                main_connection,
                organizations,
                fields=fields,
                version=request.api_version,
            )
        else:
            organizations = []

        return organizations


def _get_all_organizations(meta_connection, fields=None, query_params=None, id_list=None):
    """
    Получить список всех организаций с пейджинацией по шардам.
    """
    def get_data(shard, offset, limit):
        return _get_all_organizations_from_shard(
            meta_connection,
            shard,
            fields,
            id_list=id_list,
            skip=offset,
            limit=limit,
        )

    return _paginate_organizations_by_shards(query_params, get_data)


def _get_page_params(query_params):
    query_params = query_params or {}

    per_page = query_params.get('per_page')
    if per_page is None:
        return None, None

    try:
        per_page = max(1, int(per_page))
    except ValueError:
        raise ImmediateReturn(
            json_error_invalid_value('per_page')
        )

    try:
        page = max(1, int(query_params.get('page', '1')))
    except ValueError:
        raise ImmediateReturn(
            json_error_invalid_value('page')
        )

    return page, per_page


def _get_user_organizations(meta_connection, uid, org_ids=None, fields=None, query_params=None, is_cloud=False, organization_type=None):
    OrganizationModel.raise_on_prohibited_fields(fields, ('ip', 'karma'))

    sorted_org_ids_full_list = get_sorted_user_ready_org_ids(meta_connection, uid, org_ids, is_cloud=is_cloud)
    if not sorted_org_ids_full_list:
        return [], {}

    sorted_org_ids_page_list = sorted_org_ids_full_list

    links = {}
    page, per_page = _get_page_params(query_params)
    if per_page:
        first_item_index = (page - 1) * per_page
        last_item_index = first_item_index + per_page - 1
        sorted_org_ids_page_list = sorted_org_ids_full_list[first_item_index:last_item_index + 1]

        if last_item_index + 1 < len(sorted_org_ids_full_list):
            params = deepcopy(query_params)
            params.update(page=page+1)
            links['next'] = url_join(
                app.config['SITE_BASE_URI'],
                request.path,
                force_trailing_slash=True,
                query_params=params,
            )

    order_num_by_org_id = {}
    order_num = 0
    for org_id in sorted_org_ids_page_list:
        order_num_by_org_id[org_id] = order_num
        order_num += 1

    meta_organizations = OrganizationMetaModel(meta_connection).find(
        {'id': sorted_org_ids_page_list},
        fields=['id', 'shard', 'cloud_org_id'],
    )

    org_info_by_shard = collections.defaultdict(list)
    for meta_organization in meta_organizations:
        shard = meta_organization['shard']
        org_info_by_shard[shard].append(meta_organization)

    organizations = []
    has_cloud_org_id = fields is not None and 'cloud_org_id' in fields
    if has_cloud_org_id and fields is not None:
        fields = [field for field in fields if field != 'cloud_org_id']
        if 'id' not in fields:
            fields.append('id')

    for shard, shard_metas in org_info_by_shard.items():
        shard_org_ids = [meta['id'] for meta in shard_metas]
        id_to_cloud_org_id = {meta['id']: meta['cloud_org_id'] for meta in shard_metas}

        with get_main_connection(shard) as main_connection:
            org_filter = {'id': shard_org_ids}
            if organization_type is not None:
                org_filter['organization_type'] = organization_type.split(',')
            shard_orgs = OrganizationModel(main_connection).find(
                org_filter, fields=fields
            )
            if has_cloud_org_id:
                for org in shard_orgs:
                    org['cloud_org_id'] = id_to_cloud_org_id[org['id']]

            _process_fields(
                meta_connection,
                main_connection,
                shard_orgs,
                fields=fields,
                version=request.api_version,
            )
            organizations.extend(shard_orgs)

    organizations.sort(key=lambda organization: order_num_by_org_id[organization['id']])

    return organizations, links


def _paginate_organizations_by_shards(query_params, item_getter_func):
    query_params = query_params or {}
    per_page = query_params.get('per_page')

    # Определим, организации из какого шарда надо отдавать
    shards = get_shard_numbers()
    try:
        shard = int(query_params.get('shard', shards[0]))
    except ValueError:
        raise ImmediateReturn(
            json_error_invalid_value('shard')
        )

    try:
        page = max(1, int(query_params.get('page', '1')))
    except ValueError:
        raise ImmediateReturn(
            json_error_invalid_value('page')
        )

    try:
        per_page = max(1, int(per_page))
    except ValueError:
        raise ImmediateReturn(
            json_error_invalid_value('per_page')
        )

    try:
        results, next_shard, next_page = paginate_by_shards(
            shards=shards,
            shard=shard,
            page=page,
            per_page=per_page,
            item_getter=item_getter_func,
        )
    except ShardNotFoundError:
        # Если шард не удалось найти, то это ошибка
        # и надо вернуть BadRequest
        raise ImmediateReturn(
            json_error_invalid_value('shard')
        )

    def build_url(**kwargs):
        params = deepcopy(query_params)
        params.update(kwargs)

        return url_join(
            app.config['SITE_BASE_URI'],
            request.path,
            force_trailing_slash=True,
            query_params=params,
        )

    # Если есть следующая страница, то формируем на неё ссылку
    if next_shard and next_page:
        next_link = build_url(
            shard=next_shard,
            per_page=per_page,
            page=next_page,
        )
    else:
        # Если это последний шард, то линка на следующую
        # страницу быть не должно.
        next_link = None

    links = {}
    if next_link:
        links['next'] = next_link

    return results, links

def preprocess_requested_fields(fields):
    """
    Временный "костыль" для того, чтобы отдавать все поля о сервисах и доменах.
    В будущих версиях API мы возможно позволим выбирать отдельные поля
    прямо сейчас, отдельно можно запросить лишь services.responsible.name,
    и это было сделано для задачи: https://st.yandex-team.ru/DIR-6313
    """

    # Нехорошо менять входные параметры, поэтому мы вернём новый список полей.
    fields = copy(fields)

    services_found = False

    for item in fields:
        if item == 'services':
            services_found = True
            fields.remove('services')
        if item.startswith('services.'):
            services_found = True

    if services_found:
        fields.extend(constants.SERVICE_DEFAULT_FIELDS)

    # Про домены мы тоже отдаём все простые поля
    if 'domains' in fields:
        fields.remove('domains')
        fields.append('domains.*')

    return fields


def _process_organizations_services(meta_connection,
                                    main_connection,
                                    organizations,
                                    version=1):
    """
    Добавляет в данные об организации ключ services со списком slug подключенных сервисов
    """

    for organization in organizations:
        org_services = organization.get('services')
        if org_services:
            if version >= 2:

                services = []
                for service in org_services:
                    if version > 7:
                        days_till_expiration = None
                        if service['trial_expires']:
                            days_till_expiration = max((service['trial_expires'] - utcnow().date()).days, 0)
                        service_info = {
                            'slug': service['slug'],
                            'ready': service['ready'],
                            'enabled': service['enabled'],
                            'trial': {
                                'expiration_date': format_date(service['trial_expires'], allow_none=True),
                                'status': service['trial_status'],
                                'days_till_expiration': days_till_expiration,
                            },
                            'expires_at': format_date(service['expires_at'], allow_none=True),
                            'user_limit': service['user_limit'],
                        }
                        # Поле с ответственным может быть запрошено отдельно
                        if 'responsible' in service:
                            service_info['responsible'] = service['responsible']

                        services.append(service_info)
                    # В старых версиях API мы выдавали только
                    # включённые сервисы
                    elif service['enabled']:
                        trial_expired = None
                        if service['trial_expires']:
                            trial_expired = service['trial_expires'] < utcnow().date()
                        service_info = {
                            'slug': service['slug'],
                            'ready': service['ready'],
                            'trial_expires': format_date(service['trial_expires'], allow_none=True),
                            'trial_expired': trial_expired,
                            'expires_at': format_date(service['expires_at'], allow_none=True),
                            'user_limit': service['user_limit'],
                        }
                        # Поле с ответственным может быть запрошено отдельно
                        if 'responsible' in service:
                            service_info['responsible'] = service['responsible']
                        services.append(service_info)

                organization['services'] = services

            else:
                organization['services'] = only_attrs([serv for serv in org_services if serv['enabled']], 'slug')


def assert_not_spammer(org_name):
    if is_spammy(org_name):
        with log.fields(name=org_name):
            log.warning('Bad organization name')

        raise ImmediateReturn(
            json_error(
                422,
                'bad_name',
                'Bad organization name',
            )
        )


def remove_orgs_with_deleted_portal_user(organizations, uid, is_cloud=False):
    if not organizations:
        return []
    if is_cloud or not is_outer_uid(uid):
        return organizations
    with get_meta_connection() as connection:
        ids_to_remove = UserMetaModel(connection).filter(is_dismissed=True, id=uid).fields('org_id').scalar('org_id')
    return [org for org in organizations if org['id'] not in ids_to_remove]
