# -*- coding: utf-8 -*-
import json
import uuid
from copy import deepcopy

from flask import g, request
from google.protobuf.json_format import MessageToDict
from sqlalchemy.orm.exc import NoResultFound

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth import decorators as auth_decorators
from intranet.yandex_directory.src.yandex_directory.auth.scopes import check_scopes
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection_for_new_organization,
    get_main_connection,
    get_shard,
    get_shard_numbers,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    AuthorizationError,
    OrganizationNotReadyError,
    NotFoundError,
    InvalidValue,
    UserDoesNotExist,
    InvalidDomain,
    DomainOccupied,
    RegistrarCheckFailed,
    ParameterNotCorrect,
    InvalidEmail,
)
from intranet.yandex_directory.src.yandex_directory.common.models.types import (
    TYPE_ORGANIZATION,
    TYPE_DEPARTMENT,
)
from intranet.yandex_directory.src.yandex_directory.common.pagination import get_link_header
from intranet.yandex_directory.src.yandex_directory.common.sqlalchemy import EntityNotFoundError
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error,
    json_error_bad_request,
    json_error_forbidden,
    json_error_not_found,
    json_error_invalid_value,
    split_by_comma,
    get_boolean_param,
    url_join,
    json_error_required_field,
    get_user_id_from_passport_by_login,
)
from intranet.yandex_directory.src.yandex_directory.connect_services.cloud.grpc.client import GrpcCloudClient
from intranet.yandex_directory.src.yandex_directory.connect_services.exceptions import OrganizationHasResources
from intranet.yandex_directory.src.yandex_directory.core import dependencies
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_organization_logo_change,
    action_organization_add,
    action_domain_master_modify,
    action_organization_type_change,
    action_user_add,
    action_organization_modify,
    action_organization_delete,
)
from intranet.yandex_directory.src.yandex_directory.core.actions.organization import _notify_organization_deleted
from intranet.yandex_directory.src.yandex_directory.core.events import (
    event_department_added,
    event,
)
from intranet.yandex_directory.src.yandex_directory.core.events.utils import get_callbacks_for_events
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    OrganizationHasDebt,
    OrganizationHasBillingInfo,
)
from intranet.yandex_directory.src.yandex_directory.core.features import USE_DOMENATOR
from intranet.yandex_directory.src.yandex_directory.core.features.utils import get_organization_features_info, \
    is_multiorg_feature_enabled_for_user, is_feature_enabled, get_feature_by_slug, set_feature_value_for_organization
from intranet.yandex_directory.src.yandex_directory.core.models import (
    ImageModel,
    OrganizationMetaModel,
    OrganizationModel,
    UserModel,
    DomainModel,
    OrganizationServiceModel,
    ServiceModel,
    UserMetaModel,
    DepartmentModel,
    TaskModel,
    EventModel,
    ResourceModel,
    OrganizationRevisionCounterModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models import OrganizationBillingInfoModel
from intranet.yandex_directory.src.yandex_directory.core.models.domain import (
    get_domain_id_from_blackbox,
    DomainNotFound
)
from intranet.yandex_directory.src.yandex_directory.core.models.preset import apply_preset
from intranet.yandex_directory.src.yandex_directory.core.models.service import MAILLIST_SERVICE_SLUG
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    organization_permissions,
    global_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.registrar.client import RegistrarClient
from intranet.yandex_directory.src.yandex_directory.core.registrar.exceptions import RegistrarInteractionException
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.core.tasks import SetOrganizationNameInPassportTask
from intranet.yandex_directory.src.yandex_directory.core.tasks import SyncExternalIDS
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    get_user_info_from_blackbox,
    get_organization_admin_uid,
    ensure_integer,
    create_organization,
    only_attrs,
    is_outer_uid,
    create_maillist,
    unfreeze_or_copy,
    is_email_valid,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import (
    delete_domain_with_accounts,
    delete_domain_from_passport, get_domains_from_db_or_domenator, update_domains_in_db_or_domenator,
    DomainUpdateFilter, DomainUpdateData,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.organization import get_balance_and_current_month_price
from intranet.yandex_directory.src.yandex_directory.core.utils.tasks import (
    ChangeOrganizationOwnerTask,
    DeleteOrganizationTask,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import (
    remaining_users_number,
    is_outer_admin,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View, no_cache
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory import swagger as schema_decorators
from intranet.yandex_directory.src.yandex_directory.core.views.organization import constants
from intranet.yandex_directory.src.yandex_directory.core.views.organization.logic import (
    create_organization_with_domain,
    create_organization_without_domain
)
from intranet.yandex_directory.src.yandex_directory.core.views.organization.utils import (
    _clean_organization, _get_page_params, _get_user_organizations,
    remove_orgs_with_deleted_portal_user, _get_service_organizations, _get_all_organizations,
    assert_not_spammer, _delay_welcome_email, _process_fields, preprocess_requested_fields, _get_organizations_info,
    _get_all_admin_organizations
)

# поля, которые можно запросить в ручке /organizations/
from intranet.yandex_directory.src.yandex_directory.core.models import SupportActionMetaModel
from intranet.yandex_directory.src.yandex_directory.core.models.action import SupportActions
from intranet.yandex_directory.src.yandex_directory.core.models.meta import OrganizationMetaValuesModel, OrganizationMetaKeysModel

ORGANIZATION_AVAILABLE_FIELDS_V4 = sorted(
    set(OrganizationModel.all_fields)
    - set(constants.ORGANIZATION_PRIVATE_FIELDS + constants.ORGANIZATION_PRIVATE_FIELDS_V4)
)


class OrganizationByLabelView(View):
    @auth_decorators.no_scopes
    @auth_decorators.internal
    @auth_decorators.no_auth
    def get(self, meta_connection, main_connection, label):
        """
        Информация о существовании конкретной организации по её label.

        Для этой ручки авторизация не требуется.
        ---
        tags:
          - Организация
        parameters:
          - in: path
            name: label
            required: true
            type: string
        responses:
          200:
            description: Success
        """
        if OrganizationMetaModel(meta_connection).find({'label': label}):
            return json_response({'label': label})
        return json_error_not_found()


class OrganizationByOrgIdView(View):
    methods = ['get', 'delete']

    @auth_decorators.no_permission_required
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.scopes_required([scope.read_organization])
    def get_6(self, meta_connection, main_connection, org_id):
        """
        Информация об одной организации по org_id

        Если по org_id не удалось найти организацию, то вернется 404.
        Если сервис не включен у организации с id=org_id, то вернется 403.

        Если не передана информация о необходимых полях, то будет возвращен только id организации.
        Необходимые поля возвращаются, если они указаны в поле fields.

        Например:

        /v6/organizations/100500/?fields=head,services,domains

        Пример ответа:

            {
              "id": 1,
              "label": "yandex",
              "admin_uid": 4000811136,
              "name": {
                "en": "Yandex",
                "ru": "Яндекс"
              },
              "domains": {
                  "all": ["test.ru", "test1.ru"],
                  "owned": ["test.ru"],
                  "master": "test.ru",
              },
              "inn": "4324234324",
              "head": {
                  ...
              },
              "services": [{
                "slug": "disk",
                "ready": True,
                "trial_expires": None,
                "trial_expired": None
              }]
              ...
            }

        Поля организации:

        * **id** - идентификатор организации
        * **domains** - домены, принадлежащие данной организации (только по запросу в fields)
        * **services** - сервисы (только по запросу в fields)
        * **head** - все руководители, если есть (только по запросу в fields)
        ---
        tags:
          - Организация
        parameters:
          - in: path
            name: org_id
            required: true
            type: integer
          - in: query
            name: fields
            type: string
            description: название запрашиваемых полей через запятую
        responses:
          200:
            description: Success
          403:
            description: Request is not available for this service
          404:
            description: Organization is not found by org_id
        """

        org_id = ensure_integer(org_id, 'org_id')
        default_fields = ['id']
        fields = split_by_comma(request.args.get('fields')) or default_fields

        organization_meta = OrganizationMetaModel(meta_connection).get(id=org_id)
        if organization_meta and not organization_meta['ready']:
            raise OrganizationNotReadyError()

        if g.use_cloud_proxy:
            cloud_org_id = organization_meta['cloud_org_id']
            org_response = GrpcCloudClient().get_organization(
                org_id=cloud_org_id,
            )
            return json_response(MessageToDict(org_response))

        fields = preprocess_requested_fields(fields)

        organizations = _get_organizations_info(
            meta_connection,
            [org_id],
            fields=fields,
        )
        if not organizations:
            OrganizationMetaModel(meta_connection).raise_organization_not_found_exception(org_id)

        organization = organizations[0]
        shard = get_shard(meta_connection, org_id)

        # Если скоуп work_with_any_organization,
        # то сервису можно работать с любой организацией,
        # и он не обязательно должен быть к ней подключен.
        if not check_scopes(g.scopes, [scope.work_with_any_organization]):

            if not g.service:
                return json_error_forbidden()

            with get_main_connection(shard) as main_connection:
                # Смотрим, подключена ли запрашиваемая организация к сервису
                organization_service = OrganizationServiceModel(main_connection).find(filter_data={
                    'org_id': org_id,
                    'service_id': g.service.id,
                })
                if not organization_service:
                    raise AuthorizationError(
                        'Service is not enabled',
                        'service_is_not_enabled'
                    )
                elif not organization_service[0]['enabled']:
                    raise AuthorizationError(
                        'Service was disabled for this organization.',
                        'service_was_disabled'
                    )

        # удалим приватные поля
        _clean_organization(organization, request.api_version)

        return json_response(organization, allowed_sensitive_params=['can_users_change_password'])

    # 2 версия, в которой ещё есть поле admin_uid.
    # временно поддерживаем эту версию.
    get_2 = get_6
    get_8 = get_6

    get_8.__doc__ = """
Информация об одной организации по org_id

Если по org_id не удалось найти организацию, то вернется 404.
Если сервис не включен у организации с id=org_id, то вернется 403.

Если не передана информация о необходимых полях, то будет возвращен только id организации.
Необходимые поля возвращаются, если они указаны в поле fields.
Если запрошено поле services, то возвращаются все сервисы, которые когда-либо
были включены. Статус сервиса (включен/выключен) указан в параметре enabled.

Например:

/v8/organizations/100500/?fields=head,services,domains

Пример ответа:

    {
      "id": 1,
      "label": "yandex",
      "admin_uid": 4000811136,
      "name": {
        "en": "Yandex",
        "ru": "Яндекс"
      },
      "domains": {
          "all": ["test.ru", "test1.ru"],
          "owned": ["test.ru"],
          "master": "test.ru",
      },
      "inn": "4324234324",
      "head": {
          ...
      },
      "services": [
        {{
            "slug": "disk",
            "ready": True,
            "enabled": True,
            "trial": {{
                "expiration_date": None,
                "status": "inapplicable",
                "days_till_expiration": None
            }}
        }}
      ]
      ...
    }

Поля организации:

* **id** - идентификатор организации
* **domains** - домены, принадлежащие данной организации (только по запросу в fields)
* **services** - сервисы (только по запросу в fields)
* **head** - все руководители, если есть (только по запросу в fields)
---
tags:
  - Организация
parameters:
  - in: path
    name: org_id
    required: true
    type: integer
  - in: query
    name: fields
    type: string
    description: название запрашиваемых полей через запятую
responses:
  200:
    description: Success
  403:
    description: Request is not avaliable for this service
  404:
    description: Organization is not found by org_id
        """

    @auth_decorators.internal
    @auth_decorators.requires(org_id=False, user=True)
    @auth_decorators.permission_required([organization_permissions.delete])
    @auth_decorators.scopes_required([scope.delete_organization])
    def delete(self, meta_connection, main_connection, org_id):
        """
        Удаляет организацию и все ее данные, если нет биллиноговой информации и аккаунтов
        ---
        tags:
          - Организация
        parameters:
          - in: path
            name: org_id
            required: true
            type: integer
        responses:
          204:
            description: Огранизация удалена
          403:
            description: в органиазции остались аккаунты

        """
        # Если в организации остались учетки - возвращаем ошибку.
        # Если есть домены - удаляем оставшиеся учетки и домены в паспорте,
        # удаляем остатки организации в базе директории
        with log.name_and_fields('delete_organization', org_id=org_id):
            if bool(OrganizationBillingInfoModel(main_connection).get(org_id)):
                raise OrganizationHasBillingInfo()
            log.info('Trying to delete organizations')
            # Если в организации есть ресурсы Метрики, Директа, Справочника - возвращаем ошибку
            remaining_resources = ResourceModel(main_connection).filter(
                org_id=org_id,
                service__in=['metrika', 'direct', 'yandexsprav'],
            ).count()
            if remaining_resources:
                raise OrganizationHasResources()

            # Если в организации остались учетки, кроме последнего админа - возвращаем ошибку
            user_id = g.user.passport_uid
            accounts_number = remaining_users_number(main_connection, org_id, user_id)
            if accounts_number > 0:
                log.warning('Organization has accounts and can\'t be deleted')
                return json_error(
                    422,
                    'organization_has_accounts',
                    'Organizations has {accounts_number} accounts',
                    accounts_number=accounts_number,
                )

            log.info('Deleting organization')
            if OrganizationModel(main_connection).has_owned_domains(org_id):
                # Если домены есть - удаляем
                delete_domain_with_accounts(main_connection, org_id)

            else:
                # удаляем домены из паспорта
                delete_domain_from_passport(main_connection, org_id)

            organization = OrganizationModel(main_connection).filter(id=org_id).one()

            action_organization_delete(
                main_connection=main_connection,
                org_id=org_id,
                author_id=g.user.passport_uid if g.user else None,
                object_value=organization,
            )

            # получим всех подписчиков события удаления организации до очистки базы
            organization_deleted_event = EventModel(main_connection).fields('*', 'environment').filter(
                org_id=org_id,
                name=event.organization_deleted
            ).one()
            new_events_subscribers = get_callbacks_for_events(
                meta_connection,
                main_connection,
                [organization_deleted_event],
                organization['organization_type'],
            )

            # удаляем остатки организации в базе директории
            OrganizationModel(main_connection).remove_all_data_for_organization(org_id)

            # Создаём задачу, чтобы обновить в паспорте список org_id для учётки админа.
            # Это нужно, чтобы все сервисы понимали, что он больше не состоит в этой организации.
            try:
                SyncExternalIDS(main_connection).delay(
                    org_id=org_id,
                    user_id=user_id,
                )
            except DuplicatedTask:
                pass

            try:
                _notify_organization_deleted(
                    meta_connection,
                    main_connection,
                    new_events_subscribers,
                    organization_deleted_event,
                )
            except Exception:
                log.trace().error('Unable to notify about deleted organization')
            log.info('Notify subscribers')
            log.info('Organization has been deleted')

            OrganizationRevisionCounterModel(main_connection).increment_revisions_for_user(
                meta_connection,
                user_id,
            )

            return json_response({}, status_code=204)


class OrganizationFeaturesByOrgIdView(View):
    methods = ['get']

    @auth_decorators.internal
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.no_permission_required
    @auth_decorators.scopes_required([scope.read_organization])
    def get(self, meta_connection, main_connection, org_id):
        """
        Информация о включенных в организации фичах


        Пример ответа:
            {
                "pdd-proxy-to-connect": {"enabled": True, "default": False, "description": "Switches traffic of PDD proxy to Connect",
                "domain-auto-handover": {"enabled": False, "default": False, "description": "Auto domain handover"
            }

        tags:
          - Организация
        parameters:
          - in: path
            name: org_id
            required: true
            type: integer
        responses:
          200:
            description: Success
          403:
            description: Request is not avaliable for this service
          404:
            description: Organization is not found by org_id
        """
        org_id = ensure_integer(org_id, 'org_id')
        organization = OrganizationMetaModel(meta_connection).filter(id=org_id).one()
        if not organization:
            raise NotFoundError
        if not organization['ready']:
            raise OrganizationNotReadyError()

        features = get_organization_features_info(meta_connection, org_id)

        return json_response(features)


class OrganizationMetaView(View):
    @auth_decorators.internal
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.no_permission_required
    @auth_decorators.scopes_required([scope.write_organization, scope.read_organization])
    def get(self, meta_connection, _, org_id, key):
        key_model = OrganizationMetaKeysModel(meta_connection).filter(key=key).one()
        if key_model is None or not self._check_access(key_model['write_services'] + key_model['read_services']):
            return json_error_forbidden()

        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            value_model = OrganizationMetaValuesModel(main_connection).filter(org_id=org_id, key_id=key_model['id']).one()
            if value_model:
                return {'value': value_model['value'], 'created_at': value_model['created_at'], 'updated_at': value_model['updated_at']}
            else:
                return {'value': None, 'created_at': None, 'updated_at': None}

    @auth_decorators.internal
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.no_permission_required
    @auth_decorators.scopes_required([scope.write_organization])
    @schema_decorators.uses_schema(constants.ORGANIZATION_META_SAVE_SCHEMA)
    def post(self, meta_connection, _, data, org_id, key):
        key_model = OrganizationMetaKeysModel(meta_connection).filter(key=key).one()
        if key_model is None or not self._check_access(key_model['write_services']):
            return json_error_forbidden()

        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            OrganizationMetaValuesModel(main_connection).insert_or_update(
                org_id,
                key_model['id'],
                data['value']
            )
            return json_response({}, status_code=201)

    def _check_access(self, allowed_services):
        if not g.service:
            return False
        return g.service.id in allowed_services


class OrganizationFeaturesChangeView(View):
    @auth_decorators.internal
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.no_permission_required
    @auth_decorators.scopes_required([scope.write_organization])
    def post(self, meta_connection, _, org_id, feature_slug, action):
        return self._change_feature(
            meta_connection,
            org_id,
            feature_slug,
            action,
            f'{g.service.name} change feature'
        )

    @staticmethod
    def _change_feature(meta_connection, org_id, feature_slug, action, comment):
        org_id = ensure_integer(org_id, 'org_id')
        organization = OrganizationMetaModel(meta_connection).filter(id=org_id).one()

        if not organization:
            raise json_error_not_found()

        if not organization['ready']:
            raise OrganizationNotReadyError()

        if action not in ['enable', 'disable']:
            return json_error_invalid_value('action')
        enable = action == 'enable'

        feature_info = get_feature_by_slug(meta_connection, feature_slug)
        if not feature_info:
            return json_error_invalid_value('feature_slug')

        _, enabled_by_default = feature_info
        if enabled_by_default and not enable:
            raise InvalidValue(message='Feature enabled by default for everyone')

        already_enabled = is_feature_enabled(meta_connection, org_id, feature_slug)
        if already_enabled != enable:
            set_feature_value_for_organization(meta_connection, org_id, feature_slug, enable)
            SupportActionMetaModel(meta_connection).create(
                org_id,
                SupportActions.manage_features,
                g.user.passport_uid if g.user else 0,
                {feature_slug: action},
                'organization',
                comment=comment,
            )

        return json_response(
            {'status': 'ok'},
            status_code=201
        )


class OrganizationsView(View):
    @auth_decorators.scopes_required([scope.read_organization])
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.no_permission_required
    def get(self, meta_connection, main_connection):
        """
        Информация обо всех организациях для пользователя или сервиса.

        Если передан X-UID вернем организации пользователя.
        Если авторизация с обезличенным токеном выдаем все организации, подключившие этот сервис.

        Пример ответа:

            [
                {
                  "tech_user": {
                    "uid": 66865142
                  },
                  "id": 16,
                  "name": {
                    "ru": "Носки",
                    "en": "Noski"
                  },
                  "label": "noski",
                  "domains": {
                      "all": ["rubashki.ru", "noski.ru", "obuvka.ru", "noski.ws.yandex.ru"],
                      "owned": ["rubashki.ru"],
                      "master": "noski.ws.yandex.ru",
                      "display": "rubashki.ru",
                  },
                  "services": []
                },
                {
                  "tech_user": {
                    "uid": 111111
                  },
                  "id": 17,
                  "name": {
                    "ru": "Тест",
                    "en": "Test"
                  },
                  "label": "test",
                  "domains": {
                      "all": ["test.ru", "test1.ru"],
                      "owned": ["test.ru"],
                      "master": "test.ru",
                      "display": "test1.ru",
                  },
                  "services": ["disk"]
                },
                ...
            ]

        Поля организации:

        * **id** - идентификатор организации
        * **label** - уникальный человеко-читаемый id организации ([label поле](#label-polya))
        * **domains** - домены, принадлежащие данной организации
        * **services** - сервисы, подключенные к данной организации
        * **name** - название ([локализованное поле](#lokalizovannyie-polya))
        * **ogrn** - ОГРН
        * **inn** - ИНН
        * **trc** - КПП
        * **corr_acc** - корреспондентский счёт
        * **account** - расчетный счёт
        * **law_address** - юридический адрес
        * **real_address** - физический адрес
        * **head_name** - ФИО руководителя
        * **head_title** - должность руководителя
        * **phone_number** - контактный телефон
        * **fax** - факс
        * **email** - контактный email
        ---
        tags:
          - Организация
        parameters:
          - in: query
            name: fields
            required: False
            type: string
            description: список требуемых полей, перечисленных через запятую
        responses:
          200:
            description: Success
        """
        organizations = []

        if g.user is not None:  # пришел пользователь
            organizations, links = _get_user_organizations(
                meta_connection,
                g.user.get_cloud_or_passport_uid(),
                g.user.org_ids,
                fields=constants.V1_ORGANIZATION_FIELDS,
            )
        elif g.service:  # пришел обезличенный сервис
            organizations, links = _get_service_organizations(
                meta_connection,
                g.service.id,
                fields=constants.V1_ORGANIZATION_FIELDS,
            )

        # удалим приватные поля
        list(map(_clean_organization, organizations))
        return json_response(organizations)

    @auth_decorators.scopes_required([scope.read_organization])
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.no_permission_required
    def get_2(self, meta_connection, main_connection):
        """
        Информация обо всех организациях для пользователя или сервиса

        Если пользователь не состоит в организации, то
        будет возвращен пустой список.
        Если не передана информация о необходимых полях, будет возвращено только поле id для каждой организации.
        Необходимые поля возвращаются, если они указаны в поле fields.
        Например:
        /v2/organizations?fields=head,services,domains

        Пример ответа:

            [
                {
                  "tech_user": {
                    "uid": 66865142
                  },
                  "id": 16,
                  "name": {
                    "ru": "Носки",
                    "en": "Noski"
                  },
                  "label": "noski",
                  "domains": {
                      "all": ["rubashki.ru", "noski.ru", "obuvka.ru", "noski.ws.yandex.ru"],
                      "owned": ["rubashki.ru"],
                      "master": "noski.ws.yandex.ru",
                      "display": "rubashki.ru",
                  },
                  "services": [
                    {"slug": "disk"},
                    {"slug": "wiki"}
                  ]
                },
                {
                  "tech_user": {
                    "uid": 111111
                  },
                  "id": 17,
                  "name": {
                    "ru": "Тест",
                    "en": "Test"
                  },
                  "label": "test",
                  "domains": {
                      "all": ["test.ru", "test1.ru"],
                      "owned": ["rubashki.ru"],
                      "master": "test.ru",
                      "display": "test1.ru",
                  },
                  "services": [{
                    "slug": "disk",
                    "ready": True,
                    "trial_expires": None,
                    "trial_expired": None
                  }]
                },
                ...
            ]

        Поля организации:

        * **id** - идентификатор организации
        * **label** - уникальный человеко-читаемый id организации ([label поле](#label-polya))
        * **language** - язык по умолчанию в организации
        * **name** - название ([локализованное поле](#lokalizovannyie-polya))
        * **ogrn** - ОГРН
        * **inn** - ИНН
        * **trc** - КПП
        * **corr_acc** - корреспондентский счёт
        * **account** - расчетный счёт
        * **law_address** - юридический адрес
        * **real_address** - физический адрес
        * **head_name** - ФИО руководителя
        * **head_title** - должность руководителя
        * **phone_number** - контактный телефон
        * **fax** - факс
        * **email** - контактный email
        * **revision** - ревизия организации
        ---
        tags:
          - Организация
        parameters:
          - in: query
            name: fields
            required: False
            type: string
            description: список требуемых полей, перечисленных через запятую
        responses:
          200:
            description: Success
        """

        default_fields = ['id']
        fields = split_by_comma(request.args.get('fields')) \
                 or default_fields
        # Временный "костыль" для того, чтобы отдавать все поля о сервисах.
        # В будущих версиях API мы возможно позволим выбирать отдельные поля
        if 'services' in fields:
            fields.remove('services')
            fields.append('services.**')

        # Про домены мы тоже отдаём все простые поля
        if 'domains' in fields:
            fields.remove('domains')
            fields.append('domains.*')

        query_params = request.args.to_dict()

        if g.user is not None:  # пришел пользователь
            organizations, links = _get_user_organizations(
                meta_connection,
                g.user.get_cloud_or_passport_uid(),
                g.user.org_ids,
                fields=fields
            )
        elif g.service:  # пришел обезличенный сервис
            organizations, links = _get_service_organizations(
                meta_connection,
                g.service.id,
                fields=fields,
                ready=get_boolean_param(
                    query_params,
                    'service.ready',
                )
            )
        else:
            organizations = []

        # удалим приватные поля
        list(map(_clean_organization, organizations))
        return json_response(organizations, allowed_sensitive_params=['can_users_change_password'])

    @auth_decorators.scopes_required([scope.read_organization])
    @auth_decorators.requires(org_id=False, user=False)
    @auth_decorators.no_permission_required
    def get_6(self, meta_connection, main_connection):
        query_params = request.args.to_dict()
        query_params.setdefault('per_page', 10)

        total = None
        if g.use_cloud_proxy:
            page_token = None
            organizations = []
            links = {}
            while True:
                list_orgs_response = GrpcCloudClient().list_organizations(
                    page_size=1000,
                    page_token=page_token,
                    authorize_as='user',
                )
                organizations.extend(MessageToDict(list_orgs_response)['organizations'])
                page_token = list_orgs_response.next_page_token
                if not page_token:
                    break

            page, per_page = _get_page_params(query_params)
            if per_page:
                first_item_index = (page - 1) * per_page
                last_item_index = first_item_index + per_page - 1
                if last_item_index + 1 < len(organizations):
                    params = deepcopy(query_params)
                    params.update(page=page + 1)
                    links['next'] = url_join(
                        app.config['SITE_BASE_URI'],
                        request.path,
                        force_trailing_slash=True,
                        query_params=params,
                    )

                organizations = organizations[first_item_index:last_item_index + 1]


        else:
            default_fields = ['id']
            fields = split_by_comma(request.args.get('fields')) \
                     or default_fields

            show = request.args.get('show')
            fields = preprocess_requested_fields(fields)

            if show is None:
                if g.user is not None:
                    show = 'user'
                elif g.service:
                    show = 'service'
            else:
                # Если параметр show был передан явно, то надо
                # проверить, что нам известны пользователь или сервис
                # или что у сервиса есть скоуп, позволяющий работать с любой
                # организацией
                if show == 'user':
                    if g.user is None:
                        return json_error_invalid_value('show')
                elif show == 'service':
                    if g.service is None:
                        return json_error_invalid_value('show')
                elif show == 'all':
                    if not check_scopes(g.scopes, [scope.work_with_any_organization]):
                        return json_error_forbidden()

            if show == 'user':  # запрошены организации пользователя
                orgs_ids = g.user.org_ids
                admin_only = request.args.get('admin_only')
                if admin_only == 'true':
                    # оставим только организации где пользователь - админ
                    orgs_ids = _get_all_admin_organizations(g.user)
                    if is_outer_admin(meta_connection, g.user.passport_uid, is_cloud=g.user.is_cloud):
                        outer_admin_ids = UserMetaModel(meta_connection).filter(
                            id=g.user.passport_uid,
                            user_type='outer_admin',
                        ).scalar('org_id')
                        orgs_ids = set(orgs_ids)
                        orgs_ids.update(outer_admin_ids)

                if request.api_version < 10:
                    organizations, links = _get_user_organizations(
                        meta_connection,
                        g.user.get_cloud_or_passport_uid(),
                        orgs_ids,
                        fields=fields,
                        is_cloud=g.user.is_cloud,
                        organization_type=query_params.get('organization_type'),
                    )
                else:
                    # отдаем пагинированный список в апи >= 10
                    organizations, links = _get_user_organizations(
                        meta_connection,
                        g.user.get_cloud_or_passport_uid(),
                        orgs_ids,
                        fields=fields,
                        query_params=query_params,
                        is_cloud=g.user.is_cloud,
                    )
                    total = len(orgs_ids)

                # убрать организации, в которых пользователь был удален
                organizations = remove_orgs_with_deleted_portal_user(organizations, g.user.passport_uid, g.user.is_cloud)

            elif show == 'service':  # запрошены организации, для которых включен сервис
                organizations, links = _get_service_organizations(
                    meta_connection,
                    g.service.id,
                    fields=fields,
                    ready=get_boolean_param(
                        query_params,
                        'service.ready',
                    ),
                    query_params=query_params,
                )
            elif show == 'all':
                organizations, links = _get_all_organizations(
                    meta_connection,
                    fields=fields,
                    query_params=query_params,
                )
            else:
                organizations = []
                links = {}

        # Удалим приватные поля для 4 версии API.
        list([_clean_organization(x, request.api_version) for x in organizations])

        headers = {}

        # Заголовок со ссылками на другие страницы должен
        # добавляться только если есть какие-то ссылки. Иначе
        # он получится пустой
        if links:
            headers['Link'] = get_link_header(links)

        response = {
            'result': organizations,
            'links': links,
        }

        if request.api_version >= 10 and total is not None:
            response['total'] = total

        return json_response(
            response,
            headers=headers,
            allowed_sensitive_params=['can_users_change_password'],
        )
    available_fields_v4 = '\n'.join(['* **%s**' % i for i in ORGANIZATION_AVAILABLE_FIELDS_V4])
    get_6.__doc__ = """
Информация обо всех организациях для пользователя или сервиса

Если пользователь не состоит в организации, то
в "result" будет возвращен пустой список.

Если у сервиса есть scope ``work_with_any_organization``, то он
может указать дополнительный параметр ``show=all``. Кроме того,
теперь этот параметр сервис может использовать, чтобы запросить
не организации, в которых он подключен, а организации в которых
состоит пользователь. Для этого надо указать аргумент
``show=user``.

По умолчанию ручка сохранила то же поведение, что и было – если
пришёл сервис, то отдаются организации, в которых он
подключен. Если пришёл пользователь (а сервис id не известен),
то отдаются организации пользователя.

Если не передана информация о необходимых полях, будет
возвращено только поле id для каждой организации.

Необходимые поля возвращаются, если они указаны в поле fields.
Например:

```
/v6/organizations/?fields=head,services,domains
```

По-умолчанию, количество элементов, отдаваемых на странице – 10.

Если в ключе "links" есть словарь с ключом "next", то необходимо получить
следующую страницу по ссылке из этого ключа.

Пример ответа:

    {{
        "result": [
            {{
              "id": 16,
              "name": {{
                "ru": "Носки",
                "en": "Noski"
              }},
              "label": "noski",
              "domains": {{
                  "all": ["rubashki.ru", "noski.ru", "obuvka.ru", "noski.ws.yandex.ru"],
                  "owned": ["rubashki.ru"],
                  "master": "noski.ws.yandex.ru",
                  "display": "rubashki.ru",
              }},
              "services": [
                {{
                    "slug": "disk",
                    "ready": True,
                    "trial_expires": None,
                    "trial_expired": None
                }},
                {{
                    "slug": "wiki",
                    "ready": False,
                    "trial_expires": None,
                    "trial_expired": None
                }}
              ]
            }},
            {{
              "id": 17,
              "name": {{
                "ru": "Тест",
                "en": "Test"
              }},
              "label": "test",
              "domains": {{
                  "all": ["test.ru", "test1.ru"],
                  "owned": ["test.ru"],
                  "master": "test.ru",
                  "display": "test1.ru",
              }},
              "services": [
                {{
                    "slug": "disk",
                    "ready": True,
                    "trial_expires": None,
                    "trial_expired": None
                }}
              ]
            }},
            ...
        ],
        "links": {{}}
    }}

Поля организации:
{fields}
---
tags:
  - Организация
parameters:
  - in: query
    name: fields
    required: False
    type: string
    description: список требуемых полей, перечисленных через запятую
responses:
  200:
    description: Success
""".format(
        fields=available_fields_v4
    )

    # 4 версия, в которой ещё есть поле admin_uid.
    # временно поддерживаем эту версию.
    get_4 = get_6
    get_8 = get_6
    doc_v8 = """
Информация обо всех организациях для пользователя или сервиса

Если пользователь не состоит в организации, то
в "result" будет возвращен пустой список.

Если у сервиса есть scope ``work_with_any_organization``, то он
может указать дополнительный параметр ``show=all``. Кроме того,
теперь этот параметр сервис может использовать, чтобы запросить
не организации, в которых он подключен, а организации в которых
состоит пользователь. Для этого надо указать аргумент
``show=user``.
{user_v10}

По умолчанию ручка сохранила то же поведение, что и было – если
пришёл сервис, то отдаются организации, в которых он
подключен. Если пришёл пользователь (а сервис id не известен),
то отдаются организации пользователя.

Если не передана информация о необходимых полях, будет
возвращено только поле id для каждой организации.

Необходимые поля возвращаются, если они указаны в поле fields.
Если запрошено поле services, то возвращаются все сервисы, которые когда-либо
были включены. Статус сервиса (включен/выключен) указан в параметре enabled.
Например:

```
/v{version}/organizations/?fields=head,services,domains
```

По-умолчанию, количество элементов, отдаваемых на странице – 10.

Если в ключе "links" есть словарь с ключом "next", то необходимо получить
следующую страницу по ссылке из этого ключа.

Пример ответа:

    {{
        "result": [
            {{
              "id": 16,
              "name": {{
                "ru": "Носки",
                "en": "Noski"
              }},
              "label": "noski",
              "domains": {{
                  "all": ["rubashki.ru", "noski.ru", "obuvka.ru", "noski.ws.yandex.ru"],
                  "master": "noski.ws.yandex.ru",
                  "display": "rubashki.ru",
              }},
              "services": [
                {{
                    "slug": "disk",
                    "ready": True,
                    "enabled": True,
                    "trial": {{
                        "expiration_date": None,
                        "status": "inapplicable",
                        "days_till_expiration": None
                    }}
                }},
                {{
                    "slug": "wiki",
                    "ready": False,
                    "enabled": False,
                    "trial": {{
                        "expiration_date": "2018-10-10",
                        "status": "in_progress",
                        "days_till_expiration": 6
                    }}
                }}
              ]
            }},
            {{
              "id": 17,
              "name": {{
                "ru": "Тест",
                "en": "Test"
              }},
              "label": "test",
              "domains": {{
                  "all": ["test.ru", "test1.ru"],
                  "owned": ["test.ru"],
                  "master": "test.ru",
                  "display": "test1.ru",
              }},
              "services": [
                {{
                    "slug": "disk",
                    "ready": True,
                    "enabled": True,
                    "trial": {{
                        "expiration_date": None,
                        "status": "inapplicable",
                        "days_till_expiration": None
                    }}
                }}
              ]
            }},
            ...
        ],
        "links": {{}}
    }}

Поля организации:
{fields}
---
tags:
  - Организация
parameters:
  - in: query
    name: fields
    required: False
    type: string
    description: список требуемых полей, перечисленных через запятую
  - in: query
    name: admin_only
    required: False
    type: boolean
    description: если true - будет возвращен список организаций, где пользователь админ
responses:
  200:
    description: Success
"""

    get_8.__doc__ = doc_v8.format(
        user_v10='',
        version=8,
        fields=available_fields_v4,
    )
    get_10 = get_6
    get_10.__doc__ = doc_v8.format(
        user_v10='Для пользователя возвращается пагинированный список организаций',
        version=10,
        fields=available_fields_v4,
    )


class OrganizationView(View):
    @auth_decorators.no_permission_required
    @auth_decorators.scopes_required([scope.read_organization])
    @auth_decorators.requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Эта ручка УСТАРЕЛА и скоро будет удалена.
        Пожалуйста, используйте вместо неё ручку /organizations/.

        Информация об организации текущего пользователя

        Если пользователь не состоит в организации, то
        будет возвращен пустой словарь.

        Пример ответа:

            {
              "id": 1,
              "label": "yandex",
              "admin_uid": 4000811136,
              "name": {
                "en": "Yandex",
                "ru": "Яндекс"
              },
              "domains": ['yandex.ws.yandex.ru', 'yandex.ru'],
              "inn": "4324234324",
              "head": {
                  ...
              },
              ...
            }

        Поля организации:

        * **id** - идентификатор организации
        * **label** - уникальный человеко-читаемый id организации ([label поле](#label-polya))
        * **domains** - домены, принадлежащие данной организации
        * **name** - название ([локализованное поле](#lokalizovannyie-polya))
        * **language** - язык по умолчанию в организации
        * **ogrn** - ОГРН
        * **inn** - ИНН
        * **trc** - КПП
        * **corr_acc** - корреспондентский счёт
        * **account** - расчетный счёт
        * **law_address** - юридический адрес
        * **real_address** - физический адрес
        * **head_name** - ФИО руководителя
        * **head_title** - должность руководителя
        * **phone_number** - контактный телефон
        * **fax** - факс
        * **email** - контактный email
        ---
        tags:
          - Организация
        responses:
          200:
            description: Success
        """

        organization = OrganizationModel(main_connection).get(
            id=g.org_id,
            fields=constants.V1_ORGANIZATION_FIELDS,
        )
        _process_fields(
            meta_connection,
            main_connection,
            [organization],
            version=request.api_version,
        )
        _clean_organization(organization)
        return json_response(organization)

    @auth_decorators.internal
    @auth_decorators.no_permission_required
    @schema_decorators.uses_schema(constants.ORGANIZATION_CREATE_SCHEMA)
    @auth_decorators.scopes_required([scope.write_organization])
    @auth_decorators.requires(org_id=False, user=True)
    def post(self, meta_connection, none_main_connection, data):
        """
        Завершение регистрации новой организации.
        Начало регистрации на специальной паспортной странице.

        Это внутренняя ручка, доступная только для внутренних сервисов.

        Пример ответа:

            {{
                "org_id": 123 // id созданной организации
            }}

        ---
        tags:
            - Организация
        responses:
            201:
                description: Создали организацию.
            200:
                description: Вернули ранее созданную организацию.
        """
        uid = g.user.passport_uid
        user_ip = g.user.ip

        if not is_multiorg_feature_enabled_for_user(meta_connection, uid):
            # делаем ручку индемпотентной
            # если пользователь уже состоит в организации
            # то просто вернём эту организацию
            meta_user = UserMetaModel(meta_connection).get(user_id=uid)
            if meta_user:
                return json_response(
                    data={'org_id': meta_user['org_id']},
                    status_code=200
                )

        language = data.get(
            'language',
            'ru',  # язык по-умолчанию - русский
        )
        source = data.get('source', 'unknown')
        tld = data.get('tld', 'unknown')
        country = data.get('country', None)
        outer_id = data.get('outer_id')
        maillist_type = data.get('maillist_type', 'inbox')

        (
            admin_login,
            admin_first_name,
            admin_last_name,
            admin_gender,
            admin_birth_date,
            admin_email,
            cloud_uid,
        ) = get_user_info_from_blackbox(uid, userip=user_ip)

        if not admin_login:
            raise UserDoesNotExist()

        domains_data = app.blackbox_instance.hosted_domains(domain_admin=uid)

        with log.fields(uid=uid, blackbox_response=domains_data):
            log.debug('Blackbox response received')

        domains = domains_data['hosted_domains']

        if not domains:
            log.error('Admin %s has no hosted domains' % uid)
            return json_error_bad_request()

        domain = domains[0]
        domain_name = domain['domain']

        if not domain_name.endswith(app.config['DOMAIN_PART']):
            raise InvalidDomain()

        # домен не должен быть подтвержден в другой организации
        exists_domain = DomainModel(None).filter(name=domain_name, owned=True).one()
        if exists_domain and exists_domain['owned']:
            raise DomainOccupied(domain=domain_name)

        label = domain_name.split('.', 1)[0]
        domain_part = '.' + domain_name.split('.', 1)[1]
        options = json.loads(domain['options'])
        org_name = options['organization_name']

        assert_not_spammer(org_name)

        admin_nickname = admin_login.split('@', 1)[0]

        # добавляем организацию в свою базу
        shard = OrganizationMetaModel.get_shard_for_new_organization()
        with get_main_connection(shard, for_write=True) as main_connection:
            created_organization_info = create_organization(
                meta_connection,
                main_connection,
                name=org_name,
                label=label,
                domain_part=domain_part,
                admin_uid=uid,
                language=language,
                admin_nickname=admin_nickname,
                admin_first_name=admin_first_name,
                admin_last_name=admin_last_name,
                admin_gender=admin_gender,
                admin_birthday=admin_birth_date,
                source=source,
                tld=tld,
                country=country,
                root_dep_label='all',
                maillist_type=maillist_type,
            )
            root_department = created_organization_info['root_department']
            org_id = created_organization_info['organization']['id']

            apply_preset(
                meta_connection,
                main_connection,
                org_id,
                data.get('preset', 'default'),

            )

            # досаздадим рассылку в паспорте, если включен сервис рассылок
            # делаем это тут т.к. мы применяем присет после создания организации
            # а до этого нет информации о подключенных сервисах
            if OrganizationServiceModel(main_connection).is_service_enabled(org_id, MAILLIST_SERVICE_SLUG):
                maillist_uid = create_maillist(main_connection, org_id, 'all', ignore_login_not_available=True)
                DepartmentModel(main_connection).update_one(root_department['id'], org_id, {'uid': maillist_uid})

            user_from_db = UserModel(main_connection).get(
                user_id=uid,
                fields=[
                    '*',
                    'department.*',
                    'groups.*',
                ],
            )

            # если передан outer_id и он внешний, то создаем внешнего админа
            if outer_id and is_outer_uid(outer_id):
                UserMetaModel(meta_connection).create(
                    id=outer_id,
                    org_id=org_id
                )

            # событие создания организации
            revision = action_organization_add(
                main_connection,
                org_id=org_id,
                author_id=uid,
                object_value=created_organization_info['organization'],
            )

            # Генерируем событие про создание корневого департамента только после генерации
            # события о добавлении новой организации

            event_department_added(
                main_connection,
                org_id=org_id,
                revision=revision,
                object_value=root_department,
                object_type=TYPE_DEPARTMENT,
            )

            # так как диск заводится по событию, то тут нужно сэмулировать
            # действие, как будто пользователя завели через API
            action_user_add(
                main_connection,
                org_id=org_id,
                author_id=uid,
                object_value=user_from_db,
            )

            SetOrganizationNameInPassportTask(main_connection).delay(
                org_id=org_id,
            )

            # приветственное письмо всем пользователям организации (с задержкой)
            _delay_welcome_email(org_id, admin_nickname, uid)

            return json_response(data={'org_id': org_id}, status_code=201)

    @schema_decorators.uses_schema(constants.ORGANIZATION_UPDATE_SCHEMA)
    @auth_decorators.scopes_required([scope.write_organization])
    @auth_decorators.permission_required([organization_permissions.edit], TYPE_ORGANIZATION)
    def patch(self, meta_connection, main_connection, data):
        """
        Изменить информацию об организации

        Эта ручка на входе ожидает получить словарь с некоторыми из тех полей, которые
        получает POST. Однако вот эти поля менять запрещено:

        - domain
        - uid
        - nickname
        - first_name
        - last_name
        - gender


        Раньше смена отображаемого домена происходила путём передачи параметра `display_domain`,
        но начиная с 2016-12-01 эта возможность не поддерживается,
        а надо смену основного домена надо проводить через параметр `master_domain`.

        Зато дополнительно можно передать в поле `master_domain` имя головного домена
        и оно сменится повсюду – в паспорте, почте, адресной книге и остальных сервисах.

        Так же, дополнительно можно передать поле `default_uid` для того, чтобы установить
        ящик "по-умолчанию". Однако это работает только для организаций с доменом, и таким
        образом можно установить только доменную учётку. В случае, если учётка не доменная,
        будет выдан код ошибки `user_not_in_domain`, а если учётная запись вовсе не найдена
        в организации, то `user_not_found`.

        ---
        tags:
          - Организация
        parameters:
          - in: path
            name: organization_id
            required: true
            type: string
          - in: body
            name: body
        responses:
          200:
            description: Организация обновлена
          422:
            description: Параметры не соответствуют схеме
          403:
            description: {permissions}

        """
        # патчим всегда текущую организацию, поэтому нет нужды
        # передавать её id в урле
        org_id = g.org_id

        if g.use_cloud_proxy:
            cloud_org_id = OrganizationMetaModel(meta_connection).get(org_id)['cloud_org_id']
            org_response = GrpcCloudClient().update_organization(
                org_id=cloud_org_id,
                data=data,
            )
            return json_response(MessageToDict(org_response)['response'])

        self.check_settings_change_allowed(
            meta_connection,
            main_connection,
            org_id,
            data
        )

        # если надо, то меняем отображаемый домен
        # TODO: https://st.yandex-team.ru/DIR-2042
        if 'display_domain' in data:
            data = unfreeze_or_copy(data)
            domain = data.pop('display_domain').lower()
            old_display_domain = DomainModel(main_connection).find(
                filter_data=dict(
                    display=True,
                    org_id=org_id,
                )
            )
            try:
                DomainModel(main_connection).update_one(
                    name=domain,
                    org_id=org_id,
                    data={'display': True}
                )
            except DomainNotFound as e:
                raise InvalidValue(message=e.message, error_code='domain_not_found', **e.params)
            new_display_domain = DomainModel(main_connection).find(
                filter_data=dict(
                    display=True,
                    org_id=org_id,
                )
            )

            # https://st.yandex-team.ru/DIR-1992
            action_domain_master_modify(
                main_connection,
                org_id=org_id,
                author_id=g.user.passport_uid,
                object_value=new_display_domain,
                old_object=old_display_domain,
            )

        # если надо, то меняем мастер домен
        if 'master_domain' in data:
            # проверям наличие прав на смену master домена
            if not g.user.has_permissions(
                    meta_connection,
                    main_connection,
                    [global_permissions.change_master_domain],
                    org_id=org_id):
                return json_error_forbidden()

            data = unfreeze_or_copy(data)
            domain = data.pop('master_domain').lower()
            if is_feature_enabled(meta_connection, org_id, USE_DOMENATOR):
                update_domains_in_db_or_domenator(
                    meta_connection=meta_connection,
                    update_filter=DomainUpdateFilter(org_id=org_id, name=domain),
                    update_data=DomainUpdateData(master=True),
                    main_connection=main_connection,
                )
            else:
                DomainModel(main_connection).change_master_domain(org_id, domain)

        # патчим поля
        old_organization = None

        if 'name' in data:
            assert_not_spammer(data['name'])

        if data:
            old_organization = OrganizationModel(main_connection).get(
                org_id,
                # Пока отдаём в PATCH все данные, включая вложенные,
                # может быть в будущих версиях API это изменится.
                fields=[
                    '**',
                    # Slug надо указать отдельно, так как он подтягивается
                    # через prefetch-related и требуется для того, чтобы
                    # сериализовать ответ
                    'services.slug'
                ],
            )
            meta_patch_data = {}
            main_patch_data = unfreeze_or_copy(data)
            if 'cloud_org_id' in main_patch_data:
                meta_patch_data = {'cloud_org_id': main_patch_data.pop('cloud_org_id')}

            if 'default_uid' in data:
                default_uid = main_patch_data.pop('default_uid')
                self.change_default_uid(main_connection, org_id, default_uid)

            if 'can_users_change_password' in data:
                default_uid = main_patch_data.pop('can_users_change_password')
                self.change_can_users_change_password(main_connection, org_id, default_uid)

            if 'email' in data:
                if not is_email_valid(data['email']):
                    raise InvalidEmail()

            if main_patch_data:
                OrganizationModel(main_connection).update_one(
                    org_id,
                    main_patch_data,
                )

            if meta_patch_data:
                OrganizationMetaModel(meta_connection).update_one(
                    org_id,
                    meta_patch_data,
                )

        # готовим выходные данные
        organization = OrganizationModel(main_connection).get(
            org_id,
            # Пока отдаём в PATCH все данные, включая вложенные,
            # может быть в будущих версиях API это изменится.
            fields=[
                '**',
                # Slug надо указать отдельно, так как он подтягивается
                # через prefetch-related и требуется для того, чтобы
                # сериализовать ответ
                'services.slug'
            ],
        )
        if data:
            action_organization_modify(
                main_connection,
                org_id=org_id,
                author_id=g.user.passport_uid,
                object_value=organization,
                old_object=old_organization,
            )
            if 'organization_type' in data:
                action_organization_type_change(
                    main_connection,
                    org_id=org_id,
                    author_id=g.user.passport_uid,
                    object_value=organization,
                    old_object=old_organization,
                    content={'organization_type': data.get('organization_type')},
                )
            if 'cloud_org_id' in data:
                organization['cloud_org_id'] = data['cloud_org_id']

        _process_fields(
            meta_connection,
            main_connection,
            [organization],
            version=request.api_version,
        )

        OrganizationRevisionCounterModel(main_connection).increment_revisions_for_user(meta_connection, g.user.passport_uid)

        return json_response(organization, allowed_sensitive_params=['can_users_change_password'])

    @staticmethod
    def return_master_domain_or_error(main_connection, org_id):
        try:
            return DomainModel(main_connection).get_master(org_id)
        except DomainNotFound:
            raise ImmediateReturn(
                json_error(
                    422,
                    'no_domain',
                    'No domain in organization',
                    field='default_uid',
                )
            )

    def change_default_uid(self, main_connection, org_id, default_uid):
        domain = self.return_master_domain_or_error(main_connection, org_id)

        if default_uid:
            default_uid = int(default_uid)

            # Это должен быть обязательно действующий пользователь
            if is_outer_uid(default_uid):
                # Мы пытаемся установить default_uid в uid сотрудника из этой же организации
                # но сотрудник при этом – портальный.
                # Это не работает из-за ограничения в Паспорте. Сейчас он принимает только
                # логин сотрудника, в том же домене. Попытка передать портальный логин
                # проглатывается паспортом без ошибки, но смены default_uid не происходит.
                raise ImmediateReturn(
                    json_error(
                        422,
                        'user_not_in_domain',
                        'User not in domain',
                        field='default_uid',
                    )
                )

            user = UserModel(main_connection).filter(
                org_id=org_id,
                id=default_uid,
                is_dismissed=False,
            ).one()

            if user is None:
                raise ImmediateReturn(
                    json_error(
                        422,
                        'user_not_found',
                        'User not found',
                        field='default_uid',
                    )
                )
            else:
                default_login = user['nickname']
        else:
            # При сбросе паспорт требует, чтобы вместо логина
            # была передана пустая строка.
            default_login = ''

        dom_id = get_domain_id_from_blackbox(domain['name'])
        app.passport.domain_edit(dom_id, {'default': default_login})

    def change_can_users_change_password(self, main_connection, org_id, can_users_change_password):
        domain = self.return_master_domain_or_error(main_connection, org_id)
        dom_id = get_domain_id_from_blackbox(domain['name'])
        app.passport.domain_edit(dom_id, {'can_users_change_password': can_users_change_password})

    def check_settings_change_allowed(self, meta_connection, main_connection, org_id, data):
        """
        Проверяем можно ли изменять переданные настройки.
        Если от настройки зависти какой-либо сервис, подключенный для текущей организации, то изменение настройки заперещено.
        Если есть запрещенная для изменения настройка, то кинем исключение ImmediateReturn.
        :param meta_connection: соединие к meta базе
        :param main_connection: соединие к main базе
        :param org_id: ид организации
        :type org_id: int
        :param data: аналогино data для ручки patch
        :type data: dict
        """
        # пока можно менять только 1 настройку
        if 'shared_contacts' not in data:
            return
        org_services = OrganizationServiceModel(main_connection).find(
            filter_data={
                'org_id': org_id,
                'enabled': True,
            },
            fields=['service_id']
        )
        if not org_services:
            return

        service_ids = only_attrs(org_services, 'service_id')
        services = ServiceModel(meta_connection).find(
            filter_data={'id': service_ids},
            fields=['slug']
        )
        service_slugs = only_attrs(services, 'slug')
        shared_contact_settings = dependencies.Setting('shared-contacts', True)

        for slug in service_slugs:
            deps = dependencies.dependencies.get(
                dependencies.Service(slug), []
            )
            if shared_contact_settings in deps:
                raise InvalidValue(
                    message='Change settings "{field}" conflict with "{service}" service',
                    field='shared_contacts',
                    service=slug,
                )


class OrganizationWithDomainView(View):
    @auth_decorators.internal
    @auth_decorators.no_permission_required
    @schema_decorators.uses_schema(constants.ORGANIZATION_WITH_DOMAIN_CREATE_SCHEMA)
    @auth_decorators.scopes_required([scope.write_organization])
    @auth_decorators.requires(org_id=False, user=True)
    def post(self, meta_connection, none_main_connection, data):
        """
        Создание новой организации с привязанным доменом.

        Поле `domain_name` является обязательным и не должно быть пустым
        Если не указать поле `name`, то именем организации будет имя домена
        Если валидация имени домена не прошла, то возвращается 422 код ошибки
        и её описание в словаре `{"errors": [...]}`
        При создании организации применяется preset `no-owned-domain`,
        который включает в себя сервисы, необходимые для использования дашборда.
        При создании организации можно указать `preset`,
        и тогда после подтверждения домена этот preset применится к организации.
        Если `preset` не указать - после подтверждения домена применится preset default

        Пример ответа:

            {{
                "org_id": 123 // id созданной организации
            }}

        ---
        tags:
            - Организация

        parameters:
            - in: body
              name: body

        responses:
            201:
                description: Создали организацию.
            409:
                description: Подтвержденный домен с таки именем уже существует в Директории или в Паспорте
            422:
                description: Какая-то ошибка валидации
        """
        uid = g.user.passport_uid
        user_ip = g.user.ip
        org_id = create_organization_with_domain(meta_connection, data, uid, user_ip)

        return json_response(data={'org_id': org_id}, status_code=201)


class OrganizationFromRegistrarView(View):
    @auth_decorators.internal
    @auth_decorators.no_permission_required
    @schema_decorators.uses_schema(constants.ORGANIZATION_WITH_DOMAIN_CREATE_SCHEMA)
    @auth_decorators.scopes_required([scope.write_organization])
    @auth_decorators.requires(org_id=False, user=True)
    def post(self, meta_connection, none_main_connection, data, registrar_id):
        """
        Создание новой организации с привязанным доменом от регистратора.

        Поле `domain_name` является обязательным и не должно быть пустым
        Если не указать поле `name`, то именем организации будет имя домена
        Если валидация имени домена не прошла, то возвращается 422 код ошибки
        и её описание в словаре `{"errors": [...]}`
        При создании организации применяется preset `no-owned-domain`,
        который включает в себя сервисы, необходимые для использования дашборда.
        При создании организации можно указать `preset`,
        и тогда после подтверждения домена этот preset применится к организации.
        Если `preset` не указать - после подтверждения домена применится preset default

        Пример ответа:

            {{
                "org_id": 123 // id созданной организации
            }}

        ---
        tags:
            - Организация

        parameters:
            - in: body
              name: body

        responses:
            201:
                description: Создали организацию.
            409:
                description: Подтвержденный домен с таки именем уже существует в Директории или в Паспорте
            422:
                description: Какая-то ошибка валидации
        """
        uid = g.user.passport_uid
        user_ip = g.user.ip

        try:
            registrar = app.domenator.get_registrar(registrar_id)
        except NoResultFound:
            raise EntityNotFoundError(registrar_id)

        domain_name = data.get('domain_name')

        if registrar['pdd_version'] == 'old':
            # получаем service_id
            service_id = data.get('service_id')
            if not service_id:
                raise ParameterNotCorrect(message='Parameter "{parameter}" is not provided', parameter='service_id')

            registrar_client = RegistrarClient(registrar)

            try:
                result = registrar_client.call_check_payed_callback(domain_name, service_id)
            except RegistrarInteractionException:
                raise RegistrarCheckFailed()

            if not result:
                raise RegistrarCheckFailed()

        org_id = create_organization_with_domain(meta_connection, data, uid, user_ip, registrar_id)

        return json_response(data={'org_id': org_id}, status_code=201)


class OrganizationWithoutDomainView(View):
    @auth_decorators.internal
    @auth_decorators.no_permission_required
    @schema_decorators.uses_schema(constants.ORGANIZATION_WITHOUT_DOMAIN_CREATE_SCHEMA)
    @auth_decorators.scopes_required([scope.write_organization])
    @auth_decorators.requires(org_id=False, user=True)
    def post(self, meta_connection, none_main_connection, data):
        """
        Создание новой организации без домена.

        При создании организации применяется preset `no-domain`,
        который включает в себя сервисы, необходимые для использования дашборда.
        Пример ответа:

            {{
                "org_id": 123 // id созданной организации
            }}

        ---
        tags:
            - Организация

        parameters:
            - in: body
              name: body

        responses:
            201:
                description: Создали организацию.
            422:
                description: Какая-то ошибка валидации
        """
        uid = g.user.passport_uid
        user_ip = g.user.ip

        if not is_outer_uid(uid):
            log.warning('Organization can not be created by domain user.')
            return json_error_forbidden(code='forbidden_for_domain_user')

        with get_main_connection_for_new_organization(for_write=True) as main_connection:
            org_id = create_organization_without_domain(
                meta_connection,
                main_connection,
                data,
                uid,
                user_ip,
            )

        return json_response(data={'org_id': org_id}, status_code=201)




class OrganizationsChangeLogoView(View):
    @auth_decorators.scopes_required([scope.write_organization])
    @auth_decorators.permission_required([global_permissions.change_logo], TYPE_ORGANIZATION)
    @auth_decorators.requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, org_id):
        """Сменить лого.
        Поменять лого можно 2мя способами:
        1. По урлу:
            * **logo_url** - url с лого
            Ожидаем Content-Type: 'application/json'

        2. Через файл:
            * **logo_file** - файл с лого
            Ожидаем Content-Type, который будет начинаться с multipart/form-data.
            Например:
            multipart/form-data; boundary=<Asrf456BGe4>

        ---
        tags:
          - Организация

        parameters:
            - in: path
              name: org_id
              required: true
              type: integer

        responses:
          200:
            description: Лого изменено.
          404:
            description: Организация не найдена.
          403:
            description: {permissions}
          422:
            description: Какая-то ошибка валидации
        """
        org_id = ensure_integer(org_id, 'org_id')

        if request.headers['Content-Type'].startswith('multipart/form-data'):
            # http://flask.pocoo.org/docs/0.11/patterns/fileuploads/
            files = request.files or {}
            img = files.get('logo_file')

            if not img:
                return json_error_required_field('logo_file')

            return self._change_logo_for_organization(
                meta_connection=meta_connection,
                main_connection=main_connection,
                author_id=g.user.passport_uid,
                file=img,
            )

        # теперь проверяем, не загружают ли картинку по урлу
        data = request.get_json()
        if data and data.get('logo_url'):
            return self._change_logo_for_organization(
                meta_connection=meta_connection,
                main_connection=main_connection,
                author_id=g.user.passport_uid,
                url=data.get('logo_url'),
            )

        return json_error_required_field('logo_url')

    def _change_logo_for_organization(self,
                                      main_connection,
                                      meta_connection,
                                      author_id,
                                      file=None,
                                      url=None):
        org_id = g.org_id
        query = OrganizationModel(main_connection).filter(id=org_id)

        image = ImageModel(main_connection).create(org_id=org_id, file=file, url=url)

        old_logo_id = query.scalar('logo_id')
        new_logo_id = image['id']

        query.update(logo_id=new_logo_id)

        action_organization_logo_change(
            main_connection,
            org_id=org_id,
            author_id=author_id,
            object_value=new_logo_id,
            old_object=old_logo_id,
        )

        return json_response(
            data={},
            status_code=200
        )

    @auth_decorators.scopes_required([scope.write_organization])
    @auth_decorators.permission_required([global_permissions.change_logo], TYPE_ORGANIZATION)
    def delete(self, meta_connection, main_connection, org_id):
        """Удалить лого.
        ---
        tags:
          - Организация

        parameters:
            - in: path
              name: org_id
              required: true
              type: integer

        responses:
          200:
            description: Лого удалено.
          404:
            description: Организация не найдена.
          403:
            description: {permissions}
        """
        org_id = g.org_id
        query = OrganizationModel(main_connection).filter(id=org_id)

        old_logo_id = query.scalar('logo_id')

        query.update(logo_id=None)

        # Пока закомментировал, потому что непонятно что складывать в old_object
        action_organization_logo_change(
            main_connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value=None,
            old_object=old_logo_id,
        )

        return json_response(
            data={},
            status_code=200
        )


class OrganizationsDeleteView(View):
    @auth_decorators.internal
    @schema_decorators.uses_schema_for_get(constants.DELETE_ORG_STATUS_SCHEMA)
    @auth_decorators.scopes_required([scope.read_organization])
    @auth_decorators.no_permission_required
    @auth_decorators.requires(org_id=False, user=False)
    @no_cache
    def get(self, meta_connection, _, org_id):
        """
        Получить таск на удаление организации.
        ---
        tags:
          - Организация

        parameters:
            - in: path
              name: org_id
              required: true
              type: integer

        responses:
          200:
            description: Информация о таске id и статус.
          404:
            description: Организация не найдена.
          403:
            description: Request is not available for this service
        """
        task_id = request.args.to_dict().get('task_id')
        try:
            uuid.UUID(task_id)
        except ValueError:
            return json_error_not_found()
        shard = get_shard(meta_connection, org_id)
        current_task = None

        if shard:
            with get_main_connection(shard=shard) as main_connection:
                current_task = self._get_delete_task(main_connection, org_id, task_id)
        else:
            for shard in get_shard_numbers():
                with get_main_connection(shard=shard) as main_connection:
                    current_task = self._get_delete_task(main_connection, org_id, task_id)
                    if current_task:
                        break

        if current_task:
            return json_response(
                {
                    'task_id': task_id,
                    'status': current_task['state']
                },
            )
        else:
            return json_error_not_found()

    def _get_delete_task(self, main_connection, org_id, task_id):
        return TaskModel(main_connection).filter(
            org_id=org_id,
            id=task_id,
            task_name=DeleteOrganizationTask.get_task_name()
        ).fields('id', 'state').order_by('-created_at').one()

    @auth_decorators.internal
    @auth_decorators.requires(org_id=True, user=True)
    @auth_decorators.permission_required([organization_permissions.delete])
    @auth_decorators.scopes_required([scope.delete_organization])
    def delete(self, meta_connection, main_connection, org_id):
        """
        Запустить таск по удалению организации.
        ---
        tags:
          - Организация

        parameters:
            - in: path
              name: org_id
              required: true
              type: integer

        responses:
          202:
            description: Информация о таске id и статус.
          404:
            description: Организация не найдена.
          403:
            description: {permissions}
        """
        balance, current_month_price = get_balance_and_current_month_price(org_id)
        if current_month_price > balance:
            raise OrganizationHasDebt()

        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard, for_write=True) as main_connection:
            try:
                task = DeleteOrganizationTask(main_connection).delay(
                    org_id=org_id,
                    user_id=g.user.passport_uid,
                )
                task_id = task.task_id
                task_state = task.state
            except DuplicatedTask as exc:
                existing_task_id = exc.existing_task_id

                with log.fields(existing_task_id=existing_task_id):
                    log.warning('Background task already exists, returning it\'s id')

                current_task = self._get_delete_task(main_connection, org_id, existing_task_id)
                task_id = current_task['id']
                task_state = current_task['state']

            return json_response(
                {
                    'task_id': task_id,
                    'status': task_state
                },
                status_code=202
            )


class OrganizationsChangeOwnerView(View):
    @auth_decorators.internal
    @schema_decorators.uses_schema_for_get(constants.CHANGE_OWNER_STATUS_SCHEMA)
    @auth_decorators.scopes_required([scope.read_organization])
    @auth_decorators.no_permission_required
    @auth_decorators.requires(org_id=False, user=False)
    def get(self, meta_connection, _, org_id):
        """
        Получить таск на передачу владения организацией.
        ---
        tags:
          - Организация

        parameters:
            - in: path
              name: org_id
              required: true
              type: integer

        responses:
          200:
            description: Информация о таске id и статус.
          404:
            description: Организация не найдена.
          403:
            description: Request is not available for this service
        """
        task_id = request.args.to_dict().get('task_id')
        try:
            uuid.UUID(task_id)
        except ValueError:
            return json_error_not_found()
        shard = get_shard(meta_connection, org_id)
        with get_main_connection(shard=shard) as main_connection:
            current_task = self._get_change_owner_task(main_connection, org_id, task_id)
        if current_task:
            return json_response(
                {
                    'task_id': task_id,
                    'status': current_task['state']
                },
            )
        else:
            return json_error_not_found()

    def _get_change_owner_task(self, main_connection, org_id, task_id):
        return TaskModel(main_connection).filter(
            org_id=org_id,
            id=task_id,
            task_name=ChangeOrganizationOwnerTask.get_task_name()
        ).fields('id', 'state').order_by('-created_at').one()

    @auth_decorators.internal
    @schema_decorators.uses_schema(constants.ORGANIZATION_CHANGE_OWNER_SCHEMA)
    @auth_decorators.scopes_required([scope.change_owner])
    @auth_decorators.permission_required([organization_permissions.change_owner], TYPE_ORGANIZATION)
    @auth_decorators.requires(org_id=False, user=True)
    def post(self, meta_connection, _, data, org_id):
        """
        Запустить таск по передаче владения организацией.
        ---
        tags:
          - Организация

        parameters:
            - in: path
              name: org_id
              required: true
              type: integer
            - in: body
              name: body

        responses:
          202:
            description: Информация о таске id и статус.
          404:
            description: Организация не найдена, либо не найден новый владелец.
          403:
            description: {permissions}
        """
        shard = get_shard(meta_connection, org_id)
        login = data['login'].strip()

        with log.fields(login=login, org_id=org_id), \
             get_main_connection(shard=shard, for_write=True) as main_connection:

            if get_organization_admin_uid(main_connection, org_id) != g.user.passport_uid:
                # доп проверка на случай, если стали выдавать право change_owner кому-то еще
                log.warning('Current user is not owner of organization')
                return json_error_forbidden()

            new_owner_uid = get_user_id_from_passport_by_login(login)

            if not new_owner_uid:
                log.warning('Unable to find user by login in passport')
                return json_error_not_found()

            # если id внутренний, то он должен принадлежать той же организации
            if not (is_outer_uid(new_owner_uid)
                    or UserModel(main_connection).filter(org_id=org_id, id=new_owner_uid).one()):
                log.warning('User is not portal and does not belong to this organization')
                return json_error_not_found()

            try:
                task = ChangeOrganizationOwnerTask(main_connection).delay(
                    org_id=org_id,
                    old_owner_uid=g.user.passport_uid,       # ручку может дергать только текущий владелец, подставим его uid
                    new_owner_uid=new_owner_uid,
                )
            except DuplicatedTask as exc:
                existing_task_id = exc.existing_task_id

                with log.fields(existing_task_id=existing_task_id):
                    log.warning('Background task already exists, returning it\'s id')

                current_task = self._get_change_owner_task(main_connection, org_id, existing_task_id)
                return json_response(
                    {
                        'task_id': current_task['id'],
                        'status': current_task['state']
                    },
                    status_code=202
                )

            return json_response(
                {
                    'task_id': task.task_id,
                    'status': task.state
                },
                status_code=202
            )


class OrganizationDebtView(View):
    methods = ['get']

    @no_cache
    @auth_decorators.internal
    @auth_decorators.requires(org_id=True, user=True)
    @auth_decorators.permission_required([organization_permissions.delete])
    @auth_decorators.scopes_required([scope.read_organization])
    def get(self, meta_connection, main_connection, org_id):
        balance, current_month_price = get_balance_and_current_month_price(org_id)

        debt = 0
        if current_month_price > balance:
            debt = current_month_price - balance

        return json_response({
            'balance': balance,
            'current_month_price': current_month_price,
            'debt': round(debt, 2),
            'after_payment': round(balance - current_month_price, 2),
        })
