# coding: utf-8
from flask import g, request
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_invalid_value,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    DomainNotFound,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import get_permissions
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    no_permission_required,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.core.models.domain import DomainModel


class PermissionListView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.read_users, scope.write_users])
    def get(self, meta_connection, main_connection):
        """
        Список привилегий пользователя

        Эта ручка отдает либо гломальные права текущего пользователя,
        либо его права на некоторую сущность.

        Для запроса прав пользователя на работу с объектом в директории,
        надо указать два аргумента: `type` и `id`. Type может принимать
        следующие значения: "department", "group", "user".

        Пока поддерживается запрос прав только на группы.

        Если type и id не указаны, то выдаются "глобальные права", то есть
        права на выполнение каких-то базовых операций на портале, например на
        редактирование свойств организации или добавление новых отделов и групп.

        ### Глобальные права

        - add_users
        - add_departments
        - add_groups
        - edit_organization

        ### Права на группы

        - edit_group

        ---
        tags:
          - Права
        parameters:
              - in: query
                name: type
                type: string
                description: тип объекта, на который запрашиваются права
              - in: query
                name: id
                type: string
                description: id объекта, на который запрашиваются права

        responses:
          200:
            description: Список прав
        """
        object_type = request.args.get('type')
        object_id = request.args.get('id')
        if object_id:
            try:
                object_id = int(object_id)
            except ValueError:
                pass

        try:
            org_domain = DomainModel(main_connection).get_master(g.org_id)
            master_domain = org_domain['name']
        except DomainNotFound:
            master_domain = None
        permissions = get_permissions(
            meta_connection,
            main_connection,
            g.user.passport_uid,
            object_type,
            object_id,
            g.org_id,
        )
        return json_response(permissions)
