# coding: utf-8

from flask import (
    g,
    request,
)
from itertools import groupby
from intranet.yandex_directory.src.yandex_directory.common.utils import url_join
from collections import defaultdict
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import pmap
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_scopes,
    no_auth,
    permission_required,
    no_permission_required,
    requires,
    internal,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.utils import json_response
from intranet.yandex_directory.src.yandex_directory.common.pagination import get_link_header
from intranet.yandex_directory.src.yandex_directory import app


def get_organizations(meta_connection, user_id, page, per_page):
    offset = (page - 1) * per_page
    # items = app.components.meta_users.get_org_ids_with_shard_numbers(
    #     user_id,
    #     offset=offset,
    #     limit=per_page,
    # )
    total = meta_connection.execute(
        """
    SELECT COUNT(*)
      FROM users
     INNER JOIN organizations
             ON users.org_id = organizations.id
     WHERE users.id = %(user_id)s
       AND users.is_dismissed != True
    """,
        user_id=user_id
    ).fetchall()[0][0]

    meta_data = meta_connection.execute(
        """
    SELECT org_id, shard
      FROM users
     INNER JOIN organizations
             ON users.org_id = organizations.id
     WHERE users.id = %(user_id)s
       AND users.is_dismissed != True
     ORDER BY org_id
     LIMIT %(limit)s
    OFFSET %(offset)s
    """,
        user_id=user_id,
        # Запрашиваем на один элемент больше, чтобы точно понять, есть ли следующая страница
        limit=per_page + 1,
        offset=offset).fetchall()

    # Сюда мы сложим данные, которые достали параллельными запросами в базу
    def make_org():
        return dict(
            id=None,
            admin_id=None,
            domains=dict(
                all=[],
                master=None,
                owned=[],
            ),
            user_count=None,
            logo=None,
        )

    data_from_shards = defaultdict(make_org)

    domains_query = """
    SELECT o.id, o.admin_uid, o.user_count, o.registrar_id, d.name, d.owned, d.master
      FROM organizations as o
      LEFT JOIN domains as d
        ON o.id = d.org_id
     WHERE o.id IN %(org_ids)s
    """

    logo_query = """
    SELECT i.meta->'sizes'->'orig'->'path' as path, i.org_id
      FROM organizations as o
     INNER JOIN images as i
        ON o.id = i.org_id AND o.logo_id = i.id
     WHERE o.id IN %(org_ids)s
    """

    def get_data_from_shard(shard, org_ids):
        with get_main_connection(shard) as main_connection:
            rows = main_connection.execute(domains_query, org_ids=org_ids).fetchall()
            for org_id, admin_uid, user_count, registrar_id, name, owned, master in rows:
                org = data_from_shards[org_id]
                org['id'] = org_id
                org['admin_id'] = admin_uid
                org['user_count'] = user_count
                org['registrar_id'] = registrar_id

                if name:
                    domains = org['domains']
                    domains['all'].append(name)
                    if owned:
                        domains['owned'].append(name)
                    if master:
                        domains['master'] = name

    def get_logo_from_shard(shard, org_ids):
        with get_main_connection(shard) as main_connection:
            rows = main_connection.execute(logo_query, org_ids=org_ids).fetchall()
            for logo_path, logo_org_id in rows:
                if logo_path:
                    data_from_shards[logo_org_id]['logo'] = url_join(app.config['MDS_READ_API'], logo_path)

    by_shard = groupby(meta_data, lambda item: item[1])
    params = [
        {
            'shard': shard,
            'org_ids': tuple(i[0] for i in items),
        }
        for shard, items in by_shard
    ]
    pmap(get_data_from_shard, params)
    pmap(get_logo_from_shard, params)
    results = [
        data_from_shards[org_id]
        for org_id, shard in meta_data
    ]

    next_link = None
    if len(results) > per_page:
        params = dict(per_pag=per_page, page=page+1)
        next_link = url_join(
            app.config['SITE_BASE_URI'],
            request.path,
            query_params=params,
        )

    return results[:per_page], total, next_link


class ProxyOrganizationsView(View):
    @internal
    @scopes_required([scope.read_organization])
    @requires(org_id=False, user=True)
    @no_permission_required
    def get(self, meta_connection, main_connection):
        """
        Информация обо всех организациях и их доменах для пользователя API ПДД.

        В отличии от ручки /organizations/, у этой идёт сквозная пейджинация
        по шардам.

        Но есть и минус, организации можно фильтровать только по уиду внешнего админа,
        потому что в API ПДД доменами могли управлять только внешние админы.

        Этот фильтр является обязательным.

        Так же, в отличие от ручки /organizations/, формат выдачи фиксированный,
        потому что эта ручка сделана специально для API прокси:

        ```
        {
            "total": 42,
            "result": [
                {
                    "id": <org-id>,
                    "admin_id": <admin-uid>,
                    "registrar_id": <registrar_id>,
                    "domains": {
                        "all": ["domlocals.com", "house.ru", "example.com"],
                        "master": "domlocals.com",
                        "owned": ["domlocals.com", "house.ru"]
                    },
                    "user_count": <users_count>,
                    "logo": <logo>
                },
                ...
            ],
            "links": {
                "next": "https://api-internal.directory.yandex.net/v10/proxy/organizations/?uid=100500&page=2"
            }
        }
        ```

        ---
        tags:
          - Организация
        parameters:
          - in: query
            name: page
            required: False
            type: integer
            description: Номер страницы. Если не задан, то 1.
          - in: query
            name: per_page
            required: False
            type: integer
            description: Количество элементов на странице. Если не задано, то 20.
        responses:
          200:
            description: Success
        """
        query_params = request.args.to_dict()
        page = int(query_params.get('page', 1))
        per_page = int(query_params.get('per_page', 20))
        user_id = g.user.passport_uid

        headers = {}


        organizations, total, next_link = get_organizations(
            meta_connection,
            user_id,
            page,
            per_page,
        )

        links = {'next': next_link} if next_link else {}

        # Заголовок со ссылками на другие страницы должен
        # добавляться только если есть какие-то ссылки. Иначе
        # он получится пустой
        if links:
            headers['Link'] = get_link_header(links)

        response = {
            'result': organizations,
            'links': links,
            'total': total,
        }

        return json_response(response, headers=headers)

