# coding: utf-8
from marshmallow_sqlalchemy import (
    ModelSchema,
    field_for,
)
from marshmallow import fields, INCLUDE

from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    requires,
    scopes_required,
    no_permission_required,
)
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
)
from intranet.yandex_directory.src.yandex_directory.core.registrar import (
    CryptToken,
    RegistrarPassword,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View, no_cache
from intranet.yandex_directory.src.yandex_directory.meta.repositories import RegistrarRepository
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
)
from intranet.yandex_directory.src.yandex_directory.common.components import component_registry
from intranet.yandex_directory.src.yandex_directory.meta.models import Registrar
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    unfreeze_or_copy,
)

REGISTRAR_SCHEMA = {
    'title': 'Registrar',
    'type': 'object',
    'properties': {
        'name': schemas.STRING,
        'password': schemas.STRING,
        'oauth_client_id': schemas.STRING,
        'validate_domain_url': schemas.STRING,
        'domain_added_callback_url': schemas.STRING,
        'domain_verified_callback_url': schemas.STRING,
        'domain_deleted_callback_url': schemas.STRING,
        # Эти поля надо поддержать, чтобы менять каллбэки для old версии регистраторов
        'payed_url': schemas.STRING,
        'added_init': schemas.STRING,
        'added': schemas.STRING,
        'delete_url': schemas.STRING,
    },
    'additionalProperties': False
}

REGISTRAR_FIELDS_MAP = {
    'payed_url': 'validate_domain_url',
    'added_init': 'domain_added_callback_url',
    'added': 'domain_verified_callback_url',
    'delete_url': 'domain_deleted_callback_url',
}


class RegistrarsRepositoryMixin(object):

    @property
    def registrars(self):
        # type: (...) -> RegistrarRepository
        return component_registry().registrar_repository


class TokensMixin(RegistrarsRepositoryMixin):

    @property
    def tokens(self):
        return CryptToken(self.registrars)


class RegistrarSchema(ModelSchema):
    class Meta:
        model = Registrar

    # Это поле работает только на сериализацию
    payed_url = fields.Function(lambda obj: obj.validate_domain_url)
    # А это на десериализацию.
    validate_domain_url = field_for(Registrar, 'validate_domain_url', load_from='payed_url')

    # С остальным полями - то же самое.
    # Это работает. Но почему я не знаю.
    # Полдня убил на то, чтобы подобрать подходящее решение
    # для переименовыния поля в моменты сериализации/десериализации.
    #
    # Просто оставь это как есть, если сомниваешься :)
    added_init = fields.Function(lambda obj: obj.domain_added_callback_url)
    domain_added_callback_url = field_for(Registrar, 'domain_added_callback_url', load_from='added_init')

    added = fields.Function(lambda obj: obj.domain_verified_callback_url)
    domain_verified_callback_url = field_for(Registrar, 'domain_verified_callback_url', load_from='added')

    delete_url = fields.Function(lambda obj: obj.domain_deleted_callback_url)
    domain_deleted_callback_url = field_for(Registrar, 'domain_deleted_callback_url', load_from='delete_url')

    password = fields.Method("get_password")

    def get_password(self, obj):
        if obj.pdd_version == 'old':
            return obj.password
        return RegistrarPassword().decrypt_password(obj).decode('utf-8')

    def load(self, data, session=None, instance=None, transient=False, *args, **kwargs):
        data = unfreeze_or_copy(data)
        for field, map_field in REGISTRAR_FIELDS_MAP.items():
            field_value = data.pop(field, 'fake_value')
            if field_value != 'fake_value':
                data[map_field] = field_value
        if 'password' in data:
            password = data.pop('password')
            instance = RegistrarPassword().encrypt_password(password, instance)
        return super(RegistrarSchema, self).load(
            data, session=session,
            instance=instance, transient=transient,
            **kwargs
        )


class RegistrarView(RegistrarsRepositoryMixin, View):
    registrar_new_schema = RegistrarSchema(
        only=(
            'id',
            'pdd_id',
            'pdd_version',
            'name',
            'admin_id',
            'password',
            'oauth_client_id',
            'validate_domain_url',
            'domain_added_callback_url',
            'domain_verified_callback_url',
            'domain_deleted_callback_url',
        )
    )
    registrar_old_schema = RegistrarSchema(
        only=(
            'id',
            'pdd_id',
            'pdd_version',
            'name',
            'admin_id',
            'password',
            'oauth_client_id',
            'payed_url',
            'added_init',
            'added',
            'delete_url',
        )
    )

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_registrar])
    @no_permission_required
    def get(self, meta_connection, _, id):
        """
        Получить информацию про регистратора.

        Пример выдачи:

            {
                "id": 100500,
                "pdd_id": 42,
                "pdd_version": "new",
                "name": "regru",
                "oauth_client_id": "ee2werwsa",
                "admin_id": 2343432,
                "password": "dfdghrseadg34re423ew",
                "validate_domain_url": "http://validate_domain_url.com/",
                "domain_added_callback_url": "http://domain_added_callback_url.com/"
                "domain_verified_callback_url":  "http://domain_verified_callback_url.com/",
                "domain_deleted_callback_url":  "http://domain_deleted_callback_url.com/",
            }

        ---
        tags:
          - Регистраторы
        parameters:
          - in: path
            name: id
            required: true
            type: string
            description: "Коннектовский id регистратора или id регистратора из ПДД (вида old/new:123)"
        responses:
           200:
              description:  Данные регистратора.
        """
        data = app.domenator.get_registrar(id)
        return json_response(
            data,
            allowed_sensitive_params=('password',)
        )

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_registrar])
    @uses_schema(REGISTRAR_SCHEMA)
    @no_permission_required
    def patch(self, meta_connection, _, data, id):
        """
        Обновление данных про регистратора.

        Обновлять можно следующие поля:

        * name
        * password
        * oauth_client_id

        А так же, для регистраторов "нового" образца:

        * validate_domain_url
        * domain_added_callback_url
        * domain_verified_callback_url
        * domain_deleted_callback_url

        И для регистраторов старого образца (в Директории эти поля мапятся в те, которые приведены выше):

        * payed_url
        * added_init
        * added
        * delete_url

        ---
        tags:
          - Регистраторы
        parameters:
          - in: path
            name: id
            required: true
            type: integer
            description: "Коннектовский id регистратора"
          - in: body
            name: body
        responses:
           200:
              description: данные успешно изменены
        """
        app.domenator.patch_registrar(id, unfreeze_or_copy(data))
        return json_response({})

    def choose_schema(self, version):
        if version == 'new':
            return self.registrar_new_schema
        else:
            return self.registrar_old_schema


class RegistarV2ByTokenView(TokensMixin, View):
    registrar_by_token_schema = RegistrarSchema(
        only=(
            'id',
            'pdd_id',
            'admin_id',
        )
    )

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_registrar])
    @no_permission_required
    @no_cache
    def get(self, meta_connection, _, token):
        """
        Получить информацию про регистратора по токену.

        Пример выдачи:

            {
                "id": 100500,
                "pdd_id": 42,
                "admin_id": 2343432,
            }

        ---
        tags:
          - Регистраторы
        parameters:
          - in: path
            name: token
            required: true
            type: string
            description: "token  регистратора"
        responses:
           200:
              description:  Данные регистратора.
        """
        data = app.domenator.get_registrar_by_token(token)
        return json_response(data)


class RegistarV2TokenView(TokensMixin, View):

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_registrar])
    @no_permission_required
    def post(self, meta_connection, _, registrar_id):
        """
        Получить токену для регистратора.


        Пример выдачи:
            {
                "token": "SSFSFSFSFSDFSDF"
            }

        ---
        tags:
          - Регистраторы
        parameters:
          - in: path
            name: registrar_id
            required: true
            type: integer
            description: "id  регистратора"
        responses:
           200:
              description:  Токен регистратора.
        """
        token = app.domenator.get_or_create_registrar_token(registrar_id)
        return json_response({'token': token})

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_registrar])
    @no_permission_required
    def delete(self, meta_connection, _, registrar_id):
        """
        Удалить токен  регистратора.

        ---
        tags:
          - Регистраторы
        parameters:
          - in: path
            name: registrar_id
            required: true
            type: integer
            description: "id  регистратора"
        responses:
           200:
              description:  Токен удален.
        """
        app.domenator.delete_registrar_token(registrar_id)
        return json_response({})


class RegistarV2ByUidView(TokensMixin, View):
    registrar_new_schema = RegistrarSchema(
        only=(
            'id',
            'pdd_id',
            'pdd_version',
            'name',
            'admin_id',
            'password',
            'oauth_client_id',
            'validate_domain_url',
            'domain_added_callback_url',
            'domain_verified_callback_url',
            'domain_deleted_callback_url',
        )
    )

    @internal
    @requires(org_id=False, user=False)
    @scopes_required([scope.manage_registrar])
    @no_permission_required
    @no_cache
    def get(self, meta_connection, _, uid):
        """
        Получить информацию про регистратора по uid админа.

        Пример выдачи:

            {
                "id": 100500,
                "pdd_id": 42,
                "pdd_version": "new",
                "admin_id": 2343432,
                ...
            }

        ---
        tags:
          - Регистраторы
        parameters:
          - in: path
            name: uid
            required: true
            type: integer
            description: "uid админа регистратора"
        responses:
           200:
              description:  Данные регистратора.
        """
        data = app.domenator.get_registrar_by_admin(uid)
        return json_response(data, allowed_sensitive_params=('password',))
