# -*- coding: utf-8 -*-
from flask import g, request

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    no_permission_required,
    scopes_required,
    requires,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope, check_scopes
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ServiceNotFound,
    AuthorizationError,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import check_permissions
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    get_object_or_404,
    json_response,
)
from intranet.yandex_directory.src.yandex_directory.connect_services.idm import get_service
from intranet.yandex_directory.src.yandex_directory.connect_services.idm.base_service import OperationNotInIdmError
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_resource_relation_delete,
    action_resource_relation_add,
)
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    RelationIsAlreadyExists,
    RelationWithResourceAlreadyExists,
)
from intranet.yandex_directory.src.yandex_directory.core.models import ResourceModel
from intranet.yandex_directory.src.yandex_directory.core.models.resource import RESOURCE_OBJECT_TYPES
from intranet.yandex_directory.src.yandex_directory.core.models.resource import ResourceRelationModel
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import user_permissions
from intranet.yandex_directory.src.yandex_directory.core.utils import check_objects_exists
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    prepare_relation,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.views.resources import convert_relations, _get_service
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

RELATION_SCHEMA = {
    'title': 'RelationObject',
    'type': 'object',
    'properties': {
        'name': schemas.STRING,
        'object_type': {
            'enum': RESOURCE_OBJECT_TYPES,
        },
        'object_id': schemas.INTEGER,
        'resource_id': schemas.STRING,
        'service': schemas.STRING,
    },
    'required': [
        'name', 'object_type',
        'object_id', 'resource_id',
    ],
    'additionalProperties': False
}


class RelationDetailView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.read_resources])
    def get(self, meta_connection, main_connection, relation_id, service_slug):
        """
        Информацию о связи
        ---
        tags:
          - Связи ресурсов
        parameters:
          - in: path
            name: relation_id
            required: true
            type: string
        responses:
          200:
            description: Словарь с информацией о связи.
        """
        relation = get_object_or_404(
            model_instance=ResourceRelationModel(main_connection),
            id=relation_id,
            org_id=g.org_id,
            fields=[
                '*',
                'user.*',
                'department.*',
                'group.*',
            ]
        )
        return json_response(
            prepare_relation(relation),
        )

    @internal
    @scopes_required([scope.write_resources])
    @no_permission_required
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, relation_id,  service_slug):
        """
        Удалить связь
        ---
        tags:
          - Связи ресурсов
        parameters:
          - in: path
            name: relation_id
            required: true
            type: string
        responses:
          204:
            description: Связь удалена.
          404:
            description: Связь не найдена.
        """
        with log.fields(relation={
            'id': relation_id,
        }):
            try:
                if service_slug:
                    relation = ResourceRelationModel(main_connection).get(
                        id=relation_id, org_id=g.org_id,
                    )
                    resource_id = None
                    if relation:
                        resource = ResourceModel(main_connection).get(
                            id=relation['resource_id'],
                            org_id=g.org_id,
                        )
                        resource_id = resource['external_id']

                    check_permissions(
                        meta_connection=meta_connection,
                        main_connection=main_connection,
                        permissions=[user_permissions.can_change_relations],
                        object_type=service_slug,
                        object_id=resource_id,
                    )
                    get_service(service_slug).revoke_roles(g.org_id, g.user.passport_uid, None, relation_id)
                    relation = {
                        'id': relation_id,
                        'resource_id': relation_id,
                        'org_id': g.org_id,
                        'name': 'b2bidm',
                        'service_slug': service_slug,
                    }

                    action_resource_relation_delete(
                        main_connection,
                        org_id=g.org_id,
                        author_id=None,
                        object_value=relation,
                    )
                else:
                    raise ServiceNotFound
            except (OperationNotInIdmError, ServiceNotFound):
                relation = get_object_or_404(
                    model_instance=ResourceRelationModel(main_connection),
                    id=relation_id,
                    org_id=g.org_id,
                )
                resource = ResourceModel(main_connection).get(
                    id=relation['resource_id'],
                    org_id=g.org_id,
                )

                with log.fields(resource={
                    'id': resource['external_id'],
                    'service': {
                        'slug': service_slug,
                    },
                }):
                    if resource['service'] != g.service.identity:
                        if not check_scopes(g.scopes, [scope.change_resource_relations]):
                            raise AuthorizationError(
                                'This operation requires one of scopes: {0}'.format(
                                    ', '.join([scope.change_resource_relations])
                                )
                            )
                    check_permissions(
                        meta_connection=meta_connection,
                        main_connection=main_connection,
                        permissions=[user_permissions.can_change_relations],
                        object_type=resource['service'],
                        object_id=resource['external_id'],
                    )

                    ResourceRelationModel(main_connection).delete(
                        {
                            'id': relation_id,
                            'org_id': g.org_id
                        }
                    )
                relation['service_slug'] = resource['service']
                action_resource_relation_delete(
                    main_connection,
                    org_id=g.org_id,
                    author_id=None,
                    object_value=relation,
                )

            return json_response(
                {'message': 'No Content'},
                status_code=204,
            )


class RelationListView(View):
    @internal
    @scopes_required([scope.write_resources])
    @no_permission_required
    @requires(org_id=True, user=True)
    @uses_schema(RELATION_SCHEMA)
    def post(self, meta_connection, main_connection, data):
        """
        Создать связь
        ---
        tags:
          - Связи ресурсов
        responses:
          201:
            description: Связь создана.
          404:
            description: Ресурс не найден.
          409:
            description: Такая связь уже есть.
        """
        data = dict(data)
        data['service'] = _get_service(request, data=data)
        with log.fields(resource={
            'id': data['resource_id'],
            'service': {
                'slug': data['service'],
            },
        }):
            check_permissions(
                meta_connection=meta_connection,
                main_connection=main_connection,
                permissions=[user_permissions.can_change_relations],
                object_type=data['service'],
                object_id=data['resource_id'],
            )

            try:
                org_id = g.org_id
                author_id = g.user.passport_uid

                relation_data = convert_relations([data])[0]
                with log.fields(relation={
                    'uid': relation_data['user_id'],
                    'name': data['name'],
                }):
                    role = {
                        'path': data['name'],
                        'resource_id': data['resource_id'],
                        'uid': relation_data['user_id'],
                    }

                    role_id = get_service(data['service']).request_roles(org_id, author_id, role)[0]

                    action_resource_relation_add(
                        main_connection,
                        org_id=g.org_id,
                        author_id=author_id,
                        object_value={
                            'id': role_id,
                            'resource_id': data['resource_id'],
                            'user_id': relation_data['user_id'],
                            'name': relation_data['name'],
                            'service_slug': data['service'],
                        },
                    )

                return json_response({}, status_code=201)
            except (OperationNotInIdmError, ServiceNotFound):
                pass

            relations = [data]
            check_objects_exists(main_connection, g.org_id, relations)
            relation_data = convert_relations(relations)[0]

            with log.fields(relation={
                'uid': relation_data.get('user_id'),
                'name': relation_data['name'],
            }):
                resource = get_object_or_404(
                    model_instance=ResourceModel(main_connection),
                    service=data['service'],
                    org_id=g.org_id,
                    external_id=data['resource_id'],
                )
                user_id = relation_data.get('user_id')
                if data['service'] == settings.METRIKA_SERVICE_SLUG and user_id:
                    existed_relations = ResourceRelationModel(main_connection).filter(
                        resource_id=resource['id'],
                        user_id=user_id,
                        org_id=g.org_id,
                    ).fields('name')
                    if existed_relations:
                        raise RelationWithResourceAlreadyExists(
                            service_slug=data['service'],
                            relation_names=[rel['name'] for rel in existed_relations]
                        )

                relation = ResourceRelationModel(main_connection).create(
                    org_id=g.org_id,
                    resource_id=resource['id'],
                    name=relation_data['name'],
                    user_id=user_id,
                    group_id=relation_data.get('group_id'),
                    department_id=relation_data.get('department_id'),
                )
                if not relation.get('id'):
                    # такой relation уже был
                    raise RelationIsAlreadyExists()

                relation['service_slug'] = data['service']
                action_resource_relation_add(
                    main_connection,
                    org_id=g.org_id,
                    author_id=None,
                    object_value=relation,
                )

        return json_response({}, status_code=201)
