# -*- coding: utf-8 -*-
from flask import g, request
from intranet.yandex_directory.src.yandex_directory import app

from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope, check_scopes
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import send_metrika_resource_unbind_email, send_resource_unbind_email
from intranet.yandex_directory.src.yandex_directory.core.models.resource import (
    ResourceModel,
    RESOURCE_OBJECT_TYPES,
)
from intranet.yandex_directory.src.yandex_directory.common.schemas import STRING
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    build_list_response,
    get_object_or_404,
    json_response,
    json_error_not_found,
    json_error_user_dismissed,
    check_permissions,
)
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
    uses_out_schema,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    prepare_resource,
    check_objects_exists,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_resource_modify,
    action_resource_add,
    action_resource_delete,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    DepartmentModel,
    GroupModel,
    OrganizationServiceModel,
    ServiceModel,
)

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    no_permission_required,
    requires,
    scopes_required,
)

from intranet.yandex_directory.src.yandex_directory.common.exceptions import AuthorizationError, ResourceAlreadyExists, IsNotMember
from intranet.yandex_directory.src.yandex_directory.core.exceptions import RequiredFieldsMissed
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import user_permissions, service_permissions
from intranet.yandex_directory.src.yandex_directory.connect_services.idm import get_service
from intranet.yandex_directory.src.yandex_directory.connect_services.idm.base_service import OperationNotInIdmError
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ServiceNotFound
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

"""
Через API пользователи работают не с id ресурса, а с external_id и отдавать в поле id им нужно именно его.
Сам же id используется для внутренних связей в директории и является уникальным на всю таблицу.
"""


def prepare_comma_separated_or_simple_value(value):
    if ',' in value:
        return value.split(',')
    else:
        return value


OBJECT_SCHEMA = {
    'title': 'RelationObject',
    'type': 'object',
    'properties': {
        'name': {
            'type': 'string'
        },
        'object_type': {
            'enum': RESOURCE_OBJECT_TYPES
        },
        'object_id': {
            'type': 'integer'
        }
    },
    'required': ['name', 'object_type', 'object_id'],
    'additionalProperties': False
}

OBJECT_OUT_SCHEMA = {
    'title': 'RelationObject',
    'type': 'object',
    'properties': {
        'name': {
            'type': 'string'
        },
        'object_type': {
            'enum': RESOURCE_OBJECT_TYPES
        },
        'object': {
            'type': 'object'
        },
        'id': {
            'type': 'integer'
        },
    },
    'required': ['name', 'object_type', 'object'],
    'additionalProperties': False,
}

RESOURCE_RELATIONS_SCHEMA = {
    'title': 'RelationsList',
    'type': 'array',
    'items': OBJECT_SCHEMA
}


RESOURCE_SCHEMA = {
    'title': 'Resource',
    'type': ['object', 'null'],
    'properties': {
        'relations': RESOURCE_RELATIONS_SCHEMA,
        'id': STRING,
    },
    'additionalProperties': False
}

RESOURCE_OUT_SCHEMA = {
    'title': 'Resource',
    'type': ['object', 'null'],
    'properties': {
        'relations': {
            'title': 'RelationsList',
            'type': 'array',
            'items': OBJECT_OUT_SCHEMA
        },
        'id': {'type': ['string']},
        'metadata': {'type': ['object']},
    },
}


RESOURCE_ARRAY_OUT_SCHEMA = {
    'title': 'Resource object list',
    'type': 'array',
    'items': RESOURCE_OUT_SCHEMA
}


RESOURCE_OBJECT_TYPE_TO_RELATION_PARAM_MAP = {
    'user': 'user_id',
    'group': 'group_id',
    'department': 'department_id',
}


OPERATIONS_TYPES = [
    'add',
    'delete',  # todo: remove me
    'remove',
]


OPERATIONS_SCHEMA = {
    'title': 'Operations',
    'type': 'array',
    'items': {
        'title': 'Operation',
        'type': 'object',
        'properties': {
            'operation_type': {
                'enum': OPERATIONS_TYPES  # todo: test me
            },
            'value': OBJECT_SCHEMA,
        },
        'required': ['operation_type', 'value'],
        'additionalProperties': False
    }
}


RELATIONS_UPDATE_SCHEMA = {
    'title': 'Relation_update',
    'type': ['object'],
    'properties': {
        'operations': OPERATIONS_SCHEMA
    },
    'required': ['operations'],
    'additionalProperties': False
}


def _get_service(request, data=None, read=False, perm=None):
    data = data or {}
    if not perm:
        if read:
            perm = [scope.read_service_resources]
        else:
            perm = [scope.change_resource_relations]

    service = request.args.get('service') or data.get('service')
    if service and service != g.service.identity:
        if not check_scopes(g.scopes, perm):
            raise AuthorizationError(
                'This operation requires one of scopes: {0}'.format(
                    ', '.join(perm)
                )
            )
    return service or g.service.identity


def check_membership(main_connection, data):
    """
    Проверяет, что объект внутри организации
    """
    member = False
    if data['object_type'] == 'user':
        member = UserModel(main_connection).get(
            user_id=int(data['object_id']),
            org_id=g.org_id,
        )
    elif data['object_type'] == 'department':
        member = DepartmentModel(main_connection).get(
            department_id=int(data['object_id']),
            org_id=g.org_id,
        )
    elif data['object_type'] == 'group':
        member = GroupModel(main_connection).get(
            org_id=g.org_id,
            group_id=int(data['object_id']),
        )
    return bool(member)


def convert_relations(relations):
    for relation in relations:
        key = RESOURCE_OBJECT_TYPE_TO_RELATION_PARAM_MAP[relation['object_type']]
        relation[key] = relation['object_id']
    return relations


def metadata_client(service):
    services_clients = {
        'metrika': app.metrika,
        'yandexsprav': app.yandexsprav,
        'alice_b2b': app.alice_b2b,
    }
    return services_clients.get(service)


def skip_resource(resource):
    return resource['metadata'] == {} and resource['service'] == 'metrika'


class ResourceListView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.read_resources, scope.write_resources])
    @requires(org_id=True, user=False)
    @uses_out_schema(RESOURCE_ARRAY_OUT_SCHEMA)
    def get(self, meta_connection, main_connection):
        """
        Список ресурсов

        Пример выдачи:


            {
                ...
                "result": [
                    {
                        "id": "5552f9a1e05f8d0817ce91ce",
                        "service": "yamb",
                        "metadata": {},
                        "relations": [
                            {
                                "name": "read",
                                "object_type": "user",
                                "object": {
                                    "email": "web-chib@ya.ru",
                                    "name": {
                                        "first": {
                                            "en": "Gennady",
                                            "ru": "Геннадий"
                                        },
                                        "last": {
                                            "en": "Chibisov",
                                            "ru": "Чибисов"
                                        }
                                    },
                                    "gender": "male",
                                    "id": 66865142,
                                    "nickname": "web-chib"
                                }
                            }
                            ...
                        ]
                    }
                    ...
                ]
            }

        Поля сущности ресурса:

        * **id** - идентификатор
        * **service** - идентификатор сервиса, которому
        принадлежит ресурс.
        * **metadata** - дополнительные даные ресурса (имя, домен, ...), для разных
        сервисов набор может быть разный
        * **relations** - список отношений различных
        сущностей к данному ресурсу.
            * **name** - название отношения
            * **object_type** - тип сущности. Может
              принимать значения `department`, `group` или `user`
            * **object** - сама сущность. Разные типы
              сущностей имеют разное представление


        #### Фильтрация

        Фильтрация ресурсов возможна по идентификатору:

            /resources/?id=55521101e05f8d0814ce91ce

        Для фильтрации по списку идентификаторов необходимо перечислить их через запятую:

            /resources/?id=55521101e05f8d0814ce91ce,44421101e04f8d0814ce91dd

        Фильтрация ресурсов возможна по названию отношения:

            /resources/?relation_name=read

        Для фильтрации по списку названий отношений необходимо перечислить их через запятую:

            /resources/?relation_name=read,write

        Фильтрация ресурсов возможна по конкретному сервису (требует прав read_service_resources):

            /resources/?service=metrika

        Фильтрация ресурсов возможна по идентификатору пользователя:

            /resources/?user_id=66865142

        При этом нужно иметь ввиду следующее соглашение - при фильтрации по идентификатору пользователя в выборку попадают не
        только те ресурсы, объектом отношения с которыми является этот пользователь, но и ресурсы, объектом отношений с которыми
        являются группы/департаменты в которые входит этот пользователь.

        Конечно, все перечисленные выше фильтры можно комбинировать. Например, выборка ресурсов с id `1` или `2`, к которым пользователь
        с id `66865142` имеет отношение `write` или `admin`:

            /resources/?user_id=1,2&relation_name=write,admin&user_id=66865142

        ---
        tags:
          - Ресурсы
        parameters:
            - in: query
              name: id
              type: string
              description: список id ресурсов, перечисленных через з
            - in: query
              name: relation_name
              type: string
              description: названия отношений через запятую
            - in: query
              name: user_id
              type: string
              description: UID пользователя. Используется для того, чтобы проверить имеет ли сотрудник доступ к каким-то ресурсам.
            - in: query
              name: page
              type: integer
              description: текущая страница
            - in: query
              name: per_page
              type: integer
              description: какое кол-во объектов выведено на странице
            - in: query
              name: service
              type: string
              description: ресурсы какого сервиса выводить (если не задано - выводится для того сервиса который пришел)
        responses:
          200:
            description: Список ресурсов
        """
        model_filters = self._get_filter_data()
        service = model_filters['service']

        response = build_list_response(
            model=ResourceModel(main_connection),
            model_filters=model_filters,
            path=request.path,
            query_params=request.args.to_dict(),
            model_fields=[
                '*',
                'relations.*',
                'relations.user.*',
                'relations.department.*',
                'relations.group.*',
            ],
            prepare_result_item_func=prepare_resource,
        )
        self._populate_response_with_metadata(response, service)

        return json_response(
            response['data'],
            headers=response['headers']
        )

    def _populate_response_with_metadata(self, response, service):

        resources_data = response['data']['result']
        service_client = metadata_client(service)

        if service_client:
            resources_ids = [
                resource['id'] for resource in resources_data
            ]
            resources_metadata = service_client.get_metadata(*resources_ids)
            result_data = []

            for resource_data in resources_data:
                resource_data['metadata'] = resources_metadata.get(resource_data['id'], {})
                if not skip_resource(resource_data):
                    result_data.append(resource_data)

            response['data']['result'] = result_data

    def _get_filter_data(self):
        filter_data = {
            'service': g.service.identity,
            'org_id': g.org_id,
        }
        if 'service' in request.args:
            if not check_scopes(g.scopes, [scope.read_service_resources]):
                raise AuthorizationError(
                    'This operation requires {}'.format(scope.read_service_resources)
                )
            filter_data['service'] = request.args['service']
        if 'id' in request.args:
            filter_data['external_id'] = prepare_comma_separated_or_simple_value(request.args['id'])
        if 'relation_name' in request.args:
            filter_data['relation_name'] = prepare_comma_separated_or_simple_value(request.args['relation_name'])
        if 'user_id' in request.args:  # todo: test me
            filter_data['user_id'] = prepare_comma_separated_or_simple_value(request.args['user_id'])
        return filter_data

    @internal
    @no_permission_required
    @scopes_required([scope.write_resources])
    @requires(org_id=True, user=False)
    @uses_schema(RESOURCE_SCHEMA)
    def post(self, meta_connection, main_connection, data):
        """
        Создать новый ресурс

        В эту ручку можно делать пост с пустым body, и тогда будет создан ресурс
        без каких-либо отношений.

        Если же, при создании ресурса можно сразу передать список отношений
        сущностей к создаваемому ресурсу. Список должен состоять из
        объектов с ключами:

        * **name** - название отношения
        * **object_id** - идентификатор сущности

        Пример:

            # Request
            POST /resources/ HTTP/1.1

            {{
                "relations": [
                    {{
                        "name": "read",
                        "object_type": "user",
                        "object_id": "66865142"
                    }}
                ]
            }}

            # Response
            HTTP/1.1 201 CREATED

            {{
                "id": "5552f9a1e05f8d0817ce91ce",
                "relations": [
                    {{
                        "name": "read",
                        "object": {{
                            "email": "web-chib@ya.ru",
                            "name": {{
                                "first": {{
                                    "en": "Gennady",
                                    "ru": "Геннадий"
                                }},
                                "last": {{
                                    "en": "Chibisov",
                                    "ru": "Чибисов"
                                }}
                            }},
                            "gender": "male",
                            "id": 66865142,
                            "nickname": "web-chib"
                        }},
                        "object_type": "user"
                    }}
                ],
                "service": "yamb"
            }}
        ---
        tags:
          - Ресурсы
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Ресурс создан
        """
        data = data or {}
        relations = data.get('relations', [])
        service_slug = g.service.identity
        org_id = g.org_id
        external_id = data.get('id')
        with log.fields(resource={
            'id': external_id,
            'service': {
                'slug': service_slug,
            },
        }):

            resource_model = ResourceModel(main_connection)
            exists = resource_model \
                .filter(
                    service=service_slug,
                    org_id=org_id,
                    external_id=external_id,
                )
            if exists:
                raise ResourceAlreadyExists()

            # запрет на создание ресурса с уволенными сотрудниками
            for item in relations:
                if item['object_type'] != 'user':
                    continue
                uid = item['object_id']
                dismiss = UserModel(main_connection).count({
                    'id': uid,
                    'is_dismissed': True,
                    'org_id': org_id,
                })
                if dismiss:
                    return json_error_user_dismissed(uid)

            # если какого-то объекта нет то кидаем ConstraintValidationError
            check_objects_exists(main_connection, g.org_id, relations)

            relations = convert_relations(relations)

            resource = resource_model.create(
                service=service_slug,
                org_id=org_id,
                relations=relations,
                external_id=external_id,
            )
            action_resource_add(
                main_connection,
                org_id=g.org_id,
                author_id=None,
                object_value=resource,
            )

        response = ResourceModel(main_connection).get(
            id=resource['id'],
            org_id=g.org_id,
            service=g.service.identity,
            fields=[
                '*',
                'relations.*',
                'relations.user.*',
                'relations.department.*',
                'relations.group.*',
            ],
        )
        return json_response(
            prepare_resource(response),
            status_code=201,
        )


class ResourceDetailView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.read_resources, scope.write_resources])
    @requires(org_id=True, user=False)
    @uses_out_schema(RESOURCE_OUT_SCHEMA)
    def get(self, meta_connection, main_connection, resource_id):
        """
        Информацию о ресурсе

        ---
        tags:
          - Ресурсы
        parameters:
          - in: path
            name: resource_id
            required: true
            type: string
          - in: query
            name: service
            type: string
            description: relation ресурса какого сервиса выводить (нужен грант read_service_resources)
        responses:
          200:
            description: Словарь с информацией о ресурсе.
          404:
            description: Ресурс не найден.
        """
        service = _get_service(request, read=True)
        resource = get_object_or_404(
            model_instance=ResourceModel(main_connection),
            external_id=resource_id,
            org_id=g.org_id,
            service=service,
            fields=[
                '*',
                'relations.*',
                'relations.user.*',
                'relations.department.*',
                'relations.group.*',
            ],
        )
        prepared_resource = prepare_resource(resource)
        service_client = metadata_client(service)
        if service_client:
            resources_metadata = service_client.get_metadata(prepared_resource['id'])
            if resources_metadata:
                prepared_resource['metadata'] = resources_metadata[prepared_resource['id']]
        if skip_resource(prepared_resource):
            return json_error_not_found()
        return json_response(prepared_resource)

    @internal
    @no_permission_required
    @scopes_required([scope.write_resources])
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, resource_id):
        """
        Удаление ресурса

        ---
        tags:
          - Ресурсы
        parameters:
          - in: path
            name: resource_id
            required: true
            type: string
          - in: query
            name: service
            type: string
            description: ресурс какого сервиса удалить (если не совпадает с текущим - нужен грант change_service_resources)
        responses:
          200:
            description: Ресурс удален.
          404:
            description: Ресурс не найден.
        """

        service_slug = _get_service(request, perm=[scope.change_service_resources])

        with log.fields(resource={
            'id': resource_id,
            'service': {
                'slug': service_slug,
            },
        }):
            check_permissions(
                meta_connection=meta_connection,
                main_connection=main_connection,
                permissions=[service_permissions.update_service_data],
                object_type=service_slug,
            )
            resource = get_object_or_404(
                model_instance=ResourceModel(main_connection),
                external_id=resource_id,
                org_id=g.org_id,
                service=service_slug,
            )
            old_responsible = None

            try:
                service = get_service(service_slug)
                service.unbind_resource(
                    org_id=g.org_id,
                    author_id=g.user.passport_uid,
                    resource_id=resource_id,
                )
                if service_slug == 'direct':
                    # TODO Это место нужно будет убрать когда разрешим
                    #  несколько клиентов в организации
                    organization_service = OrganizationServiceModel(main_connection).get_by_slug(
                        org_id=g.org_id, service_slug=service_slug,
                        fields=['id', 'responsible_id'],
                    )
                    old_responsible = organization_service['responsible_id']

                    OrganizationServiceModel(main_connection).update_one(
                        id=organization_service['id'],
                        update_data={'responsible_id': None}
                    )
            except (OperationNotInIdmError, ServiceNotFound):
                pass

            ResourceModel(main_connection).delete(
                filter_data={'id': resource['id']}
            )

            action_resource_delete(
                main_connection,
                org_id=g.org_id,
                author_id=g.user.passport_uid,
                object_value=resource,
                old_object=resource,
            )

            service_model = ServiceModel(meta_connection).filter(slug=service_slug).one()
            organization_service = OrganizationServiceModel(main_connection).filter(
                org_id=g.org_id,
                service_id=service_model['id'],
            ).one()

            if organization_service:
                responsible_id = old_responsible or organization_service['responsible_id']
                if g.user.passport_uid != responsible_id:
                    responsible = UserModel(main_connection).filter(
                        org_id=g.org_id,
                        id=responsible_id,
                    ).one()
                    if responsible:
                        send_resource_unbind_email(
                            main_connection, service_slug,
                            g.org_id, responsible['email'],
                            resource['external_id'],
                        )

            return json_response({'status': 'ok'}, status_code=204)


class ResourceDetailRelationsView(View):

    @internal
    @no_permission_required
    @scopes_required([scope.read_resources, scope.write_resources])
    @requires(org_id=True, user=True)
    def get(self, meta_connection, main_connection, resource_id):
        """
        Список связей

        Пример выдачи:

            [
                {
                    "name": "read",
                    "object": {
                        "email": "web-chib@ya.ru",
                        "name": {
                            "first": {
                                "en": "Gennady",
                                "ru": "Геннадий"
                            },
                            "last": {
                                "en": "Chibisov",
                                "ru": "Чибисов"
                            }
                        },
                        "gender": "male",
                        "id": 66865142,
                        "nickname": "web-chib"
                    },
                    "object_type": "user"
                }
            ]

        ---
        tags:
          - Связи ресурсов
        parameters:
          - in: query
            name: service
            type: string
            description: relation ресурса какого сервиса выводить (нужен грант read_service_resources)
        responses:
          200:
            description: Список отношений данного ресурса.

        """
        service = _get_service(request, read=True)
        try:
            org_id = g.org_id
            author_id = g.user.passport_uid

            roles = get_service(service).roles_by_resource_id(org_id, author_id, resource_id)
            user_ids = [role['user']['username'] for role in roles]

            if not user_ids:
                return json_response([])

            users = UserModel(main_connection).filter(org_id=org_id, id=user_ids).all()
            users_by_ids = {user['id']: user for user in users}

            service_id = ServiceModel(meta_connection).get_by_slug(service)['id']
            responsible_id = OrganizationServiceModel(main_connection).filter(
                org_id=org_id, service_id=service_id,
            ).one()['responsible_id']

            result = []
            for role in roles:
                user_id = int(role['user']['username'])
                if user_id == responsible_id:
                    continue

                user = users_by_ids[user_id]

                result.append({
                    'id': role['id'],
                    'name': role['human'],
                    'state': role['state'],
                    'object': {
                        'email': user['email'],
                        'name': user['name'],
                        'gender': user['gender'],
                        'id': user['id'],
                        'nickname': user['nickname'],
                    },
                    'object_type': 'user',
                })

            return json_response(result)

        except (OperationNotInIdmError, ServiceNotFound):
            pass
        resource = get_object_or_404(
            model_instance=ResourceModel(main_connection),
            external_id=resource_id,
            org_id=g.org_id,
            service=service,
            fields=[
                '*',
                'relations.*',
                'relations.user.*',
                'relations.department.*',
                'relations.group.*',
            ],
        )
        return json_response(
            prepare_resource(resource)['relations'],
        )

    @internal
    @no_permission_required
    @scopes_required([scope.write_resources])
    @requires(org_id=True, user=False)
    @uses_schema(RESOURCE_RELATIONS_SCHEMA)
    def put(self, meta_connection, main_connection, data, resource_id):
        """
        Полное обновление связей.
        Если передан несуществующий resource_id, то он будет создан, а переданные relations свяжутся с этим resource_id.

        Пример ответа:

            [
                {{
                    "name": "write",
                    "object": {{
                        "email": "web-chib@ya.ru",
                        "name": {{
                            "first": {{
                                "en": "Gennady",
                                "ru": "Геннадий"
                            }},
                            "last": {{
                                "en": "Chibisov",
                                "ru": "Чибисов"
                            }}
                        }},
                        "gender": "male",
                        "id": 66865142,
                        "nickname": "web-chib"
                    }},
                    "object_type": "user"
                }}
            ]

        ---
        tags:
          - Связи ресурсов
        parameters:
          - in: path
            name: resource_id
            required: true
            type: string
          - in: query
            name: service
            type: string
            description: relation ресурса какого сервиса создавать (нужен грант read_service_resources)
          - in: body
            name: body
        responses:
          200:
            description: Связи обновлены
        """
        service = _get_service(request)
        relations = request.get_json()

        # запрет на создание ресурса с уволенными сотрудниками
        for item in relations:
            if item['object_type'] != 'user':
                continue
            uid = item['object_id']
            dismiss = UserModel(main_connection).count({
                'id': uid,
                'is_dismissed': True,
                'org_id': g.org_id,
            })
            if dismiss:
                return json_error_user_dismissed(uid)

        # если какого-то объекта нет то кидаем ConstraintValidationError
        check_objects_exists(main_connection, g.org_id, relations)
        relations = convert_relations(relations)

        resource = ResourceModel(main_connection).get(
            external_id=resource_id,
            org_id=g.org_id,
            service=service,
            fields=[
                '*',
                'relations.*',
            ],
        )

        # https://st.yandex-team.ru/DIR-2277
        # если ресурс с передаваемым id (external_id не найден), то создаем этот ресурс и связываем
        # существующие relations вместе с ним
        if not resource:
            resource = ResourceModel(main_connection).create(
                external_id=resource_id,
                org_id=g.org_id,
                service=service,
                relations=relations
            )
            old_object = None
            object_value = resource
            action_func = action_resource_add
        else:
            check_permissions(
                meta_connection=meta_connection,
                main_connection=main_connection,
                permissions=[user_permissions.can_change_relations],
                object_type=resource['service'],
                object_id=resource_id,
            )
            old_object = resource
            ResourceModel(main_connection).update_relations(
                resource_id=resource['id'],
                org_id=resource['org_id'],
                relations=relations
            )
            object_value = ResourceModel(main_connection).get(
                id=resource['id'],
                org_id=resource['org_id'],
                service=service,
                fields=[
                    '*',
                    'relations.*',
                ],
            )
            action_func = action_resource_modify

        action_func(
            main_connection,
            org_id=g.org_id,
            author_id=None,
            object_value=object_value,
            old_object=old_object,
        )

        response = prepare_resource(
            ResourceModel(main_connection).get(
                id=resource['id'],
                org_id=g.org_id,
                service=service,
                fields=[
                    'external_id',
                    'service',
                    'relations.*',
                    'relations.user.*',
                    'relations.department.*',
                    'relations.group.*',
                ],
            )
        )['relations']

        return json_response(response)


class ResourceBulkUpdateRelationsView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.write_resources])
    @requires(org_id=True, user=False)
    @uses_schema(RELATIONS_UPDATE_SCHEMA)
    def post(self, meta_connection, main_connection, data, resource_id):
        """
        Балковое обновление.
        Если передан несуществующий resource_id, то он будет создан, а переданные relations в операции added
        свяжутся с этим resource_id.

        ---
        tags:
          - Связи ресурсов
        parameters:
          - in: path
            name: resource_id
            required: true
            type: string
          - in: query
            name: service
            type: string
            description: ресурсы какого сервиса выводить (нужен грант read_service_resources)
          - in: body
            name: body
        responses:
          200:
            description: Операция выполнена успешно
        """
        service = _get_service(request)
        resource = ResourceModel(main_connection).get(
            external_id=resource_id,
            service=service,
            org_id=g.org_id,
            fields=[
                '*',
                'relations.*',
            ],
        )
        operations = (data or {}).get('operations', [])

        # запрет на создание ресурса с уволенными сотрудниками
        for item in operations:
            item = item["value"]
            if item.get("operation_type") != "add":
                continue
            if item['object_type'] != 'user':
                continue
            uid = item['object_id']
            dismiss = UserModel(main_connection).count({
                'id': uid,
                'is_dismissed': True,
                'org_id': g.org_id,
            })
            if dismiss:
                return json_error_user_dismissed(uid)

        bulk_operation = {}
        for operation_type in OPERATIONS_TYPES:
            bulk_operation[operation_type] = []
        for operation in operations:
            item = operation.get('value')
            operation_type = operation.get('operation_type')

            if not check_membership(main_connection, item):
                raise IsNotMember(type=item['object_type'], id=item['object_id'])
            bulk_operation[operation_type].append(item)
        add_relations = bulk_operation['add']

        action_func = action_resource_modify
        if not resource:
            resource = ResourceModel(main_connection).create(
                external_id=resource_id,
                org_id=g.org_id,
                service=service,
            )
            action_func = action_resource_add
            old_object = None
        else:
            old_object = resource

        ResourceModel(main_connection).add_relations(
            resource_id=resource['id'],
            org_id=resource['org_id'],
            relations=convert_relations(add_relations)
        )
        delete_relations = bulk_operation['delete'] + bulk_operation['remove']  # todo: use only remove
        ResourceModel(main_connection).delete_relations(
            resource_id=resource['id'],
            org_id=resource['org_id'],
            relations=convert_relations(delete_relations)
        )

        object_value = ResourceModel(main_connection).get(
            id=resource['id'],
            org_id=resource['org_id'],
            service=service,
            fields=[
                '*',
                'relations.*',
            ],
        )
        action_func(
            main_connection,
            org_id=g.org_id,
            author_id=None,
            object_value=object_value,
            old_object=old_object,
        )

        resource = ResourceModel(main_connection).get(
            id=resource['id'],
            org_id=resource['org_id'],
            service=service,
            fields=[
                '*',
                'relations.*',
                'relations.user.*',
                'relations.department.*',
                'relations.group.*',
            ],
        )

        return json_response(
            prepare_resource(resource)['relations']
        )


class ResourceCountView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.read_resources, scope.write_resources])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Отдаёт число ресурсов, привязанных к заданному сервису.

        Если сервис не задан, то возвращает 422 ошибку

        Пример выдачи:


            {
                "count": 100500
            }

        Фильтрация по сервису делается с помощью GET параметра:

            /resources/count?service=metrika

        ---
        tags:
          - Ресурсы
        parameters:
            - in: query
              name: service
              type: string
              description: ресурсы какого сервиса выводить
        responses:
          200:
            description: Список ресурсов
        """
        service = request.args.get('service')
        if not service:
            raise RequiredFieldsMissed(['service'])
        count = ResourceModel(main_connection) \
            .filter(service=service, org_id=g.org_id,) \
            .count()
        return json_response({'count': count})
