# coding: utf-8
from datetime import datetime, timedelta

from copy import deepcopy
from flask import (
    g,
    request,
)
from dateutil.relativedelta import relativedelta

from intranet.yandex_directory.src.yandex_directory.core.utils.services import (
    x_host_for_service,
    try_replace_url_host,
)
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log

from collections import defaultdict
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_scopes,
    no_permission_required,
    permission_required,
    internal,
    requires,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.core.sms.utils import send_sms_to_admins
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_invalid_value,
    json_error_forbidden,
    utcnow,
    check_permissions,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import ALL_FIELDS
from intranet.yandex_directory.src.yandex_directory.common.exceptions import ServiceNotFound
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import (
    send_email_to_admins,
    send_email_to_all_async
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_fields,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    enable_service,
    disable_service,
    TRACKER_SERVICE_SLUG,
    WIKI_SERVICE_SLUG,
    reason,
    get_service_field_by_tld,
    trial_status,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    ServiceModel,
    ServicesLinksModel,
    OrganizationServiceModel,
    OrganizationModel,
    DomainModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_service_set_ready,
    action_service_responsible_change,
)
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema_for_get,
    uses_schema,
    uses_out_schema,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    global_permissions,
    service_permissions,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    DomainNotFound,
    UserNotFoundInOrganizationError,
)

from intranet.yandex_directory.src.yandex_directory.connect_services.idm import get_service
from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type
from intranet.yandex_directory.src.settings import TRIAL_SHUTDOWN_DATE_FOR_CLOUD_ORGANIZATIONS

SERVICE_GET_SCHEMA = {
    'title': 'Get service',
    'type': 'object',
    'properties': {
        'tld': schemas.STRING,
        'language': schemas.STRING,
    },
    'required': ['tld', 'language'],
    'additionalProperties': True
}

SERVICE_READY_OUT_SCHEMA = {
    'title': 'Get service ready',
    'type': 'object',
    'properties': {
        'ready': schemas.BOOLEAN,
        'enabled': schemas.BOOLEAN,
        'slug': schemas.STRING,
    },
}

ORGANIZATION_SERVICE_CREATE_SCHEMA = {
    'title': 'Add service',
    'type': 'object',
    'properties': {
        'slug': schemas.STRING
    }
}

ACTION_SERVICE_SCHEMA = {
    'type': ['object', 'null'],
    'title': 'Enable/disable service',
    'properties': {
        'drop_licenses': schemas.BOOLEAN,
    },
    'additionalProperties': False
}

RESPONSIBLE_SERVICE_SCHEMA = {
    'type': 'object',
    'title': 'Change service responsible',
    'properties': {
        'responsible_id': schemas.INTEGER,
    },
    'required': ['responsible_id', ],
    'additionalProperties': False
}


def is_service_available(meta_connection, main_connection, org_id, service_slug):
    """
    Проверяем доступен ли сервис для организации
    :param meta_connection:
    :param main_connection:
    :param org_id:
    :param service_slug:
    :return:
    """

    # пока все сервисы доступны для организации
    return True


def get_services_data(meta_connection,
                      main_connection,
                      tld,
                      language,
                      org_id):
    """Возвращает словарь с данными про сервисы известные Директории.

    Результатом является такой словарь из словарей:

    {
        'slug': {
            'url': url,
            'icon': icon,
            'name': name,
            'show_in_header': True,
            'priority': 10,
        }
    }

    Приоритет используется для того, чтобы сортировать сервисы при отрисовке
    в шапке. Чем меньше число, тем ближе к началу списка должен стоять сервис.
    """


    fallback_language = OrganizationModel(main_connection).get(org_id, ['language'])['language']

    services = ServiceModel(meta_connection).find(fields=ALL_FIELDS)
    data = defaultdict(dict)
    for service in services:
        slug = service['slug']
        show_in_header = service['show_in_header']

        # выбираем настройки для tld
        url = get_service_field_by_tld(service, tld, 'url')
        icon = get_service_field_by_tld(service, tld, 'icon')
        if not url or not icon:
            if show_in_header:
                # Если данные про этот сервис должны быть показаны в шапке,
                # а данных нет, то считаем это ошибкой
                with log.fields(slug=slug, tld=tld):
                    log.error('No service data for TLD')
            continue

        # выбираем настройки для языка
        language_data = service['data_by_language']
        name = language_data.get(language, {}).get('name') or language_data.get(fallback_language, {}).get('name')
        if not name:
            if show_in_header:
                # Если данные про этот сервис должны быть показаны в шапке,
                # а данных нет, то считаем это ошибкой
                with log.fields(slug=slug, language=language, fallback_language=fallback_language):
                    log.error('No data for language')
            continue

        data[slug] = {
            'url': url,
            'icon': icon,
            'name': name,
            'show_in_header': show_in_header,
            'in_new_tab':  service['in_new_tab'],
            'priority': service['priority'],
            'id': service['id'],
            'available': is_service_available(meta_connection, main_connection, org_id, slug)
        }

    return data


class ServicesListView(View):

    @internal
    @scopes_required([scope.read_services])
    @no_permission_required
    @uses_schema_for_get(SERVICE_GET_SCHEMA)
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Информация о сервисах.

        Пример ответа:

            {
                "wiki": {
                    "name": "Вики",
                    "url": "https://wiki.yandex.ru/",
                    "icon": "https://…",
                    "available": "True|False", // Доступен для организации
                }
            }

        ---
        tags:
          - Внутренние ручки
        parameters:
          - in: query
            name: tld
            required: true
            type: string
            description: tld для ссылок на сервис и иконок
          - in: query
            name: language
            required: true
            type: string
            description: язык для названий сервисов
        responses:
          200:
            description: Словарь с информацией о сервиса.
          422:
            description: Не указаны необходимые праметры.
        """
        tld = request.args.get('tld')
        language = request.args.get('language')
        data = get_services_data(
            meta_connection,
            main_connection,
            tld,
            language,
            g.org_id,
        )
        data = dict(
            (slug, only_fields(service, 'url', 'icon', 'name', 'available', 'in_new_tab'))
             for slug, service
             in list(data.items())
        )
        for slug, service in list(data.items()):
            host = x_host_for_service(slug)
            if host:
                service['url'] = try_replace_url_host(service['url'], host)
        return json_response(data)


class OrganizationServiceDetailView(View):

    @internal
    @no_permission_required
    @scopes_required([scope.write_services])
    @uses_schema(RESPONSIBLE_SERVICE_SCHEMA)
    @requires(org_id=True, user=True)
    def patch(self, meta_connection, main_connection, data, service_slug):
        """
        Изменить ответственного у сервиса

        Возможные коды ошибок:
        * **service_is_not_enabled** - сервис не включен в организации
        * **user_not_in_organization** - пользователь не состоит в организации

        ---
        tags:
          - Сервисы
        parameters:
          - in: path
            name: service_slug
            required: true
            type: string
            description: slug сервиса
          - in: body
            name: body
        responses:
          200:
            description: Ответственный изменен
          422:
            description: Ошибка валидации
        """
        check_permissions(
            meta_connection=meta_connection,
            main_connection=main_connection,
            permissions=[service_permissions.update_service_data],
            object_type=service_slug,
        )
        responsible_id = data['responsible_id']
        service = OrganizationServiceModel(main_connection).get_by_slug(
            org_id=g.org_id, service_slug=service_slug,
            fields=['id', 'responsible_id'],
        )
        user = UserModel(main_connection).get(
            user_id=responsible_id,
            org_id=g.org_id,
            fields=['id'],
        )
        if not user:
            raise UserNotFoundInOrganizationError()

        try:
            connect_service = get_service(service_slug)
            connect_service.change_responsible(
                main_connection=main_connection,
                org_id=g.org_id,
                author_id=g.user.passport_uid,
                responsible_id=user['id'],
            )
        except ServiceNotFound:
            log.info(
                'IDM service for {} is not implemented'.format(service_slug)
            )

        OrganizationServiceModel(main_connection).update_one(
            id=service['id'],
            update_data={'responsible_id': user['id']}
        )

        service_new = deepcopy(service)
        service_new['responsible_id'] = user['id']
        action_service_responsible_change(
            main_connection,
            g.org_id,
            None,
            object_value=service_new,
            old_object=service,
        )
        return json_response(service_new, status_code=200)


class OrganizationServiceActionView(View):

    @internal
    @uses_schema(ACTION_SERVICE_SCHEMA)
    @no_permission_required
    @scopes_required([scope.write_services])
    @requires(org_id=True, user=False)
    def post(self, meta_connection, main_connection, data, service_slug, action):
        """
        Включить/выключить сервис в организации.

        Action может принимать значения enable/disable.

        Возможные коды ошибок:
        * **unknown_service** - сервис не найден

        ---
        tags:
          - Сервисы
        parameters:
          - in: path
            name: service_slug
            required: true
            type: string
            description: slug включаемого или отключаемого сервиса
          - in: path
            name: action
            required: true
            type: string
            description: enable - включить сервис, disable - отключить сервис
          - in: body
            name: body
        responses:
          201:
            description: Включен/выключен сервис
          422:
            description: Какая-то ошибка валидации
        """
        service = ServiceModel(meta_connection).get_by_slug(service_slug)
        if service:
            is_paid_by_license = service['paid_by_license']
        else:
            raise ServiceNotFound()

        is_partner_organization = OrganizationModel(main_connection).is_partner_organization(g.org_id)

        if is_partner_organization and is_paid_by_license:
            return json_error_forbidden()

        is_cloud_organization = OrganizationModel(main_connection).is_cloud_organization(g.org_id)

        action = action.lower()
        params = [
            meta_connection,
            main_connection,
            g.org_id,
            service_slug,
            g.user.passport_uid,
        ]
        if action == 'enable':
            if is_cloud_organization and service_slug in ('tracker', 'wiki'):
                permissions_to_check = [global_permissions.manage_cloud_services_enable]
            else:
                permissions_to_check = [global_permissions.manage_services]

            check_permissions(
                meta_connection,
                main_connection,
                permissions_to_check,
            )
            func = enable_service
            if data:
                params.append(data.get('drop_licenses'))
        elif action == 'disable':
            check_permissions(
                meta_connection,
                main_connection,
                [global_permissions.manage_services],
            )
            func = disable_service
            # надо указать причину отключения
            # при отключении через ручку считаем что отключено пользователем
            params.insert(4, reason.disabled_by_user)
        else:
            return json_error_invalid_value('action')

        try:
            service = func(*params)
        except ServiceNotFound:
            return json_error_invalid_value('service_slug')

        return json_response(service, status_code=201)


class OrganizationServiceReadyView(View):

    @internal
    @no_permission_required
    @no_scopes
    @uses_out_schema(SERVICE_READY_OUT_SCHEMA)
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, service_slug):
        """
        Узнать статус готовности сервиса.

        Организация определяется заголовком X-Org-ID.

        ---
        tags:
          - Сервисы
        parameters:
          - in: path
            name: service_slug
            required: true
            type: string
            description: slug сервиса
        responses:
          200:
            description: Действие произведено успешно
        """
        service = ServiceModel(meta_connection).get_by_slug(service_slug)
        if not service:
            return json_error_invalid_value('service_slug')

        org_service = OrganizationServiceModel(main_connection).find(
            filter_data={
                'org_id': g.org_id,
                'service_id': service['id'],
            },
            fields=[
                'ready', 'enabled',
            ],
            one=True,
        )
        if not org_service:
            raise ServiceNotFound()
        data = {
            'slug': service_slug,
            'ready': org_service['ready'],
            'enabled': org_service['enabled'],
        }
        return json_response(data)

    @internal
    @no_permission_required
    # скоупы не нужны, поскольку сервис может проставить ready только для себя
    @no_scopes
    @requires(org_id=True, user=False)
    def post(self, meta_connection, main_connection, service_slug):
        """
        Пометить, что сервис готов обслуживать текущую организацию.

        Организация определяется заголовком X-Org-ID.

        ---
        tags:
          - Сервисы
        parameters:
          - in: path
            name: service_slug
            required: true
            type: string
            description: slug сервиса
        responses:
          200:
            description: Действие произведено успешно
        """
        if service_slug != g.service.identity:
            return json_error_forbidden()

        service = ServiceModel(meta_connection).get_by_slug(service_slug)
        if not service:
            return json_error_invalid_value('service_slug')

        # Проверим, что сервис ещё не зарепортил о своей готовности к работе
        org_service = OrganizationServiceModel(main_connection).find(
            filter_data={
                'org_id': g.org_id,
                'service_id': service['id'],
            },
            fields=['ready', 'trial_expires'],
            one=True,
        )

        # Если сервис уже "готов", то делать ничего не надо.
        # Иначе будут дубликаты писем админам:
        # https://st.yandex-team.ru/DIR-3654
        if not org_service['ready']:
            update_data = {
                'ready': True,
                'ready_at': utcnow(),
            }

            if service['paid_by_license'] and not org_service['trial_expires']:
                trial_expires = (update_data['ready_at'] + relativedelta(months=service['trial_period_months'])).date()

                if service_slug == 'tracker' and datetime.utcnow() > TRIAL_SHUTDOWN_DATE_FOR_CLOUD_ORGANIZATIONS:
                    trial_expires = update_data['ready_at'].date() - timedelta(days=1)
                    update_data['trial_status'] = trial_status.expired

                update_data['trial_expires'] = trial_expires

            OrganizationServiceModel(main_connection).update(
                update_data=update_data,
                filter_data={
                    'org_id': g.org_id,
                    'service_id': service['id'],
                }
            )

            action_service_set_ready(
                main_connection,
                g.org_id,
                None,
                object_value=service,
                old_object={},
            )

            if service_slug == TRACKER_SERVICE_SLUG:
                self.on_tracker_enable(meta_connection, main_connection, service['id'])

        return json_response(service, status_code=200)

    def on_tracker_enable(self, meta_connection, main_connection, service_id):
        # для трекера после активации отправляем письма админам
        # https://st.yandex-team.ru/DIR-3404

        org_service = OrganizationServiceModel(main_connection).find(
            filter_data={
                'org_id': g.org_id,
                'service_id': service_id,
                'ready': True,
            },
            one=True,
            limit=1,
        )
        first_activation = org_service['trial_expires'] >= utcnow().date()
        trial_days = (org_service['trial_expires'] - utcnow().date()).days + 1
        trial_days = max(trial_days, 0)

        # отправляем письма админам об активации трекера
        if first_activation:
            # активация в триале
            send_params = dict(
                campaign_slug=app.config['SENDER_CAMPAIGN_SLUG']['TRACKER_ACTIVATED_EMAIL'],
                organization_name=OrganizationModel(main_connection).get(g.org_id)['name'],
                trial_days=trial_days,
            )
        else:
            # активация по лицензии
            send_params = dict(
                campaign_slug=app.config['SENDER_CAMPAIGN_SLUG']['TRACKER_SUBSCRIPTION_ACTIVATED_EMAIL'],
                organization_name=OrganizationModel(main_connection).get(g.org_id)['name'],
            )

        send_email_to_admins(
            meta_connection,
            main_connection,
            g.org_id,
            **send_params
        )

        # отправляем  СМС админам если с момента подключения прошло N секунд
        enabled_at = org_service['enabled_at']
        delta = (utcnow() - enabled_at).total_seconds()
        if delta > app.config['SMS_NOTIFY_TRACKER_HOLD_TIME']:
            # от tld зависит ссылка на сервис в тексте СМС
            tld = OrganizationModel(main_connection).get_tld_for_email(g.org_id)
            # 'tracker_enabled' - ключ перевода в танкере
            # https://tanker.yandex-team.ru/?project=directory&branch=master&keyset=Sms
            send_sms_to_admins(meta_connection, main_connection, g.org_id, 'tracker_enabled', tld=tld)
