# -*- coding: utf-8 -*-
import datetime

from flask import (
    g,
    request,
    Response,
)

from intranet.yandex_directory.src import settings
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_shard,
    get_shard_numbers,
    get_main_connection,
)
from intranet.yandex_directory.src.yandex_directory.common.models.base import Values
from intranet.yandex_directory.src.yandex_directory.core.sms.tasks import (
    sms_metrika_counter_request_confirmed,
    sms_metrika_counter_request_denied,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import create_portal_user
from intranet.yandex_directory.src.yandex_directory.core.task_queue.exceptions import DuplicatedTask
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.common.billing.invoice_client import (
    BillingInvoiceClient,
    BillingInvoiceApiException,
)

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    permission_required,
    no_permission_required,
    requires,
    internal,
    scopes_required,
    no_scopes,
)
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope
from intranet.yandex_directory.src.yandex_directory.common.schemas import (
    STRING,
    INTEGER,
    BILLING_PERSON_TYPE,
    BILLING_CURRENCY_TYPE,
    BILLING_SUBSCRIPTION_PLAN_TYPE,
    NUMBER,
    NUMBER_OR_NULL,
    DATE_OR_NULL,
    DATE,
    STRING_OR_NULL,
    BOOLEAN,
    LIST_OF_STRINGS,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    prepare_resource_for_licenses,
    check_objects_exists,
    prepare_requests_for_licenses,
    add_user_data_from_blackbox,
    get_object_type,
    only_ids,
    unfreeze_or_copy,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.subscription import format_promocode
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import (
    send_email_to_admins,
    send_request_access_email_to_responsible_for_service,
    send_confirm_access_email_to_user,
    send_deny_access_email_to_user,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    json_error_invalid_value,
    json_error_user_dismissed,
    utcnow,
    format_date,
    get_object_or_404,
    build_list_response,
    split_by_comma,
    check_permissions,
    json_error,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    OrganizationModel,
    ResourceModel,
    UserModel,
    OrganizationServiceModel,
    ServiceModel,
    OrganizationPromocodeModel,
    RequestedUserServiceLicenses,
    DepartmentModel,
    UserMetaModel,
    GroupModel,
    UserServiceLicenses,
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    organization_type,
    OrganizationBillingInfoModel,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    SubscriptionIncompatibleContainerType,
    SubscriptionLimitExceeded,
    UnsupportedService,
    ParameterNotCorrect,
    ResourceNotFound,
    InvalidValue,
    AuthorizationError,
    UserNotFoundInBlackbox,
    BillingInvoiceServiceInteractionError,
    UserIsNotMemberOfThisOrganization,
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    get_price_and_product_id_for_service,
    promocode_type,
)
from intranet.yandex_directory.src.yandex_directory.core.models.resource import (
    RESOURCE_OBJECT_TYPES,
    ResourceRelationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import (
    check_has_debt,
    get_promocode_by_id,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    global_permissions,
    user_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.swagger import (
    uses_schema,
    uses_out_schema,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_resource_relation_add,
    action_service_license_change,
    action_resource_license_request,
    action_resource_license_request_action,
)
from intranet.yandex_directory.src.yandex_directory.core.models.service import update_service_licenses
from intranet.yandex_directory.src.yandex_directory.connect_services.partner_disk.tasks import (
    DeleteSpacePartnerDiskTask,
    AddSpacePartnerDiskTask,
)
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    OrganizationHasDebt,
    BillingUnknownError,
    BillingClientIdMismatch,
    BillingClientIdNotFound,
    NoActivePerson,
    OrganizationIsWithoutContract,
)
from intranet.yandex_directory.src.yandex_directory.mds import MdsS3ApiClient

ORGANIZATION_PAY_SCHEMA = {
    'title': 'OrganizationPay',
    'type': 'object',
    'properties': {
        'amount': NUMBER,
        'return_path': STRING,
    },
    'required': ['amount'],
    'additionalProperties': False
}

ORGANIZATION_TRUST_PAY_SCHEMA = {
    'title': 'OrganizationTrustPay',
    'type': 'object',
    'properties': {
        'amount': NUMBER,
    },
    'required': ['amount'],
    'additionalProperties': False
}

ACTIVATE_PROMOCODE_SCHEMA = {
    'title': 'OrganizationPromocodeActivateView',
    'type': 'object',
    'properties': {
        'id': STRING,
    },
    'required': ['id'],
    'additionalProperties': False,
}

ACTIVATE_PROMOCODE_OUT_SCHEMA = {
    'title': 'OrganizationPromocodeActivateView',
    'type': 'object',
    'properties': {
        'id': STRING,
        'expires': DATE,
        'description': STRING,
    },
    'required': ['id', 'expires'],
    'additionalProperties': False,
}

ORGANIZATION_SUBSCRIPTION_INFO_OUT_SCHEMA = {
    'title': 'OrganizationSubscriptionInfo',
    'type': 'object',
    'properties': {
        'balance': NUMBER_OR_NULL,
        'person_type': STRING_OR_NULL,
        'subscription_plan': BILLING_SUBSCRIPTION_PLAN_TYPE,
        'debt_start_date': DATE_OR_NULL,
        'has_contract': BOOLEAN,
        'next_act_date': DATE_OR_NULL,
        'payment_due_date': DATE_OR_NULL,
        'subscription_plan_change_requirements': {
            'why_not_changeable': LIST_OF_STRINGS,
        },
        'days_until_disable_by_debt': NUMBER_OR_NULL,
    },
    'additionalProperties': False
}


PROMOCODE = {
    'title': 'Promocode',
    'type': ['object', 'null'],
    'properties': {
        'id': STRING,
        'expires': DATE,
        'description': STRING,
    },
    'additionalProperties': False,
}

SERVICE_PRICING = {
    'type': 'object',
    'properties': {
        'total': NUMBER,
        'total_with_discount': NUMBER_OR_NULL,
        'per_user': NUMBER,
        'per_user_with_discount': NUMBER_OR_NULL,
        'users_count': INTEGER,
    },
}

ORGANIZATION_PRICING_OUT_SCHEMA = {
    'title': 'OrganizationPricingView',
    'type': 'object',
    'properties': {
        'currency': BILLING_CURRENCY_TYPE,
        'total': NUMBER,
        'total_with_discount': NUMBER_OR_NULL,
        'services': {
            'type': 'object',
            'patternProperties': {
                '^.*$': SERVICE_PRICING,
            },
        },
        'promocode': PROMOCODE,
        'connect_upgrade': SERVICE_PRICING,
    },
    'required': ['currency', 'services', 'connect_upgrade'],
    'additionalProperties': False
}


ORGANIZATION_CHANGE_SUBSCRIPTION_PLAN_SCHEMA = {
    'title': 'ChangeOrganizationsSubscriptionPlan',
    'type': 'object',
    'oneOf': [
        {
            'properties': {
                'person_type': BILLING_PERSON_TYPE,
                'subscription_plan': {'enum': ['paid']},
                'first_name': STRING,
                'last_name': STRING,
                'middle_name': STRING,
                'email': STRING,
                'phone': STRING,
            },
            'required': [
                'person_type',
                'subscription_plan',
                'first_name',
                'last_name',
                'middle_name',
                'email',
                'phone',
            ],
            'additionalProperties': False,
        },
        {
            'properties': {
                'person_type': BILLING_PERSON_TYPE,
                'subscription_plan': {'enum': ['paid']},
                'long_name': STRING,
                'postal_code': STRING,
                'postal_address': STRING,
                'legal_address': STRING,
                'inn': STRING,
                'kpp': STRING_OR_NULL,
                'bik': STRING,
                'account': STRING,
                'phone': STRING,
                'email': STRING,
                'contract': BOOLEAN,
            },
            'required': [
                'person_type',
                'subscription_plan',
                'long_name',
                'postal_code',
                'postal_address',
                'legal_address',
                'inn',
                'bik',
                'account',
                'phone',
                'email',
            ],
            'additionalProperties': False,
        },
        {
            'properties': {
                'subscription_plan': {'enum': ['free', 'paid']},
            },
            'required': [
                'subscription_plan',
            ],
            'additionalProperties': False,
        },
        {
            'properties': {
                'subscription_plan': {'enum': ['paid']},
                'person_id': INTEGER,
                'person_type': BILLING_PERSON_TYPE,
            },
            'required': [
                'subscription_plan',
                'person_id',
                'person_type',
            ],
            'additionalProperties': True,
        },
    ],
}

ORGANIZATION_CREATE_CONTRACT_INFO_SCHEMA = {
    'title': 'OrganizationCreateContractInfoView',
    'type': 'object',
    'oneOf': [
        {
            'properties': {
                'person_type': BILLING_PERSON_TYPE,
                'first_name': STRING,
                'last_name': STRING,
                'middle_name': STRING,
                'email': STRING,
                'phone': STRING,
            },
            'required': [
                'person_type',
                'first_name',
                'last_name',
                'middle_name',
                'email',
                'phone',
            ],
            'additionalProperties': False,
        },
        {
            'properties': {
                'person_type': BILLING_PERSON_TYPE,
                'long_name': STRING,
                'postal_code': STRING,
                'postal_address': STRING,
                'legal_address': STRING,
                'inn': STRING,
                'kpp': STRING_OR_NULL,
                'bik': STRING,
                'account': STRING,
                'phone': STRING,
                'email': STRING,
            },
            'required': [
                'person_type',
                'long_name',
                'postal_code',
                'postal_address',
                'legal_address',
                'inn',
                'bik',
                'account',
                'phone',
                'email',
            ],
            'additionalProperties': False,
        },
        {
            'properties': {
                'person_id': INTEGER,
                'person_type': BILLING_PERSON_TYPE,
            },
            'required': [
                'person_id',
                'person_type',
            ],
            'additionalProperties': True,
        },
    ],
}


OBJECT_SCHEMA = {
    'title': 'LicenseObject',
    'type': 'object',
    'properties': {
        'type': {
            'enum': RESOURCE_OBJECT_TYPES
        },
        'id': {
            'type': 'integer'
        }
    },
    'required': ['type', 'id'],
    'additionalProperties': False
}


LICENSES_SCHEMA = {
    'title': 'LicensesList',
    'type': 'array',
    'items': OBJECT_SCHEMA
}


REQUESTS_SCHEMA = {
    'title': 'RequestsList',
    'type': 'object',
    'properties': {
        'objects': LICENSES_SCHEMA,
        'comment': STRING_OR_NULL,
        'resource_ids': LIST_OF_STRINGS,
    },
    'required': ['objects'],
    'additionalProperties': False
}


OBJECT_TYPE_TO_RELATION_PARAM_MAP = {
    'user': 'user_id',
    'group': 'group_id',
    'department': 'department_id',
}


def convert_licenses_to_relations(licenses, relation_name=True):
    for lic in licenses:
        key = OBJECT_TYPE_TO_RELATION_PARAM_MAP[lic['type']]
        lic[key] = lic['id']
        if relation_name:
            lic['name'] = 'member'
    return licenses


class OrganizationSubscriptionInfoView(View):
    methods = ['get']

    @internal
    @no_permission_required
    @scopes_required([scope.can_view_billing_info])
    @uses_out_schema(ORGANIZATION_SUBSCRIPTION_INFO_OUT_SCHEMA)
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Возвращает информацию о тарифе организации и её статусе в биллинге.

        Пример выдачи:
            {{
                "balance": 1006,
                "person_type": "legal",
                "debt_start_date": "2018-03-26",
                "has_contract": true,
                "next_act_date": "2018-06-01",
                "payment_due_date": "2018-04-25",
                "subscription_plan": "paid",
                "subscription_plan_change_requirements":
                    {{
                        "why_not_changeable": []
                    }},
                "days_until_disable_by_debt": 0,
            }}

        Поля ответа:
        * **balance** - баланс организации
        * **debt_start_date** - дата первого неоплаченного акта, дата или None
        * **has_contract** - у организации есть договор True/False
        * **next_act_date** - дата выставления следующего акта, дата или None
        * **payment_due_date** - дата до которой необходимо погасить задолженность, дата или None
        * **subscription_plan** - тарифный план организации paid или free
        * **subscription_plan_change_requirements** - причины почему нельзя перейти на платный режим,
        сейчас только одна - organization_has_debt
        * **days_until_disable_by_debt** - сколько дней осталось до выклюения платных услуг за долги,
        количество дней, 0 - если услуги отключены, None - если нет долга

        ---
        tags:
          - Организация
        responses:
          200:
            description: Success
        """

        organization = OrganizationModel(main_connection).get(
            g.org_id,
            fields=[
                'billing_info.balance',
                'billing_info.client_id',
                'billing_info.first_debt_act_date',
                'billing_info.person_type',
                'subscription_plan',
            ],
        )

        balance = None
        person_type = None
        has_contract = False
        debt_start_date = None
        next_act_date = None
        payment_due_date = None
        days_until_disable_by_debt = None
        why_not_changeable = []

        is_free_plan = organization['subscription_plan'] == 'free'

        billing_info = organization.get('billing_info')
        if billing_info:
            current_balance_info = None
            try:
                current_balance_info = app.billing_client.get_balance_info(
                    billing_info['client_id']
                )
            except BillingClientIdNotFound as error:
                log.warning(error)
                shard = get_shard(meta_connection, g.org_id)
                with get_main_connection(shard=shard, for_write=True) as main_connection:
                    OrganizationBillingInfoModel(main_connection).delete(
                        {'org_id': g.org_id}
                    )
                    OrganizationModel(main_connection).increment_revision(g.org_id)
            if current_balance_info is not None:
                balance = current_balance_info['balance']
                person_type = billing_info['person_type']
                has_contract = True
                if billing_info['first_debt_act_date']:
                    debt_start_date = format_date(billing_info['first_debt_act_date'])
                    payment_due_date = billing_info['first_debt_act_date'] + \
                                    datetime.timedelta(days=app.config['BILLING_PAYMENT_TERM'])
                    days_until_disable_by_debt = (payment_due_date - utcnow().date()).days
                    days_until_disable_by_debt = max(days_until_disable_by_debt + 1, 0)
                    payment_due_date = format_date(payment_due_date)
                    if is_free_plan and check_has_debt(
                            first_debt_act_date=billing_info['first_debt_act_date'],
                            balance=billing_info['balance']
                    )[0]:
                        why_not_changeable.append('organization_has_debt')
                next_act_date = format_date(get_next_act_date())

        return json_response({
            'balance': balance,
            'person_type': person_type,
            'subscription_plan': organization['subscription_plan'],
            'debt_start_date': debt_start_date,
            'has_contract': has_contract,
            'next_act_date': next_act_date,
            'payment_due_date': payment_due_date,
            'subscription_plan_change_requirements': {
                'why_not_changeable': why_not_changeable,
            },
            'days_until_disable_by_debt': days_until_disable_by_debt,
        })


class OrganizationSubscriptionPricingView(View):
    methods = ['get']

    @internal
    @no_permission_required
    @scopes_required([scope.can_view_billing_info])
    @uses_out_schema(ORGANIZATION_PRICING_OUT_SCHEMA)
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Возвращает информацию о стоимости платного тарифа для организации: общая цена, валюта, стоимость сервисов
        """
        return json_response(
            OrganizationModel(main_connection).get_pricing_info(
                org_id=g.org_id,
            ),
        )


class OrganizationSubscriptionAdminPersonsView(View):
    methods = ['get']

    @internal
    @no_permission_required
    @scopes_required([scope.can_view_billing_info])
    @requires(org_id=True, user=True)
    def get(self, meta_connection, main_connection):
        """
        Возвращает информацию о наличии плательщиков у пользователя
        """
        user_id = g.user.passport_uid
        return json_response(
            app.billing_client.get_user_persons(user_id)
        )


class OrganizationPayView(View):
    methods = ['post']

    @internal
    @uses_schema(ORGANIZATION_PAY_SCHEMA)
    @requires(org_id=True, user=True)
    @scopes_required([scope.can_pay])
    @permission_required(permissions=[global_permissions.can_pay])
    def post(self, meta_connection, main_connection, data):
        """
        Внесение денег на счет организации.
        ---
        tags:
          - Организация
        parameters:
          - in: body
            name: body
        responses:
          200:
            description: Success
          422:
            description: Organization is without contract
        """
        log_fields = {
            'org_id': g.org_id,
            'author_id': g.user.passport_uid,
        }
        log_fields.update(data)
        with log.name_and_fields('billing_pay_view', **log_fields):
            payment_url = OrganizationModel(main_connection).get_url_for_paying(
                org_id=g.org_id,
                author_id=g.user.passport_uid,
                amount=data['amount'],
                return_path=data.get('return_path'),
            )
            return json_response({'payment_url': payment_url})


class OrganizationTrustPayView(View):
    methods = ['post']

    @internal
    @uses_schema(ORGANIZATION_TRUST_PAY_SCHEMA)
    @requires(org_id=True, user=True)
    @scopes_required([scope.can_pay])
    @permission_required(permissions=[global_permissions.can_pay])
    def post(self, meta_connection, main_connection, data):
        """
        Формирование ссылки на форму оплаты Траста https://wiki.yandex-team.ru/trust/
        ---
        tags:
          - Организация
        parameters:
          - in: body
            name: body
        responses:
          200:
            description: Success
          422:
            description: Organization is without contract
        """
        log_fields = {
            'org_id': g.org_id,
            'author_id': g.user.passport_uid,
        }
        log_fields.update(data)
        with log.name_and_fields('trust_pay_view', **log_fields):
            try:
                payment_url = OrganizationModel(main_connection).get_url_for_paying_trust(
                    org_id=g.org_id,
                    author_id=g.user.passport_uid,
                    amount=data['amount'],
                )
            except NoActivePerson:
                return json_error(
                    status_code=NoActivePerson.status_code,
                    error_code=NoActivePerson.code,
                    error_message=NoActivePerson.message,
                )
            return json_response({'payment_url': payment_url})


class OrganizationInvoiceView(View):
    methods = ['post']

    @internal
    @uses_schema(ORGANIZATION_PAY_SCHEMA)
    @requires(org_id=True, user=True)
    @scopes_required([scope.can_pay])
    @permission_required(permissions=[global_permissions.can_pay])
    def post(self, meta_connection, main_connection, data):
        """
        Генерация pdf инвойса и возврат его.
        ---
        tags:
          - Организация
        parameters:
          - in: body
            name: body
        responses:
          200:
            description: Success
          422:
            description: Organization is without contract
        """
        log_fields = {
            'org_id': g.org_id,
            'author_id': g.user.passport_uid,
        }
        log_fields.update(data)
        with log.name_and_fields('billing_invoice_view', **log_fields):
            invoice_id = OrganizationModel(main_connection).create_invoice(
                org_id=g.org_id,
                author_id=g.user.passport_uid,
                amount=data['amount'],
            )
            invoice_data = self._get_invoice_data(invoice_id)

            pdf_invoice = self._get_pdf_invoice(invoice_data['mds_link'])
            return Response(
                pdf_invoice,
                content_type='application/pdf; charset=utf-8',
                headers={'Content-Disposition': 'attachment;filename=invoice.pdf'},
                status=201,
            )

    def _get_pdf_invoice(self, link):
        client = MdsS3ApiClient()
        *_, bucket, invoice_id = link.split('/')
        return client.get_object_content(bucket=bucket, object_id=invoice_id)

    def _get_invoice_data(self, invoice_id):
        """
        Получаем в отдельном апи билинга данные счета
        """
        client = BillingInvoiceClient()
        try:
            return client.get_invoice_data(invoice_id)
        except BillingInvoiceApiException:
            raise BillingInvoiceServiceInteractionError()


class OrganizationChangeSubscriptionPlanView(View):
    methods = ['post']

    @internal
    @uses_schema(ORGANIZATION_CHANGE_SUBSCRIPTION_PLAN_SCHEMA)
    @requires(org_id=True, user=True)
    @scopes_required([scope.can_change_subscription_plan])
    @permission_required(permissions=[global_permissions.change_subscription_plan])
    def post(self, meta_connection, main_connection, data):
        """
        Включение платного режима для организации.
        Набор необходимых полей зависит от типа лица (физическое/юридическое).
        Тип лица указывается в поле person_type и может принимать два значения: natural/legal

        Для физических лиц обязательны поля:
            person_type
            first_name
            last_name
            middle_name
            phone
            email

        Для юридических:
            person_type
            long_name
            postal_code
            postal_address
            legal_address
            inn
            kpp
            bik
            account
            phone
            email

        ---
        tags:
          - Организация
        parameters:
          - in: body
            name: body
        responses:
          200:
            description: Success
          422:
            description: Invalid filed value or missing mandatory field
          503:
            description: Unknown billing error
        """
        log_fields = {
            'org_id': g.org_id,
            'author_id': g.user.passport_uid,
        }
        log_fields.update(data)
        # TODO: partner

        with log.name_and_fields('billing_view', **log_fields):

            if data['subscription_plan'] == 'free':
                return self._disable_paid_mode(main_connection, g.org_id)
            elif data['subscription_plan'] == 'paid' and 'person_type' not in data:
                # если person_type нет в данных, значит пытаются
                # включить платный режим для организации которая уже
                # подписала договор или приняла оферту, то есть у нас есть
                # все биллинговые данные и их не надо заново просить
                result = self._enable_paid_mode_for_initiated_in_billing_organization(
                    main_connection,
                    g.org_id,
                )
                self._send_paid_mode_enable_email_to_admins(meta_connection, main_connection)
                return result
            if data.get('person_id'):
                result = self._enable_paid_mode_for_existing_client_id(
                    main_connection=main_connection,
                    org_id=g.org_id,
                    data=data,
                )
            elif data['person_type'] == 'natural':
                result = self._enable_paid_mode_for_natural_person(
                    main_connection=main_connection,
                    org_id=g.org_id,
                    data=data,
                )
            elif data['person_type'] == 'legal':
                result = self._enable_paid_mode_for_legal_person(
                    main_connection=main_connection,
                    org_id=g.org_id,
                    data=data,
                )
            else:
                result = json_error_invalid_value('person_type')

            self._send_paid_mode_enable_email_to_admins(meta_connection, main_connection)
            return result

    def _send_paid_mode_enable_email_to_admins(self, meta_connection, main_connection):
        """
        Отправляем письма всем админам о включении платного режима
        """
        try:
            send_email_to_admins(
                meta_connection,
                main_connection,
                g.org_id,
                app.config['SENDER_CAMPAIGN_SLUG']['PAID_MODE_ENABLE_EMAIL'],
            )
        except:
            log.trace().error('Error during sending email about enabled paid mode')

    def _disable_paid_mode(self, main_connection, org_id):
        """
        Выключаем платный режим
        """
        log.info('Customer requested a change the subscription plan into a free mode')
        OrganizationModel(main_connection).disable_paid_mode(
            org_id=org_id,
            author_id=g.user.passport_uid,
        )
        return json_response({'subscription_plan': 'free'})

    def _enable_paid_mode_for_initiated_in_billing_organization(self, main_connection, org_id):
        """
        Включаем платный режим для организации, которая уже подписала договор или заключила оферту
        """
        log.info('Customer requested a change the subscription plan into a paid mode for initiated in Billing organziation')
        OrganizationModel(main_connection).enable_paid_mode_for_initiated_in_billing_organization(
            org_id=org_id,
            author_id=g.user.passport_uid,
        )

        return json_response({'subscription_plan': 'paid'})

    def _enable_paid_mode_for_natural_person(self, main_connection, org_id, data):
        """
        Включаем платный режим в организации для физического лица
        """
        log.info('Customer requested a change the subscription plan into a paid mode for natural person')
        OrganizationModel(main_connection).enable_paid_mode_for_natural_person(
            org_id=org_id,
            author_id=g.user.passport_uid,
            first_name=data['first_name'],
            last_name=data['last_name'],
            middle_name=data['middle_name'],
            phone=data['phone'],
            email=data['email'],
        )
        return json_response({'subscription_plan': 'paid'})

    def _enable_paid_mode_for_legal_person(self, main_connection, org_id, data):
        """
        Включаем платный режим в организации для юридического лица
        """
        log.info('Customer requested a change the subscription plan into a paid mode mode for legal person')
        OrganizationModel(main_connection).enable_paid_mode_for_legal_person(
            org_id=org_id,
            author_id=g.user.passport_uid,
            long_name=data['long_name'],
            phone=data['phone'],
            email=data['email'],
            postal_code=data['postal_code'],
            postal_address=data['postal_address'],
            legal_address=data['legal_address'],
            inn=data['inn'],
            kpp=data['kpp'],
            bik=data['bik'],
            account=data['account'],
            contract=data.get('contract', False),
        )
        return json_response({'subscription_plan': 'paid'})

    def _enable_paid_mode_for_existing_client_id(self, main_connection, org_id, data):
        """
        Включаем платный режим в организации для существующего client_id
        """
        log.info('Enabling paid mode for existing client_id and person_id')
        OrganizationModel(main_connection).enable_paid_mode_for_existing_client_id(
            org_id=org_id,
            author_id=g.user.passport_uid,
            person_id=data['person_id'],
            person_type=data['person_type'],
        )
        return json_response({'subscription_plan': 'paid'})


def get_next_act_date(current_date=utcnow().date()):
    """
    Дата выставления следующего акта - первое число следюущего месяца или сегодня, если сегодня первое
    """
    if current_date.day == 1:
        return current_date
    else:
        year, month = divmod(current_date.month, 12)
        return current_date.replace(
            year=current_date.year + year,
            month=month + 1,
            day=1
        )


class OrganizationCreateContractInfoView(View):
    methods = ['post']

    @internal
    @uses_schema(ORGANIZATION_CREATE_CONTRACT_INFO_SCHEMA)
    @requires(org_id=True, user=True)
    @scopes_required([scope.can_change_subscription_plan])
    @permission_required(permissions=[global_permissions.can_pay])
    def post(self, meta_connection, main_connection, data):
        """
        Заведение биллинговой информации для организации для оплаты сервисов по лицензиям.
        Тарифный план остается 'free'.
        Набор необходимых полей зависит от типа лица (физическое/юридическое).
        Тип лица указывается в поле person_type и может принимать два значения: natural/legal

        Для физических лиц обязательны поля:
            person_type
            first_name
            last_name
            middle_name
            phone
            email

        Для юридических:
            person_type
            long_name
            postal_code
            postal_address
            legal_address
            inn
            kpp
            bik
            account
            phone
            email

        ---
        tags:
          - Организация
        parameters:
          - in: body
            name: body
        responses:
          200:
            description: Success
          422:
            description: Invalid filed value or missing mandatory field
          503:
            description: Unknown billing error
        """
        log_fields = {
            'org_id': g.org_id,
            'author_id': g.user.passport_uid,
        }
        log_fields.update(data)
        with log.name_and_fields('billing_view', **log_fields):
            try:
                if data.get('person_id'):
                    return self._create_contract_info_for_existing_person(
                        main_connection=main_connection,
                        org_id=g.org_id,
                        data=data,
                    )
                elif data['person_type'] == 'natural':
                    return self._create_contract_info_for_natural_person(
                        main_connection=main_connection,
                        org_id=g.org_id,
                        data=data,
                    )
                elif data['person_type'] == 'legal':
                    return self._create_contract_info_for_legal_person(
                        main_connection=main_connection,
                        org_id=g.org_id,
                        data=data,
                    )
                else:
                    return json_error_invalid_value('person_type')
            except BillingUnknownError:
                raise BillingClientIdMismatch()


    def _create_contract_info_for_natural_person(self, main_connection, org_id, data):
        """
        Сохраняем биллинговую инфомарцию для физического лица
        """
        log.info('Customer requested a change the subscription plan into a paid mode for natural person')
        OrganizationModel(main_connection).create_contract_info_for_natural_person(
            org_id=org_id,
            author_id=g.user.passport_uid,
            first_name=data['first_name'],
            last_name=data['last_name'],
            middle_name=data['middle_name'],
            phone=data['phone'],
            email=data['email'],
        )
        return json_response(
            data={},
            status_code=200
        )

    def _create_contract_info_for_legal_person(self, main_connection, org_id, data):
        """
        Сохраняем биллинговую инфомарцию для юридического лица
        """
        log.info('Customer requested a change the subscription plan into a paid mode mode for legal person')
        OrganizationModel(main_connection).create_contract_info_for_legal_person(
            org_id=org_id,
            author_id=g.user.passport_uid,
            long_name=data['long_name'],
            phone=data['phone'],
            email=data['email'],
            postal_code=data['postal_code'],
            postal_address=data['postal_address'],
            legal_address=data['legal_address'],
            inn=data['inn'],
            kpp=data['kpp'],
            bik=data['bik'],
            account=data['account'],
        )
        return json_response(
            data={},
            status_code=200
        )

    def _create_contract_info_for_existing_person(self, main_connection, org_id, data):
        """
        Сохраняем биллинговую инфомарцию для существующего плательщика
        """
        OrganizationModel(main_connection).create_contract_info_for_existing_person(
            org_id=org_id,
            author_id=g.user.passport_uid,
            person_id=data['person_id'],
            person_type=data['person_type'],
        )
        return json_response(
            data={},
            status_code=200
        )


class OrganizationDeleteServiceLicensesView(View):
    methods = ['delete']

    @internal
    @permission_required(
        permissions=[global_permissions.manage_licenses, global_permissions.manage_tracker],
        any_permission=True,
    )
    @scopes_required([scope.manage_service_licenses])
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, service_slug, obj_type, obj_id):
        """
        Удалить лицензию на объект

        ---
        tags:
          - Лицензии сервиса
        parameters:
          - in: path
            name: service_slug
            type: string
            required: true
          - in: path
            name: obj_type
            type: string
            required: true
          - in: path
            name: obj_id
            type: string
            required: true
        responses:
          204:
            description: Лицензия удалена
          422:
            description: Сервис без лицензий
          403:
            description: Сервис не включен в организации
          404:
            description: Сервис не найден
        """
        resource_id = OrganizationServiceModel(main_connection).get_licensed_service_resource_id(g.org_id, service_slug)
        data = [{
            'type': obj_type,
            'id': obj_id,
        }]
        relations = convert_licenses_to_relations(data)
        ResourceModel(main_connection).delete_relations(
            resource_id=resource_id,
            org_id=g.org_id,
            relations=relations
        )
        update_service_licenses(main_connection, g.org_id, service_slug, g.user.passport_uid, sync=True)

        return json_response(
            {'message': 'No Content'},
            status_code=204,
        )


class OrganizationChangeServiceLicensesView(View):
    methods = ['get', 'put', 'post']

    @internal
    @no_permission_required
    @scopes_required([scope.read_service_licenses])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, service_slug):
        """
        Список лицензий сервиса

        Пример выдачи:

            [
                {
                    "object": {
                        "email": "web-chib@ya.ru",
                        "name": {
                            "first": {
                                "en": "Gennady",
                                "ru": "Геннадий"
                            },
                            "last": {
                                "en": "Chibisov",
                                "ru": "Чибисов"
                            }
                        },
                        "gender": "male",
                        "id": 66865142,
                        "nickname": "web-chib"
                    },
                    "object_type": "user"
                }
            ]

        ---
        tags:
          - Лицензии сервиса
        parameters:
          - in: path
            name: service_slug
            type: string
            required: true
        responses:
          200:
            description: Список лицензий данного сервиса.
        """
        resource_id = OrganizationServiceModel(main_connection).get_licensed_service_resource_id(g.org_id, service_slug)
        resource = get_object_or_404(
            model_instance=ResourceModel(main_connection),
            external_id=resource_id,
            org_id=g.org_id,
            service=app.config['DIRECTORY_SERVICE_NAME'],
            fields=[
                'external_id',
                'service',
                'relations.*',
                'relations.user.*',
                'relations.department.*',
                'relations.group.*',
            ],
        )
        return json_response(
            prepare_resource_for_licenses(resource),
        )

    def _check_licenses(self, main_connection, meta_connection, service_slug, data, org_id):
        check_license_objects(main_connection, data, org_id)

        # для партнерских организаций можно выдвать лицензии только для пользователей
        org_type = OrganizationModel(main_connection).get_organization_type(org_id)
        if (org_type in organization_type.partner_types or service_slug == 'disk') and \
                check_has_non_users_objects(data):
            raise SubscriptionIncompatibleContainerType()

        billing_info = OrganizationModel(main_connection).get(
            org_id,
            fields=[
                'billing_info.first_debt_act_date',
                'billing_info.balance',
            ],
        )['billing_info']
        if billing_info and check_has_debt(
                first_debt_act_date=billing_info['first_debt_act_date'],
                balance=billing_info['balance']
        )[0]:
            raise OrganizationHasDebt()
        relations = convert_licenses_to_relations(data)
        resource_id = OrganizationServiceModel(main_connection).get_licensed_service_resource_id(org_id, service_slug)

        if service_slug == 'tracker':
            if org_type in organization_type.partner_types:
                service_id = ServiceModel(meta_connection).get_by_slug(service_slug, ['id'])['id']
                user_limit = OrganizationServiceModel(main_connection).get_by_org_service(
                    org_id,
                    service_id,
                    ['user_limit']
                )['user_limit']
                users_count = get_users_count_from_relations(main_connection, g.org_id, relations)
                if users_count > user_limit:
                    raise SubscriptionLimitExceeded()
        return resource_id, relations, billing_info

    @internal
    @permission_required(
        permissions=[global_permissions.manage_licenses, global_permissions.manage_tracker],
        any_permission=True,
    )
    @scopes_required([scope.manage_service_licenses])
    @requires(org_id=True, user=True)
    @uses_schema(OBJECT_SCHEMA)
    def post(self, meta_connection, main_connection, data, service_slug):
        """
        Выдать лицензию на объект

        * **id** - идентификатор сущности
        * **type** - тип сущности

        Пример:

             {{
                "type": "user",
                "id": 66865142
             }}

        ---
        tags:
          - Лицензии сервиса
        parameters:
          - in: path
            name: service_slug
            type: string
            required: true
        responses:
          201:
            description: Лицензия добавлена
          422:
            description: Сервис без лицензий
          403:
            description: Сервис не включен в организации
          404:
            description: Сервис не найден
        """
        data = [unfreeze_or_copy(data)]
        org_id = g.org_id
        cloud_billing_account = bool(request.args.get('cloud_billing_account', 0, int))

        resource_id, relations, billing_info = self._check_licenses(
            main_connection=main_connection,
            meta_connection=meta_connection,
            data=data,
            service_slug=service_slug,
            org_id=org_id,
        )
        if not billing_info and service_slug == 'tracker' and not cloud_billing_account:
            if relations[0]['type'] != 'user':
                raise OrganizationIsWithoutContract()
            service = OrganizationServiceModel(main_connection) \
                .get_by_slug(org_id, service_slug, fields=['service_id'])
            user_count = UserServiceLicenses(main_connection) \
                .filter(org_id=org_id, service_id=service['service_id']) \
                .count()
            if user_count >= settings.TRACKER_FREE_LICENSES:
                raise OrganizationIsWithoutContract()

        ResourceModel(main_connection).add_relations(
            resource_id=resource_id,
            org_id=org_id,
            relations=relations
        )
        update_service_licenses(main_connection, org_id, service_slug, g.user.passport_uid, sync=True)

        response = prepare_resource_for_licenses(
            ResourceModel(main_connection).get(
                id=resource_id,
                org_id=org_id,
                service=app.config['DIRECTORY_SERVICE_NAME'],
                fields=[
                    'external_id',
                    'service',
                    'relations.*',
                    'relations.user.*',
                    'relations.department.*',
                    'relations.group.*',
                ],
            )
        )

        return json_response(
            response,
            status_code=201,
        )

    @internal
    @permission_required(
        permissions=[global_permissions.manage_licenses, global_permissions.manage_tracker],
        any_permission=True,
    )
    @scopes_required([scope.manage_service_licenses])
    @requires(org_id=True, user=True)
    @uses_schema(LICENSES_SCHEMA)
    def put(self, meta_connection, main_connection, data, service_slug):
        """
        Обновить список лицензий сервиса

        Можно сразу передать список лицензий.
        Список должен состоять из объектов с ключами:

        * **object_id** - идентификатор сущности
        * **object_type** - тип сущности

        Пример:

            {{
                [
                    {{
                        "type": "user",
                        "id": 66865142
                    }}
                ]
            }}

        ---
        tags:
          - Лицензии сервиса
        parameters:
          - in: path
            name: service_slug
            type: string
            required: true
        responses:
          200:
            description: Лицензии созданы
          422:
            description: Сервис без лицензий
          403:
            description: Сервис не включен в организации
          404:
            description: Сервис не найден
        """
        org_id = g.org_id
        cloud_billing_account = bool(request.args.get('cloud_billing_account', 0, int))

        resource_id, relations, billing_info = self._check_licenses(
            main_connection=main_connection,
            meta_connection=meta_connection,
            data=data,
            service_slug=service_slug,
            org_id=org_id,
        )

        organization = OrganizationModel(main_connection).get(org_id, fields=['organization_type'])
        org_type = organization['organization_type']
        if not billing_info and service_slug == 'tracker' and not cloud_billing_account:
            if len(relations) > settings.TRACKER_FREE_LICENSES and org_type not in organization_type.partner_types:
                raise OrganizationIsWithoutContract()
            for relation in relations:
                if relation['type'] != 'user':
                    raise OrganizationIsWithoutContract()

        if service_slug == 'disk':
            current_lic_uid = set(ResourceRelationModel(main_connection).filter(
                resource_id=resource_id,
                org_id=org_id,
                name='member'
            ).scalar('user_id'))
            new_lic_uid = set([item['user_id'] for item in relations])

            to_delete = current_lic_uid - new_lic_uid
            to_add = new_lic_uid - current_lic_uid

        ResourceModel(main_connection).update_relations(
            resource_id=resource_id,
            org_id=org_id,
            relations=relations
        )

        if service_slug == 'disk':
            for uid in to_delete:
                try:
                    DeleteSpacePartnerDiskTask(main_connection).delay(
                        org_id=org_id,
                        uid=uid,
                        resource_id=resource_id,
                        author_id=g.user.passport_uid,
                    )
                except DuplicatedTask:
                    log.info('Task for this uid is already in the queue')

            for uid in to_add:
                try:
                    AddSpacePartnerDiskTask(main_connection).delay(
                        org_id=org_id,
                        uid=uid,
                        resource_id=resource_id,
                        author_id=g.user.passport_uid,
                    )
                except DuplicatedTask:
                    log.info('Task for this uid is already in the queue')
        update_service_licenses(main_connection, org_id, service_slug, g.user.passport_uid, sync=True)

        response = prepare_resource_for_licenses(
            ResourceModel(main_connection).get(
                id=resource_id,
                org_id=org_id,
                service=app.config['DIRECTORY_SERVICE_NAME'],
                fields=[
                    'external_id',
                    'service',
                    'relations.*',
                    'relations.user.*',
                    'relations.department.*',
                    'relations.group.*',
                ],
            )
        )

        return json_response(
            response,
            status_code=200,
        )


class OrganizationCalculatePriceServiceLicensesView(View):
    methods = ['post']

    @internal
    @permission_required(
        permissions=[global_permissions.manage_licenses, global_permissions.manage_tracker],
        any_permission=True,
    )
    @requires(org_id=True, user=True)
    @uses_schema(LICENSES_SCHEMA)
    @scopes_required([scope.manage_service_licenses])
    def post(self, meta_connection, main_connection, data, service_slug):
        """
        Считаем общую стоимость контейнера и цену за одного пользователя.
        Контейнер - это список сущностей вида:
        [
            {{
                "type": "user",
                "id": 1
            }},
            {{
                "type": "department",
                "id": 2
            }},
            {{
                "type": "group",
                "id": 3
            }}
        ]
        Пример выдачи:
        {{
            "per_user": 90,
            "total": 7110,
            "user_count": 79
        }}

        ---
        tags:
          - Лицензии сервиса
        responses:
          200:
            description: Стоимость контейнера и цена за каждого пользователя
          422:
            description: Сервис без лицензий
          404:
            description: Сервис не найден

        """
        check_license_objects(main_connection, data, g.org_id)
        ServiceModel(meta_connection).get_licensed_service_by_slug(service_slug)
        if service_slug == 'disk' and check_has_non_users_objects(data):
            # для диска доступны только пользователи
            raise SubscriptionIncompatibleContainerType()

        relations = convert_licenses_to_relations(data)
        users_count = get_users_count_from_relations(main_connection, g.org_id, relations)

        promocode = OrganizationPromocodeModel(main_connection).get_active_for_organization(
            org_id=g.org_id,
            fields=['promocode_id', 'expires_at'],
        )
        internal_promocode = False

        if promocode:
            promocode = get_promocode_by_id(promocode['promocode_id'])
            internal_promocode = promocode['promocode_type'] == promocode_type.internal
            promocode = format_promocode(
                main_connection,
                promocode,
                g.org_id,
            )

        if service_slug == 'tracker' and app.config['NEW_TRACKER_PRICING']:
            from intranet.yandex_directory.src.yandex_directory.core.billing.utils import get_price_for_users

            total_price, total_price_with_discount = get_price_for_users(
                users_count=users_count,
                promocode_id=promocode['id'] if promocode else None,
            )
            # в новой схеме цену за пользователя не считаем
            per_user = 0
            per_user_with_discount = 0

            if total_price_with_discount == total_price:
                total_price_with_discount = None

        else:

            price = get_price_and_product_id_for_service(
                users_count,
                service_slug,
                promocode_id=promocode['id'] if promocode else None,
            )
            total_price = price['price'] * users_count

            price_with_discount = price['price_with_discount']

            # если у нас нет скидки или цена со скидкой равна цене без скидки - вернем None
            total_price_with_discount = price_with_discount * users_count if price_with_discount is not None else None
            if total_price_with_discount == total_price:
                total_price_with_discount = None

            per_user = price['price']
            per_user_with_discount = price['price_with_discount']

        # если прокод внутренний, не будем показывать его на фронте
        if internal_promocode:
            promocode = None

        return json_response({
            'per_user': per_user,
            'per_user_with_discount': per_user_with_discount,
            'total': total_price,
            'total_with_discount': total_price_with_discount,
            'user_count': users_count,
            'promocode': promocode,
        })


def check_for_dismissed_users(main_connection, licenses, org_id):
    uid_list = []
    for item in licenses:
        if item['type'] == 'user':
            uid_list.append(item['id'])
    if uid_list:
        dismiss = UserModel(main_connection).find(filter_data={
            'id': uid_list,
            'is_dismissed': True,
            'org_id': org_id,
        })
        if dismiss:
            return dismiss[0]['id']


def check_has_non_users_objects(licenses):
    """
    Если ли в списках на лицензию объекты отличные от пользователь
    :param licenses:
    :rtype: bool
    """
    for item in licenses:
        if item['type'] != 'user':
            return True
    return False


def check_over_limit_for_partner_organizations(main_connection, org_id, service, add_license_count):
    pass


def get_users_count_from_relations(main_connection, org_id, relations):
    """
    Получаем количество пользователей, у которых есть доступ к ресурсу
    напрямую или через членство в отделах/группах.
    relations: список объектов ResourceRelationModel,
               либо список лицензий, выданных на пользователей/группу/отдел вида:
               [
                    {
                        "user_id": 66865142
                    },
                    {
                        "group_id": 11334
                    },
                    {
                        "department_id": 987765
                    }
               ]
    """

    total_user_ids = set()
    group_ids = []
    department_ids = []
    for item in relations:
        if item.get('user_id'):
            total_user_ids.add(item['user_id'])
        elif item.get('group_id'):
            group_ids.append(item['group_id'])
        elif item.get('department_id'):
            department_ids.append(item['department_id'])
    if department_ids:
        # добавляем в total_user_ids всех пользователей, которые входят в департаменты с учетом вложенности
        total_user_ids.update(set(only_ids(
            UserModel(main_connection).find(
                filter_data={
                    'recursive_department_id': department_ids,
                    'org_id': org_id,
                },
                fields=['id'],
            )
        )))
    if group_ids:
        # добавляем в total_user_ids всех пользователей, которые входят в группы с учетом вложенности
        total_user_ids.update(set(only_ids(
            UserModel(main_connection).find(
                filter_data={
                    'group_id__in': group_ids,
                    'org_id': org_id,
                },
                fields=['id'],
            )
        )))
    return len(total_user_ids)


class OrganizationPromocodeActivateView(View):
    methods = ['post']

    @internal
    @permission_required(permissions=[global_permissions.activate_promocode])
    @requires(org_id=True, user=True)
    @uses_schema(ACTIVATE_PROMOCODE_SCHEMA)
    @uses_out_schema(ACTIVATE_PROMOCODE_OUT_SCHEMA)
    @scopes_required([scope.activate_promocode])
    def post(self, meta_connection, main_connection, data):
        """
        Активация промокода

        ---
        tags:
          - Промокоды
        responses:
          200:
            description: Промокод успешно активирован.
          422:
            description: Ошибка активации промокода
        """
        promocode_id = data['id']

        OrganizationModel(main_connection).activate_promocode(
            org_id=g.org_id,
            promocode_id=promocode_id,
            author_id=g.user.passport_uid,
            internal_activation=False,
        )

        promocode = get_promocode_by_id(promocode_id)

        return json_response(
            format_promocode(
                main_connection,
                promocode,
                g.org_id,
            ),
        )


def check_license_objects(main_connection, services_licenses, org_id):
    """
    Проверяем, нет ли среди переданных для лицензирования объектов несуществующих или уволенных пользователей
    services_licenses - словарь вида:
        [
            {'type': OBJECT_TYPE, 'id': id},
        ]
    """
    check_objects_exists(main_connection, org_id, services_licenses)
    dismissed_user = check_for_dismissed_users(main_connection, services_licenses, org_id)
    if dismissed_user:
        raise ImmediateReturn(json_error_user_dismissed(dismissed_user))


def process_tracker_requested_licenses(
        main_connection,
        meta_connection,
        org_id,
        uid,
        action,
        data,
):
    check_license_objects(main_connection, data, org_id)
    relations = convert_licenses_to_relations(data)

    if action == 'confirm':
        # для партнерских организаций можно выдвать лицензии только для пользователей
        org_type = OrganizationModel(main_connection).get_organization_type(org_id)
        if org_type in organization_type.partner_types and \
                check_has_non_users_objects(data):
            raise SubscriptionIncompatibleContainerType()

        resource_id = OrganizationServiceModel(main_connection).get_licensed_service_resource_id(
            org_id,
            'tracker'
        )

        if org_type in organization_type.partner_types:
            resource = ResourceModel(main_connection).filter(
                external_id=resource_id,
                org_id=org_id,
                service=app.config['DIRECTORY_SERVICE_NAME'],
            ).fields(
                'external_id',
                'service',
                'relations.*',
                'relations.user.*',
                'relations.department.*',
                'relations.group.*',
            ).one()

            # для partner_organization лицензии выдаются только на пользователя
            licenses = prepare_resource_for_licenses(resource)
            current_count = len(licenses)

            service_id = ServiceModel(meta_connection).get_by_slug('tracker', ['id'])['id']
            user_limit = OrganizationServiceModel(main_connection). \
                get_by_org_service(g.org_id, service_id, ['user_limit'])['user_limit']
            if current_count + len(relations) > user_limit:
                raise SubscriptionLimitExceeded()

        ResourceModel(main_connection).add_relations(
            resource_id=resource_id,
            org_id=org_id,
            relations=relations
        )
        update_service_licenses(main_connection, org_id, 'tracker', g.user.passport_uid)

    for obj in relations:
        obj_key = '{}_id'.format(get_object_type(obj))
        RequestedUserServiceLicenses(main_connection).delete(
            filter_data={
                'org_id': org_id,
                'service_slug': 'tracker',
                obj_key: obj.get(obj_key),
            }
        )


def process_metrika_requested_licenses(
        main_connection,
        meta_connection,
        org_id,
        author_id,
        action,
        data,
        external_resource_id,
        user_ip,
):
    def get_user_lang(uid):
        user_lang = users_info_in_blackbox[uid]['fields'].get('language')
        if user_lang:
            return user_lang

        if uid in organization_uids:
            org_lang = organization_info.get('language')
            if org_lang:
                return org_lang

        return 'en'

    if not OrganizationServiceModel(main_connection).is_service_enabled(org_id, 'metrika'):
        raise AuthorizationError(
            'Service is not enabled',
            'service_is_not_enabled',
        )

    organization_info = OrganizationModel(main_connection).get(org_id)

    resource = ResourceModel(main_connection).find(filter_data={
        'org_id': org_id,
        'service': 'metrika',
        'external_id': external_resource_id,
    }, one=True)

    if not resource:
        raise ResourceNotFound()

    uids_for_process = set([
        int(request_object['id'])
        for request_object in data
        if request_object['type'] == 'user'
    ])

    organization_uids = set(
        UserModel(main_connection).filter(org_id=org_id, id=uids_for_process).fields('id').scalar('id')
    )

    users_info_in_blackbox = UserModel(main_connection).get_batch_userinfo(list(uids_for_process))
    uids_from_blackbox = set(users_info_in_blackbox.keys())

    not_found_in_blackbox_uids = set(map(str, uids_for_process)) - set(map(str, uids_from_blackbox))
    if not_found_in_blackbox_uids:
        raise UserNotFoundInBlackbox(
            message='Users {uids} not found in blackbox',
            uids=','.join(map(str, not_found_in_blackbox_uids)),
        )

    if action == 'confirm':
        outstaff = DepartmentModel(main_connection).get_or_create_outstaff(org_id)

        for uid_for_process in uids_for_process:
            if uid_for_process not in organization_uids:
                create_portal_user(
                    meta_connection=meta_connection,
                    main_connection=main_connection,
                    uid=uid_for_process,
                    org_id=org_id,
                    department_id=outstaff['id'],
                    user_ip=user_ip,
                    author_id=author_id,
                )

        relations = [
            {
                'org_id': org_id,
                'resource_id': resource['id'],
                'name': 'view',
                'user_id': uid,
            }
            for uid in uids_for_process
        ]

        ResourceRelationModel(main_connection).bulk_create(
            data=relations,
            strategy=Values(on_conflict=Values.do_nothing),
        )

        for relation in relations:
            relation['service_slug'] = 'metrika'
            relation['resource_id'] = external_resource_id
            action_resource_relation_add(
                main_connection,
                org_id=org_id,
                author_id=author_id,
                object_value=relation,
            )

        for uid_for_process in uids_for_process:
            try:
                send_confirm_access_email_to_user(
                    main_connection=main_connection,
                    org_id=org_id,
                    email=users_info_in_blackbox[uid_for_process]['default_email'],
                    lang=get_user_lang(uid_for_process),
                    resource_id=external_resource_id,
                )
                sms_metrika_counter_request_confirmed(
                    meta_connection=meta_connection,
                    main_connection=main_connection,
                    org_id=org_id,
                    uid=uid_for_process,
                    external_id=external_resource_id,
                )
            except DuplicatedTask:
                pass
    else:
        for uid_for_process in uids_for_process:
            try:
                send_deny_access_email_to_user(
                    main_connection=main_connection,
                    org_id=org_id,
                    email=users_info_in_blackbox[uid_for_process]['default_email'],
                    lang=get_user_lang(uid_for_process),
                    resource_id=external_resource_id,
                )
                sms_metrika_counter_request_denied(
                    meta_connection=meta_connection,
                    main_connection=main_connection,
                    org_id=org_id,
                    uid=uid_for_process,
                    external_id=external_resource_id,
                )
            except DuplicatedTask:
                pass

    RequestedUserServiceLicenses(main_connection).delete(
        filter_data={
            'org_id': org_id,
            'service_slug': 'metrika',
            'user_id': uids_for_process,
            'external_id': external_resource_id,
        }
    )


class OrganizationManageRequestedLicensesView(View):
    methods = ['post']

    @internal
    @no_permission_required
    @scopes_required([scope.manage_service_licenses])
    @requires(org_id=True, user=True)
    @uses_schema(LICENSES_SCHEMA)
    def post(self, meta_connection, main_connection, data, service_slug, action):
        """
        Подтвердить/отклонить заявки на получение лицензий
        Action может принимать значения confirm/deny.

        Входные данные:

            [
                {{
                    "type": "user",
                    "id": 66865142
                }},
                {{
                    "type": "user",
                    "id": 66865156
                }}
            ]

        ---
        tags:
          - Лицензии сервиса
        parameters:
          - in: path
            name: service_slug
            required: true
            type: string
          - in: path
            name: action
            required: true
            type: string
          - in: query
            name: resource_id
            required: false
            type: string
          - in: body
            name: body
        responses:
          200:
            description: Заявки на лицензии обработаны
          422:
            description: Сервис без лицензий
          403:
            description: Сервис не включен в организации
          404:
            description: Сервис не найден
        """

        action = action.lower()
        if action not in ['confirm', 'deny']:
            return json_error_invalid_value('action')

        if service_slug == 'tracker':
            check_permissions(
                meta_connection=meta_connection,
                main_connection=main_connection,
                permissions=[global_permissions.manage_licenses, global_permissions.manage_tracker],
                any_permission=True,
            )

            process_tracker_requested_licenses(
                main_connection=main_connection,
                meta_connection=meta_connection,
                org_id=g.org_id,
                uid=g.user.passport_uid,
                action=action,
                data=data,
            )

            model_filters = {
                'org_id': g.org_id,
                'service_slug': service_slug,
            }

        elif service_slug == 'metrika':
            external_resource_id = request.args.get('resource_id', None, str)
            if external_resource_id is None:
                raise InvalidValue(message='Invalid value for {field}', field='resource_id')

            check_permissions(
                meta_connection=meta_connection,
                main_connection=main_connection,
                permissions=[user_permissions.can_change_relations],
                object_type='metrika',
                object_id=external_resource_id,
            )

            process_metrika_requested_licenses(
                main_connection=main_connection,
                meta_connection=meta_connection,
                org_id=g.org_id,
                author_id=g.user.passport_uid,
                action=action,
                data=data,
                external_resource_id=external_resource_id,
                user_ip=g.user.ip,
            )

            model_filters = {
                'org_id': g.org_id,
                'service_slug': service_slug,
                'external_id': external_resource_id,
            }

            action_resource_license_request_action(
                main_connection,
                org_id=g.org_id,
                object_value={
                    'action': action,
                    'service_slug': service_slug,
                    'external_id': external_resource_id,
                },
                author_id=g.user.passport_uid,
            )

        else:
            raise UnsupportedService()

        response = build_list_response(
            model=RequestedUserServiceLicenses(main_connection),
            model_fields=[
                '*',
                'author.*',
                'group.*',
                'department.*',
                'user.*',
            ],
            model_filters=model_filters,
            path=request.path,
            query_params=request.args.to_dict(),
            prepare_result_item_func=prepare_requests_for_licenses,
            order_by=['-created_at'],
            max_per_page=1000,
        )

        return json_response(
            response['data'],
            headers=response['headers'],
        )


class OrganizationTrackerDetailLicensesView(View):
    @internal
    @no_permission_required
    @scopes_required([scope.read_service_licenses])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Данные о подписках на трекер

        Пример выдачи:
        {
            'users':  50,
            'licences': 24
            'groups': [{'id': 1, 'label': 'smth', 'users': 12}, ...],
            'departments': [{'id': 1, 'label': 'smth', 'users': 12}, ...],
        }
        ---
        responses:
          200:
            description: Словарь со списками данных о лицензиях трекера.
        """
        organization = OrganizationModel(main_connection).get(
            id=g.org_id,
            fields=[
                'user_count',
            ]
        )
        groups = GroupModel(main_connection).filter(
            org_id=g.org_id,
            type='generic',
        ).fields('id', 'name', 'members_count')

        departments = DepartmentModel(main_connection).filter(
            org_id=g.org_id
        ).fields('id', 'name', 'members_count')
        licenses = UserServiceLicenses(main_connection).count(
            filter_data={
                'org_id': g.org_id,
                'service_slug': 'tracker',
            },
            distinct_field='user_id'
        )

        response = {
            'users': organization['user_count'],
            'licenses': licenses,
            'groups': [
                {
                    'id': group['id'],
                    'name': group['name'],
                    'users': group['members_count'],
                }
                for group in groups
            ],
            'departments': [
                {
                    'id': department['id'],
                    'name': department['name'],
                    'users': department['members_count'],
                }
                for department in departments
            ]
        }

        return json_response(response)


class OrganizationRequestLicensesView(View):
    methods = ['get', 'post']

    @internal
    @no_permission_required
    @scopes_required([scope.read_service_licenses])
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, service_slug):
        """
        Список заявок на лицензии сервиса

        Пример выдачи:

            [
                {
                    "object": {
                        "email": "web-chib@ya.ru",
                        "name": {
                            "first": {
                                "en": "Gennady",
                                "ru": "Геннадий"
                            },
                            "last": {
                                "en": "Chibisov",
                                "ru": "Чибисов"
                            }
                        },
                        "gender": "male",
                        "id": 66865142,
                        "nickname": "web-chib",
                        "position": none,
                        "avatar_id": none,
                        "external": true,
                    },
                    "object_type": "user",
                    "author": {
                        "email": "web-chib@ya.ru",
                        "name": {
                            "first": {
                                "en": "Gennady",
                                "ru": "Геннадий"
                            },
                            "last": {
                                "en": "Chibisov",
                                "ru": "Чибисов"
                            }
                        },
                        "gender": "male",
                        "id": 66865142,
                        "nickname": "web-chib",
                        "position": none,
                        "avatar_id": "1234456",
                        "external": false,
                    },
                    "comment": "some text",
                    "service_slug": "tracker",
                    "name: "owner",
                    "created_at": '2018-01-01T10:00:00.123Z',
                    "resource_id": "123345"
                }
            ]

        ---
        tags:
          - Лицензии сервиса
        parameters:
          - in: path
            name: service_slug
            type: string
            required: true
          - in: query
            name: resource_id
            type: string
          - in: query
            name: fields
            type: string
            description: список требуемых полей, перечисленных через запятую
        responses:
          200:
            description: Словарь со списками заявок на лицензии сервиса.
        """
        fields = split_by_comma(request.args.to_dict().get('fields'))
        model_fields = [
            'user_id', 'department_id', 'group_id',
            'service_slug', 'external_id',
        ]
        model_fields.extend(fields)
        filter_data = {
            'org_id': g.org_id,
            'service_slug': service_slug,
        }
        uid = request.args.to_dict().get('user_id')
        if uid:
            filter_data['user_id'] = uid

        external_id = request.args.to_dict().get('resource_id')
        if external_id:
            filter_data['external_id'] = external_id

        response = build_list_response(
            model=RequestedUserServiceLicenses(main_connection),
            model_fields=model_fields,
            model_filters=filter_data,
            path=request.path,
            query_params=request.args.to_dict(),
            prepare_result_item_func=prepare_requests_for_licenses,
            preprocess_results=add_user_data_from_blackbox,
            order_by=['-created_at'],
            max_per_page=1000,
        )

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    @internal
    @no_permission_required
    @no_scopes
    @requires(org_id=False, user=True)
    @uses_schema(REQUESTS_SCHEMA)
    def post(self, meta_connection, main_connection, data, service_slug):
        """
        Создание заявки на получение лицензии
        Входные данные:
        {{
            "objects": [
                {{
                    "type": "user",
                    "id": 1
                }},
                {{
                    "type": "department",
                    "id": 2
                }},
                {{
                    "type": "group",
                    "id": 3
                }}
            ],
            "comment": "some text",
            "resource_id": ["1", "2"]
        }}
        ---
        tags:
          - Лицензии сервиса
        parameters:
          - in: path
            name: service_slug
            type: string
            required: true
          - in: body
            name: body
        responses:
          200:
            description: Заявка на лицензию успешно создана.
          422:
            description: Сервис без лицензий
          403:
            description: Сервис не включен в организации
          404:
            description: Сервис не найден
        """
        with log.fields(resource={
            'id': ','.join(data.get('resource_ids', [])),
            'service': {
                'slug': service_slug,
            },
        }):
            if service_slug == 'tracker':
                if g.org_id is None:
                    raise AuthorizationError('Organization is required for this operation')

                check_license_objects(main_connection, data['objects'], g.org_id)

                # для партнерских организаций можно выдвать лицензии только для пользователей
                org_type = OrganizationModel(main_connection).get_organization_type(g.org_id)
                if org_type in organization_type.partner_types and \
                        check_has_non_users_objects(data['objects']):
                    raise SubscriptionIncompatibleContainerType()

                lic_requests = convert_licenses_to_relations(data['objects'], relation_name=False)

                # проверяем, что сервис подключен и лицензионный
                OrganizationServiceModel(main_connection).get_licensed_service_resource_id(
                    org_id=g.org_id,
                    service_slug=service_slug,
                )
                RequestedUserServiceLicenses(main_connection).create_requests(
                    lic_requests=lic_requests,
                    comment=data.get('comment'),
                    org_id=g.org_id,
                    service_slug=service_slug,
                    author_id=g.user.passport_uid,
                )

            elif service_slug == 'metrika':
                external_resource_ids = list(map(str, data.get('resource_ids', [])))
                if not external_resource_ids:
                    raise ParameterNotCorrect('Parameter "{parameter}" not filled', parameter='resource_ids')

                try:
                    uid = data['objects'][0]['id']
                    # для метрики запрос на доступ к ресурсу может придти только от одного пользователя
                except KeyError:
                    raise InvalidValue(message='Value "{value_path}" was not specified', value_path='objects.0.id')

                comment = data.get('comment')

                organization_shard = None
                shards = get_shard_numbers()
                for shard in shards:
                    with get_main_connection(shard=shard) as connection:
                        resources = ResourceModel(connection).filter(
                            service=service_slug,
                            external_id=external_resource_ids,
                        ).fields('external_id', 'org_id').all()
                        if resources:
                            organization_shard = shard
                            break

                if not resources:
                    raise ResourceNotFound(
                        message='Resources {resources} not found',
                        resources=','.join(external_resource_ids)
                    )

                # Проверяем что все ресурсы принадлежат одной организации
                org_id = resources[0]['org_id']
                for resource in resources:
                    if org_id != resource['org_id']:
                        raise InvalidValue(message='Requested resources are in different organizations')

                existent_external_resource_ids = [resource['external_id'] for resource in resources]

                not_found_resource_ids = set(external_resource_ids) - set(existent_external_resource_ids)
                if not_found_resource_ids:
                    raise ResourceNotFound(
                        message='Resources {resources} not found',
                        resources=','.join(not_found_resource_ids)
                    )

                # проверим что пользователь состоит в той же организации что и ресурсы
                user_orgs = UserMetaModel(meta_connection).filter(id=uid, is_dismissed=False).scalar('org_id')
                if user_orgs and org_id not in user_orgs:
                    raise UserIsNotMemberOfThisOrganization()

                with get_main_connection(shard=organization_shard, for_write=True) as main_connection:
                    # проверяем, что сервис подключен
                    if not OrganizationServiceModel(main_connection).is_service_enabled(org_id, service_slug):
                        raise AuthorizationError(
                            'Service is not enabled',
                            'service_is_not_enabled',
                        )

                    requested_user_service_licenses = [
                        {
                            'org_id': org_id,
                            'author_id': uid,
                            'user_id': uid,
                            'service_slug': service_slug,
                            'external_id': external_id,
                            'comment': comment,
                        }
                        for external_id in external_resource_ids
                    ]

                    # создание запросов на получение доступа
                    RequestedUserServiceLicenses(main_connection).bulk_create(
                        data=requested_user_service_licenses,
                        strategy=Values(on_conflict=Values.do_nothing),
                    )

                    # отправка письма о запросе получения доступа ответственному за сервис
                    try:
                        send_request_access_email_to_responsible_for_service(
                            main_connection=main_connection,
                            org_id=org_id,
                            service_slug=service_slug,
                            uid=uid,
                            resource_ids=external_resource_ids,
                        )
                    except DuplicatedTask:
                        pass

                    for license_request in requested_user_service_licenses:
                        action_resource_license_request(
                            main_connection,
                            org_id=org_id,
                            object_value=license_request,
                            author_id=g.user.passport_uid,
                        )
            else:
                raise UnsupportedService

        return json_response(
            data={},
            status_code=200,
        )


class OrganizationDownloadContractView(View):
    methods = ['get']

    @internal
    @no_permission_required
    @no_scopes
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Возвращает печатную версию договора
        """
        return Response(
            OrganizationModel(main_connection).get_contract_print_form(org_id=g.org_id),
            content_type='application/pdf; charset=utf-8',
            headers={'Content-Disposition': 'attachment;filename=contract.pdf'}
        )
