# -*- coding: utf-8 -*-

from flask import request, g
from functools import partial
from collections import defaultdict

from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope, check_scopes
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.models.user import UserModel
from intranet.yandex_directory.src.yandex_directory.core.models.department import DepartmentModel
from intranet.yandex_directory.src.yandex_directory.core.models.group import GroupModel
from intranet.yandex_directory.src.yandex_directory.core.models.service import ServiceModel
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    prepare_user,
    prepare_department,
    prepare_group,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import global_permissions
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
)
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    internal,
    no_permission_required,
    scopes_required,
    requires,
    permission_required,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    prepare_for_query,
)
DEFAULT_LIMIT = 3
MAX_LIMIT = 50

SUGGEST_SCOPES = [scope.read_users, scope.write_users, scope.read_groups, scope.write_groups, scope.read_departments, scope.write_departments]


class SuggestView(View):
    @internal
    @no_permission_required
    @scopes_required(SUGGEST_SCOPES)
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Саджест

        Отдает заданное кол-во людей, департаментов, групп,
        содержащих заданную подстроку.
        Умеет искать по полям:
        логин пользователя, имя пользователя, название организации,
        название группы.

        Формат ответа:

            {
                "users": [user_object1, user_object2, ..],
                "departments": [department_object1, department_object2, ...],
                "groups": [group_object1, group_object1, ...],
            }

        Пример ответа:

            {
             "departments": [],
             "groups": [],
             "users": [
                    {
                        "department": {
                            "id": "555a3313e05f8d3869d6a35d",
                            "label": "ya6",
                            "name": "test"
                        },
                        "email": "mabubuka123@yandex.ru",
                        "gender": "male",
                        "groups": [],
                        "id": 123,
                        "nickname": "mabubuka123",
                        "name": {
                            "first": {
                                "ru": "Светлана"
                            },
                            "last": {
                                "ru": "Попова"
                            }
                        }
                    },
                    ]
             }

        По умолчанию, выдаются группы всех типов. Если нужно показывать только
        пользовательские, но не служебные группы, то необходимо воспользоваться
        параметром `group_type`.

        ---
        tags:
          - Другое
        parameters:
          - in: query
            name: text
            required: true
            type: string
            description: поисковая строка
          - in: query
            name: group_type
            required: false
            type: string
            description: тип групп (по умолчанию выдаются все)
          - in: query
            name: limit
            required: false
            type: integer
            default: 3
            maximum: 50
            description: ограничение выдачи
          - in: query
            name: domain_only
            required: false
            type: boolean
            default: false
            description: выдавать только доменные учетки
        responses:
          200:
            description: Success

        """

        query_parameters = request.args.to_dict()
        text = query_parameters.get('text', '').lower().strip()
        domain_only = query_parameters.get('domain_only', '') in {'true', '1', }
        group_type = query_parameters.get('group_type', '').lower()
        limit = int(query_parameters.get('limit', DEFAULT_LIMIT))
        org_id = g.org_id

        if limit <= 0:
            limit = DEFAULT_LIMIT
        else:
            limit = min(limit, MAX_LIMIT)

        service_scopes = g.scopes
        def check_permissions(*required_scopes):
            """
            Проверям право на поиск по сущностям
            :param required_scopes: необходимые OAuth скоупы
            :type required_scopes: list
            :rtype: bool
            """
            return check_scopes(service_scopes, required_scopes)

        result = {}
        if check_permissions(scope.read_users, scope.write_users):
            result['users'] = self.suggest_users(main_connection, org_id, text, limit, domain_only) if text else []
        if check_permissions(scope.read_departments, scope.write_departments):
            result['departments'] = self.suggest_departments(main_connection, org_id, text, limit) if text else []
        if check_permissions(scope.read_groups, scope.write_groups):
            result['groups'] = self.suggest_groups(main_connection, org_id, text, group_type, limit) if text else []
        return json_response(result)

    def suggest_groups(self, main_connection, org_id, text, group_type, limit):
        group_model = GroupModel(main_connection)
        init_query = prepare_for_query(text)

        group_filter = {
            'suggest': text,
            'org_id': org_id
        }
        if group_type:
            group_filter['type'] = group_type
        extra_projections = [
            "(lower(groups.name ->>'ru') = %(init_query)s)::BOOLEAN",
            "(lower(groups.name ->>'en') = %(init_query)s)::BOOLEAN"
        ]
        mogrify_extra_projections = [
            group_model.mogrify(order_by, {'init_query': init_query})
            for order_by in extra_projections
            ]

        order_by_set = ["- " + x for x in extra_projections]
        mogrify_order_by_set = [
            group_model.mogrify(order_by, {'init_query': init_query})
            for order_by in order_by_set
            ]

        groups = list(map(
            partial(
                prepare_group,
                main_connection,
                api_version=request.api_version,
            ),
            group_model.find(
                group_filter,
                fields=[
                    '*',
                    'members.*',
                ],
                limit=limit,
                order_by=mogrify_order_by_set,
                extra_projections=mogrify_extra_projections
            )
        ))
        return groups

    def suggest_departments(self, main_connection, org_id, text, limit):
        department_filter = {
                'suggest': text,
                'org_id': org_id
            }
        init_query = prepare_for_query(text)
        extra_projections = [
            "(lower(departments.name_plain) = %(init_query)s)::BOOLEAN"
        ]
        department_model = DepartmentModel(main_connection)
        mogrify_extra_projections = [
            department_model.mogrify(order_by, {'init_query': init_query})
            for order_by in extra_projections
        ]

        order_by_set = ["- " + x for x in extra_projections]

        mogrify_order_by = [
            department_model.mogrify(order_by, {'init_query': init_query})
            for order_by in order_by_set
        ]
        departments = list(map(
            partial(
                prepare_department,
                main_connection,
                api_version=request.api_version,
            ),
            department_model.find(
                department_filter,
                limit=limit,
                fields=[
                    '*',
                    'parent.*',
                ],
                order_by=mogrify_order_by,
                extra_projections=mogrify_extra_projections
            )
        ))
        return departments

    def suggest_users(self, main_connection, org_id, text, limit, domain_only=False):
        def expand_prepare_user(u):
            return prepare_user(
                main_connection,
                u,
                expand_contacts=True,
                api_version=request.api_version,
            )

        user_model = UserModel(main_connection)
        query = {
            'suggest': text,
            'org_id': org_id
        }
        if domain_only:
            query['id__gt'] = 113 * 10 ** 13
            query['id__lt'] = 900 * 10 ** 13

        users = list(map(
            expand_prepare_user,
            user_model.find(
                query,
                limit=limit,
                fields=[
                    '*',
                    'department.*',
                    'groups.*',
                ],
            )
        ))
        return users


class SuggestLicenseView(View):
    @internal
    @permission_required(
        permissions=[global_permissions.manage_licenses, global_permissions.manage_tracker],
        any_permission=True,
    )
    @scopes_required([scope.manage_service_licenses])
    @requires(org_id=True, user=True)
    def get(self, meta_connection, main_connection, service_slug):
        """
        Саджест

        Отдает заданное кол-во людей, у которых есть лицензии на сервис и в имени/логине содержится заданная подстрока.
        Если лицензия выдана пользователю через группу/отдел, то его тоже вернем.
        Умеет искать по полям:
        логин пользователя, имя пользователя.

        Пример ответа:

            {
             "departments": [],
             "groups": [
                {
                    "members_count": 2,
                    "type": "generic",
                    "users": [
                        {
                            "email": "abracadabra@yandex.ru",
                            "gender": "female",
                            "id": 5678,
                            "nickname": "abracadabra",
                            "name": {
                                "first": {
                                    "ru": "Lady"
                                },
                                "last": {
                                    "ru": "Gaga"
                                }
                            }
                        }
                    ]
                }
             ],
             "users": [
                {
                    "email": "mabubuka123@yandex.ru",
                    "gender": "male",
                    "id": 123,
                    "nickname": "mabubuka123",
                    "name": {
                        "first": {
                            "ru": "Abram"
                        },
                        "last": {
                            "ru": "Ivanov"
                        }
                    }
                },
             ]
             }


        ---
        tags:
          - Другое
        parameters:
          - in: path
            name: service_slug
            required: true
            type: string
          - in: query
            name: text
            required: true
            type: string
            description: поисковая строка
          - in: query
            name: limit
            required: false
            type: integer
            default: 3
            maximum: 50
            description: ограничение выдачи
        responses:
          200:
            description: Success

        """

        query_parameters = request.args.to_dict()
        text = query_parameters.get('text', '').lower().strip()
        limit = int(query_parameters.get('limit', DEFAULT_LIMIT))
        org_id = g.org_id

        if limit <= 0:
            limit = DEFAULT_LIMIT
        else:
            limit = min(limit, MAX_LIMIT)
        service_id = ServiceModel(meta_connection).get_licensed_service_by_slug(service_slug)['id']

        def expand_prepare_user(u):
            return prepare_user(
                main_connection,
                u,
                expand_contacts=True,
                api_version=request.api_version,
            )

        users = UserModel(main_connection).user_license_suggest(org_id, service_id, text, limit)

        result = {}
        via_group = defaultdict(list)
        via_department = defaultdict(list)
        direct_users = []
        for user in users:
            if user['via_department_id']:
                via_department[user['via_department_id']].append(expand_prepare_user(user))
            elif user['via_group_id']:
                via_group[user['via_group_id']].append(expand_prepare_user(user))
            else:
                direct_users.append(expand_prepare_user(user))

        if via_group:
            groups = list(map(
                partial(
                    prepare_group,
                    main_connection,
                    api_version=request.api_version,
                ),
                GroupModel(main_connection).find(
                    filter_data={'id': list(via_group.keys()), 'org_id': org_id},
                    fields=['*'],
                )
            ))
            for gr in groups:
                gr['users'] = via_group[gr['id']]
            result['groups'] = groups
        else:
            result['groups'] = []

        if via_department:
            departments = list(map(
                partial(
                    prepare_department,
                    main_connection,
                    api_version=request.api_version,
                ),
                DepartmentModel(main_connection).find(
                    filter_data={'id': list(via_department.keys()), 'org_id': org_id},
                    fields=['*'],
                )
            ))
            for dep in departments:
                dep['users'] = via_department[dep['id']]
            result['departments'] = departments
        else:
            result['departments'] = []

        result['users'] = direct_users
        return json_response(result)
