# coding: utf-8
from flask import (
    g,
    request,
)
from functools import partial

from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_fields,
    prepare_fields,
)

from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    get_object_or_404,
    json_error_forbidden,
    build_list_response,
    split_by_comma,
)
from intranet.yandex_directory.src.yandex_directory.core.models import TaskModel
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_permission_required,
    requires,
    internal,
    scopes_required,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import scope

ALLOWED_FIELDS = [
    'id',
    'task_name',
    'state',
    'start_at',
    'created_at',
    'finished_at',
    'author_id',
]


def prepare_task(task):
    """
    Подготовка задачи к выдаче в ручках.
    """
    return only_fields(task, *ALLOWED_FIELDS)


class TaskDetailView(View):
    @internal
    @scopes_required([scope.read_tasks])
    @no_permission_required
    @requires(org_id=False, user=True)
    def get(self, meta_connection, main_connection, task_id):
        """
        Возвращаем состояние задачи

        Пример ответа:

            {
                'id': '038bbc21-579d-4509-9477-392fac485eb1',
                'created_at': 'free',
                'state': '2017-08-25T12:17:26.638279+00:00',
                'start_at': null,
                'finished_at': null,
                'author_id': 123123123123,
            }

        ---
        tags:
            - Очередь задач
        responses:
            200:
                description: Получаем данные о задаче
            403:
                description: Доступ запрещен
            404:
                description: Задача не найдена
        """
        fields = prepare_fields(request.args.to_dict().get('fields'))

        _fields = fields[:]
        _fields.append('author_id')
        task = get_object_or_404(
            TaskModel(main_connection),
            id=task_id,
            fields=_fields,
        )

        if task['author_id'] != g.user.passport_uid:
            return json_error_forbidden()

        if 'author_id' not in fields:
            del task['author_id']
        task = prepare_task(task)
        return json_response(task)


class TaskListView(View):
    @internal
    @scopes_required([scope.read_tasks])
    @no_permission_required
    @requires(org_id=False, user=True)
    def get(self, meta_connection, main_connection):
        """
        Возвращаем состояние всех задач для текущего пользователя
        Пример ответа:
            [
                {
                    'id': '038bbc21-579d-4509-9477-392fac485eb1',
                    'state': 'free',
                    'created_at': '2017-08-25T12:17:26.638279+00:00',
                    'start_at': null,
                    'finished_at': null,
                    'author_id': 123123123123,
                },
                ...
            ]
        ---
        tags:
            - Очередь задач
        responses:
            200:
                description: Получаем данные обо всех задачах для пришедшего пользователя
        """
        fields = prepare_fields(request.args.to_dict().get('fields'))
        _fields = fields[:]
        _fields.append('author_id')

        query_params = request.args.to_dict()

        response = build_list_response(
            model=TaskModel(main_connection),
            model_filters={'author_id': g.user.passport_uid},
            path=request.path,
            query_params=query_params,
            prepare_result_item_func=partial(
                prepare_task,
            ),
            model_fields=_fields,
            max_per_page=1000,
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )
