# coding: utf-8
import os
from cachetools import (
    cached,
    TTLCache,
)
from threading import RLock
from itertools import groupby

from flask import (
    g,
    request,
)

from operator import itemgetter

from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_scopes,
    no_permission_required,
    internal,
    requires,
)

from intranet.yandex_directory.src.yandex_directory.common.db import get_main_connection
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
)
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    json_response,
    get_user_data_from_blackbox_by_uid,
    get_user_login_from_passport_by_uid,
    Ignore,
    make_simple_strings,
    utcnow,
    format_date,
    check_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    only_fields,
    only_attrs,
    is_org_admin,
    is_deputy_admin,
    get_user_role,
    get_organization_admin_uid,
    is_outer_uid,
    build_email,
    pmap,
    objects_map_by_id,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.services import (
    x_host_for_service,
    try_replace_url_host,
)

from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.views.service import get_services_data
from intranet.yandex_directory.src.yandex_directory.core.models import (
    UserModel,
    UserMetaModel,
    GroupModel,
    DomainModel,
    DepartmentModel,
    OrganizationServiceModel,
    ServiceModel,
    ServicesLinksModel,
    OrganizationModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.domain import DomainNotFound

from intranet.yandex_directory.src.yandex_directory.core.models.service import (
    get_service_field_by_tld,
    get_service_url,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    global_permissions,
    department_permissions,
    get_permissions,
)

from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema_for_get
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import domain_is_tech, get_domains_from_db_or_domenator, DomainFilter

SERVICES_FEATURES = [
    'with_subscriptions', 'with_resources', 'highlighted', 'beta',
]

HEADER_GET_SCHEMA = {
    'title': 'Get service',
    'type': 'object',
    'properties': {
        'tld': schemas.STRING,
        'language': schemas.STRING,
    },
    'required': ['tld', 'language'],
    'additionalProperties': True
}

SERVICE_GET_SCHEMA = {
    'title': 'Get service',
    'type': 'object',
    'properties': {
        'tld': schemas.STRING,
        'language': schemas.STRING,
        'slug': schemas.STRING,
    },
    'additionalProperties': True,
}


if os.environ.get('ENVIRONMENT') == 'autotests':
    import sys
    external_admin_services_cache_size = 0  # для тестов размер кэша == 0
    # установим бесконечное время жизни,
    # чтобы TTLCache не пытался вызвать expire
    # на ключ all-services через 300с
    external_admin_services_cache_ttl = sys.maxsize
else:
    external_admin_services_cache_size = 1000
    external_admin_services_cache_ttl = 30 * 60  # 30 минут


external_admin_services_cache = TTLCache(
    maxsize=external_admin_services_cache_size,
    ttl=external_admin_services_cache_ttl,
)
external_admin_services_lock = RLock()


@cached(cache=external_admin_services_cache,
        key=lambda *args, **kwargs: 'all-external-admin-services',
        lock=external_admin_services_lock)
def visible_for_external_admin_services(meta_connection):
    show_to_outer_admin = ServiceModel(meta_connection).find(
        filter_data={
            'available_for_external_admin': True,
            'show_in_header': True,
        },
        fields=['slug'],
    )
    return set(only_attrs(show_to_outer_admin, 'slug'))


def is_visible(
    slug,
    role,
    enabled,
    has_owned_domain,
    permissions,
    is_configurable,
):
    """
    Решаем показывать сервис или нет
    """

    has_permissions_manage_services = global_permissions.manage_services in permissions
    has_permissions_department_edit = department_permissions.edit in permissions
    if slug == 'dashboard':
        return False
    elif slug == 'portal':
        return role in ['admin', 'deputy_admin', 'outer_admin'] or has_permissions_department_edit
    elif enabled:
        return True
    elif (
        not is_configurable
        or not has_permissions_manage_services
        or (slug in ['mail', 'calendar'] and not has_owned_domain)
    ):
        return False
    else:
        return True


def is_enabled(service, org_services, role):
    slug = service['slug']
    if service['always_enabled']:
        return True
    if slug in ('webmaster', 'portal'):
        return role in ['admin', 'deputy_admin', 'outer_admin']

    return org_services.get(slug, {}).get('enabled', False)


def set_header_based_host(services):
    """
    Для относительных путей ведем на портал
    :param services:
    :param tld:
    """

    result = []
    for service in services:
        host = x_host_for_service(service['slug'])
        if host:
            service['url'] = try_replace_url_host(service['url'], host)
        result.append(service)
    return result


def get_actions(
        main_connection,
        permissions,
        service,
        tld,
        slug,
        org_id,
        role,
):
    """
    Проставляется поле actions в зависимости от slug сервиса
    """
    actions = service['actions']

    if slug == 'webmaster':
        has_permissions = global_permissions.add_domains in permissions
        if not has_permissions:
            actions = []

    if slug == 'staff' and role in ['deputy_admin', 'outer_admin']:
        actions = []

    for action in actions:
        if slug == 'portal':
            has_owned_domains = OrganizationModel(main_connection).has_owned_domains(org_id)
            if has_owned_domains:
                action_slug = 'add-user'
            else:
                action_slug = 'invite-user'
            action['url'] = action['url'].format(tld, action_slug)
        elif slug == 'calendar':
            action['url'] = action['url'].format(tld, g.user.passport_uid)
        elif slug == 'staff':
            nickname = UserModel(main_connection).get(g.user.passport_uid)['nickname']
            action['url'] = action['url'].format(tld, nickname, org_id, g.user.passport_uid)
        elif action['url']:
            action['url'] = action['url'].format(tld)
    return actions


def add_features(service_data, service):
    for feature in SERVICES_FEATURES:
        service_data[feature] = True if feature in service['features'] else False

    return service_data


def get_services_links_data(
        meta_connection,
        main_connection,
        tld,
        language,
        org_id,
        slug,
):
    """
    Возвращает словарь с данными про сервисы известные Директории.

    Результатом является следующий словарь из словарей:
   {
            'id': id,
            'slug': slug,
            'name': name,
            'url': url,
            'icon': url,
            'can_be_enabled': bool,
            'can_be_disabled': bool,
            'available': bool,
            'ready': bool,
            'enabled': bool,
            'trial': {'expiration_date': None, 'status': 'inapplicable', 'days_till_expiration': None} или {},
            'options': {"free_subscriptions": 5}, # существует пока только у Трекера
            'priority': 1,

            # (начиная с //) (брать хост из заголовка X-<slug>-SERVICE-HOST если передан)
            # + поддержка заголовка X-TLD, если X-<slug>-SERVICE-HOST не указан - то брать tld из него.
            'actions': [{'url': 'url', main: True, 'slug': 'delete_smth'}],

            'settings_url': url,
            # Поля указанные ниже являются развернутым списком поля features
            'with_subscriptions': bool, данной фичей обладают сервисы с подпиской, сейчас только у Tracker
            'with_resources': bool, данная фичей обладают многие сервисы, которые с ресурсами, например, webmaster
            'highlighted': bool, данную фичу быделение сервиса не используется ни в одном сервисе
            'beta': bool,  данную фичей обладают сервисы, которые находятся в режиме beta
    }
    """

    services_filter = None
    if slug:
        services_filter = {'slug': slug}

    services = ServicesLinksModel(meta_connection).find(filter_data=services_filter)
    data = []

    # Определим, какие сервисы подключены в организации
    org_services = OrganizationServiceModel(main_connection).find(
        filter_data={
            'org_id': org_id,
            'enabled': Ignore,
        },
        fields=['slug', 'ready', 'trial_expires', 'trial_status', 'enabled', ],
        order_by=['org_id'],
    )
    org_services = objects_map_by_id(org_services, key='slug', remove_key='slug')

    # определяем роль пользователя
    role = g.user.role

    try:
        org_domain = DomainModel(main_connection).get_master(g.org_id)
        master_domain = org_domain['name']
    except DomainNotFound:
        master_domain = None
    permissions = get_permissions(
        meta_connection,
        main_connection,
        g.user.passport_uid,
        org_id=org_id,
    )
    for service in services:
        slug = service['slug']

        # выбираем настройки для tld
        main_url = get_service_field_by_tld(service, tld, 'url')
        icon = get_service_field_by_tld(service, tld, 'icon')

        # определим подключен ли сервис
        enabled = is_enabled(service, org_services, role)

        # определяем доспупность сервиса
        available = is_visible(
            slug=slug,
            role=role,
            enabled=enabled,
            has_owned_domain=master_domain,
            permissions=permissions,
            is_configurable=service['is_configurable'],
        )

        # выбираем настройки для языка
        language_data = service['data_by_language']
        name = language_data.get(language, {}).get('name')

        # проверим готов ли сервис
        # если сервис не подключен, то по дефолту он будет True
        ready = True
        if org_services.get(slug, {}).get('enabled', False):
            ready = org_services[slug]['ready']

        # определяем пробный период
        if slug == 'tracker' and org_services.get(slug):
            days_till_expiration = None
            if org_services[slug]['trial_expires']:
                days_till_expiration = max((org_services[slug]['trial_expires'] - utcnow().date()).days, 0)
            trial = {
                    'expiration_date': format_date(org_services[slug]['trial_expires'], allow_none=True),
                    'status': org_services[slug]['trial_status'],
                    'days_till_expiration': days_till_expiration,
            }
        else:
            trial = {}

        # определить action
        actions = get_actions(
            main_connection,
            permissions,
            service,
            tld,
            slug,
            org_id,
            role,
        )

        # определяем settings_url

        if slug == 'disk':
            settings_url = service['settings_url'].format(tld, org_id, g.user.passport_uid)
        else:
            settings_url = service['settings_url'].format(tld)

        service_data = {
                'id': service['id'],
                'slug': slug,
                'name': name,
                'url': main_url,
                'icon': icon,
                'can_be_enabled': service['can_be_enabled'],
                'can_be_disabled': service['can_be_disabled'],
                'available': available,
                'ready': ready,
                'enabled': enabled,
                'trial': trial,
                'options': service['options'],
                'priority': service['priority'],
                'actions': actions,
                'settings_url': settings_url,
            }

        # разварачиваем и добавляем поле feature
        service_data = add_features(service_data, service)
        data.append(service_data)
    return data


class UIHeaderView(View):
    @internal
    @no_scopes
    @no_permission_required
    @uses_schema_for_get(HEADER_GET_SCHEMA)
    @requires(org_id=True, user=True)
    def get(self, meta_connection, main_connection):
        """
        Информация для отрисовки шапки сервиса.

        Отдаёт описания сервисов, которые доступны текущему пользователю,
        а так же другую информацию, необходимую для отрисовки шапки.

        ---
        tags:
          - Внутренние ручки
        parameters:
          - in: query
            name: tld
            required: true
            type: string
            description: tld для ссылок на сервис и иконок
          - in: query
            name: language
            required: true
            type: string
            description: язык для названий сервисов
        responses:
          200:
            description: Словарь с информацией.
          422:
            description: Не указаны необходимые параметры.
        """
        tld = request.args.get('tld')
        language = request.args.get('language')

        services = get_services_data(
            meta_connection,
            main_connection,
            tld,
            language,
            g.org_id,
        )

        # Определим, какие сервисы подключены в организации
        org_services = OrganizationServiceModel(main_connection).find(
            filter_data={
                'org_id': g.org_id,
                'ready': True,
            },
            fields=['service_id']
        )
        org_services = set(only_attrs(org_services, 'service_id'))

        show_to_outer_admin = {}

        user_meta = UserMetaModel(meta_connection).get(
            user_id=g.user.passport_uid,
            org_id=g.org_id,
            is_outer=Ignore,
            fields=['user_type'],
            is_cloud=g.user.is_cloud,
        )
        is_outer = user_meta['user_type'] != 'inner_user'

        if is_outer:
            show_to_outer_admin = visible_for_external_admin_services(meta_connection)

        # Правила отображения сервисов в шапке:
        #
        # Сотруднику показываем сервис, если этот сервис
        # подключён в организации.
        #
        # Внешнему админу показываем только избранные сервисы,
        # выбирая их по признаку в базе - available_for_external_admin.
        #
        # Обычному админу дополнительно добавляем в список "portal"
        # Во всех этих случаях у сервиса должен быть флаг show_in_header.

        if g.user.role:
            role = g.user.role
        else:
            role = get_user_role(meta_connection, main_connection, g.org_id, g.user.get_cloud_or_passport_uid(),
                                 is_cloud=g.user.is_cloud)
        user_is_admin_or_deputy = is_org_admin(role=role) or is_deputy_admin(role=role)

        def is_visible(slug, service):
            """Решаем показывать сервис или нет."""
            if service['show_in_header']:
                if slug == 'portal' and user_is_admin_or_deputy:
                    # для админов и заместителей подмешиваем портал
                    return True
                if is_outer:
                    if slug in show_to_outer_admin and \
                                    service['id'] in org_services:
                        return True
                elif service['id'] in org_services:
                    return True

        services = [
            dict(service, slug=slug)
            for slug, service in services.items()
            if is_visible(slug, service)
        ]
        services = set_header_based_host(services)
        services.sort(key=itemgetter('priority'), reverse=True)

        # Теперь оставим лишь нужные поля
        services = [
            only_fields(service, 'slug', 'url', 'icon', 'name', 'in_new_tab')
            for service in services
        ]

        organization = OrganizationModel(main_connection).get(
            g.org_id,
            ['logo', 'subscription_plan'],
        )
        logo = organization['logo']

        # Временно вернул поле company_logo, до тех пор, пока сервисы
        # не поддержат лого с разными размерами.
        if logo:
            company_logo = logo['orig']['url']
        else:
            company_logo = None

        response = {
            'services': services,
            'logo': logo,
            'company_logo': company_logo,
        }
        return json_response(response)


class UIOrganizationProfileInfoView(View):
    @internal
    @no_scopes
    @no_permission_required
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Информация для профиля организации.

        Отдаёт такую информацию:
        1. Подключен ли хотя бы один естественный домен
        2. Количество учетных записей на домене (за вычетом роботных)
        3. Количество отделов (за исключением удаленных и роботных)
        4. Количество команд (только generic)
        5. Количество администраторов организации (считая внешнего админа, если он есть)

        ---
        tags:
          - Внутренние ручки

        responses:
          200:
            description: Словарь с информацией про профиль организации.
        """
        domains = get_domains_from_db_or_domenator(
            meta_connection=meta_connection,
            domain_filter=DomainFilter(org_id=g.org_id),
        )
        domain_names = [domain['name'] for domain in domains]

        # Если хотя бы один домен не совпадает с техническим, то у
        # организации - есть естественные домены

        has_natural_domains = any([
            domain for domain in domain_names
            if not domain_is_tech(domain)
        ])

        orgmodel = OrganizationModel(main_connection)

        users_count = UserModel(main_connection).count(
            filter_data={
                'org_id': g.org_id,
                'is_dismissed': False,
            }
        )
        robot_count = len(orgmodel.get_robots(g.org_id))
        users_without_robots = users_count - robot_count

        department_count = DepartmentModel(main_connection).count(
            filter_data={'org_id': g.org_id}
        )

        group_count = GroupModel(main_connection).count(
            filter_data={
                'type': 'generic',
                'org_id': g.org_id,
            }
        )

        admin_count = len(orgmodel.get_admins(g.org_id))
        admin_count += UserMetaModel(meta_connection).count(
            filter_data={
                'org_id': g.org_id,
                'user_type': 'outer_admin'
            }
        )
        owner_uid = get_organization_admin_uid(main_connection, g.org_id)
        if is_outer_uid(owner_uid):
            owner_login = get_user_data_from_blackbox_by_uid(owner_uid)['login']
        else:
            owner_login = build_email(
                main_connection,
                UserModel(main_connection).filter(id=owner_uid).fields('nickname').scalar()[0],
                g.org_id
            )

        response = {
            'has_natural_domains': has_natural_domains,
            'user_count': users_without_robots,
            'department_count': department_count,
            'group_count': group_count,
            'admin_count': admin_count,
            'owner_login': owner_login,
        }

        return json_response(response)


class UIOrganizationsView(View):
    @internal
    @no_scopes
    @no_permission_required
    @requires(org_id=False, user=True)
    def get(self, meta_connection, main_connection):
        """
        Отдаёт список организаций, в которых пользователь является сотрудником.

        Формат там простой:

        ```
        [
            {"id": 100500, "name": "Копа и Рогыта"},
            {"id": 200000, "name": "ООО Ромашка"}
        ]
        ```

        ---
        tags:
          - Внутренние ручки
        responses:
          200:
            description: Список с информацией об организациях.
        """

        meta_orgs = UserMetaModel(meta_connection) \
            .filter(
                id=g.user.passport_uid,
                is_dismissed=False,
                user_type='inner_user') \
            .fields('organization.shard') \
            .all()
        grouped = groupby(
            meta_orgs,
            key=lambda item: item['organization']['shard']
        )

        response = []

        def get_org_names(shard, users):
            org_ids = [u['organization']['id'] for u in users]
            with get_main_connection(shard=shard) as conn:
                orgs = OrganizationModel(conn) \
                    .filter(id__in=org_ids) \
                    .fields('id', 'name')
                response.extend(orgs)

        chunk_results = pmap(
            get_org_names,
            (
                {'shard': shard, 'users': users}
                for shard, users in grouped
            )
        )

        return json_response(make_simple_strings(response))


class UIServicesView(View):
    @internal
    @no_scopes
    @no_permission_required
    @uses_schema_for_get(SERVICE_GET_SCHEMA)
    @requires(org_id=True, user=True)
    def get(self, meta_connection, main_connection):
        """
        Информация для отрисовки данных о доступных пользователю сервисах.

        Отдаёт описания сервисов, которые доступны текущему пользователю,
        а так же другую информацию, необходимую для отрисовки информации о сервисе.
        Упорядочены по убыванию приоритета.
        В формате следующем:
        id                              1
        slug                            example
        name                            Привер
        url                             http://slug.example.yandex.com
        icon                            http://icons.example.yandex.com/slug-com.png
        can_be_enabled:                 True
        can_be_disabled:                False
        available:                      True
        ready:                          True
        enabled:                        False
        trial:                          {'expiration_date': None, 'status': 'inapplicable', 'days_till_ex': None}
        options:                        {"free_subscriptions": 5}
        priority:                       10
        actions:                        [{'url': 'url', main: True, 'slug': 'delete_smth'}]
        settings_url:                   '/portal/admin'
        with_subscriptions:             True
        with_resources:                 False
        highlighted:                    True
        beta:                           True

        Поля service:
        * **url** - основной урл сервиса
        * **can_be_enabled** - можно включать
        * **can_be_disabled** - можно выключать
        * **available** - отображать ли сервис
        * **ready** - в процессе включения/отключения
        * **enabled** - включён/выключен
        * **trial** - информация о пробном периоде
        * **options** - дополнительные параметры для сервиса
        * **priority** - приоритет сервиса, необходим для упорядочивания сервисов при отображении
        * **actions** - действие которое необходимо выполнить при нажатии на плашку
        * **settings_url** - урл настроек, ведущий на настройки сервиса либо в коннекте, либо внутри сервиса
        * **with_subscriptions** - это фича для сервиса, фичей обладают сервисы с подпиской, сейчас только у Tracker
        * **with_resources** - это фича для сервиса, фичей обладают сервисы, которые с ресурсами, например, webmaster
        * **highlighted** - это фича для сервиса, данную фичу быделение сервиса не используется ни в одном сервисе
        * **beta** - это фича для сервиса, фичей обладают сервисы, которые находятся в режиме beta
        ---
        tags:
          - Внутренние ручки
        parameters:
          - in: query
            name: tld
            required: false
            type: string
            description: tld для ссылок на сервис и иконок
          - in: query
            name: language
            required: false
            type: string
            description: язык для названий сервисов
        responses:
          200:
            description: Список словарей с информацией о сервисах.
          422:
            description: Не указаны необходимые параметры.
        """

        tld = request.args.get('tld', 'ru')
        language = request.args.get('language', 'ru')
        slug = request.args.get('slug')

        services = get_services_links_data(
            meta_connection,
            main_connection,
            tld,
            language,
            g.org_id,
            slug,
        )
        services = set_header_based_host(services)

        return json_response(services)
