# -*- coding: utf-8 -*-
import pdb
from copy import (
    copy,
    deepcopy,
)
from time import sleep
import requests
from flask import request, g
from requests.exceptions import HTTPError
from retrying import retry
from datetime import timedelta

from intranet.yandex_directory.src.yandex_directory.common.utils import utcnow, make_simple_strings
from intranet.yandex_directory.src.yandex_directory import app
from intranet.yandex_directory.src.yandex_directory.core.features import (
    is_feature_enabled,
    NO_DOREGISTRATION,
)
from intranet.yandex_directory.src.yandex_directory.auth.decorators import (
    no_scopes,
    internal,
    requires,
    permission_required,
    no_permission_required,
    scopes_required,
    no_auth,
)
from intranet.yandex_directory.src.yandex_directory.auth.scopes import (
    scope,
    get_oauth_service_data,
    check_scopes
)
from intranet.yandex_directory.src.yandex_directory.common import schemas
from intranet.yandex_directory.src.yandex_directory.common.cache import cached_in_memory_with_ttl
from intranet.yandex_directory.src.yandex_directory.common.compat import session_compat
from intranet.yandex_directory.src.yandex_directory.common.db import (
    get_main_connection,
    get_meta_connection,
    get_shard,
)
from intranet.yandex_directory.src.yandex_directory.core.exceptions import (
    ExternalIDAlreadyUsed,
)
from intranet.yandex_directory.src.yandex_directory.common.exceptions import (
    ImmediateReturn,
    AuthorizationError,
    NoScopesError,
    InvalidGroupTypeError,
    ConstraintValidationError,
    InvalidValue,
    CannotChangeOrganizationOwner,
    DomainUserCreationInSsoOrgsNotAllowed,
    SsoPersonalInfoPatchNotAllowed,
    SsoPasswordPatchNotAllowed,
    SsoUserUnblockNotAllowed,
    SsoUserBlockInProvisionedOrgNotAllowed,
    SsoUserUnblockInProvisionedOrgNotAllowed,
    SsoUserAliasesNotAllowed,
    LastOuterAdminDismissNotAllowed,
    CannotDismissOwner,
    TooManyAliases,
    UsersLimitApiError,
    UsersLimitError,
    MasterDomainNotFound,
    UserAlreadyExists,
)

from intranet.yandex_directory.src.yandex_directory.common.models.types import TYPE_USER
from intranet.yandex_directory.src.yandex_directory.common.components import component_registry
from intranet.yandex_directory.src.yandex_directory.common.utils import (
    build_list_response,
    build_fast_list_response,
    parse_birth_date,
    get_object_or_404,
    json_response,
    json_error,
    json_error_unknown,
    json_error_forbidden,
    json_error_required_field,
    json_error_service_unavailable,
    check_permissions,
    check_label_or_nickname_or_alias_is_uniq_and_correct,
    strip_object_values,
    Ignore,
    split_by_comma,
    get_user_data_from_blackbox_by_uid,
    ensure_date,
    get_user_data_from_blackbox_by_login,
    get_user_id_from_passport_by_login,
    json_error_not_found,
)
from intranet.yandex_directory.src.yandex_directory.core.actions import (
    action_user_add,
    action_user_modify,
    action_user_alias_add,
    action_user_alias_delete,
    action_security_user_grant_organization_admin,
    action_security_user_revoke_organization_admin,
    action_security_user_avatar_changed,
    action_organization_outer_deputy_delete,
    action_organization_outer_deputy_add,
)
from intranet.yandex_directory.src.yandex_directory.core.mailer.utils import (
    # send_welcome_email,
    send_change_password_email,
)
from intranet.yandex_directory.src.yandex_directory.core.models import (
    RobotServiceModel,
    ServiceModel,
    OrganizationModel,
    DomainModel,
    OrganizationRevisionCounterModel,
)
from intranet.yandex_directory.src.yandex_directory.core.models.group import (
    GroupModel
)
from intranet.yandex_directory.src.yandex_directory.core.models.organization import organization_type
from intranet.yandex_directory.src.yandex_directory.core.models.user import (
    UserModel,
    UserMetaModel,
    CONTACT_TYPES, UserRoles,
)
from intranet.yandex_directory.src.yandex_directory.core.permission.permissions import (
    global_permissions,
    user_permissions,
    get_permissions,
    all_users_permissions,
)
from intranet.yandex_directory.src.yandex_directory.core.utils import (
    build_email,
    ensure_integer,
    prepare_user,
    prepare_user_with_service,
    prepare_user_with_fields,
    create_user,
    only_ids,
    get_organization_admin_uid,
    is_yandex_team_uid,
    is_org_admin,
    is_deputy_admin,
    get_random_password,
    change_object_alias,
    get_user_role,
    is_common_user,
    strip_id,
    is_outer_uid,
    is_domain_uid,
    get_master_domain,
)
from intranet.yandex_directory.src.yandex_directory.core.utils.users.base import (
    is_organization_owner
)
from intranet.yandex_directory.src.yandex_directory.core.utils.organization import (
    is_sso_turned_on,
    is_provisioning_turned_on
)
from intranet.yandex_directory.src.yandex_directory.core.views.base import View
from intranet.yandex_directory.src.yandex_directory.core.views.departments import check_that_department_exists
from intranet.yandex_directory.src.yandex_directory.limits.models import OrganizationLimit
from intranet.yandex_directory.src.yandex_directory.directory_logging.logger import log
from intranet.yandex_directory.src.yandex_directory.passport.exceptions import (
    LoginNotavailable,
)
from intranet.yandex_directory.src.yandex_directory.swagger import uses_schema, uses_schema_for_get
from intranet.yandex_directory.src.yandex_directory.core.utils.domain import get_master_domain_from_db_or_domenator
from intranet.yandex_directory.src.yandex_directory.common import http_client

from intranet.yandex_directory.src.yandex_directory.sso.utils import UserPatchDataValidator

TYPE_USER_NAME = {
    'type': 'object',
    'properties': {
        'first': schemas.I18N_STRING,
        'last': schemas.I18N_STRING,
        'middle': schemas.I18N_STRING
    },
    'additionalProperties': False
}

USERS_OUT_SCHEMA = {
    'title': 'User',
    'type': 'object',
    'properties': {
        'name': TYPE_USER_NAME,
        'email': schemas.STRING_OR_NULL,
        'department_id': schemas.INTEGER,
        'position': schemas.I18N_STRING_OR_NULL,
        'about': schemas.I18N_STRING_OR_NULL,
        'gender': schemas.GENDER_TYPE,
        'birthday': schemas.DATE_OR_NULL,
        'external_id': schemas.STRING_OR_NULL,
        'nickname': {
            'type': 'string'
        },
        'login': {
            'type': 'string'
        },
        'aliases': {
            'type': ['array', 'null']
        },
        'contacts': {
            'type': ['array', 'null'],
            'items': {
                'type': 'object',
                'properties': {
                    'type': {
                        'enum': CONTACT_TYPES  # todo: test me
                    },
                    'label': schemas.I18N_STRING,
                    'value': {
                        'type': 'string'
                    },
                    'main': {
                        'type': 'boolean',
                    },
                    'alias': {
                        'type': 'boolean',
                    },
                    'synthetic': {
                        'type': 'boolean',
                    },
                },
                'required': [
                    'type',
                    'value',
                ]
            }
        },
        'is_dismissed': {
            'type': 'boolean',
        },
        'is_admin': {
            'type': 'boolean',
        },
        'role': {
            'enum': ['user', 'admin', 'deputy_admin'],
        },
        'timezone': schemas.STRING,
        'language': schemas.STRING,
    },
    'required': [
        'id'
    ],
}

USER_BASE_SCHEMA = USERS_OUT_SCHEMA.copy()
USER_BASE_SCHEMA['additionalProperties'] = False
USER_BASE_SCHEMA['properties']['department'] = {'type': ['object', 'null']}
USER_BASE_SCHEMA['required'] = ['name']
USER_BASE_SCHEMA['anyOf'] = [
    {'required': ['department_id']},
    {'required': ['department']}
]

USER_CREATE_SCHEMA = deepcopy(USER_BASE_SCHEMA)
USER_CREATE_SCHEMA['title'] = 'Create user'
USER_CREATE_SCHEMA['properties']['nickname'] = {
    'type': 'string',
}
USER_CREATE_SCHEMA['properties']['password'] = {
    'type': 'string',
}
USER_CREATE_SCHEMA['required'].remove('name')
USER_CREATE_SCHEMA['required'] += [
    'password',
    # нужен ли тут nickname?
    # пока что он проверяется внутри вьюшки,
    # совместно с логином. Проверку можно
    # будет выпилить, когда мы выпилим в поле
    # логин. При этом в схеме параметр nickname
    # нужно сделать обязательным.
]
# DIR-5330 мы добавляем новый режим hash|plain
USER_CREATE_SCHEMA['properties']['password_mode'] = {
    'enum': ['hash', 'plain']
}
USER_CREATE_SCHEMA['properties']['is_yamb_bot'] = {
    'type': 'boolean',
}
# запрещено задавать эти признаки для контакта при его создании
del USER_CREATE_SCHEMA['properties']['contacts']['items']['properties']['synthetic']
del USER_CREATE_SCHEMA['properties']['contacts']['items']['properties']['alias']

USER_UPDATE_SCHEMA = deepcopy(USER_BASE_SCHEMA)
USER_UPDATE_SCHEMA['title'] = 'Update user'
USER_UPDATE_SCHEMA['properties']['password_change_required'] = {
    'type': 'boolean',
    'description': 'Должен ли пользователь изменить пароль при первом входе.',
}
USER_UPDATE_SCHEMA['properties']['groups'] = schemas.LIST_OF_INTEGERS
USER_UPDATE_SCHEMA['properties']['password'] = {
    'type': 'string'
}
USER_UPDATE_SCHEMA['properties']['is_enabled'] = {'type': 'boolean'}
USER_UPDATE_SCHEMA['properties']['recovery_email'] = schemas.STRING_OR_NULL
USER_UPDATE_SCHEMA['properties']['external_email'] = schemas.STRING_OR_NULL
# для редактирования алиасов отдельные ручки
del USER_UPDATE_SCHEMA['properties']['aliases']
del USER_UPDATE_SCHEMA['properties']['nickname']
del USER_UPDATE_SCHEMA['required']
del USER_UPDATE_SCHEMA['anyOf']

USER_UPDATE_SCHEMA_V6 = deepcopy(USER_UPDATE_SCHEMA)
USER_UPDATE_SCHEMA_V6['properties']['contacts']['items']['additionalProperties'] = False
del USER_UPDATE_SCHEMA_V6['properties']['email']
# запрещено задавать эти признаки для контакта при его редактировании
del USER_UPDATE_SCHEMA_V6['properties']['contacts']['items']['properties']['synthetic']
del USER_UPDATE_SCHEMA_V6['properties']['contacts']['items']['properties']['alias']

USER_BULK_UPDATE_SCHEMA_ITEM = deepcopy(USER_UPDATE_SCHEMA)
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['password']
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['password_change_required']
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['external_email']
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['timezone']
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['language']
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['birthday']
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['is_enabled']
del USER_BULK_UPDATE_SCHEMA_ITEM['properties']['is_dismissed']
USER_BULK_UPDATE_SCHEMA_ITEM['properties']['id'] = schemas.INTEGER
USER_BULK_UPDATE_SCHEMA_ITEM['required'] = ['id']

USER_BULK_UPDATE_SCHEMA = {
    'title': 'User bulk update',
    'type': 'array',
    'items': USER_BULK_UPDATE_SCHEMA_ITEM
}

USERS_GET_SCHEMA = {
    'title': 'User',
    'type': 'object',
    'properties': {
        'resource_id': {
            'type': 'string'
        },
        'service_slug': {
            'type': 'string'
        },
        'nickname': {
            'type': ['string', 'array', 'integer'],
            'items': {
                'type': ['string', 'integer']
            }
        },
        'resource_relation_name': {
            'type': 'string'
        },
        'department_id': {
            'type': ['integer', 'array'],
            'items': {
                'type': 'integer'
            }
        },
        'recursive_department_id': {
            'type': ['integer', 'array'],
            'items': {
                'type': 'integer'
            }
        },
        'group_id': {
            'type': ['integer', 'array'],
            'items': {
                'type': 'integer'
            }
        },
        'recursive_group_id': {
            'type': ['integer', 'array'],
            'items': {
                'type': 'integer'
            }
        },
        'id': {
            'type': ['integer', 'array'],
            'items': {
                'type': 'integer'
            }
        }
    }
}

USERS_CREATE_ALIAS_SCHEMA = {
    'type': 'object',
    'title': 'Add user alias',
    'properties': {
        'name': schemas.STRING,
    },
    'required': [
        'name',
    ],
    'additionalProperties': False
}

USER_CREATE_OUTER_DEPUTY = {
    'type': 'object',
    'title': 'Add outer deputy admin',
    'properties': {
        'nickname': schemas.STRING,
    },
    'required': [
        'nickname',
    ],
    'additionalProperties': False
}


class UserView(View):
    def normalization(self, data, schema=None):
        """
        Метод, который вызывается в user_schema для нормализации входных
        данных, определяет эквивалентные параметры
        """
        if 'department_id' not in data and 'department' in data:
            data['department_id'] = data.get('department', {}).get('id')
            del data['department']
        return super(UserView, self).normalization(data, schema=schema)


class UserListView(UserView):
    allowed_ordering_fields = ['name', 'tracker_licenses']

    @uses_schema_for_get(USERS_GET_SCHEMA)
    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    # пользователя знать не обязательно
    # и эту ручку используют разные сервисы, чтобы по org_id
    # получить список всех пользователей
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        user_model = UserModel(main_connection)
        org_domain = get_master_domain_from_db_or_domenator(
            org_id=g.org_id,
            main_connection=main_connection,
            meta_connection=meta_connection,
            raise_on_empty_result=False,
        )
        org_domain = org_domain['name'] if org_domain else None

        def prepare_user_ex(u):
            return prepare_user(
                connection=main_connection,
                user=u,
                expand_contacts=True,
                api_version=request.api_version,
                org_domain=org_domain,
            )

        response = build_list_response(
            model=user_model,
            model_fields=[
                '*',
                'department.*',
                'groups.*',
                'is_admin',
                'is_robot',
                'service_slug',
                'role',
            ],
            model_filters=self._get_filters(main_connection, request.api_version),
            path=request.path,
            query_params=request.args.to_dict(),
            prepare_result_item_func=prepare_user_ex,
            # здесь задан специальный лимит, потому что без этого
            # плохо работает портал:
            # https://st.yandex-team.ru/DIR-974
            max_per_page=1000,
            distinct=True,
            order_by=self._get_ordering_fields()
        )
        return json_response(
            response['data'],
            headers=response['headers'],
        )

    get.__doc__ = """
Список сотрудников

Максимально, на странице может выводиться до 1000 сотрудников. Количество
данных на страницу, задается параметром per_page.

Поля сущности пользователя:

* **id** - идентификатор. Совпадает с UID пользователя в яндексе. Можно перечислять несколько id через запятую, чтобы запросить данные по нескольким пользователям сразу.
* **email** - электронный адрес
* **name** - имя. Содержит имя в ключе `first`, фамилию в ключе `last` и отчество в `middle` .
Все поля являются [локализованными](#lokalizovannyie-polya)
* **gender** - пол. Может принимать значения `male` и `female`
* **department** - департамент пользователя.
Содержит идентификатор и его название.
* **groups** - список групп, в которые входит
пользователей. Каждый элемент представляет из
себя объект с идентификатором и названием группы.
* **position** - должность ([локализованное поле](#lokalizovannyie-polya))
* **about** - о себе ([локализованное поле](#lokalizovannyie-polya))
* **birthday** - [дата](#data-i-vremya) рождения
* **nickname** - nickname пользователя (уникальный, назначается при создании)
* **aliases** - пользовательские паспортные алиасы (без доменной части)
* **contacts** - список контактов. Каждый контакт представляет из себя объект с ключами:
    * **type** - тип контакта. Ограничен следующим списком значений:
{contact_types}
    * **label** - любое текстовое описание контакта ([локализованное поле](#lokalizovannyie-polya))
    * **value** - значение контакта (например, email адрес)
    * **main** - главный рабочий контакт в пределах данного type-а
    * **alias** - пользовательский рабочий email-алиас
    * **synthetic** - означает, что этот контакт был сформирован искусственно,
    т.е. такой контакт в базе не хранится, и был искусственно склеен

---
tags:
  - Сотрудники
parameters:
      - in: query
        name: fields
        type: string
        description: список требуемых полей, перечисленных через запятую
      - in: query
        name: resource
        type: string
        description: список id ресурсов, перечисленных через запятую
      - in: query
        name: resource_relation_name
        type: string
        description: названия отношений, через запятую
      - in: query
        name: nickname
        type: string
        description: nickname сотрудника/сотрудников через запятую
      - in: query
        name: recursive_department_id
        type: string
        description: список id департаментов. Фильтр пользователей, входящих в департемант с учётом вложенности
      - in: query
        name: department_id
        type: string
        description: список id департаментов, перечисленных через запятую. Фильтр пользователей, входящих в департемант напрямую
      - in: query
        name: recursive_group_id
        type: string
        description: список id групп, перечисленных через запятую. Фильтр пользователей, входящих в группы с учётом вложенности
      - in: query
        name: group_id
        type: string
        description: список id групп, перечисленных через запятую. Фильтр пользователей, входящих в непосредственно в перечисленные группы
      - in: query
        name: id
        type: string
        description: список id пользователей, перечисленных через запятую
      - in: query
        name: page
        type: integer
        description: текущая страница
      - in: query
        name: per_page
        type: integer
        description: какое кол-во объектов выведено на странице
      - in: query
        name: is_dismissed
        type: string
        description: выводить удаленных ("ignore" - удаленные+текущие , "true" - только удаленные)
      - in: query
        name: service_slug
        type: string
        description: отдает роботов указанного сервиса


responses:
  200:
    description: Список сотрудников
        """.format(
        contact_types='\n'.join(['        * `%s`' % i for i in CONTACT_TYPES])
    )

    @uses_schema_for_get(USERS_GET_SCHEMA)
    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    # пользователя знать не обязательно
    # и эту ручку используют разные сервисы, чтобы по org_id
    # получить список всех пользователей
    @requires(org_id=True, user=False)
    def get_3(self, meta_connection, main_connection):
        fields = split_by_comma(request.args.to_dict().get('fields'))
        real_fields = fields[:]
        if 'services' in fields:
            if 'org_id' not in fields:
                real_fields.append('org_id')

        return self._get(meta_connection, main_connection, request.path, request.api_version, real_fields)

    get_3.__doc__ = """
Список сотрудников

Максимально, на странице может выводиться до 1000 сотрудников. Количество
данных на страницу, задается параметром per_page.

По умолчанию, отдаётся только id пользователя, однако можно
указать аргумент fields в котором через запятую перечислены
необходимые поля.

Поля доступные для пользователя:

* **org_id** - id организации.
* **id** - идентификатор. Совпадает с UID пользователя в яндексе. Можно перечислять несколько id через запятую, чтобы запросить данные по нескольким пользователям сразу.
* **email** - электронный адрес
* **name** - имя. Содержит имя в ключе `first`, фамилию в ключе `last` и отчество в `middle` .
Все поля являются [локализованными](#lokalizovannyie-polya)
* **gender** - пол. Может принимать значения `male` и `female`
* **department** - департамент пользователя.
Содержит идентификатор и его название.
* **groups** - список групп, в которые входит
пользователей. Каждый элемент представляет из
себя объект с идентификатором и названием группы.
* **position** - должность ([локализованное поле](#lokalizovannyie-polya))
* **about** - о себе ([локализованное поле](#lokalizovannyie-polya))
* **birthday** - [дата](#data-i-vremya) рождения
* **nickname** - nickname пользователя (уникальный, назначается при создании)
* **aliases** - пользовательские паспортные алиасы (без доменной части)
* **contacts** - список контактов. Каждый контакт представляет из себя объект с ключами:
    * **type** - тип контакта. Ограничен следующим списком значений:
{contact_types}
    * **label** - любое текстовое описание контакта ([локализованное поле](#lokalizovannyie-polya))
    * **value** - значение контакта (например, email адрес)
    * **main** - главный рабочий контакт в пределах данного type-а
    * **alias** - пользовательский рабочий email-алиас
    * **synthetic** - означает, что этот контакт был сформирован искусственно,
    т.е. такой контакт в базе не хранится, и был искусственно склеен

---
tags:
  - Сотрудники
parameters:
      - in: query
        name: fields
        type: string
        description: список требуемых полей, перечисленных через запятую
      - in: query
        name: resource
        type: string
        description: список id ресурсов, перечисленных через запятую
      - in: query
        name: resource_relation_name
        type: string
        description: названия отношений, через запятую
      - in: query
        name: nickname
        type: string
        description: nickname сотрудника/сотрудников через запятую
      - in: query
        name: recursive_department_id
        type: string
        description: список id департаментов. Фильтр пользователей, входящих в департемант с учётом вложенности
      - in: query
        name: department_id
        type: string
        description: список id департаментов, перечисленных через запятую. Фильтр пользователей, входящих в департемант напрямую
      - in: query
        name: recursive_group_id
        type: string
        description: список id групп, перечисленных через запятую. Фильтр пользователей, входящих в группы с учётом вложенности
      - in: query
        name: group_id
        type: string
        description: список id групп, перечисленных через запятую. Фильтр пользователей, входящих в непосредственно в перечисленные группы
      - in: query
        name: id
        type: string
        description: список id пользователей, перечисленных через запятую
      - in: query
        name: page
        type: integer
        description: текущая страница
      - in: query
        name: per_page
        type: integer
        description: какое кол-во объектов выведено на странице
      - in: query
        name: is_dismissed
        type: string
        description: выводить удаленных ("ignore" - удаленные+текущие , "true" - только удаленные)
      - in: query
        name: service_slug
        type: string
        description: отдает роботов указанного сервиса


responses:
  200:
    description: Список сотрудников
        """.format(
        contact_types='\n'.join(['        * `%s`' % i for i in CONTACT_TYPES])
    )

    @uses_schema_for_get(USERS_GET_SCHEMA)
    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get_7(self, meta_connection, main_connection):
        fields = split_by_comma(request.args.to_dict().get('fields'))
        real_fields = fields[:]
        if 'services' in fields:
            real_fields.extend(['services.slug', 'services.name'])

        return self._get(meta_connection, main_connection, request.path, request.api_version, real_fields)

    get_7.__doc__ = """
    Список сотрудников

    Максимально, на странице может выводиться до 1000 сотрудников. Количество
    данных на страницу, задается параметром per_page.

    По умолчанию, отдаётся только id пользователя, однако можно
    указать аргумент fields в котором через запятую перечислены
    необходимые поля.

    Поля доступные для пользователя:

    * **org_id** - id организации.
    * **id** - идентификатор. Совпадает с UID пользователя в яндексе. Можно перечислять несколько id через запятую, чтобы запросить данные по нескольким пользователям сразу.
    * **email** - электронный адрес
    * **name** - имя. Содержит имя в ключе `first`, фамилию в ключе `last` и отчество в `middle` .
    Все поля являются [локализованными](#lokalizovannyie-polya)
    * **gender** - пол. Может принимать значения `male` и `female`
    * **department** - департамент пользователя.
    Содержит идентификатор и его название.
    * **groups** - список групп, в которые входит
    пользователей. Каждый элемент представляет из
    себя объект с идентификатором и названием группы.
    * **position** - должность ([локализованное поле](#lokalizovannyie-polya))
    * **about** - о себе ([локализованное поле](#lokalizovannyie-polya))
    * **birthday** - [дата](#data-i-vremya) рождения
    * **nickname** - nickname пользователя (уникальный, назначается при создании)
    * **aliases** - пользовательские паспортные алиасы (без доменной части)
    * **services** - спсиок доступных пользователю сервисов, которые подключены в организации и находятся в состоянии ready.
    Сервис достпуен пользователю в следующих случаях:
    - сервис бесплатный
    - сервис платный в триальном периоде
    - сервис платный, триальный период закончился (или его нет), у пользователя есть лицензия

    Из-за особенности работы трекера временно отдаем все подключенные сервисы, доступные пользователю, в не зависимости от состояния ready.
    Так сделано потому, что трекер запрашивает всех пользователей в состоянии ready=False и мы не можем сравнить дату истечения триала,
    потому что она проставляется, когда сервис присылает нам ready. При этом считаем, что если ready=False, то сервис доступен всем.
    В этом месте будет неправильно работать для платных сервисов, у которых нет триала и trial_expires is NULL всегда.

    При повторном включении сервиса ручка работает аналогично: считаем, что сервис с ready=False доступен всем пользователям.
    Это место нужно исправить, потому что доступность сервиса нужно отдавать по наличию лицензий.
    * **contacts** - список контактов. Каждый контакт представляет из себя объект с ключами:
        * **type** - тип контакта. Ограничен следующим списком значений:
    {contact_types}
        * **label** - любое текстовое описание контакта ([локализованное поле](#lokalizovannyie-polya))
        * **value** - значение контакта (например, email адрес)
        * **main** - главный рабочий контакт в пределах данного type-а
        * **alias** - пользовательский рабочий email-алиас
        * **synthetic** - означает, что этот контакт был сформирован искусственно,
        т.е. такой контакт в базе не хранится, и был искусственно склеен

    ---
    tags:
      - Сотрудники
    parameters:
          - in: query
            name: fields
            type: string
            description: список требуемых полей, перечисленных через запятую
          - in: query
            name: resource
            type: string
            description: список id ресурсов, перечисленных через запятую
          - in: query
            name: resource_relation_name
            type: string
            description: названия отношений, через запятую
          - in: query
            name: nickname
            type: string
            description: nickname сотрудника/сотрудников через запятую
          - in: query
            name: recursive_department_id
            type: string
            description: список id департаментов. Фильтр пользователей, входящих в департемант с учётом вложенности
          - in: query
            name: department_id
            type: string
            description: список id департаментов, перечисленных через запятую. Фильтр пользователей, входящих в департемант напрямую
          - in: query
            name: recursive_group_id
            type: string
            description: список id групп, перечисленных через запятую. Фильтр пользователей, входящих в группы с учётом вложенности
          - in: query
            name: group_id
            type: string
            description: список id групп, перечисленных через запятую. Фильтр пользователей, входящих в непосредственно в перечисленные группы
          - in: query
            name: id
            type: string
            description: список id пользователей, перечисленных через запятую
          - in: query
            name: page
            type: integer
            description: текущая страница
          - in: query
            name: per_page
            type: integer
            description: какое кол-во объектов выведено на странице
          - in: query
            name: is_dismissed
            type: string
            description: выводить удаленных ("ignore" - удаленные+текущие , "true" - только удаленные)
          - in: query
            name: service_slug
            type: string
            description: отдает роботов указанного сервиса
          - in: query
            name: service
            type: string
            description: отдает пользователей, у который есть доступ к указанному сервису с данным slug, можно указать "service_slug".license_issued чтобы получить только пользователей с выданными лицензиями на сервис
          - in: query
            name: is_robot
            type: string
            description: если указывать 1, то отдает пользователей, кто является роботом. если 0, то всех не роботов
          - in: query
            name: user_type
            type: string
            description: отдает пользователей по типам user, robot


    responses:
      200:
        description: Список сотрудников
            """.format(
        contact_types='\n'.join(['        * `%s`' % i for i in CONTACT_TYPES])
    )

    @uses_schema_for_get(USERS_GET_SCHEMA)
    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get_9(self, meta_connection, main_connection):
        fields = split_by_comma(request.args.to_dict().get('fields'))
        real_fields = fields[:]
        if 'services' in fields:
            real_fields.extend(['services.slug', 'services.name'])
        keyset_pagination = False
        if not self._get_ordering_fields():
            keyset_pagination = True
        return self._get(
            meta_connection,
            main_connection,
            request.path,
            request.api_version,
            real_fields,
            keyset_pagination=keyset_pagination,
        )
    # Чтобы ручка отображалась в Playground, у неё должен быть докстринг
    get_9.__doc__ = get_7.__doc__

    def _get(self,
             meta_connection,
             main_connection,
             path,
             api_version,
             fields,
             real_fields=None,
             keyset_pagination=False,
             ):
        from intranet.yandex_directory.src.yandex_directory.core.utils import check_data_relevance

        org_domain = get_master_domain_from_db_or_domenator(
            org_id=g.org_id,
            main_connection=main_connection,
            meta_connection=meta_connection,
            raise_on_empty_result=False,
        )
        org_domain = org_domain['name'] if org_domain else None

        def prepare_user_ex(user):
            return prepare_user_with_fields(
                meta_connection,
                main_connection,
                user,
                fields=fields,
                api_version=api_version,
                org_domain=org_domain,
            )

        # Поле 'contacts' зависит от 'org_id' и 'nickname',
        # поэтому мы должны их запросить, но не отдавать при выдаче,
        # если их не запросили явно (в fields).
        real_fields = real_fields or fields[:]
        if 'contacts' in fields:
            if 'nickname' not in fields:
                real_fields.append('nickname')
            if 'org_id' not in fields:
                real_fields.append('org_id')

        if keyset_pagination:
            response = build_fast_list_response(
                model=UserModel(main_connection),
                model_fields=real_fields,
                model_filters=self._get_filters(main_connection, api_version),
                path=path,
                query_params=request.args.to_dict(),
                prepare_result_item_func=prepare_user_ex,
                max_per_page=1000,
            )
        else:
            response = build_list_response(
                model=UserModel(main_connection),
                model_fields=real_fields,
                model_filters=self._get_filters(main_connection, api_version),
                path=path,
                query_params=request.args.to_dict(),
                prepare_result_item_func=prepare_user_ex,
                # здесь задан специальный лимит, потому что без этого
                # плохо работает портал:
                # https://st.yandex-team.ru/DIR-974
                max_per_page=1000,
                order_by=self._get_ordering_fields()
            )

        org = OrganizationModel(main_connection).get(id=g.org_id)
        if not org['last_passport_sync'] or org['last_passport_sync'] < (utcnow() - timedelta(hours=12)):
            check_data_relevance(g.org_id, response['data']['result'])

        return json_response(
            response['data'],
            headers=response['headers'],
        )

    def _get_ordering_fields(self):
        # временно переопределим метод, чтобы сортировать по нужным полям

        ordering_fields = super(UserListView, self)._get_ordering_fields()
        if ordering_fields:
            if 'name' in ordering_fields:
                ind = ordering_fields.index('name')
                ordering_fields = ordering_fields[:ind] \
                                  + ['last_name', 'first_name', 'middle_name', 'nickname'] \
                                  + ordering_fields[ind + 1:]
            if '-name' in ordering_fields:
                ind = ordering_fields.index('-name')
                ordering_fields = ordering_fields[:ind] \
                                  + ['-last_name', '-first_name', '-middle_name', '-nickname'] \
                                  + ordering_fields[ind + 1:]
            if 'tracker_licenses' in ordering_fields:
                ind = ordering_fields.index('tracker_licenses')
                ordering_fields = ordering_fields[:ind] + [
                                      'tracker_licenses.exists',
                                      'id',
                                  ] + ordering_fields[ind + 1:]
            if '-tracker_licenses' in ordering_fields:
                ind = ordering_fields.index('-tracker_licenses')
                ordering_fields = ordering_fields[:ind] + [
                                      '-tracker_licenses.exists',
                                      'id',
                                  ] + ordering_fields[ind + 1:]

        return ordering_fields

    def _get_filters(self, main_connection, api_version):
        filters = {
            'org_id': g.org_id,
        }

        query_params = request.args.to_dict()

        # если ищем пользователей-роботов для сервиса
        # ограничим поиск uid-ми пользователей роботов
        if 'service_slug' in query_params:
            robots_uids = RobotServiceModel(main_connection) \
                          .filter(
                              # Раньше мы тут не фильтровали по org_id,
                              # потому что этого поля не было в табличке
                              # и это могло вызывать проблемы в производительности
                              # таких запросов.
                              org_id=g.org_id,
                              slug=query_params['service_slug']
                          ) \
                     .scalar('uid')
            if not robots_uids:
                robots_uids = [-1]

            filters['id'] = robots_uids

        is_dismissed = query_params.get('is_dismissed', '').lower()
        # игнорирум признак уволенности
        if is_dismissed == 'ignore':
            filters['is_dismissed'] = Ignore
        # отдаем только уволенных
        elif is_dismissed == 'true':
            filters['is_dismissed'] = True

        def process_list_parameter(name):
            """Добавляет параметр с именем `name` в словарь `filters`.
            Если параметр содержит запятые, то он будет посплитан
            и добавлен как список.
            """
            value = request.args.get(name)
            if value:
                value = value.split(',')
                value = (item.strip() for item in value)
                value = [_f for _f in value if _f]

                if len(value) == 1:
                    value = value[0]
                filters[name] = value

        # список параметров, по которым можно фильтровать
        filter_params = [
            'id',
            'nickname',
            'resource',
            'resource_relation_name',
            'department_id',
            'group_id',
            'recursive_department_id',
            'recursive_group_id',
            'role',
            'id__gt',
            'id__lt',
            'created',
            'created__gt',
            'created__lt',
            'updated_at',
            'updated_at__gt',
            'updated_at__lt',
            'is_robot',
            'user_type',
            'tracker_licenses',
            'suggest',
        ]
        for param in filter_params:
            process_list_parameter(param)

        if api_version > 6:
            process_list_parameter('service')

        if 'resource' in filters:
            # заполняем текущим сервисом
            filters['resource_service'] = g.service.identity

        return filters

    @uses_schema(USER_CREATE_SCHEMA)
    @scopes_required([scope.write_users, scope.manage_yamb_bots])
    @permission_required([global_permissions.add_users])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Создать нового сотрудника


        * **login** - устарело, в будущем данного поля не будет. Используй вместо него nickname.

        ---
        tags:
          - Сотрудники
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Сотрудник добавлен
          422:
            description: Какая-то ошибка валидации или пользователь существует с данным id
          403:
            description: {permissions}
        """
        # поле login мы собираемся выпилить со временем
        # https://st.yandex-team.ru/DIR-2128

        # TODO: change this when user_type refactor
        if is_sso_turned_on(main_connection, g.org_id):
            raise DomainUserCreationInSsoOrgsNotAllowed()

        if not data.get('login') and not data.get('nickname'):
            return json_error_required_field('nickname')

        is_yamb_bot = data.get('is_yamb_bot', False)
        nickname = data.get('nickname', '') or data.get('login', '')
        if is_yamb_bot:
            if not check_scopes(g.scopes, [scope.manage_yamb_bots]):
                raise NoScopesError(scope.manage_yamb_bots)

            has_appropriate_prefix = nickname.startswith(
                app.config['ROBOT_ACCOUNT_NICKNAME_PREFIX']
            )
            if not has_appropriate_prefix:
                raise InvalidValue(
                    message='Field "{field}" must start with "{prefix}".',
                    field='nickname',
                    prefix=app.config['ROBOT_ACCOUNT_NICKNAME_PREFIX'],
                )
        else:
            # запрещено создавать аккаунты с префиксом "robot-"
            if nickname.startswith(app.config['ROBOT_ACCOUNT_NICKNAME_PREFIX']):
                raise InvalidValue(
                    message='Field "{field}" can\'t start with "{prefix}".',
                    field='nickname',
                    prefix=app.config['ROBOT_ACCOUNT_NICKNAME_PREFIX'],
                )

        user = self._post_user(
            meta_connection,
            main_connection,
            g.org_id,
            data,
            g.user.passport_uid,
            user_type='yamb_bot' if is_yamb_bot else 'user',
            service_slug=g.service.identity if is_yamb_bot else None,
        )
        return json_response(
            prepare_user(
                main_connection,
                user,
                api_version=request.api_version,
                expand_contacts=True,
            ),
            status_code=201,
        )

    @staticmethod
    def _post_user(meta_connection,
                   main_connection,
                   org_id,
                   data,
                   auth_uid,
                   ignore_login_not_available=True,
                   user_type='user',
                   service_slug=None,
                   api_version=None,
                   ):
        """Этот метод используется во вью заведения нового сотрудника, а
        так же, в процедуре создания робота.

        При создании робота, в метод надо передать признак is_robot и service_slug,
        Это нужно для того, чтобы событие про создание пользователя
        содержало эти данные: DIR-2878.
        """

        is_robot = user_type != 'user'

        nickname = data.get('nickname')
        if 'login' in data and nickname is None:
            nickname = data['login'].split('@', 1)[0]

        if data.get('contacts'):  # Контакты могут быть null, в таком случае проверять и не надо
            validate_contacts(data['contacts'])

        external_id = data.get('external_id')

        user_id = data.get('id')

        if not is_robot and 'name' in data:
            name_plain = make_simple_strings(data['name'])
            first_name = name_plain['first']
            app.passport.validate_firstname(first_name)
            last_name = name_plain.get('last', '')
            if last_name:
                app.passport.validate_lastname(last_name)
        user_data = {
            'name': data.get('name'),
            'id': user_id,
            'nickname': nickname,
            'gender': data.get('gender', None),
            'department_id': data.get('department_id'),
            'position': data.get('position'),
            'about': data.get('about'),
            'birthday': parse_birth_date(data.get('birthday')),
            'contacts': data.get('contacts'),
            'aliases': data.get('aliases'),
            'external_id': external_id,
            'timezone': data.get('timezone'),
            'language': data.get('language'),

        }
        strip_object_values(user_data)

        # Проверяем логин на уникальность только если
        # учётка ещё не существует
        if not user_id:
            check_label_or_nickname_or_alias_is_uniq_and_correct(
                main_connection,
                user_data['nickname'],
                org_id,
            )

        if external_id:
            check_external_id_is_free(main_connection, org_id, external_id)

        try:
            created_user_info = create_user(
                meta_connection,
                main_connection,
                org_id,
                user_data,
                nickname=user_data['nickname'],
                password=data.get('password'),
                password_mode=data.get('password_mode', 'plain'),
                ignore_login_not_available=ignore_login_not_available,
                user_type=user_type,
                is_outer=data.get('is_outer')
            )

            # Для избранных организаций, мы не требуем от пользоватей подтверждения принятия
            # пользовательского соглашения (DIR-7410).
            if is_feature_enabled(meta_connection, org_id, NO_DOREGISTRATION):
                try:
                    app.passport.accept_eula(created_user_info['user']['id'])
                    sleep(0.5)  # даем шанс репликам паспорта
                except:
                    # Тут мы специально только логгируем и игнорируем ошибку,
                    # потому что самое плохое , что может случиться - это то,
                    # что мы попросим пользователя подтвердить ПС.
                    log.trace().error('Unable to auto-accept EULA')
        except UsersLimitError:
            raise UsersLimitApiError()
        except LoginNotavailable:
            # если аккаунт уже занят, то удалять его из Паспорта не нужно
            # Это плохой случай, который свидетельствует о рассинхронизации между Директорией и Паспортом
            # т.к. если логин на домене занят, то ошибка должна возникнуть выше в
            # check_label_or_nickname_or_alias_is_uniq_and_correct
            with log.name_and_fields('views.users.post', nickname=user_data['nickname']):
                log.trace().error('POST /user/: LoginNotavailable')

                # Создаем пользователя в Директории
                # но явным образом игнорируя ошибку из Паспорта о занятости этого пользователя.
                # https://st.yandex-team.ru/PDDADM-3650
                created_user_info = create_user(
                    meta_connection,
                    main_connection,
                    org_id,
                    user_data,
                    nickname=user_data['nickname'],
                    password=data.get('password'),
                    password_mode=data.get('password_mode', 'plain'),
                    ignore_login_not_available=True,
                    user_type=user_type,
                )
        except UserAlreadyExists:
            raise
        except Exception as e:
            with log.name_and_fields('views.users.post', nickname=user_data['nickname']):
                if getattr(e, 'log_level', 'ERROR') == 'WARNING':
                    log.warning('POST /user/: {}'.format(str(e)))
                else:
                    log.trace().error('POST /user/: unexpected error')
                raise

        user_id = created_user_info['user']['id']
        aliases = data.get('aliases')
        if aliases:
            for alias in aliases:
                change_object_alias(
                    main_connection=main_connection,
                    obj_id=user_id,
                    obj_type=TYPE_USER,
                    org_id=org_id,
                    alias=alias,
                    action_name='add',
                    action_func=action_user_alias_add,
                    author_id=auth_uid,
                    skip_connect=True,
                )

        user = UserModel(main_connection).get(
            user_id=user_id,
            org_id=org_id,
            fields=[
                '*',
                'timezone',
                'language',
                'department.*',
                'groups.*',
            ]
        )
        user['is_robot'] = is_robot
        user['user_type'] = user_type
        user['service_slug'] = service_slug

        if is_robot and not is_yandex_team_uid(user['id']):
            app.passport.accept_eula(user['id'])
        log.info('eula accepted')

        action_user_add(
            main_connection,
            org_id=org_id,
            author_id=auth_uid,
            object_value=user
        )

        # # приветственное письмо
        # portal = OrganizationModel(main_connection).get_organization_type(org_id) == organization_type.portal
        # if not portal and not is_robot:
        #     send_welcome_email(
        #         meta_connection=meta_connection,
        #         main_connection=main_connection,
        #         org_id=user['org_id'],
        #         uid=user['id']
        #     )

        return user

class BaseUserDetail(UserView):

    def _prepare_fields(self, args_dict):
        fields = split_by_comma(args_dict.get('fields'))
        if 'services.disk.has_paid_space' in fields and 'services' not in fields:
            fields.append('services')
        if 'id' not in fields:
            fields.append('id')
        return fields

    def _get(self, meta_connection, main_connection, user_id, api_version, fields, query_fields=None, is_cloud=False):
        if not is_cloud:
            user_id = ensure_integer(user_id, 'user_id')
        query_fields = query_fields or deepcopy(fields)
        if 'contacts' in query_fields or 'email' in query_fields:
            # Для того, чтобы отдать контакты, или поле email
            # нужно получить из базы ещё и эти поля,
            # потому что на их основе строится основной email
            # и сейчас эта логика дублируется в этой вьюшке,
            # и в той, что список отдаёт
            if 'nickname' not in query_fields:
                query_fields.append('nickname')
            if 'org_id' not in query_fields:
                query_fields.append('org_id')
            if is_cloud and 'cloud_uid' not in query_fields:
                query_fields.append('is_cloud')

        user = self._get_user_or_404(
            meta_connection,
            main_connection,
            UserModel(main_connection),
            g.org_id,
            user_id,
            fields=query_fields,
            api_version=api_version,
            is_cloud=is_cloud,
        )

        prepared_user = prepare_user_with_fields(
            meta_connection,
            main_connection,
            user,
            fields=fields,
            api_version=api_version,
        )
        return json_response(prepared_user)

    @staticmethod
    def _get_user_or_404(meta_connection,
                         main_connection,
                         user_model,
                         org_id,
                         user_id,
                         fields=None,
                         api_version=1,
                         is_cloud=False,
                         ):

        if fields:
            # TODO:
            # По идее, загрузку services, надо уносить в prefetch_related модели,
            # но пока оставим так, потому что этот метод использует get первой
            # версии
            if 'services' in fields:
                fields = deepcopy(fields)
                fields.append('org_id')  # для раскрытия сервисов пользователя, в его данных должно быть поле org_id
        try:
            user = get_object_or_404(
                model_instance=user_model,
                org_id=org_id,
                user_id=user_id,
                fields=fields,
                is_cloud=is_cloud,
            )
        except ImmediateReturn as err:
            # костыль для порталов, чтобы досоздавать пользователей,
            # которые есть в паспорте, но нет у нас, т.к. мы не можем синхронизировать свежие данные
            org_type = OrganizationModel(main_connection).get(org_id, fields=['organization_type'])['organization_type']
            if err.response.status_code == 404 and org_type == 'portal':
                user_data = get_user_data_from_blackbox_by_uid(user_id, attributes=['1017'])
                if not user_data or user_data['is_maillist']:
                    # на всякий случай еще защитимся от создания рассылок
                    raise
                create_user_from_bb_info(user_data, org_id)
                user = get_object_or_404(
                    model_instance=user_model,
                    org_id=org_id,
                    user_id=user_id,
                    fields=fields,
                )
            else:
                raise

        return user


class UserDetailView(BaseUserDetail):
    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    # пользователя знать не обязательно
    # но запрос должен быть аунтентифицирован
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection, user_id):
        """
        Информация о сотруднике
        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
          - in: query
            name: fields
            type: string
            description: название запрашиваемых полей через запятую
        responses:
          200:
            description: Словарь с информацией о сотруднике.
          404:
            description: Сотрудник не найден.
        """
        default_fields = [
            '*',
            'groups.*',
            'department.*',
            'is_admin',
            'is_enabled',
            'is_robot',
            'service_slug',
            'role',
        ]
        fields = split_by_comma(request.args.to_dict().get('fields')) or []
        user_id = ensure_integer(user_id, 'user_id')

        #multiorg_conflict
        user = self._get_user_or_404(
            meta_connection,
            main_connection,
            UserModel(main_connection),
            g.org_id,
            user_id,
            fields=default_fields,
        )

        if 'services' in fields:
            return json_response(
                prepare_user_with_service(
                    meta_connection,
                    main_connection,
                    user,
                    api_version=request.api_version,
                )
            )

        return json_response(prepare_user(
            main_connection,
            user,
            expand_contacts=True,
            api_version=request.api_version,
        ))

    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    # пользователя знать не обязательно
    # но запрос должен быть аунтентифицирован
    @requires(org_id=True, user=False)
    def get_3(self, meta_connection, main_connection, user_id):
        """
        Информация о сотруднике
        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
          - in: query
            name: fields
            type: string
            description: |
                Название запрашиваемых полей через запятую, возможные варианты:
                    id,
                    org_id,
                    login,
                    email,
                    department_id,
                    name,
                    gender,
                    position,
                    about,
                    birthday,
                    contacts,
                    aliases,
                    nickname,
                    is_dismissed,
                    external_id
        responses:
          200:
            description: Словарь с информацией о сотруднике.
          404:
            description: Сотрудник не найден.
        """
        fields = self._prepare_fields(request.args.to_dict())
        query_fields = deepcopy(fields)
        if 'organization.organization_type' in fields:
            query_fields.remove('organization.organization_type')
        return self._get(meta_connection, main_connection, user_id, request.api_version, fields, query_fields=query_fields)

    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    # пользователя знать не обязательно
    # но запрос должен быть аунтентифицирован
    @requires(org_id=True, user=False)
    def get_7(self, meta_connection, main_connection, user_id):
        """
        Информация о сотруднике
        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
          - in: query
            name: fields
            type: string
            description: |
                Название запрашиваемых полей через запятую, возможные варианты:
                    id
                    org_id
                    login
                    email
                    department_id
                    name
                    gender
                    position
                    about
                    birthday
                    contacts
                    aliases
                    nickname
                    is_dismissed
                    external_id
                    timezone
                    language
        responses:
          200:
            description: Словарь с информацией о сотруднике.
          404:
            description: Сотрудник не найден.
        """
        fields = self._prepare_fields(request.args.to_dict())
        query_fields = deepcopy(fields)
        if 'services' in fields:
            query_fields.remove('services')
            query_fields.extend(['services.name', 'services.slug'])
        if 'organization.organization_type' in fields:
            query_fields.remove('organization.organization_type')

        return self._get(meta_connection, main_connection, user_id, request.api_version, fields, query_fields)

    def _patch(self, meta_connection, main_connection, data, user_id):
        patched_user_id = ensure_integer(user_id, 'user_id')

        if g.user.role:
            user_role = g.user.role
        else:
            user_role = get_user_role(meta_connection, main_connection, g.org_id, g.user.passport_uid)

        if is_common_user(role=user_role):
            self._check_simple_user_permissions(data)

        patched_user = self._get_user_or_404(
            meta_connection,
            main_connection,
            UserModel(main_connection),
            g.org_id,
            patched_user_id,
            fields=['role', 'user_type', 'is_sso'],
        )

        is_sso_org = is_sso_turned_on(main_connection, g.org_id)
        is_provisioned = is_provisioning_turned_on(main_connection, g.org_id)

        # пачка запретов для SSO пользователей
        # FIXME: поправить при рефакторинге типов пользователей

        user_type_fieldname = 'is_sso'
        sso_user_type = True

        if is_provisioned:
            if patched_user[user_type_fieldname] == sso_user_type and UserPatchDataValidator.has_block(data):
                raise SsoUserBlockInProvisionedOrgNotAllowed()

            if patched_user[user_type_fieldname] == sso_user_type and UserPatchDataValidator.has_unblock(data):
                raise SsoUserUnblockInProvisionedOrgNotAllowed()

        if patched_user[user_type_fieldname] == sso_user_type and UserPatchDataValidator.has_personal_info(data):
            raise SsoPersonalInfoPatchNotAllowed()

        if patched_user[user_type_fieldname] == sso_user_type and UserPatchDataValidator.has_password(data):
            raise SsoPasswordPatchNotAllowed()

        if not is_sso_org:
            if patched_user[user_type_fieldname] == sso_user_type and UserPatchDataValidator.has_unblock(data):
                raise SsoUserUnblockNotAllowed()

        # DIR-5672
        # запрещено отбирать права у владельца организации
        if data.get('role') == 'user' or data.get('is_admin') is False:
            if is_organization_owner(main_connection, g.org_id, patched_user_id):
                raise CannotChangeOrganizationOwner()

            # запрещено отбирать роль у последнего портального админа в sso организации
            if is_sso_turned_on(main_connection, g.org_id) and \
               UserModel(main_connection).is_last_admin(g.org_id, patched_user_id):
                raise LastOuterAdminDismissNotAllowed()

        # Заместителю админа нельзя редактировать админа
        if is_deputy_admin(role=user_role) and is_org_admin(role=patched_user['role']):
            return json_error_forbidden()

        # Не разрешаем редактировать робота напрямую через API
        if patched_user['user_type'] == 'robot':
            return json_error_forbidden()

        if data.get('is_dismissed', False):
            return self._dismiss(
                meta_connection=meta_connection,
                main_connection=main_connection,
                org_id=g.org_id,
                user_id=patched_user_id,
                author_id=g.user.passport_uid,
                role=user_role,
            )

        with log.fields(user_id=patched_user_id, org_id=g.org_id):
            # В случае чего, _patch_user выбросит исключение, которое
            # будет обработано на уровне базовой вьюшки
            result = self._patch_user(
                meta_connection,
                main_connection,
                g.org_id,
                data,
                patched_user_id,
                g.user.passport_uid,
                user_is_admin=is_org_admin(user_role),
                api_version=request.api_version,
                fields=split_by_comma(request.args.to_dict().get('fields')),
            )
            return json_response(result)

    @uses_schema(USER_UPDATE_SCHEMA)
    @scopes_required([scope.write_users])
    @permission_required([
        user_permissions.edit,
        user_permissions.edit_contacts,
        user_permissions.edit_birthday,
        global_permissions.leave_organization,
    ], TYPE_USER, any_permission=True)
    @requires(org_id=True, user=True)
    def patch(self, meta_connection, main_connection, data, user_id):
        """
        Изменить сотрудника
        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          200:
            description: Информация обновлена.
          404:
            description: Сотрудник не найден.
          403:
            description: {permissions}
          422:
            description: Какая-то ошибка валидации
        """
        return self._patch(meta_connection, main_connection, data, user_id)

    @uses_schema(USER_UPDATE_SCHEMA_V6)
    @scopes_required([scope.write_users])
    @permission_required([
        user_permissions.edit,
        user_permissions.edit_contacts,
        user_permissions.edit_birthday,
        global_permissions.leave_organization,
    ], TYPE_USER, any_permission=True)
    @requires(org_id=True, user=True)
    def patch_6(self, meta_connection, main_connection, data, user_id):
        """
        Изменить сотрудника
        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          200:
            description: Информация обновлена.
          404:
            description: Сотрудник не найден.
          403:
            description: {permissions}
          422:
            description: Какая-то ошибка валидации
        """
        return self._patch(meta_connection, main_connection, data, user_id)

    def _check_simple_user_permissions(self, data):
        """
        Обычный пользователь может только редактировать свои контакты, день рождения и удалять себя из групп
        """
        allowed_keys = ('contacts', 'groups', 'birthday', 'is_dismissed')

        if not all(key in allowed_keys for key in data):
            raise AuthorizationError(
                'User can only edit these fields: {0}'.format(
                    ', '.join(allowed_keys)
                )
            )

    @classmethod
    def _patch_user(cls,
                    meta_connection,
                    main_connection,
                    org_id,
                    data,
                    user_id,
                    auth_uid,
                    user_is_admin,
                    api_version,
                    fields=None,
                    skip_check_permissions=False,
                    ):
        strip_object_values(data)
        user_model = UserModel(main_connection)

        # для старших версий выбираем из базы только изменяемые поля и те,
        # которые были переданы в параметрах
        if api_version > 7:
            # Из базы мы достаём не только те поля, которые
            # были изменены, но и все "простые" поля, потому что
            # нам надо чтобы они были сохранены с событии и
            # не пострадала обратная совместимость и пользователи API.
            # Инцидент с поломкой в этом месте описан в тикете:
            # https://st.yandex-team.ru/DIR-5828
            query_fields = set(['*'])
            query_fields.update(fields or [])

            changed_fields = list(map(strip_id, list(data.keys())))
            for field in changed_fields:
                if UserModel.prefetch_related_fields.get(field):
                    query_fields.add('{}.*'.format(field))
                elif field in UserModel.all_fields:
                    query_fields.add(field)

            # Чтобы сгенерились события department_user_added, department_user_deleted
            # когда сотрудника перевели из отдела в отдел, в данных о пользователе должна
            # быть полная цепочка событий. Так же departments будет включён в diff
            if 'department' in changed_fields:
                query_fields.add('departments')

        else:
            query_fields = ['**']

        # Так как мы по ходу дела будем убирать из data те поля, которые
        # не хранятся в базе, а потом из остатка сделаем UPDATE, то тут надо
        # разморозить словарь
        data = dict(data)

        old_user = cls._get_user_or_404(
            meta_connection,
            main_connection,
            user_model,
            org_id,
            user_id,
            fields=query_fields,
        )

        org_domain = DomainModel(main_connection).find(
            filter_data=dict(org_id=org_id),
            one=True,
        )
        if org_domain:
            org_domain = org_domain['name']
        else:
            org_domain = None
        # Возьмем тут все права 1 раз, тчтобы потом не вызывать несколько раз ф-ю проверки прав
        # и не делать лишних запросов в базу, чтобы получить домен
        if skip_check_permissions:
            # dirty hack
            # Нужно, что работала крон-команда синка со стаффом import-organization
            # Она запускается для yandex-team организации, а для таких организаций нет многих прав
            permissions = copy(all_users_permissions)
        else:
            permissions = get_permissions(
                meta_connection,
                main_connection,
                auth_uid,
                TYPE_USER,
                user_id,
                org_id,
            )

        is_enabled = data.pop('is_enabled', None)
        if isinstance(is_enabled, bool):
            cls._raise_if_no_permission(permissions, user_permissions.block)
            UserDetailView._change_is_enabled_status_for_user(
                main_connection,
                user_id=old_user['id'],
                is_enabled=is_enabled,
                org_id=org_id,
                author_id=auth_uid,
            )

        if 'name' in data:
            name_plain = make_simple_strings(data['name'])
            first_name = name_plain['first']
            app.passport.validate_firstname(first_name)
            last_name = name_plain.get('last', '')
            if last_name:
                app.passport.validate_lastname(last_name)

        if 'password_change_required' in data and 'password' not in data:
            raise ImmediateReturn(response=json_error_required_field('password'))

        if 'external_email' in data:
            if data.get('password_change_required'):
                # отправляем письмо с временным паролем и просьбой установить новый пароль
                login = build_email(main_connection, old_user['nickname'], org_id)
                send_change_password_email(
                    main_connection,
                    to_email=data.pop('external_email'),
                    password=data['password'],
                    login=login,
                    org_id=org_id,
                )
            else:
                raise ImmediateReturn(response=json_error_required_field('password_change_required'))

        if 'password' in data:
            cls._raise_if_no_permission(permissions, user_permissions.change_password)
            UserDetailView._change_password_for_user(
                main_connection,
                user=old_user,
                new_password=data.pop('password'),
                force_next_login_password_change=data.pop('password_change_required', False),
                org_id=org_id,
                author_id=auth_uid,
            )

        if 'is_admin' in data:
            cls._raise_if_no_permission(permissions, user_permissions.make_admin)
            if data['is_admin']:
                UserModel(main_connection).make_admin_of_organization(org_id, user_id)
                # логируем
                action_security_user_grant_organization_admin(
                    main_connection,
                    org_id=org_id,
                    author_id=auth_uid,
                    object_value=old_user,
                )
            else:
                user_model.revoke_admin_permissions(org_id=org_id, user_id=user_id)
                # логируем
                action_security_user_revoke_organization_admin(
                    main_connection,
                    org_id=org_id,
                    author_id=auth_uid,
                    object_value=old_user,
                )
            del data['is_admin']

        if 'role' in data:
            cls._raise_if_no_permission(permissions, user_permissions.change_role)
            UserModel(main_connection).change_user_role(org_id, user_id, data['role'], auth_uid, old_user)
            del data['role']

        if 'groups' in data:
            groups = GroupModel(main_connection).find({
                'id': data['groups'],
                'org_id': org_id,
            })
            for group in groups:
                if group['type'] != 'generic':
                    raise InvalidGroupTypeError(
                        group['type']
                    )

                old_user_group_ids = only_ids(old_user['groups'])
                if not user_is_admin and group['id'] not in old_user_group_ids:
                    # обычный пользователь не может сам себя добавлять в команды
                    raise AuthorizationError('User can\'t add himself to groups')

        if 'department_id' in data:
            check_that_department_exists(
                main_connection,
                org_id,
                data['department_id'],
                error_code='department_not_found',
                error_message='Unable to find department with id={id}',
            )

        if 'birthday' in data:
            cls._raise_if_no_permission(permissions, user_permissions.edit_birthday)
            data['birthday'] = parse_birth_date(data['birthday'])

        if data.get('contacts'):  # Контакты могут быть null, в таком случае проверять и не надо
            cls._raise_if_no_permission(permissions, user_permissions.edit_contacts)
            validate_contacts(data['contacts'])

        # Если есть поля, которые будем отправлять в паспорт - проверям, не порталный ли пользователь.
        # Для портальных пользователей изменения пасспортных данных запрещено
        passport_fields = ('timezone', 'language', 'gender', 'name')
        if set(data.keys()).intersection(passport_fields):
            cls._raise_if_no_permission(permissions, user_permissions.edit_info)

        if 'timezone' in data:
            UserModel(main_connection).change_timezone(old_user['id'], data['timezone'])
            del data['timezone']

        if 'language' in data:
            UserModel(main_connection).change_language(old_user['id'], data['language'])
            del data['language']

        if data:
            user_model.update_one(
                filter_data={
                    'id': old_user['id'],
                    'org_id': org_id,
                },
                update_data=data
            )

        user = cls._get_user_or_404(
            meta_connection,
            main_connection,
            user_model,
            org_id,
            user_id,
            fields=query_fields,
        )
        action_user_modify(
            main_connection,
            org_id=org_id,
            author_id=auth_uid,
            object_value=user,
            old_object=old_user,
        )

        if api_version > 7:
            return prepare_user_with_fields(
                meta_connection,
                main_connection,
                user,
                fields=fields,
                api_version=api_version,
            )
        else:
            return prepare_user(
                main_connection,
                user,
                expand_contacts=True,
                api_version=api_version,
            )

    @staticmethod
    def _change_password_for_user(main_connection,
                                  user,
                                  new_password,
                                  force_next_login_password_change,
                                  org_id,
                                  author_id):
        UserModel(main_connection).change_password(
            org_id=org_id,
            author_id=author_id,
            user_id=user['id'],
            new_password=new_password.strip(),
            force_next_login_password_change=force_next_login_password_change,
        )

    @staticmethod
    def _change_is_enabled_status_for_user(main_connection,
                                           user_id,
                                           is_enabled,
                                           org_id,
                                           author_id):
        try:
            blocked_status_changed = UserModel(main_connection).change_is_enabled_status(
                org_id=org_id,
                author_id=author_id,
                user_id=user_id,
                is_enabled=is_enabled,
            )
        except (requests.exceptions.ConnectionError, requests.exceptions.Timeout) as exc:
            with log.name_and_fields('views.users.block', exception=exc, is_enabled=is_enabled):
                log.error('User blocking error')
            raise ImmediateReturn(json_error_service_unavailable())

        if not blocked_status_changed:
            if is_enabled:
                raise ImmediateReturn(
                    json_error(
                        422,
                        'unable_to_unblock',
                        'Unable to unblock user',
                    )
                )
            else:
                raise ImmediateReturn(
                    json_error(
                        422,
                        'unable_to_block',
                        'Unable to block user',
                    )
                )

    @staticmethod
    def _change_avatar_for_user(meta_connection,
                                main_connection,
                                user,
                                author_id,
                                file_img=None,
                                url=None):
        # пользователь может сменить аватарку сам себе,
        # админ можем сменить аватарку любому сотруднику,
        # пользователь не может сменить аватарку дргуим пользователям

        check_permissions(
            meta_connection,
            main_connection,
            [user_permissions.change_avatar],
            object_type=TYPE_USER,
            object_id=user['id']
        )
        app.passport.change_avatar(user['id'], file_img, url)

        action_security_user_avatar_changed(
            main_connection,
            org_id=g.org_id,
            author_id=author_id,
            # TODO: выяснить у Вики, почему тут не весь объект, а только id
            object_value=user['id']
        )

        action_user_modify(
            main_connection,
            org_id=g.org_id,
            author_id=author_id,
            object_value=user,
            old_object=user,
        )

        return json_response(
            prepare_user(
                main_connection,
                user,
                api_version=request.api_version,
            )
        )

    @classmethod
    def _dismiss(cls,
                 meta_connection,
                 main_connection,
                 org_id,
                 user_id,
                 author_id,
                 skip_check_permissions=False,
                 api_version=None,
                 role=None):
        """
        Увольнение сотрудника
        """
        if not skip_check_permissions:
            if author_id == user_id:
                # Пользователь хочет уволить себя
                permission = global_permissions.leave_organization
            else:
                permission = user_permissions.dismiss
            check_permissions(
                meta_connection,
                main_connection,
                [permission],
                object_type=TYPE_USER,
                object_id=user_id,
            )
        # Не позволяем удалить (уволить) админа создателя организации.
        # Временное решение, потом поменяем
        admin_uid = get_organization_admin_uid(
            main_connection,
            org_id
        )
        if user_id == admin_uid:
            raise CannotDismissOwner()

        if role and role == 'admin':
            if not UserModel(main_connection).is_user_admin(user_id=user_id, exclude={org_id}):
                app.passport.reset_admin_option(user_id)

        user_model = UserModel(main_connection)

        if is_sso_turned_on(main_connection, org_id) and user_model.is_last_admin(g.org_id, user_id):
            raise LastOuterAdminDismissNotAllowed()

        old_user = cls._get_user_or_404(
            meta_connection,
            main_connection,
            user_model,
            org_id,
            user_id,
            fields=[
                '*',
                'department.*',
                'groups.*',
            ],
            api_version=api_version
        )
        user_model.dismiss(org_id, user_id, author_id=author_id, old_user=old_user)
        user = old_user.copy()
        user['is_dismissed'] = True

        return json_response(
            prepare_user(
                main_connection,
                user,
                expand_contacts=True,
                api_version=api_version or request.api_version,
            ),
            status_code=200
        )


class CloudUserDetailView(BaseUserDetail):
    methods = ['get']

    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get_6(self, meta_connection, main_connection, cloud_uid):
        """
        Информация о сотруднике по облачному uid
        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: cloud_uid
            required: true
            type: string
          - in: query
            name: fields
            type: string
            description: |
                Название запрашиваемых полей через запятую, возможные варианты:
                    id,
                    org_id,
                    login,
                    email,
                    department_id,
                    name,
                    gender,
                    position,
                    about,
                    birthday,
                    contacts,
                    aliases,
                    nickname,
                    is_dismissed,
                    external_id,
                    cloud_uid,
        responses:
          200:
            description: Словарь с информацией о сотруднике.
          404:
            description: Сотрудник не найден.
        """
        fields = self._prepare_fields(request.args.to_dict())
        return self._get(meta_connection, main_connection, cloud_uid, request.api_version, fields, is_cloud=True)



class UserChangeAvatarView(UserView):
    @scopes_required([scope.write_users])
    @permission_required([user_permissions.change_avatar],
                         TYPE_USER, any_permission=True)
    def post(self, meta_connection, main_connection, user_id):
        """Сменить аватарку (проксируем аватарку в паспорт).
        Поменять аватарку можно 2мя способами:
        1. По урлу:
            * **avatar_url** - url с аватаркой
            Ожидаем Content-Type: 'application/json'

        2. Через файл:
            * **avatar_file** - файл с аватаркой
            Ожидаем Content-Type, который будет начинаться с multipart/form-data.
            Например:
            multipart/form-data; boundary=<Asrf456BGe4>

        ---
        tags:
          - Сотрудники

        parameters:
            - in: path
              name: user_id
              required: true
              type: integer

        responses:
          200:
            description: Аватарка изменена.
          404:
            description: Сотрудник не найден.
          403:
            description: {permissions}
          422:
            description: Какая-то ошибка валидации
        """
        user_id = ensure_integer(user_id, 'user_id')

        user = UserModel(main_connection).get(
            user_id,
            org_id=g.org_id,
            fields=[
                '*',
                'department.*',
                'groups.*',
            ]
        )
        if request.headers['Content-Type'].startswith('multipart/form-data'):
            # http://flask.pocoo.org/docs/0.11/patterns/fileuploads/
            data = request.files or {}
            img = data.get('avatar_file')

            if not img:
                return json_error_required_field('avatar_file')

            return UserDetailView._change_avatar_for_user(
                meta_connection=meta_connection,
                main_connection=main_connection,
                user=user,
                author_id=g.user.passport_uid,
                file_img=img,
            )

        # теперь проверяем, не загружают ли картинку по урлу
        data = request.get_json()
        if data and data.get('avatar_url', False):
            return UserDetailView._change_avatar_for_user(
                meta_connection=meta_connection,
                main_connection=main_connection,
                user=user,
                author_id=g.user.passport_uid,
                url=data.get('avatar_url'),
            )

        return json_error_required_field('avatar_url')


class UserAliasesListView(UserView):

    @uses_schema(USERS_CREATE_ALIAS_SCHEMA)
    @scopes_required([scope.write_users])
    @permission_required([user_permissions.change_alias])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data, user_id):
        """
        Добавить алиас пользователю

        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
          - in: body
            name: body
        responses:
          201:
            description: Алиас добавлен
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """

        if is_sso_turned_on(main_connection, g.org_id):
            raise SsoUserAliasesNotAllowed()

        aliases = UserModel(main_connection)\
            .filter(id=user_id)\
            .fields('aliases').\
            one()

        components = component_registry()
        with session_compat(meta_connection) as session:
            user_aliases_max_count = components.organization_limiter.get_limit(
                g.org_id,
                OrganizationLimit.USER_ALIASES_MAX_COUNT,
                session=session,
            )

        if len(aliases['aliases']) >= user_aliases_max_count:
            raise TooManyAliases()
        updated_user = change_object_alias(
            main_connection=main_connection,
            obj_id=user_id,
            obj_type=TYPE_USER,
            org_id=g.org_id,
            alias=data['name'],
            action_name='add',
            action_func=action_user_alias_add,
            author_id=g.user.passport_uid,
        )

        return json_response(
            data=dict(updated_user),
            status_code=201
        )


class UserAliasesDetailView(UserView):

    @permission_required([user_permissions.change_alias])
    @scopes_required([scope.write_users])
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, user_id, alias):
        """
        Удалить алиас у пользователя

        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
          - in: path
            name: alias
            required: true
            type: string
        responses:
          204:
            description: Алиас удален
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """

        change_object_alias(
            main_connection=main_connection,
            obj_id=user_id,
            obj_type=TYPE_USER,
            org_id=g.org_id,
            alias=alias,
            action_name='delete',
            action_func=action_user_alias_delete,
            author_id=g.user.passport_uid,
        )

        return json_response({}, status_code=204)


class UserSettingsListView(UserView):

    @internal
    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    @requires(org_id=True, user=True)
    def get(self, meta_connection, main_connection):
        """
        Настройки текущего пользователя
        Сейчас всегда отдаем
            {
                "header": "connect",
                "shared_contacts": False
            }
        ---
        tags:
          - Сотрудники
        responses:
          200:
            description: Словарь с настройками сотрудника.
        """
        # тут отдаем только 1 настройку (какую шапку показывать)
        # это нужно для других сервисов типа Диска и Почты
        settings = OrganizationModel(main_connection).get(
            g.org_id,
            fields=['header', 'shared_contacts']
        )
        del settings['id']
        return json_response(settings)


class UserTokenView(UserView):

    @cached_in_memory_with_ttl(600)
    def singleton_robots_uids(self, meta_connection):
        """
        @rtype: set([int])
        """
        from intranet.yandex_directory.src.yandex_directory.core.utils import robots
        return set(robots.singleton_robot_uids(meta_connection))

    @internal
    @no_permission_required
    @no_scopes
    @requires(org_id=False, user=False)
    def get(self, meta_connection, main_connection, user_id):
        """
        Получаем токен роботного пользователя для сервиса

        Пример ответа:

            {
                "token": "435ref3242r32143",
                "expires_in": 31536000
            }

        Где expires_in, это время жизни токена в секундах.

        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: user_id
            required: true
            type: integer
        responses:
          200:
            description: Словарь с токеном для пользователя.
          404:
            description: Сотрудник-робот не найден.
          403:
            description: Нет прав.
        """
        # пускаем только доверенные сервисы
        if not g.service:
            return json_error_forbidden()

        user_id = ensure_integer(user_id, 'user_id')

        if user_id in self.singleton_robots_uids(meta_connection):
            return json_error_forbidden()

        meta_user = get_object_or_404(
            UserMetaModel(meta_connection),
            user_id=user_id
        )
        org_id = meta_user['org_id']
        shard = get_shard(meta_connection, org_id=org_id)

        with get_main_connection(shard) as connection:
            # если пользователь не робот то 403
            if not RobotServiceModel(connection) \
               .filter(org_id=org_id, uid=user_id, service_id=g.service.id) \
               .count():
                return json_error_forbidden()
            nickname = UserModel(connection).get(user_id)['nickname']
            # Ручка OAuth требует полного логина, который включает в себя
            # nickname и доменную часть.
            username = build_email(
                connection,
                nickname,
                org_id,
            )

        client_id = ServiceModel(meta_connection).get(g.service.id)['client_id']
        internal_client_id, internal_client_secret = get_oauth_service_data(client_id)
        # если не заданы внутренние client_id и secret то 403
        if internal_client_id is None or internal_client_secret is None:
            return json_error_forbidden()

        # мы у себя не храним пароль, а меняем каждый раз на новый
        new_password = get_random_password()
        app.passport.change_password(
            uid=user_id,
            new_password=new_password,
        )
        # принимаем ПС, потому что метод сброса пароля сбрасывает и ПС
        app.passport.accept_eula(user_id)

        # получаем OAuth токен для пользователя
        # https://wiki.yandex-team.ru/oauth/token/#granttypepassword
        status_code, token_info = self._get_token(
            username=username,
            password=new_password,
            client_id=internal_client_id,
            client_secret=internal_client_secret,
        )

        # не логируем авторизационные данные
        if 'access_token' in token_info:
            token = token_info['access_token']
            token_info['access_token'] = "****"

        with log.fields(client_id=internal_client_id,
                        username=username,
                        oauth_response=token_info,
                        status_code=status_code):

            if status_code == 200:
                log.debug('User token response')
            else:
                log.error('User token response')

        if status_code != 200 or 'error' in token_info:
            return json_error_unknown()

        return json_response({
            'token': token,
            'expires_in': token_info['expires_in'],
        })

    @retry(stop_max_attempt_number=3, retry_on_exception=lambda x: isinstance(x, HTTPError))
    def _get_token(self, username, password, client_id, client_secret):
        url = app.config['OAUTH_BASE_URL'] + 'token'

        response = http_client.request(
            'post',
            url,
            data={
                'grant_type': 'password',
                'client_id': client_id,
                'client_secret': client_secret,
                'username': username,
                'password': password,
            },
        )
        response.raise_for_status()
        return response.status_code, response.json()


class UsersBulkUpdateView(UserView):
    @uses_schema(USER_BULK_UPDATE_SCHEMA)
    @scopes_required([scope.write_users])
    @permission_required([
        user_permissions.edit,
        user_permissions.edit_contacts,
    ], TYPE_USER, any_permission=True)
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Балковое обновление пользователей
        При ошибки валидации одного обновления, не меняется никто

        Пример входных данных:

            [
                {
                    "id": id пользователя,
                    "role": "deputy_admin",
                    "department_id": ...,
                },
                {
                    "id": id пользователя,
                    "role": "admin",
                },
            ]

        ---
        responses:
          200:
            description: Операция выполнена успешно
          422:
            description: Какая-то ошибка валидации
          403:
            description: {permissions}
        """

        with main_connection.begin_nested():
            # проверим, что у владельца не отнимают роль админа
            users_to_change_role = {user['id'] for user in data if user.get('role') == 'user'}

            if users_to_change_role:
                for uid in users_to_change_role:
                    if is_organization_owner(main_connection, g.org_id, uid):
                        raise CannotChangeOrganizationOwner()

            for user in data:
                UserDetailView()._patch(meta_connection, main_connection, data=user, user_id=user['id'])

        return json_response(
            data={},
            status_code=200
        )


def validate_site_contact(value):
    if value.lstrip().startswith('javascript:'):
        raise ConstraintValidationError(
            'invalid_value',
            'Value {value} is invalid for contact type "site"',
            value=value,
        )


CONTACT_VALIDATORS = {
    'site': validate_site_contact,
}


def validate_contacts(contacts):
    """Проверяет, что со списокм контактов всё хорошо, и можно сохранить его в базу.
       Пока что проверяем лишь то, что контакт site не начинается с javascript:

       Если что не так, функция генерит исключение ConstraintValidationError,
       вызывающее 422 ответ.
    """
    for contact in contacts:
        contact_type = contact['type']
        validator = CONTACT_VALIDATORS.get(contact_type)
        if validator:
            validator(contact['value'])


def create_user_from_bb_info(user_data, org_id):
    # создаем пользователя на основе данных паспорта, используется только для порталов
    with log.fields(user_data=user_data, org_id=org_id):
        with get_meta_connection(for_write=True) as meta:
            shard = get_shard(meta, org_id)
            with get_main_connection(shard=shard, for_write=True) as main:
                # сначала проверим, что пользователь имеет отношение к организации
                login = user_data['login']
                domain_name = login[login.index('@')+1:]
                if not (org_id in user_data['org_ids'] or
                        DomainModel(main).filter(name=domain_name, owned=True, org_id=org_id).one()):
                    return

                gender_map = {
                    '1': 'male',
                    '2': 'female',
                }
                name = {
                    'first': {'ru': user_data['first_name']},
                    'last': {'ru': user_data['last_name']},
                }
                uid = int(user_data['uid'])
                try:
                    birthday = ensure_date(user_data['birth_date']) if user_data['birth_date'] else None
                except ValueError:
                    birthday = None

                prepared_data = {
                    'birthday': birthday,
                    'name': name,
                    'id': uid,
                    'nickname': login[:login.index('@')],
                    'email': login,
                    'gender': gender_map.get(user_data['sex']) if user_data['sex'] else None,
                    'org_id': org_id,
                    'department_id': 1,
                }

                log.info('Creating {} user'.format(prepared_data['nickname']))
                meta_user_data = {
                    'id': prepared_data['id'],
                    'org_id': org_id,
                }
                existing_meta_user = UserMetaModel(meta).filter(**meta_user_data).one()
                if not existing_meta_user:
                    UserMetaModel(meta).create(**meta_user_data)

                user_model = UserModel(main)
                existing_user = user_model.filter(id=uid, org_id=org_id).one()

                if existing_user:
                    (
                        user_model
                        .filter(id=uid, org_id=org_id)
                        .update(**prepared_data)
                    )
                else:
                    user_model.create(
                        calc_disk_usage=False,
                        update_groups_leafs_cache=False,
                        **prepared_data
                    )

                # Получим объект из базы, чтобы поля были одинаковыми независимот от того,
                # обновляли мы запись или создавали с нуля.
                user = user_model.filter(id=uid, org_id=org_id).one()

                admin_uid = get_organization_admin_uid(
                    main,
                    org_id
                )
                action_user_add(
                    main,
                    org_id=org_id,
                    author_id=admin_uid,
                    object_value=user
                )


class UserByNicknameView(UserView):
    methods = ['get']

    @internal
    @uses_schema_for_get(USERS_GET_SCHEMA)
    @scopes_required([scope.read_users, scope.write_users])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get_8(self, meta_connection, main_connection, nickname):
        """
        Ручка используется прослойкой для апи ПДД, чтобы получать id пользователей по nickname.
        Нужно для порталов, потому что у нас может не быть новых пользователей,
        а в паспорте они есть и мы их досоздадим.
        Используется только для порталов.
        """
        user_model = UserModel(main_connection)
        org_id = g.org_id

        fields = split_by_comma(request.args.to_dict().get('fields'))
        real_fields = fields[:]
        real_fields.extend(['org_id', 'nickname'])

        if 'services' in fields:
            real_fields.extend(['services.slug', 'services.name'])
        master_domain = False
        user = None
        try:
            master_domain = DomainModel(main_connection).get_master(org_id)
        except MasterDomainNotFound:
            pass
        if master_domain:
            user_email = build_email(
                    main_connection,
                    nickname,
                    org_id=org_id,
                )
            user = user_model.filter(
                email=user_email,
                org_id=org_id,
            ).fields(*real_fields).one()
        if not user:
            user = user_model.filter(
                nickname=nickname,
                org_id=org_id,
            ).fields(*real_fields).one()
        if not user:
            login = build_email(main_connection, nickname, org_id)
            user_data = get_user_data_from_blackbox_by_login(login, attributes=['1017'])
            if not user_data:
                return json_error_not_found()
            user = user_model.filter(id=user_data['uid'], org_id=org_id).fields(*real_fields).one()
        if not user:
            org_type = OrganizationModel(main_connection).get(org_id, fields=['organization_type'])['organization_type']
            if org_type != 'portal' or not user_data or user_data['is_maillist']:
                # на всякий случай еще защитимся от создания рассылок
                return json_error_not_found()
            create_user_from_bb_info(user_data, org_id)
            user = user_model.filter(nickname=nickname, org_id=org_id).fields(*real_fields).one()

        prepared_user = prepare_user_with_fields(
            meta_connection,
            main_connection,
            user,
            fields=fields,
            api_version=request.api_version,
        )
        return json_response(prepared_user)


class UserOuterDeputyAdminListView(View):
    methods = ['get', 'post']

    @scopes_required([scope.read_users])
    @no_permission_required
    @requires(org_id=True, user=False)
    def get(self, meta_connection, main_connection):
        """
        Получить список внешних заместителей администраторов.
        ---
        tags:
          - Сотрудники
        responses:
          200:
            description: Список внешних заместителей админа
        """
        org_id = g.org_id
        deputy_admins = only_ids(
            UserMetaModel(meta_connection).get_outer_deputy_admins(org_id=org_id, fields=['id'])
        )
        deputy_logins = []
        for uid in deputy_admins:
            u_info = get_user_data_from_blackbox_by_uid(uid)
            deputy_logins.append(u_info['login'].replace('-', '.'))

        return json_response({
            'deputies': deputy_logins
        })

    @uses_schema(USER_CREATE_OUTER_DEPUTY)
    @scopes_required([scope.write_users])
    @permission_required([user_permissions.make_admin])
    @requires(org_id=True, user=True)
    def post(self, meta_connection, main_connection, data):
        """
        Добавить внешнего заместителя админа.
        ---
        tags:
          - Сотрудники
        parameters:
          - in: body
            name: body
        responses:
          201:
            description: Заместитель добавлен
          422:
            description: Какая-то ошибка валидации
          403:
            description: Нет прав, либо передан внутренний логин
          404:
            description: Аккаунт с переданным логином не найден.

        """
        user_id = get_user_id_from_passport_by_login(data['nickname'])

        if not user_id:
            return json_error_not_found()

        if not is_outer_uid(user_id):
            # защитимся от добавления внутренних пользователей через эту ручку
            return json_error_forbidden()

        org_id = g.org_id
        if user_id == get_organization_admin_uid(main_connection, org_id):
            # админу нельзя становиться заместителем
            return json_error_forbidden()

        if not bool(UserMetaModel(meta_connection).get_outer_deputy_admins(
            org_id=org_id,
            uid=user_id
        )):
            UserMetaModel(meta_connection).create(
                id=user_id,
                org_id=g.org_id,
                user_type='deputy_admin'
            )
            action_organization_outer_deputy_add(
                main_connection,
                org_id=org_id,
                author_id=g.user.passport_uid,
                object_value={'id': org_id},
                content={'new_deputy': user_id},
            )
        return json_response({}, status_code=201)


class UserOuterDeputyAdminDetailView(View):

    @scopes_required([scope.write_users])
    @permission_required([user_permissions.make_admin])
    @requires(org_id=True, user=True)
    def delete(self, meta_connection, main_connection, nickname):
        """
        Удалить внешнего зама.
        ---
        tags:
          - Сотрудники
        parameters:
          - in: path
            name: nickname
            required: true
            type: string
        responses:
          204:
            description: Заместитель удален
          404:
            description: Заместитель не найден.
          403:
            description: Нет права на удаление.
        """
        user_id = get_user_id_from_passport_by_login(nickname)
        if not user_id:
            return json_error_not_found()

        org_id = g.org_id
        # проверим, что переданный пользователь действительно внешний зам
        is_outer_deputy = bool(UserMetaModel(meta_connection).get_outer_deputy_admins(
            org_id=org_id,
            uid=user_id
        ))
        if not is_outer_deputy:
            return json_error_forbidden()

        UserMetaModel(meta_connection).delete(
            filter_data={
                'org_id': org_id,
                'id': user_id,
                'is_outer': True
            }
        )
        action_organization_outer_deputy_delete(
            main_connection,
            org_id=org_id,
            author_id=g.user.passport_uid,
            object_value={'id': org_id},
            content={'deleted_deputy': user_id},
        )
        return json_response({}, status_code=204)


class UserTypeView(View):
    @internal
    @scopes_required([scope.read_users])
    @no_permission_required
    @requires(org_id=True, user=True)
    def get(self, meta_connection, main_connection):
        """
        Возвращает признак внутренний/внешний.

        Пример ответа:

            {
                "internal": True
            }
        ---
        tags:
          - Сотрудники
        responses:
          200:
            description: Тип пользователя
          403:
            description: Пользователь не состоит в этой организации
          404:
            description: Пользователь не найден.
        """
        user = get_object_or_404(
            UserMetaModel(meta_connection),
            user_id=g.user.passport_uid,
            org_id=g.org_id,
            is_outer=Ignore,
        )
        is_internal = False
        if user['user_type'] == 'inner_user':
            # пользователь может быть удален
            user_model = UserModel(main_connection)
            dismissed_user = bool(user_model.filter(id=g.user.passport_uid, org_id=g.org_id, is_dismissed=True).one())
            is_internal = not dismissed_user

        return json_response({'internal': is_internal})


def check_external_id_is_free(
        main_connection,
        org_id,
        external_id,
):
    """
    Проверяет, что external_id не занят существующим и действительным сотрудником.
    """

    found = UserModel(main_connection) \
        .filter(
            org_id=org_id,
            external_id=external_id,
        ) \
        .count()

    if found:
        raise ExternalIDAlreadyUsed()


class OuterAdminMigrationView(View):
    methods = ['get']

    @no_scopes
    @no_permission_required
    @no_auth
    def get(self, meta_connection, main_connection):
        """
        Добавляет/удаляет внешнего админа по токену

        ---
        parameters:
          - in: query
            name: token
            required: True
            type: string
        responses:
          200:
            description: Действие было выполнено успешно
        """
        token = request.args.get('token', None)
        if not token:
            data = {
                'status': 'error',
                'text': 'No token found'
            }
        else:
            try:
                org_id, user_id, action = self._get_token_data(meta_connection, token)
            except TypeError:
                data = {
                    'status': 'error',
                    'text': 'Token has been already used'
                }
            else:
                user = UserMetaModel(meta_connection).get(user_id=user_id, org_id=org_id, is_outer=Ignore)
                if not user or user['user_type'] != 'outer_admin':
                    data = {
                        'status': 'error',
                        'text': 'Incorrect user'
                    }
                else:
                    with get_meta_connection(for_write=True) as meta_connection:
                        shard = get_shard(meta_connection, org_id)
                        with get_main_connection(for_write=True, shard=shard) as main_connection:
                            if action == 'added':
                                self._add_outer_admin(
                                    meta_connection, main_connection,
                                    org_id, user_id, shard,
                                )
                            else:
                                UserMetaModel(meta_connection).delete(
                                    filter_data=dict(id=user_id, org_id=org_id)
                                )
                                OrganizationRevisionCounterModel(main_connection).increment_revisions_for_user(
                                    meta_connection,
                                    user_id,
                                )

                            OrganizationRevisionCounterModel(main_connection).increment_revision(org_id)
                            self._delete_token_after_use(meta_connection, org_id)
                    data = {
                        'status': 'ok',
                        'text': f'Admin was successfully {action}',
                    }
        return json_response(data)

    def _delete_token_after_use(self, meta_connection, org_id):
        query = 'delete from outer_admin_migration where org_id = %(org_id)s'
        meta_connection.execute(query, org_id=org_id)

    def _add_outer_admin(self, meta_connection, main_connection, org_id, user_id, shard):
        from intranet.yandex_directory.src.yandex_directory.core.tasks import SyncExternalIDS
        from intranet.yandex_directory.src.yandex_directory.core.utils import get_user_info_from_blackbox

        (login, first, last, gender, birthday, email, cloud_uid) = get_user_info_from_blackbox(user_id)

        log.info('adding user to org', user_id, org_id, shard)

        UserMetaModel(meta_connection) \
            .update(
            filter_data={'id': user_id, 'org_id': org_id},
            update_data={'is_outer': False, 'user_type': 'inner_user'},
        )
        exist = UserModel(main_connection).filter(id=user_id, org_id=org_id, is_dismissed=Ignore).count()
        if not exist:
            UserModel(main_connection) \
                .create(
                id=user_id,
                nickname=login,
                name={'first': first or login, 'last': last or login},
                email=email,
                gender=gender,
                org_id=org_id,
                department_id=1,
                birthday=birthday,
            )
        else:
            UserModel(main_connection).update(
                filter_data={'id': user_id, 'org_id': org_id, 'is_dismissed': Ignore},
                update_data={'is_dismissed': False}
            )

        user = UserModel(main_connection).get(
            user_id=user_id,
            org_id=org_id,
            fields=[
                '*',
                'department.*',
            ]
        )
        author_id = user_id

        try:
            action_user_add(
                main_connection,
                org_id=org_id,
                author_id=author_id,
                object_value=user
            )
        except Exception as exc:
            log.warning('Got exc while sending action', exc)

        UserModel(main_connection).make_admin_of_organization(org_id, user_id)
        action_security_user_grant_organization_admin(
            main_connection,
            org_id=org_id,
            author_id=user_id,
            object_value={},
        )

        SyncExternalIDS(main_connection).do(
            org_id=org_id,
            user_id=user_id,
        )

    def _get_token_data(self, meta_connection, token):
        query = '''
        select org_id, user_id, action
        from outer_admin_migration
        where token = %(token)s
        '''
        org_id, user_id, action = meta_connection.execute(query, token=token).fetchone()
        return org_id, user_id, action
